/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package test;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.URL;
import java.util.*;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.plaf.FontUIResource;

import org.jdesktop.swingx.*;
import org.jvnet.lafwidget.LafWidget;
import org.jvnet.lafwidget.tabbed.DefaultTabPreviewPainter;
import org.jvnet.lafwidget.utils.LafConstants.TabOverviewKind;
import org.jvnet.substance.*;
import org.jvnet.substance.button.*;
import org.jvnet.substance.fonts.FontSet;
import org.jvnet.substance.painter.*;
import org.jvnet.substance.painter.decoration.DecorationAreaType;
import org.jvnet.substance.tabbed.*;
import org.jvnet.substance.theme.*;
import org.jvnet.substance.utils.SubstanceConstants;
import org.jvnet.substance.utils.SubstanceConstants.Side;
import org.jvnet.substance.utils.SubstanceConstants.TabCloseKind;
import org.jvnet.substance.watermark.*;

import test.check.*;
import test.check.statusbar.FontSizePanel;

public class Check extends JFrame {
	private JTabbedPane jtp;

	private MyMainTabPreviewPainter mainTabPreviewPainter;

	private static class WrapperFontSet implements FontSet {
		private int extra;

		private FontSet delegate;

		public WrapperFontSet(FontSet delegate, int extra) {
			super();
			this.delegate = delegate;
			this.extra = extra;
		}

		private FontUIResource getWrappedFont(FontUIResource systemFont) {
			return new FontUIResource(systemFont.getFontName(), systemFont
					.getStyle(), systemFont.getSize() + extra);
		}

		public FontUIResource getControlFont() {
			return getWrappedFont(delegate.getControlFont());
		}

		public FontUIResource getMenuFont() {
			return getWrappedFont(delegate.getMenuFont());
		}

		public FontUIResource getMessageFont() {
			return getWrappedFont(delegate.getMessageFont());
		}

		public FontUIResource getSmallFont() {
			return getWrappedFont(delegate.getSmallFont());
		}

		public FontUIResource getTitleFont() {
			return getWrappedFont(delegate.getTitleFont());
		}

		public FontUIResource getWindowTitleFont() {
			// FontUIResource f = this.getWrappedFont(this.delegate
			// .getWindowTitleFont());
			// return new FontUIResource(f.deriveFont(Font.BOLD, f.getSize() +
			// 1));
			return getWrappedFont(delegate.getWindowTitleFont());
		}
	}

	private class MyThemeChangeListener implements ThemeChangeListener {
		public void themeChanged() {
			if (UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel) {
				out("Substance theme is '"
						+ SubstanceLookAndFeel.getCurrentThemeName() + "'");

				ClassLoader cl = Thread.currentThread().getContextClassLoader();
				setIconImage(SubstanceImageCreator
						.getThemeImage(
								null,
								new ImageIcon(
										cl
												.getResource("test/resource/image-x-generic.png")),
								SubstanceLookAndFeel.getTheme(), false));

			}
		}
	}

	private class MyWatermarkChangeListener implements WatermarkChangeListener {
		public void watermarkChanged() {
			if (UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel) {
				out("Substance watermark is '"
						+ SubstanceLookAndFeel.getCurrentWatermarkName() + "'");
			}
		}
	}

	private class MyButtonShaperChangeListener implements
			ButtonShaperChangeListener {
		public void buttonShaperChanged() {
			if (UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel) {
				out("Substance button shaper is '"
						+ SubstanceLookAndFeel.getCurrentButtonShaperName()
						+ "'");
			}
		}
	}

	private class MyGradientPainterChangeListener implements
			GradientPainterChangeListener {
		public void gradientPainterChanged() {
			if (UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel) {
				out("Substance gradient painter is '"
						+ SubstanceLookAndFeel.getCurrentGradientPainterName()
						+ "'");
			}
		}
	}

	private boolean toUseThemeObjs;

	private boolean toUseWatermarksObjs;

	private boolean toUseButtonShapersObjs;

	private boolean toUseGradientPaintersObjs;

	private JXTaskPaneContainer taskPaneContainer;

	private JXTaskPane currentSpecificTaskPane;

	private JXTaskPane mainTaskPane;

	private JToolBar toolbar;

	public Check(boolean toRegisterListeners) {
		super(
				"Substance test with very very very very very very very very very very very very very very long title");

		ClassLoader cl = Thread.currentThread().getContextClassLoader();
		setIconImage(SubstanceImageCreator.getThemeImage(null, new ImageIcon(cl
				.getResource("test/resource/image-x-generic.png")),
				SubstanceLookAndFeel.getTheme(), false));
		setLayout(new BorderLayout());

		jtp = new JTabbedPane();
		try {
			mainTabPreviewPainter = new MyMainTabPreviewPainter();
			jtp.putClientProperty(LafWidget.TABBED_PANE_PREVIEW_PAINTER,
					mainTabPreviewPainter);
		} catch (Throwable e) {
		}
		jtp.getModel().addChangeListener(new TabSwitchListener());

		// TransitionLayoutManager.getInstance().track(this.jtp, true);
		// UIManager.addPropertyChangeListener(new PropertyChangeListener() {
		// public void propertyChange(PropertyChangeEvent evt) {
		// if ("lookAndFeel".equals(evt.getPropertyName())) {
		// LookAndFeel newLaf = (LookAndFeel) evt.getNewValue();
		// if (newLaf.getID().contains("Substance")) {
		// if (!(jtp.getLayout() instanceof TransitionLayout)) {
		// TransitionLayoutManager.getInstance().track(jtp,
		// true);
		// }
		// } else {
		// TransitionLayoutManager.getInstance().untrack(jtp);
		// }
		// }
		// }
		// });

		final JXPanel jxPanel = new JXPanel(new BorderLayout());
		toolbar = getToolbar("", 22, true);
		jxPanel.add(toolbar, BorderLayout.NORTH);

		JXStatusBar statusBar = getStatusBar(jxPanel, jtp);
		this.add(statusBar, BorderLayout.SOUTH);

		// JPanel buttons = new JPanel(new FlowLayout());

		taskPaneContainer = new JXTaskPaneContainer() {
			@Override
			public boolean getScrollableTracksViewportWidth() {
				return false;
			}
		};
		taskPaneContainer.setScrollableTracksViewportHeight(false);
		taskPaneContainer.setScrollableTracksViewportWidth(false);
		taskPaneContainer.putClientProperty(
				SubstanceLookAndFeel.BUTTON_SHAPER_PROPERTY,
				new ClassicButtonShaper());
		// this.taskPaneContainer.setOpaque(false);

		mainTaskPane = new JXTaskPane();
		mainTaskPane.setLayout(new BorderLayout());
		JPanel mainControlPanel = ControlPanelFactory.getMainControlPanel(this,
				jtp, mainTabPreviewPainter, toolbar);
		// mainControlPanel.setOpaque(false);
		mainTaskPane.add(mainControlPanel, BorderLayout.CENTER);
		mainTaskPane.setTitle("General settings");
		mainTaskPane.setIcon(getIcon("JFrameColor16"));
		mainTaskPane.setExpanded(false);
		taskPaneContainer.add(mainTaskPane);

		JPanel dialogControlPanel = ControlPanelFactory
				.getDialogControlPanel(this);
		JXTaskPane dialogTaskPane = new JXTaskPane();
		dialogTaskPane.setLayout(new BorderLayout());
		dialogTaskPane.add(dialogControlPanel, BorderLayout.CENTER);
		dialogTaskPane.setTitle("Dialogs");
		dialogTaskPane.setIcon(getIcon("JDialogColor16"));
		dialogTaskPane.setExpanded(false);
		// dialogTaskPane.setOpaque(false);
		taskPaneContainer.add(dialogTaskPane);

		currentSpecificTaskPane = null;

		final JScrollPane scrollPane = new JScrollPane(taskPaneContainer,
				JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
				JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		SubstanceLookAndFeel.setDecorationType(scrollPane,
				DecorationAreaType.GENERAL);
		scrollPane.setOpaque(false);
		scrollPane.getViewport().setOpaque(false);

		JPanel mainPanel = new JPanel();
		mainPanel.add(scrollPane);
		mainPanel.add(jtp);
		mainPanel.setLayout(new LayoutManager() {
			public void addLayoutComponent(String name, Component comp) {
			}

			public Dimension minimumLayoutSize(Container parent) {
				Dimension min1 = scrollPane.getMinimumSize();
				Dimension min2 = jtp.getMinimumSize();
				return new Dimension(min1.width + min2.width, min1.height
						+ min2.height);
			}

			public void layoutContainer(Container parent) {
				// give 30% width to task pane container and
				// 70% width to the tabbed pane with controls.
				int width = parent.getWidth();
				int height = parent.getHeight();
				scrollPane.setBounds(0, 0, (int) (0.3 * width), height);
				jtp.setBounds((int) (0.3 * width), 0, width
						- (int) (0.3 * width), height);
			}

			public Dimension preferredLayoutSize(Container parent) {
				Dimension pref1 = scrollPane.getPreferredSize();
				Dimension pref2 = jtp.getPreferredSize();
				return new Dimension(pref1.width + pref2.width, pref1.height
						+ pref2.height);
			}

			public void removeLayoutComponent(Component comp) {
			}
		});
		jxPanel.add(mainPanel, BorderLayout.CENTER);

		this.add(jxPanel, BorderLayout.CENTER);

		setPreferredSize(new Dimension(400, 400));
		this.setSize(getPreferredSize());
		setMinimumSize(getPreferredSize());

		ButtonsPanel buttonsPanel = new ButtonsPanel();
		jtp.addTab("Buttons", getIcon("JButtonColor16"), buttonsPanel);
		getRootPane().setDefaultButton(buttonsPanel.defaultButton);

		jtp.addTab("Combo box", getIcon("JComboBoxColor16"), new CombosPanel());

		jtp.addTab("Scroll pane", getIcon("JScrollPaneColor16"),
				new ScrollPanel());

		TabCloseCallback closeCallback = new TabCloseCallback() {
			public TabCloseKind onAreaClick(JTabbedPane tabbedPane,
					int tabIndex, MouseEvent mouseEvent) {
				if (mouseEvent.getButton() != MouseEvent.BUTTON3)
					return TabCloseKind.NONE;
				if (mouseEvent.isShiftDown()) {
					return TabCloseKind.ALL;
				}
				return TabCloseKind.THIS;
			}

			public TabCloseKind onCloseButtonClick(JTabbedPane tabbedPane,
					int tabIndex, MouseEvent mouseEvent) {
				if (mouseEvent.isAltDown()) {
					return TabCloseKind.ALL_BUT_THIS;
				}
				if (mouseEvent.isShiftDown()) {
					return TabCloseKind.ALL;
				}
				return TabCloseKind.THIS;
			}

			public String getAreaTooltip(JTabbedPane tabbedPane, int tabIndex) {
				return null;
			}

			public String getCloseButtonTooltip(JTabbedPane tabbedPane,
					int tabIndex) {
				StringBuffer result = new StringBuffer();
				result.append("<html><body>");
				result.append("Mouse click closes <b>"
						+ tabbedPane.getTitleAt(tabIndex) + "</b> tab");
				result
						.append("<br><b>Alt</b>-Mouse click closes all tabs but <b>"
								+ tabbedPane.getTitleAt(tabIndex) + "</b> tab");
				result.append("<br><b>Shift</b>-Mouse click closes all tabs");
				result.append("</body></html>");
				return result.toString();
			}
		};

		try {
			TabPanel tp = new TabPanel();
			tp.jtp.putClientProperty(
					SubstanceLookAndFeel.TABBED_PANE_CLOSE_CALLBACK,
					closeCallback);
			jtp.addTab("Tabbed pane", getIcon("JTabbedPaneColor16"), tp);

			TabPanel tp2 = new TabPanel();
			tp2.jtp
					.putClientProperty(
							SubstanceLookAndFeel.TABBED_PANE_VERTICAL_ORIENTATION,
							true);

			tp2.jtp.putClientProperty(
					SubstanceLookAndFeel.TABBED_PANE_CLOSE_CALLBACK,
					closeCallback);
			tp2.jtp.setTabPlacement(JTabbedPane.RIGHT);
			SubstanceLookAndFeel.registerTabCloseChangeListener(tp2.jtp,
					new MultipleTabCloseListener() {
						public void tabsClosed(JTabbedPane tabbedPane,
								Set<Component> tabComponents) {
							out("Closed " + tabComponents.size()
									+ " vertical tabs - specific");
						}

						public void tabsClosing(JTabbedPane tabbedPane,
								Set<Component> tabComponents) {
							out("Closing " + tabComponents.size()
									+ " vertical tabs - specific");
						}
					});
			SubstanceLookAndFeel.registerTabCloseChangeListener(tp2.jtp,
					new TabCloseListener() {
						public void tabClosed(JTabbedPane tabbedPane,
								Component tabComponent) {
							out("Closed vertical tab - specific");
						}

						public void tabClosing(JTabbedPane tabbedPane,
								Component tabComponent) {
							out("Closing vertical tab - specific");
						}
					});

			jtp.addTab("Tabbed pane vertical", tp2);
		} catch (NoClassDefFoundError ncdfe) {
		}

		jtp.addTab("Split", new SplitPanel());

		jtp.addTab("Desktop", getIcon("JDesktopPaneColor16"),
				new DesktopPanel());

		jtp.addTab("Text fields", getIcon("JTextPaneColor16"),
				new TextFieldsPanel());

		jtp.addTab("Table", getIcon("JTableColor16"), new TablePanel());

		try {
			jtp.addTab("List", getIcon("JListColor16"), new ListPanel());
		} catch (NoClassDefFoundError ncdfe) {
		}

		jtp.addTab("Slider", getIcon("JSliderColor16"), new SliderPanel());

		jtp.addTab("Progress bar", getIcon("JProgressBarColor16"),
				new ProgressBarPanel());

		jtp.addTab("Spinner", getIcon("JSpinnerColor16"), new SpinnerPanel());

		jtp.addTab("Tree", getIcon("JTreeColor16"), new TreePanel());

		jtp.addTab("File tree", getIcon("JTreeColor16"), new FileTreePanel());

		jtp.addTab("Cards", new CardPanel());

		JPanel bigButtonPanel = new JPanel();
		bigButtonPanel.setLayout(new BorderLayout());
		bigButtonPanel.add(new JButton("One big button"));
		jtp.addTab("Big button", bigButtonPanel);

		JPanel verticalButtonPanel = new JPanel();
		verticalButtonPanel.setLayout(new GridLayout(1, 3));
		verticalButtonPanel.add(new JButton("Vert button 1"));
		verticalButtonPanel.add(new JButton("Vert button 2"));
		JPanel smallVerticalButtonPanel = new JPanel();
		smallVerticalButtonPanel.setLayout(new GridLayout(4, 4));
		for (int row = 0; row < 4; row++) {
			for (int col = 0; col < 4; col++) {
				JButton vertButton = new JButton("vert");
				vertButton.setToolTipText("Vertical button " + row + ":" + col);
				smallVerticalButtonPanel.add(vertButton);
			}
		}
		verticalButtonPanel.add(smallVerticalButtonPanel);
		jtp.addTab("Vertical buttons", verticalButtonPanel);

		jtp.addTab("Colored", new ColoredControlsPanel());

		jtp.addTab("Colorized", new ColorizedControlsPanel());

		JPanel buttonStripPanel = new JPanel();
		buttonStripPanel.setLayout(new FlowLayout());

		jtp.addTab("Sizes", new SizesPanel());

		jtp.addTab("Alignment", AlignmentPanel.getAlignmentPanel());

		if (UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel) {
			jtp.addTab("Mixed Buttons", getIcon("JColorChooserColor16"),
					new MixedButtonsPanel());
		}

		// sample menu bar
		JMenuBar jmb = new JMenuBar();

		if (UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel) {

			jmb.add(SampleMenuFactory.getThemeMenu());
			jmb.add(SampleMenuFactory.getSkinMenu());

			JMenu jmWatermark = new JMenu("Watermark");
			JCheckBoxMenuItem jmiWatermarkObjs = new JCheckBoxMenuItem(
					"Use objects");
			jmiWatermarkObjs.setSelected(true);
			jmiWatermarkObjs.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					JCheckBoxMenuItem src = (JCheckBoxMenuItem) e.getSource();
					toUseWatermarksObjs = src.isSelected();
				}
			});
			jmWatermark.add(jmiWatermarkObjs);
			toUseWatermarksObjs = jmiWatermarkObjs.isSelected();

			Map<String, WatermarkInfo> allWatermarks = SubstanceLookAndFeel
					.getAllWatermarks();
			for (Map.Entry<String, WatermarkInfo> watermarkEntry : allWatermarks
					.entrySet()) {
				String watermarkName = watermarkEntry.getKey();
				final String watermarkClassName = watermarkEntry.getValue()
						.getClassName();

				JMenuItem jmiWatermark = new JMenuItem(watermarkName);
				try {
					Class watermarkClass = Class.forName(watermarkClassName);
					SubstanceWatermark watermark = (SubstanceWatermark) watermarkClass
							.newInstance();
					jmiWatermark.setIcon(SubstanceImageCreator
							.getWatermarkIcon(watermark));
				} catch (Exception exc) {
					continue;
				}
				jmiWatermark.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						SwingUtilities.invokeLater(new Runnable() {
							public void run() {
								if (toUseWatermarksObjs) {
									try {
										SubstanceLookAndFeel
												.setCurrentWatermark((SubstanceWatermark) Class
														.forName(
																watermarkClassName)
														.newInstance());
									} catch (Exception exc) {
										exc.printStackTrace();
									}
								} else {
									SubstanceLookAndFeel
											.setCurrentWatermark(watermarkClassName);
								}
								SwingUtilities
										.updateComponentTreeUI(Check.this);
							}
						});
					}
				});
				if (watermarkName.equals(SubstanceLookAndFeel
						.getCurrentWatermarkName())) {
					jmiWatermark.setSelected(true);
				}
				jmWatermark.add(jmiWatermark);
			}

			jmWatermark.addSeparator();

			JMenuItem jmiCoffeeBeansWatermark = new JMenuItem("Coffee Beans");
			jmiCoffeeBeansWatermark.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							try {
								SubstanceLookAndFeel
										.setCurrentWatermark(new SubstanceCoffeeBeansWatermark());
							} catch (Exception exc) {
								exc.printStackTrace();
							}
							SwingUtilities.updateComponentTreeUI(Check.this);
						}
					});
				}
			});
			if ("Coffee Beans".equals(SubstanceLookAndFeel
					.getCurrentWatermarkName())) {
				jmiCoffeeBeansWatermark.setSelected(true);
			}
			jmWatermark.add(jmiCoffeeBeansWatermark);

			JMenuItem jmiVerticalGradientWatermark = new JMenuItem(
					"Vertical Gradient");
			jmiVerticalGradientWatermark
					.addActionListener(new ActionListener() {
						public void actionPerformed(ActionEvent e) {
							SwingUtilities.invokeLater(new Runnable() {
								public void run() {
									try {
										SubstanceLookAndFeel
												.setCurrentWatermark(new SubstanceVerticalGradientWatermark());
									} catch (Exception exc) {
										exc.printStackTrace();
									}
									SwingUtilities
											.updateComponentTreeUI(Check.this);
								}
							});
						}
					});
			if ("Vertical Gradient".equals(SubstanceLookAndFeel
					.getCurrentWatermarkName())) {
				jmiVerticalGradientWatermark.setSelected(true);
			}
			jmWatermark.add(jmiVerticalGradientWatermark);

			JMenuItem jmiDiagonalGradientWatermark = new JMenuItem(
					"Diagonal Gradient");
			jmiDiagonalGradientWatermark
					.addActionListener(new ActionListener() {
						public void actionPerformed(ActionEvent e) {
							SwingUtilities.invokeLater(new Runnable() {
								public void run() {
									try {
										SubstanceLookAndFeel
												.setCurrentWatermark(new SubstanceDiagonalGradientWatermark());
									} catch (Exception exc) {
										exc.printStackTrace();
									}
									SwingUtilities
											.updateComponentTreeUI(Check.this);
								}
							});
						}
					});
			if ("Diagonal Gradient".equals(SubstanceLookAndFeel
					.getCurrentWatermarkName())) {
				jmiDiagonalGradientWatermark.setSelected(true);
			}
			jmWatermark.add(jmiDiagonalGradientWatermark);

			jmb.add(jmWatermark);

			JMenu jmButtonShaper = new JMenu("Shapers");
			JCheckBoxMenuItem jmiButtonShaperObjs = new JCheckBoxMenuItem(
					"Use objects");
			jmiButtonShaperObjs.setSelected(true);
			jmiButtonShaperObjs.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					JCheckBoxMenuItem src = (JCheckBoxMenuItem) e.getSource();
					toUseButtonShapersObjs = src.isSelected();
				}
			});
			jmButtonShaper.add(jmiButtonShaperObjs);
			toUseButtonShapersObjs = jmiButtonShaperObjs.isSelected();

			ButtonGroup bgButtonShaper = new ButtonGroup();
			Map<String, ButtonShaperInfo> allButtonShapers = SubstanceLookAndFeel
					.getAllButtonShapers();
			for (Map.Entry<String, ButtonShaperInfo> ButtonShaperEntry : allButtonShapers
					.entrySet()) {
				String buttonShaperName = ButtonShaperEntry.getKey();
				final String buttonShaperClassName = ButtonShaperEntry
						.getValue().getClassName();

				JRadioButtonMenuItem jmiButtonShaper = new JRadioButtonMenuItem(
						buttonShaperName);
				jmiButtonShaper.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						SwingUtilities.invokeLater(new Runnable() {
							public void run() {
								if (toUseButtonShapersObjs) {
									try {
										SubstanceLookAndFeel
												.setCurrentButtonShaper((SubstanceButtonShaper) Class
														.forName(
																buttonShaperClassName)
														.newInstance());
									} catch (Exception exc) {
										exc.printStackTrace();
									}
								} else {
									SubstanceLookAndFeel
											.setCurrentButtonShaper(buttonShaperClassName);
								}
								SwingUtilities
										.updateComponentTreeUI(Check.this);
							}
						});
					}
				});
				if (buttonShaperName.equals(SubstanceLookAndFeel
						.getCurrentButtonShaperName())) {
					jmiButtonShaper.setSelected(true);
				}
				bgButtonShaper.add(jmiButtonShaper);
				jmButtonShaper.add(jmiButtonShaper);
			}

			jmb.add(jmButtonShaper);

			JMenu jmGradientPainter = new JMenu("Painters");
			JCheckBoxMenuItem jmiGradientPainterObjs = new JCheckBoxMenuItem(
					"Use objects");
			jmiGradientPainterObjs.setSelected(true);
			jmiGradientPainterObjs.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					JCheckBoxMenuItem src = (JCheckBoxMenuItem) e.getSource();
					toUseGradientPaintersObjs = src.isSelected();
				}
			});
			jmGradientPainter.add(jmiGradientPainterObjs);
			toUseGradientPaintersObjs = jmiGradientPainterObjs.isSelected();

			ButtonGroup bgGradientPainter = new ButtonGroup();
			Map<String, GradientPainterInfo> allGradientPainters = SubstanceLookAndFeel
					.getAllGradientPainters();
			for (Map.Entry<String, GradientPainterInfo> GradientPainterEntry : allGradientPainters
					.entrySet()) {
				String gradientPainterName = GradientPainterEntry.getKey();
				final String gradientPainterClassName = GradientPainterEntry
						.getValue().getClassName();

				JRadioButtonMenuItem jmiGradientPainter = new JRadioButtonMenuItem(
						gradientPainterName);
				jmiGradientPainter.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						SwingUtilities.invokeLater(new Runnable() {
							public void run() {
								if (toUseGradientPaintersObjs) {
									try {
										SubstanceLookAndFeel
												.setCurrentGradientPainter((SubstanceGradientPainter) Class
														.forName(
																gradientPainterClassName)
														.newInstance());
									} catch (Exception exc) {
										exc.printStackTrace();
									}
								} else {
									SubstanceLookAndFeel
											.setCurrentGradientPainter(gradientPainterClassName);
								}
								SwingUtilities
										.updateComponentTreeUI(Check.this);
							}
						});
					}
				});
				if (gradientPainterName.equals(SubstanceLookAndFeel
						.getCurrentGradientPainterName())) {
					jmiGradientPainter.setSelected(true);
				}
				bgGradientPainter.add(jmiGradientPainter);
				jmGradientPainter.add(jmiGradientPainter);
			}

			jmb.add(jmGradientPainter);
		}
		JMenu coloredMenu = new JMenu("Colors");
		coloredMenu.setMnemonic('0');
		final JMenuItem coloredMI = new JMenuItem("Italic red");
		// coloredMI.addPropertyChangeListener(new PropertyChangeListener() {
		// public void propertyChange(PropertyChangeEvent evt) {
		// if ("font".equals(evt.getPropertyName()))
		// System.out
		// .println("Font changed to " + coloredMI.getFont());
		// }
		// });
		coloredMI.setFont(coloredMI.getFont().deriveFont(Font.ITALIC));
		coloredMI.setForeground(Color.red);
		coloredMenu.add(coloredMI);
		JRadioButtonMenuItem coloredRBMI = new JRadioButtonMenuItem(
				"Bold green");
		coloredRBMI.setFont(coloredRBMI.getFont().deriveFont(Font.BOLD));
		coloredRBMI.setForeground(Color.green);
		coloredMenu.add(coloredRBMI);
		JCheckBoxMenuItem coloredCBMI = new JCheckBoxMenuItem("Big blue");
		coloredCBMI.setFont(coloredCBMI.getFont().deriveFont(32f));
		coloredCBMI.setForeground(Color.blue);
		coloredMenu.add(coloredCBMI);
		JMenu coloredM = new JMenu("Always big magenta");
		coloredM.setForeground(Color.magenta);
		coloredM.setFont(coloredM.getFont().deriveFont(24f));
		coloredMenu.add(coloredM);
		jmb.add(coloredMenu);

		JMenu testMenu = SampleMenuFactory.getTestMenu();
		jmb.add(testMenu);

		JMenu jm4 = new JMenu("Disabled");
		jm4.add(new JMenuItem("dummy4"));
		jm4.setMnemonic('4');
		jmb.add(jm4);
		jm4.setEnabled(false);

		// if (UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel) {
		// JMenu fontSizeMenu = new JMenu("Font size");
		// JMenuItem sizeM2 = new JMenuItem("Minus 2 points");
		// sizeM2.addActionListener(new FontPolicyChanger(this, -2));
		// fontSizeMenu.add(sizeM2);
		//
		// JMenuItem sizeM1 = new JMenuItem("Minus 1 point");
		// sizeM1.addActionListener(new FontPolicyChanger(this, -1));
		// fontSizeMenu.add(sizeM1);
		//
		// JMenuItem sizeB = new JMenuItem("Base");
		// sizeB.addActionListener(new FontPolicyChanger(this, 0));
		// fontSizeMenu.add(sizeB);
		//
		// JMenuItem sizeP1 = new JMenuItem("Plus 1 point");
		// sizeP1.addActionListener(new FontPolicyChanger(this, 1));
		// fontSizeMenu.add(sizeP1);
		//
		// JMenuItem sizeP2 = new JMenuItem("Plus 2 points");
		// sizeP2.addActionListener(new FontPolicyChanger(this, 2));
		// fontSizeMenu.add(sizeP2);
		//
		// jmb.add(fontSizeMenu);
		// }

		// LAF changing
		jmb.add(SampleMenuFactory.getLookAndFeelMenu(this));
		setJMenuBar(jmb);

		// sample flat toolbar (does nothing)
		// JToolBar toolBar = getToolbar("Flat ");
		// this.add(toolBar, AKDockLayout.NORTH);

		// // sample non-flat toolbar (does nothing)
		// JToolBar toolBar2 = getToolbar("Non-flat ");
		// toolBar2.putClientProperty(SubstanceLookAndFeel.TOOLBAR_BUTTON_FLAT,
		// Boolean.FALSE);
		// this.add(toolBar2, AKDockLayout.NORTH);
		//
		// final JToolBar toolBar3 = getToolbar("Translucent ");
		// toolBar3.putClientProperty(SubstanceLookAndFeel.TOOLBAR_BUTTON_FLAT,
		// Boolean.FALSE);
		// toolBar3.putClientProperty(SubstanceLookAndFeel.BACKGROUND_COMPOSITE,
		// new AlphaControlBackgroundComposite(0.5f));
		// final JSlider translucencySlider = new JSlider(0, 100, 50);
		// translucencySlider.addChangeListener(new ChangeListener() {
		// public void stateChanged(ChangeEvent e) {
		// toolBar3.putClientProperty(
		// SubstanceLookAndFeel.BACKGROUND_COMPOSITE,
		// new AlphaControlBackgroundComposite(translucencySlider
		// .getValue() / 100.f));
		// toolBar3.repaint();
		// }
		// });
		// toolBar3.add(translucencySlider);
		// this.add(toolBar3, AKDockLayout.NORTH);

		if (toRegisterListeners) {
			// register sample listeners
			SubstanceLookAndFeel
					.registerThemeChangeListener(new MyThemeChangeListener());
			SubstanceLookAndFeel
					.registerWatermarkChangeListener(new MyWatermarkChangeListener());
			SubstanceLookAndFeel
					.registerButtonShaperChangeListener(new MyButtonShaperChangeListener());
			SubstanceLookAndFeel
					.registerGradientPainterChangeListener(new MyGradientPainterChangeListener());
		}

		TabCloseCallback closeCallbackMain = new TabCloseCallback() {
			public TabCloseKind onAreaClick(JTabbedPane tabbedPane,
					int tabIndex, MouseEvent mouseEvent) {
				if (mouseEvent.getButton() != MouseEvent.BUTTON2)
					return TabCloseKind.NONE;
				if (mouseEvent.isShiftDown()) {
					return TabCloseKind.ALL;
				}
				return TabCloseKind.THIS;
			}

			public TabCloseKind onCloseButtonClick(JTabbedPane tabbedPane,
					int tabIndex, MouseEvent mouseEvent) {
				if (mouseEvent.isAltDown()) {
					return TabCloseKind.ALL_BUT_THIS;
				}
				if (mouseEvent.isShiftDown()) {
					return TabCloseKind.ALL;
				}
				return TabCloseKind.THIS;
			}

			public String getAreaTooltip(JTabbedPane tabbedPane, int tabIndex) {
				return null;
			}

			public String getCloseButtonTooltip(JTabbedPane tabbedPane,
					int tabIndex) {
				StringBuffer result = new StringBuffer();
				result.append("<html><body>");
				result.append("Mouse click closes <b>"
						+ tabbedPane.getTitleAt(tabIndex) + "</b> tab");
				result
						.append("<br><b>Alt</b>-Mouse click closes all tabs but <b>"
								+ tabbedPane.getTitleAt(tabIndex) + "</b> tab");
				result.append("<br><b>Shift</b>-Mouse click closes all tabs");
				result.append("</body></html>");
				return result.toString();
			}
		};

		jtp.putClientProperty(SubstanceLookAndFeel.TABBED_PANE_CLOSE_CALLBACK,
				closeCallbackMain);
		SubstanceLookAndFeel
				.registerTabCloseChangeListener(new TabCloseListener() {
					public void tabClosed(JTabbedPane tabbedPane,
							Component tabComponent) {
						out("Closed tab");
					}

					public void tabClosing(JTabbedPane tabbedPane,
							Component tabComponent) {
						out("Closing tab");
					}
				});

		SubstanceLookAndFeel.registerTabCloseChangeListener(jtp,
				new VetoableTabCloseListener() {
					public void tabClosed(JTabbedPane tabbedPane,
							Component tabComponent) {
						out("Closed tab - specific");
					}

					public void tabClosing(JTabbedPane tabbedPane,
							Component tabComponent) {
						out("Closing tab - specific");
					}

					public boolean vetoTabClosing(JTabbedPane tabbedPane,
							Component tabComponent) {
						int userCloseAnswer = JOptionPane
								.showConfirmDialog(
										Check.this,
										"Are you sure you want to close '"
												+ tabbedPane
														.getTitleAt(tabbedPane
																.indexOfComponent(tabComponent))
												+ "' tab?", "Confirm dialog",
										JOptionPane.YES_NO_OPTION);
						return (userCloseAnswer == JOptionPane.NO_OPTION);
					}
				});

		SubstanceLookAndFeel.registerTabCloseChangeListener(jtp,
				new VetoableMultipleTabCloseListener() {
					public void tabsClosed(JTabbedPane tabbedPane,
							Set<Component> tabComponents) {
						out("Closed " + tabComponents.size()
								+ " tabs - specific");
					}

					public void tabsClosing(JTabbedPane tabbedPane,
							Set<Component> tabComponents) {
						out("Closing " + tabComponents.size()
								+ " tabs - specific");
					}

					public boolean vetoTabsClosing(JTabbedPane tabbedPane,
							Set<Component> tabComponents) {
						int userCloseAnswer = JOptionPane.showConfirmDialog(
								Check.this, "Are you sure you want to close "
										+ tabComponents.size() + " tabs?",
								"Confirm dialog", JOptionPane.YES_NO_OPTION);
						return (userCloseAnswer == JOptionPane.NO_OPTION);
					}
				});

		addComponentListener(new ComponentAdapter() {
			@Override
			public void componentResized(ComponentEvent e) {
				System.out.println("Size " + getSize());
			}
		});

	}

	protected static JXStatusBar getStatusBar(final JXPanel jxPanel,
			final JTabbedPane mainTabbedPane) {
		JXStatusBar statusBar = new JXStatusBar();

		try {
			ClassLoader cl = Thread.currentThread().getContextClassLoader();
			Enumeration<URL> urls = cl.getResources("META-INF/MANIFEST.MF");
			String substanceVer = null;
			String substanceBuildStamp = null;
			while (urls.hasMoreElements()) {
				InputStream is = urls.nextElement().openStream();
				BufferedReader br = new BufferedReader(
						new InputStreamReader(is));
				while (true) {
					String line = br.readLine();
					if (line == null)
						break;
					int firstColonIndex = line.indexOf(":");
					if (firstColonIndex < 0)
						continue;
					String name = line.substring(0, firstColonIndex).trim();
					String val = line.substring(firstColonIndex + 1).trim();
					if (name.compareTo("Substance-Version") == 0)
						substanceVer = val;
					if (name.compareTo("Substance-BuildStamp") == 0)
						substanceBuildStamp = val;
				}
				try {
					br.close();
				} catch (IOException ioe) {
				}
			}
			if (substanceVer != null) {
				JLabel statusLabel = new JLabel(substanceVer + " [built on "
						+ substanceBuildStamp + "]");
				JXStatusBar.Constraint cStatusLabel = new JXStatusBar.Constraint();
				cStatusLabel.setFixedWidth(300);
				statusBar.add(statusLabel, cStatusLabel);
			}
		} catch (IOException ioe) {
		}

		// final JLabel statusLabel = new JLabel("");
		// JXStatusBar.Constraint cStatusLabel = new JXStatusBar.Constraint();
		// cStatusLabel.setFixedWidth(100);
		// statusBar.add(statusLabel, cStatusLabel);
		// try {
		// Toolkit.getDefaultToolkit().addAWTEventListener(
		// new AWTEventListener() {
		// public void eventDispatched(AWTEvent event) {
		// if (event instanceof MouseEvent) {
		// MouseEvent me = (MouseEvent) event;
		// if (me.getID() == MouseEvent.MOUSE_MOVED) {
		// Point p = me.getPoint();
		// statusLabel.setText(p.x + ":" + p.y);
		// }
		// }
		// }
		// }, AWTEvent.MOUSE_MOTION_EVENT_MASK);
		// } catch (AccessControlException ace) {
		// // running in JNLP - ignore
		// }

		JXStatusBar.Constraint c2 = new JXStatusBar.Constraint(
				JXStatusBar.Constraint.ResizeBehavior.FILL);
		final JLabel tabLabel = new JLabel("");
		statusBar.add(tabLabel, c2);
		mainTabbedPane.getModel().addChangeListener(new ChangeListener() {
			public void stateChanged(ChangeEvent e) {
				int selectedIndex = mainTabbedPane.getSelectedIndex();
				if (selectedIndex < 0)
					tabLabel.setText("No selected tab");
				else
					tabLabel.setText("Tab "
							+ mainTabbedPane.getTitleAt(selectedIndex)
							+ " selected");
			}
		});

		JPanel fontSizePanel = FontSizePanel.getPanel();
		JXStatusBar.Constraint fontSizePanelConstraints = new JXStatusBar.Constraint();
		fontSizePanelConstraints.setFixedWidth(190);
		statusBar.add(fontSizePanel, fontSizePanelConstraints);

		JPanel alphaPanel = new JPanel(new FlowLayout(FlowLayout.LEADING, 0, 0));
		// alphaPanel.setOpaque(false);

		final JLabel alphaLabel = new JLabel("100%");
		final JSlider alphaSlider = new JSlider(0, 100, 100);
		alphaSlider.setFocusable(false);
		// alphaSlider.setOpaque(false);
		alphaSlider.addChangeListener(new ChangeListener() {
			public void stateChanged(ChangeEvent e) {
				int currValue = alphaSlider.getValue();
				alphaLabel.setText(currValue + "%");
				jxPanel.setAlpha(currValue / 100.0f);
			}
		});
		alphaSlider
				.setToolTipText("Changes the global opacity. Is not Substance-specific");
		alphaSlider.setPreferredSize(new Dimension(120, alphaSlider
				.getPreferredSize().height));

		alphaPanel.add(alphaLabel);
		alphaPanel.add(alphaSlider);

		JXStatusBar.Constraint alphaPanelConstraints = new JXStatusBar.Constraint();
		alphaPanelConstraints.setFixedWidth(160);
		statusBar.add(alphaPanel, alphaPanelConstraints);
		return statusBar;
	}

	public static JButton getPill(SubstanceTheme theme1, SubstanceTheme theme2) {
		JButton result = new JButton("Pill");
		result.putClientProperty(
				SubstanceLookAndFeel.GRADIENT_PAINTER_PROPERTY,
				SpecularGradientPainter.class.getName());
		result.putClientProperty(SubstanceLookAndFeel.PAINT_ACTIVE_PROPERTY,
				Boolean.TRUE);
		result.putClientProperty(SubstanceLookAndFeel.THEME_PROPERTY,
				new SubstanceMixTheme(theme1, theme2));
		result.putClientProperty(SubstanceLookAndFeel.BUTTON_SHAPER_PROPERTY,
				StandardButtonShaper.class.getName());
		return result;
	}

	public static JButton getMultiPill(SubstanceTheme... themes) {
		JButton result = new JButton(themes.length + "Pill");
		result.putClientProperty(
				SubstanceLookAndFeel.GRADIENT_PAINTER_PROPERTY,
				SpecularGradientPainter.class.getName());
		result.putClientProperty(SubstanceLookAndFeel.PAINT_ACTIVE_PROPERTY,
				Boolean.TRUE);
		result.putClientProperty(SubstanceLookAndFeel.THEME_PROPERTY,
				new SubstanceMixTheme(themes));
		result.putClientProperty(SubstanceLookAndFeel.BUTTON_SHAPER_PROPERTY,
				StandardButtonShaper.class.getName());
		return result;
	}

	public static void main(String[] args) {
		boolean hasLafSpecified = false;
		try {
			hasLafSpecified = (System.getProperty("swing.defaultlaf") != null);
		} catch (Throwable t) {
			// JNLP sandbox
		}

		try {
			if (!hasLafSpecified) {
				out(" CREATING LAF ");
				long time0 = System.currentTimeMillis();
				LookAndFeel laf = (LookAndFeel) Class
						.forName(
								"org.jvnet.substance.skin.SubstanceBusinessBlackSteelLookAndFeel")
						.newInstance();
				long time1 = System.currentTimeMillis();
				out(" LAF CREATED " + (time1 - time0));
				out(" SETTING LAF ");
				long time2 = System.currentTimeMillis();
				UIManager.setLookAndFeel(laf);
				long time3 = System.currentTimeMillis();
				out(" LAF SET " + (time3 - time2));
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		SubstanceLookAndFeel.setToUseConstantThemesOnDialogs(true);

		UIManager.put(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_PROPERTY,
				Boolean.TRUE);
		// try {
		// if (System.getProperty("substancelaf.useDecorations") == null) {
		JFrame.setDefaultLookAndFeelDecorated(true);
		JDialog.setDefaultLookAndFeelDecorated(true);
		// }
		// } catch (AccessControlException ace) {
		// JFrame.setDefaultLookAndFeelDecorated(true);
		// JDialog.setDefaultLookAndFeelDecorated(true);
		// }
		// System.setProperty("sun.awt.noerasebackground", "false");

		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				long time2 = System.currentTimeMillis();
				Check c = new Check(true);
				c.addComponentListener(new ComponentAdapter() {
					@Override
					public void componentResized(ComponentEvent e) {
						super.componentResized(e);
						((JFrame) e.getComponent()).getRootPane().repaint();
					}
				});
				c.setPreferredSize(new Dimension(820, 560));
				c.setMinimumSize(new Dimension(150, 100));
				c.pack();
				Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
				// center the frame in the physical screen
				c.setLocation((d.width - c.getWidth()) / 2, (d.height - c
						.getHeight()) / 2);

				c.setVisible(true);
				c
						.setDefaultCloseOperation(System
								.getProperty("javawebstart.version") != null ? JFrame.EXIT_ON_CLOSE
								: JFrame.DISPOSE_ON_CLOSE);
				long time3 = System.currentTimeMillis();
				out("App " + (time3 - time2));
			}
		});
	}

	public static void out(Object obj) {
		try {
			System.out.println(obj);
		} catch (Exception exc) {
			// ignore - is thrown on Mac in WebStart (security access)
		}
	}

	public static void setTheme(final SubstanceTheme theme) {
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				try {
					SubstanceLookAndFeel.setCurrentTheme(theme);
					UIManager.setLookAndFeel(new SubstanceLookAndFeel());
				} catch (Exception exc) {
					exc.printStackTrace();
				}
				for (Frame frame : Frame.getFrames())
					SwingUtilities.updateComponentTreeUI(frame);
				out(UIManager.getColor("TabbedPane.selectHighlight"));
			}
		});
	}

	public static Icon getIcon(String iconName) {
		ClassLoader cl = Thread.currentThread().getContextClassLoader();
		URL url = cl.getResource("test/check/icons/" + iconName + ".gif");
		if (url != null)
			return new ImageIcon(url);
		url = cl.getResource("test/check/icons/" + iconName + ".png");
		if (url != null)
			return new ImageIcon(url);
		return null;
	}

	public static JToolBar getToolbar(String label, int size, boolean hasStrings) {
		JToolBar toolBar = new JToolBar();
		// toolBar.setLayout(new BoxLayout(toolBar,BoxLayout.LINE_AXIS));
		// toolBar.setFloatable(false);

		JButton buttonCut = new JButton(hasStrings ? "cut" : null, getIcon(size
				+ "/edit-cut"));
		buttonCut.putClientProperty(
				SubstanceLookAndFeel.BUTTON_NO_MIN_SIZE_PROPERTY, Boolean.TRUE);
		toolBar.add(buttonCut);
		JButton buttonCopy = new JButton(hasStrings ? "copy" : null,
				getIcon(size + "/edit-copy"));
		buttonCopy.putClientProperty(
				SubstanceLookAndFeel.BUTTON_NO_MIN_SIZE_PROPERTY, Boolean.TRUE);
		buttonCopy.setEnabled(false);
		toolBar.add(buttonCopy);
		JButton buttonPaste = new JButton(getIcon(size + "/edit-paste"));
		toolBar.add(buttonPaste);
		JButton buttonSelectAll = new JButton(
				getIcon(size + "/edit-select-all"));
		toolBar.add(buttonSelectAll);
		JButton buttonDelete = new JButton(getIcon(size + "/edit-delete"));
		toolBar.add(buttonDelete);
		toolBar.addSeparator();

		// add an inner toolbar to check the painting of toolbar
		// gradient and drop shadows under different skins.
		JToolBar innerToolbar = new JToolBar(JToolBar.HORIZONTAL);
		innerToolbar.setFloatable(false);
		JToggleButton buttonFormatCenter = new JToggleButton(getIcon(size
				+ "/format-justify-center"));
		buttonFormatCenter.putClientProperty(
				SubstanceLookAndFeel.CORNER_RADIUS, Float.valueOf(5.0f));
		innerToolbar.add(buttonFormatCenter);
		JToggleButton buttonFormatLeft = new JToggleButton(getIcon(size
				+ "/format-justify-left"));
		innerToolbar.add(buttonFormatLeft);
		JToggleButton buttonFormatRight = new JToggleButton(getIcon(size
				+ "/format-justify-right"));
		innerToolbar.add(buttonFormatRight);
		JToggleButton buttonFormatFill = new JToggleButton(getIcon(size
				+ "/format-justify-fill"));
		buttonFormatFill.putClientProperty(SubstanceLookAndFeel.CORNER_RADIUS,
				Float.valueOf(0.0f));
		innerToolbar.add(buttonFormatFill);

		toolBar.add(innerToolbar);
		toolBar.addSeparator();

		if (size > 20) {
			// JPanel groupPanel = new JPanel();
			// BoxLayout groupLayout = new BoxLayout(groupPanel,
			// BoxLayout.LINE_AXIS);
			// groupPanel.setLayout(groupLayout);
			JToolBar innerToolbar2 = new JToolBar(JToolBar.HORIZONTAL);
			innerToolbar2.setFloatable(false);

			JPanel innerPanel = new JPanel(
					new FlowLayout(FlowLayout.LEFT, 0, 0));// new
			// BorderLayout());
			innerToolbar2.add(innerPanel, BorderLayout.CENTER);

			final JToggleButton buttonStyleBold = new JToggleButton(
					getIcon(size + "/format-text-bold"));
			Set<Side> rightSide = new HashSet<Side>();
			rightSide.add(Side.RIGHT);
			buttonStyleBold.putClientProperty(
					SubstanceLookAndFeel.BUTTON_OPEN_SIDE_PROPERTY, rightSide);
			buttonStyleBold.putClientProperty(
					SubstanceLookAndFeel.CORNER_RADIUS, Float.valueOf(3.0f));

			final JToggleButton buttonStyleItalic = new JToggleButton(
					getIcon(size + "/format-text-italic"));
			buttonStyleItalic.putClientProperty(
					SubstanceLookAndFeel.CORNER_RADIUS, Float.valueOf(0.0f));
			buttonStyleItalic.putClientProperty(
					SubstanceLookAndFeel.BUTTON_OPEN_SIDE_PROPERTY, rightSide);

			final JToggleButton buttonStyleUnderline = new JToggleButton(
					getIcon(size + "/format-text-underline"));
			buttonStyleUnderline.putClientProperty(
					SubstanceLookAndFeel.CORNER_RADIUS, Float.valueOf(0.0f));
			buttonStyleUnderline.putClientProperty(
					SubstanceLookAndFeel.BUTTON_OPEN_SIDE_PROPERTY, rightSide);

			final JToggleButton buttonStyleStrikethrough = new JToggleButton(
					getIcon(size + "/format-text-strikethrough"));
			buttonStyleStrikethrough.putClientProperty(
					SubstanceLookAndFeel.BUTTON_SIDE_PROPERTY,
					SubstanceConstants.Side.LEFT.name());
			buttonStyleStrikethrough.putClientProperty(
					SubstanceLookAndFeel.CORNER_RADIUS, Float.valueOf(3.0f));
			buttonStyleBold.setSelected(true);

			innerPanel.add(buttonStyleBold);
			innerPanel.add(buttonStyleItalic);
			innerPanel.add(buttonStyleUnderline);
			innerPanel.add(buttonStyleStrikethrough);

			toolBar.add(innerToolbar2);
		}
		// toolBar.add(groupPanel);

		toolBar.add(Box.createGlue());
		JButton buttonExit = new JButton(getIcon(size + "/process-stop"));
		buttonExit.setToolTipText("Closes the test application");
		buttonExit.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				System.exit(0);
			}
		});
		toolBar.add(buttonExit);

		return toolBar;
	}

	public static void setTheme(final ThemeInfo themeInfo,
			final boolean useInstance) {
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				try {
					if (useInstance)
						SubstanceLookAndFeel.setCurrentTheme(SubstanceTheme
								.createInstance(themeInfo));
					else
						SubstanceLookAndFeel.setCurrentTheme(themeInfo
								.getClassName());
					// UIManager.setLookAndFeel(new SubstanceLookAndFeel());
				} catch (Exception exc) {
					exc.printStackTrace();
				}
				for (Frame frame : Frame.getFrames())
					SwingUtilities.updateComponentTreeUI(frame);
			}
		});
	}

	public void setSpecificTaskPane(JPanel contents, String title, Icon icon) {
		if (currentSpecificTaskPane != null) {
			taskPaneContainer.remove(currentSpecificTaskPane);
		}
		currentSpecificTaskPane = new JXTaskPane();
		currentSpecificTaskPane.setLayout(new BorderLayout());
		currentSpecificTaskPane.setTitle(title);
		currentSpecificTaskPane.setIcon(icon);
		// contents.setOpaque(false);
		SwingUtilities.updateComponentTreeUI(contents);
		// currentSpecificTaskPane.setOpaque(false);
		currentSpecificTaskPane.add(contents, BorderLayout.CENTER);

		// this.mainTaskPane.setExpanded(false);
		taskPaneContainer.add(currentSpecificTaskPane);
	}

	public void clearSpecificTaskPane() {
		if (currentSpecificTaskPane != null) {
			taskPaneContainer.remove(currentSpecificTaskPane);
		}
		currentSpecificTaskPane = null;
		// this.mainTaskPane.setExpanded(true);
	}

	public class TabSwitchListener implements ChangeListener {
		public void stateChanged(ChangeEvent e) {
			Component selected = jtp.getSelectedComponent();
			if (selected instanceof Controllable) {
				JPanel controlPanel = ((Controllable) selected)
						.getControlPanel();
				if (controlPanel == null)
					clearSpecificTaskPane();
				else
					setSpecificTaskPane(controlPanel, jtp.getTitleAt(jtp
							.getSelectedIndex()), jtp.getIconAt(jtp
							.getSelectedIndex()));
			} else {
				clearSpecificTaskPane();
			}
		}
	}

	public static class MyMainTabPreviewPainter extends
			DefaultTabPreviewPainter {
		protected TabOverviewKind tabOverviewKind;

		public void setTabOverviewKind(TabOverviewKind tabOverviewKind) {
			this.tabOverviewKind = tabOverviewKind;
		}

		@Override
		public TabOverviewKind getOverviewKind(JTabbedPane tabPane) {
			if (tabOverviewKind == null)
				return super.getOverviewKind(tabPane);
			return tabOverviewKind;
		}

		@Override
		public void previewTab(JTabbedPane tabPane, int tabIndex, Graphics g,
				int x, int y, int w, int h) {
			if (tabPane.getComponentAt(tabIndex) instanceof MixedButtonsPanel) {
				Graphics2D graphics = (Graphics2D) g.create();
				int fontSize = Math.min(w, h) / 6;
				Font font = new Font("Arial", Font.PLAIN, fontSize);
				graphics.setFont(font);
				graphics.setColor(Color.black);
				graphics.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
						RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
				graphics.drawString("Mixed buttons", x + 5, y + w / 2
						- fontSize);
				graphics.drawRect(x, y, w - 1, h - 1);
				graphics.dispose();
			} else {
				super.previewTab(tabPane, tabIndex, g, x, y, w, h);
			}
		}
	}

	public JTabbedPane getMainTabbedPane() {
		return this.jtp;
	}
}
