/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance.painter.text;

import java.awt.Component;
import java.awt.Rectangle;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JComponent;

import org.jvnet.substance.utils.SubstanceCoreUtilities;

/**
 * Base implementation of {@link SubstanceTextPainter} that provides common
 * functionality for core text painters.
 * 
 * @author Kirill Grouchnikov
 */
public abstract class AbstractTextPainter implements SubstanceTextPainter {
	/**
	 * Information on a single text line.
	 * 
	 * @author Kirill Grouchnikov
	 */
	public static class TextLineInfo {
		/**
		 * Text rectangle.
		 */
		public Rectangle textRect;

		/**
		 * The text itself.
		 */
		public String text;

		/**
		 * Mnemonic index. Can be -1 if no mnemonic.
		 */
		public int mnemonicIndex;

		/**
		 * Font for the text.
		 */
		public java.awt.Font font;

		/**
		 * Color for the text.
		 */
		public java.awt.Color color;

		/**
		 * Clipping rectangle for the text.
		 */
		public java.awt.Rectangle clip;

		/**
		 * Transformation for the text.
		 */
		public java.awt.geom.AffineTransform transform;

		/**
		 * Creates a new text line info object.
		 * 
		 * @param textRect
		 *            Text rectangle.
		 * @param text
		 *            The text itself.
		 * @param mnemonicIndex
		 *            Mnemonic index. Can be -1 if no mnemonic.
		 * @param font
		 *            Font for the text.
		 * @param color
		 *            Color for the text.
		 * @param clip
		 *            Clipping rectangle for the text.
		 */
		public TextLineInfo(Rectangle textRect, String text, int mnemonicIndex,
				java.awt.Font font, java.awt.Color color,
				java.awt.Rectangle clip, java.awt.geom.AffineTransform transform) {
			this.textRect = textRect;
			this.text = text;
			this.mnemonicIndex = mnemonicIndex;
			this.font = font;
			this.color = color;
			this.clip = clip;
			this.transform = transform;
		}
	}

	/**
	 * Attached background painting callbacks.
	 */
	protected List<SubstanceTextPainter.BackgroundPaintingCallback> callbackList;

	/**
	 * Attached texts.
	 */
	protected List<TextLineInfo> textLines;

	/**
	 * If <code>true</code>, the painter implementation will fill the
	 * background and invoke all callbacks even when there are no attached texts
	 * or all texts are empty.
	 */
	protected boolean toEnforceRenderOnNoTexts;
	
	protected Component comp;

	/**
	 * Creates a new abstract text painter. Is protected to enforce that it
	 * can't be directly created.
	 */
	protected AbstractTextPainter() {
		this.callbackList = new LinkedList<SubstanceTextPainter.BackgroundPaintingCallback>();
		this.textLines = new LinkedList<TextLineInfo>();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.text.SubstanceTextPainter#init(javax.swing.JComponent,
	 *      java.awt.Rectangle, boolean)
	 */
	public void init(JComponent comp, Rectangle clip,
			boolean toEnforceRenderOnNoTexts) {
		if (!this.toIgnoreBackgroundFill(comp))
			this.callbackList.clear();
		this.comp = comp;
		this.textLines.clear();
		this.toEnforceRenderOnNoTexts = toEnforceRenderOnNoTexts;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.text.SubstanceTextPainter#attachCallback(org.jvnet.substance.painter.text.SubstanceTextPainter.BackgroundPaintingCallback)
	 */
	public void attachCallback(
			SubstanceTextPainter.BackgroundPaintingCallback backgroundPaintingCallback) {
		this.callbackList.add(backgroundPaintingCallback);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.text.SubstanceTextPainter#attachText(javax.swing.JComponent,
	 *      java.awt.Rectangle, java.lang.String, int, java.awt.Font,
	 *      java.awt.Color, java.awt.Rectangle, java.awt.geom.AffineTransform)
	 */
	public void attachText(JComponent comp, Rectangle textRect, String text,
			int mnemonicIndex, java.awt.Font font, java.awt.Color color,
			java.awt.Rectangle clip) {
		this.textLines.add(new TextLineInfo(textRect, text, mnemonicIndex,
				font, color, clip, null));
	}

	/**
	 * Checks whether the background fill should be skipped for the specified
	 * component.
	 * 
	 * @param comp
	 *            Component.
	 * @return <code>true</code> if the background fill should be skipped for
	 *         the specified component, <code>false</code> otherwise.
	 */
	protected boolean toIgnoreBackgroundFill(Component comp) {
		Component c = comp;
		while (c != null) {
			if (c instanceof JComponent) {
				JComponent jc = (JComponent) c;
				if (Boolean.TRUE
						.equals(jc
								.getClientProperty(SubstanceCoreUtilities.DO_NOT_FILL_BACKGROUND))) {
					return true;
				}
			}
			c = c.getParent();
		}
		return false;
	}
}
