/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.painter;

import java.awt.*;
import java.awt.geom.Point2D;
import java.awt.image.*;

import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.utils.SubstanceColorUtilities;
import org.jvnet.substance.utils.SubstanceCoreUtilities;

/**
 * Gradient painter that returns images with 3D appearance and specular shine
 * spot. This class is part of officially supported API.
 * 
 * @author Kirill Grouchnikov
 */
public class SpecularGradientPainter extends BaseGradientPainter {
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.SubstanceGradientPainter#getDisplayName()
	 */
	public String getDisplayName() {
		return "Specular";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.SubstanceGradientPainter#getContourBackground(int,
	 *      int, java.awt.Shape, boolean, org.jvnet.substance.color.ColorScheme,
	 *      org.jvnet.substance.color.ColorScheme, float, boolean, boolean)
	 */
	public BufferedImage getContourBackground(int width, int height,
			Shape contour, boolean isDark, ColorScheme colorScheme1,
			ColorScheme colorScheme2, float cyclePos, boolean hasShine,
			boolean useCyclePosAsInterpolation) {

		BufferedImage mixResult = this.getMixContourBackground(width, height,
				contour, isDark, colorScheme1, colorScheme2, cyclePos,
				hasShine, useCyclePosAsInterpolation);
		if (mixResult != null)
			return mixResult;

		// long millis = System.nanoTime();

		BufferedImage image = SubstanceCoreUtilities.getBlankImage(width,
				height);
		Graphics2D graphics = (Graphics2D) image.getGraphics();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		ColorScheme interpolationScheme1 = colorScheme1;
		ColorScheme interpolationScheme2 = useCyclePosAsInterpolation ? colorScheme2
				: colorScheme1;

		double cycleCoef = 1.0 - cyclePos / 10.0;

//		Color topBorderColor = SubstanceColorUtilities.getTopBorderColor(
//				interpolationScheme1, interpolationScheme2, cycleCoef);
//		Color midBorderColor = SubstanceColorUtilities.getMidBorderColor(
//				interpolationScheme1, interpolationScheme2, cycleCoef);
//		Color bottomBorderColor = SubstanceColorUtilities.getBottomBorderColor(
//				interpolationScheme1, interpolationScheme2, cycleCoef);
		Color topFillColor = SubstanceColorUtilities.getTopFillColor(
				interpolationScheme1, interpolationScheme2, cycleCoef,
				useCyclePosAsInterpolation);
		Color midFillColor = SubstanceColorUtilities.getMidFillColor(
				interpolationScheme1, interpolationScheme2, cycleCoef,
				useCyclePosAsInterpolation);
		Color bottomFillColor = SubstanceColorUtilities.getBottomFillColor(
				interpolationScheme1, interpolationScheme2, cycleCoef,
				useCyclePosAsInterpolation);
		Color topShineColor = SubstanceColorUtilities.getTopShineColor(
				interpolationScheme1, interpolationScheme2, cycleCoef,
				useCyclePosAsInterpolation);
		Color bottomShineColor = SubstanceColorUtilities.getBottomShineColor(
				interpolationScheme1, interpolationScheme2, cycleCoef,
				useCyclePosAsInterpolation);

		graphics.setStroke(new BasicStroke((float) 1.3, BasicStroke.CAP_ROUND,
				BasicStroke.JOIN_ROUND));

		// Fill background
		// long millis000 = System.nanoTime();
		graphics.setClip(contour);
		GradientPaint gradientTop = new GradientPaint(0, 0, topFillColor, 0,
				height / 2, midFillColor);
		graphics.setPaint(gradientTop);
		graphics.fillRect(0, 0, width, height / 2);

		GradientPaint gradientBottom = new GradientPaint(0, height / 2,
				midFillColor, 0, height - 2, bottomFillColor);
		graphics.setPaint(gradientBottom);
		graphics.fillRect(0, height / 2, width, height / 2);
		// long millis001 = System.nanoTime();

		// long millis003 = 0, millis004 = 0, millis005 = 0;

//		// Draw border
//		GradientPaint gradientBorderTop = new GradientPaint(0, 0,
//				topBorderColor, 0, height / 2, midBorderColor);
//		graphics.setPaint(gradientBorderTop);
//		graphics.setClip(0, 0, width, height / 2);
//		graphics.draw(contour);
//
//		GradientPaint gradientBorderBottom = new GradientPaint(0, height / 2,
//				midBorderColor, 0, height - 2, bottomBorderColor);
//		graphics.setPaint(gradientBorderBottom);
//		graphics.setClip(0, height / 2, width, 1 + height / 2);
//		graphics.draw(contour);

		if (hasShine) {
			graphics.setClip(null);
			int shineHeight = (int) (height / 1.8);

			double[][] dists = new double[width][height];
			ColorModel srcColorModel = image.getColorModel();
			WritableRaster srcRaster = image.getRaster();
			int[][] alphas = new int[width][height];
			for (int x = 0; x < width; x++) {
				for (int y = 0; y < height; y++) {
					alphas[x][y] = srcColorModel.getAlpha(srcRaster
							.getDataElements(x, y, null));
					dists[x][y] = Math.max(width, height);
				}
			}

			// scan rows
			for (int y = 0; y < height; y++) {
				// from left
				int x = 0;
				double span = 0;
				while (x < width) {
					if (alphas[x][y] == 0) {
						// transparent pixel
						span = 0;
					} else {
						if ((span == 0) || (alphas[x][y] < 255))
							span = alphas[x][y] / 255.0;
						else
							span += 1.0;
					}
					dists[x][y] = Math.min(span, dists[x][y]);
					x++;
				}
				// from right
				x = width - 1;
				span = 0;
				while (x >= 0) {
					if (alphas[x][y] == 0) {
						// transparent pixel
						span = 0;
					} else {
						if ((span == 0) || (alphas[x][y] < 255))
							span = alphas[x][y] / 255.0;
						else
							span += 1.0;
					}
					dists[x][y] = Math.min(span, dists[x][y]);
					x--;
				}
			}

			// scan columns
			for (int x = 0; x < width; x++) {
				// from top
				int y = 0;
				double span = 0;
				while (y < height) {
					if (alphas[x][y] == 0) {
						// transparent pixel
						span = 0;
					} else {
						if ((span == 0) || (alphas[x][y] < 255))
							span = alphas[x][y] / 255.0;
						else
							span += 1.0;
					}
					dists[x][y] = Math.min(span, dists[x][y]);
					y++;
				}
				// from bottom
				y = height - 1;
				span = 0;
				while (y >= 0) {
					if (alphas[x][y] == 0) {
						// transparent pixel
						span = 0;
					} else {
						if ((span == 0) || (alphas[x][y] < 255))
							span = alphas[x][y] / 255.0;
						else
							span += 1.0;
					}
					dists[x][y] = Math.min(span, dists[x][y]);
					y--;
				}
			}

			for (int y = 0; y < shineHeight; y++) {
				double coef = (double) y / (double) height;
				double minDist = (0.8 + height / 20.0) * (1.0 + 3 * coef);
				double maxDist = (0.8 + height / 20.0) * (1.0 + 6 * coef);

				double colorCoef = (double) y / (double) shineHeight;
				int alpha = (int) (255 * (1.0 - 0.1 * colorCoef));
				alpha = Math.max(alpha, 0);

				int shineR = (int) (topShineColor.getRed() + colorCoef
						* (bottomShineColor.getRed() - topShineColor.getRed()));
				int shineG = (int) (topShineColor.getGreen() + colorCoef
						* (bottomShineColor.getGreen() - topShineColor
								.getGreen()));
				int shineB = (int) (topShineColor.getBlue() + colorCoef
						* (bottomShineColor.getBlue() - topShineColor.getBlue()));

				for (int x = 0; x < width; x++) {
					Point2D p = new Point2D.Double(x + 0.5, y + 0.5);
					if (!contour.contains(p)) {
						continue;
					}

					double dist = dists[x][y];
					if (dist <= minDist)
						continue;

					double delta = dist - minDist;
					int dAlpha = alpha;
					if (delta < 1.0)
						dAlpha = (int) (delta * dAlpha);

					if (dist < maxDist) {
						dAlpha *= (dist - minDist) / (maxDist - minDist);
					}

					graphics
							.setColor(new Color(shineR, shineG, shineB, dAlpha));
					graphics.drawLine(x, y, x, y);
				}
			}
		}

		// long millis006 = System.nanoTime();

		// long millis2 = System.nanoTime();
		// if (width * height > 5000) {
		// System.out.println("new - " + width + "*" + height + " = "
		// + format(millis2 - millis));
		// System.out.println("\tfill : " + format(millis001 - millis000));
		// System.out.println("\tcontour : " + format(millis003 - millis001));
		// System.out.println("\trevert : " + format(millis004 - millis003));
		// System.out.println("\toverlay : " + format(millis005 - millis004));
		// System.out.println("\tborder : " + format(millis006 - millis005));
		// }

		return image;
	}
}
