/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.plaf.*;
import javax.swing.plaf.basic.BasicBorders;
import javax.swing.plaf.basic.BasicTextAreaUI;
import javax.swing.text.*;

import org.jvnet.lafwidget.animation.FadeStateListener;
import org.jvnet.substance.text.*;
import org.jvnet.substance.utils.SubstanceCoreUtilities;
import org.jvnet.substance.utils.SubstanceSizeUtils;

/**
 * UI for text areas in <b>Substance</b> look and feel.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceTextAreaUI extends BasicTextAreaUI {
	/**
	 * Listener for fade animations.
	 */
	protected FadeStateListener substanceFadeStateListener;

	/**
	 * The associated text area.
	 */
	protected JTextArea textArea;

	/**
	 * Property change listener.
	 */
	protected PropertyChangeListener substancePropertyChangeListener;

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.ComponentUI#createUI(javax.swing.JComponent)
	 */
	public static ComponentUI createUI(JComponent c) {
		return new SubstanceTextAreaUI(c);
	}

	/**
	 * Simple constructor.
	 * 
	 * @param c
	 *            Component (text area).
	 */
	public SubstanceTextAreaUI(JComponent c) {
		super();
		this.textArea = (JTextArea) c;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicTextUI#installListeners()
	 */
	@Override
	protected void installListeners() {
		super.installListeners();

		this.substanceFadeStateListener = new FadeStateListener(this.textArea,
				null, null);
		this.substanceFadeStateListener.registerListeners(false);

		this.substancePropertyChangeListener = new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				if ("font".equals(evt.getPropertyName())) {
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							textArea.updateUI();
						}
					});
				}
			}
		};
		this.textArea
				.addPropertyChangeListener(this.substancePropertyChangeListener);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicTextUI#uninstallListeners()
	 */
	@Override
	protected void uninstallListeners() {
		this.substanceFadeStateListener.unregisterListeners();
		this.substanceFadeStateListener = null;

		this.textArea
				.removePropertyChangeListener(this.substancePropertyChangeListener);
		this.substancePropertyChangeListener = null;

		super.uninstallListeners();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicTextAreaUI#installDefaults()
	 */
	@Override
	protected void installDefaults() {
		super.installDefaults();
		Border b = this.textArea.getBorder();
		if (b == null || b instanceof UIResource) {
			Border newB = new BorderUIResource.CompoundBorderUIResource(
					new SubstanceBorder(SubstanceSizeUtils
							.getTextBorderInsets(SubstanceSizeUtils
									.getComponentFontSize(this.textArea))),
					new BasicBorders.MarginBorder());
			this.textArea.setBorder(newB);
		}
	}

	// /*
	// * (non-Javadoc)
	// *
	// * @see
	// javax.swing.plaf.basic.BasicTextUI#paintBackground(java.awt.Graphics)
	// */
	// @Override
	// protected void paintBackground(Graphics g) {
	// SubstanceCoreUtilities.paintCompBackground(g, this.textArea);
	// }
	//
	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicTextUI#createHighlighter()
	 */
	@Override
	protected Highlighter createHighlighter() {
		return new SubstanceHighlighter();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicTextAreaUI#create(javax.swing.text.Element)
	 */
	@Override
	public View create(Element elem) {
		View superResult = super.create(elem);
		if (superResult instanceof PlainView) {
			return new SubstancePlainView(elem);
		}
		if (superResult instanceof WrappedPlainView) {
			return new SubstanceWrappedPlainView(elem,
					((JTextArea) getComponent()).getWrapStyleWord());
		}
		return superResult;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicTextUI#modelChanged()
	 */
	@Override
	protected void modelChanged() {
		super.modelChanged();

		JTextComponent component = this.getComponent();
		View currView = getRootView(component);
		Element defaultRootElement = component.getDocument()
				.getDefaultRootElement();
		if (currView instanceof PlainView) {
			setView(new SubstancePlainView(defaultRootElement));
		}
		if (currView instanceof WrappedPlainView) {
			setView(new SubstanceWrappedPlainView(defaultRootElement,
					((JTextArea) getComponent()).getWrapStyleWord()));
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicTextUI#paintSafely(java.awt.Graphics)
	 */
	@Override
	protected void paintSafely(Graphics _g) {
		// // Have to call the super implementation since it sets a
		// // private "painted" flag which affects many other places.
		// // Without this there will be many visual artifacts with
		// // painting the caret and highlights
		// super.paintSafely(_g);
		//
		// SubstanceTextPainter textPainter = SubstanceLookAndFeel
		// .getCurrentTextPainter();
		//
		// SubstanceTextPainter.BackgroundPaintingCallback callback = new
		// SubstanceTextPainter.BackgroundPaintingCallback() {
		// public void paintBackground(Graphics g) {
		// Highlighter highlighter = getComponent().getHighlighter();
		//
		// // paint the background
		// if (getComponent().isOpaque()) {
		// SubstanceTextAreaUI.this.paintBackground(g);
		// }
		//
		// // paint the highlights
		// if (highlighter != null) {
		// highlighter.paint(g);
		// }
		// }
		// };
		// textPainter.init(textArea, null, true);
		// if (textPainter.needsBackgroundImage()) {
		// textPainter.setBackgroundFill(textArea, textArea.getBackground(),
		// false, 0, 0);
		// textPainter.attachCallback(callback);
		// } else {
		// callback.paintBackground(_g);
		// }
		//
		// // paint the view hierarchy
		// Rectangle alloc = getVisibleEditorRect();
		// if (alloc != null) {
		// getRootView(textArea).paint(_g, alloc);
		// }
		// textPainter.renderSurface(_g);
		//
		// // paint the caret
		// Caret caret = getComponent().getCaret();
		// if (caret != null) {
		// caret.paint(_g);
		// }
		// Have to call the super implementation since it sets a
		// private "painted" flag which affects many other places.
		// Without this there will be many visual artifacts with
		// painting the caret and highlights
		Graphics2D dummy = (Graphics2D) _g.create(0, 0, 0, 0);
		super.paintSafely(dummy);
		dummy.dispose();

		SubstanceCoreUtilities.paintTextComponent(_g, this.textArea, this
				.getRootView(this.textArea), this.getVisibleEditorRect());
	}
}
