/*  -*- c -*-  */
/* -------------------------------------------------------------------- *
**  copyright (c) 1995 ipvr stuttgart and thomas harrer
** -------------------------------------------------------------------- *
**
**  libhelp
**
**  a comprehensive hypertext help system for OSF/Motif(tm) applications. 
**  based on libhtmlw from NCSA Mosaic(tm) version 2.4
**
**  written by thomas harrer
**  e-mail: Thomas.Harrer@rus.uni-stuttgart.de
**  
** -------------------------------------------------------------------- *
*h  $Id: load.c,v 1.32 1995/06/28 12:59:30 thomas Exp $
** -------------------------------------------------------------------- *
**
*h  module:		load.c
**
**  contents:		routines to load files or directories
**
**  interface:		procedures 
**			* read_html_file
**
** -------------------------------------------------------------------- *
**  license and copying issues:
**
**  this software is free; you can redistribute it and/or modify it 
**  under terms similar to the gnu general public license (version 1 
**  or any later version published by the free software foundation). 
**  see the file Licence for more details.
**
**  this program is distributed in the hope that it will be useful,
**  but without any warranty; without even the implied warranty of
**  merchantability or fitness for a particular purpose.  
** -------------------------------------------------------------------- */
/*----------------------------------------------------------------------*
*g  include section
**----------------------------------------------------------------------*/
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>

/* application include files  */
#include "helpp.h"
#include "language.h"
#include "load.h"
#include "buffer.h"
#include "util.h"
#include "path.h"
#include "bcache.h"

/* -------------------------------------------------------------------- *
*g  module identification
** -------------------------------------------------------------------- */
#ifdef RCSID
static char rcsid [] =
    "$Id: load.c,v 1.32 1995/06/28 12:59:30 thomas Exp $";
#endif /* RCSID */

/* -------------------------------------------------------------------- *
*g  module definitions
** -------------------------------------------------------------------- */
#define PLAINTEXT 	"<plaintext>\n"

/* -------------------------------------------------------------------- *
*g  global type: 	dirfile_t
**			currently we take care about the filenames
**			other search attributes can be added.
** -------------------------------------------------------------------- */
typedef struct dirfile_s {

    char*	filename;	      /* name of a file in the dir.    */
    
} dirfile_t;

/* -------------------------------------------------------------------- *
*g  prototypes
** -------------------------------------------------------------------- */
static buffer_t* 	add_plaintext (char*);
static buffer_t* 	format_binary (char*);
static buffer_t* 	format_dir (char*);
static buffer_t* 	format_gif (char*);
static buffer_t* 	format_warning (char*);
static int 		cmp_files (dirfile_t*, dirfile_t*);

/* -------------------------------------------------------------------- *
*p  procedure-name:	read_html_file
**
**  purpose:		reads the contents of a file into a newly
**			allocated buffer (which is returned).
**			
**			if the buffer is already in the cache 
**			just returns it.
** -------------------------------------------------------------------- *
**  args:		string: name of html-file (name#anchor).
**  return type:	char*
**  postcondition:	file is opened, read into allocated 
**			buffer (should not be freed directly),
**			file is closed,	contents are returned. 
**  error handling.:	returns error message instead of file.
**			always returns some valid text.
** -------------------------------------------------------------------- */
buffer_t*
read_html_file (/* i  */ char* file_anchor)
{
    /* local data.  */
    char* 	filename;	       /* absolute path name.		*/
    buffer_t* 	buf = NULL;	       /* return buffer.		*/
    int		ftype;		       /* file classification.		*/
    char* 	path = NULL;	       /* the prefered path.            */
    
    execute ("read_html_file");

    /* preprocessing of the name.  */
    convert_newlines_to_spaces (file_anchor);

    /* if the buffer is already in the cache, we just return it.  */
    buf = bcache_find (file_anchor);
    if (buf) return buf;
    
    /* else we need actions to read and format the file.  */
    filename = path_get_helppath (file_anchor); /* scope local. */

    if (filename) {

	ftype = file_type (filename);

	/* normal files we just read in.  */
	if (ftype & file_regular) {

	    /* regular file.  */
	    FILE* 	html_stream = NULL; /* help stream (file or pipe). */

	    /* filename + path.   */
	    int		flen = c_strlen (filename);
	    int 	gz_pos = (flen > 2 ? (flen - 3) : 0);
	    int 	was_pipe = 0; /* popen: 1, fopen: 0 (for ?close) */

	    /* if ending in .gz open a pipe to gunzip process...  */
	    if (0 == c_strcmp (filename + gz_pos, ".gz")){

		/* but only if the file is really accessible.  */
		if (access (filename, R_OK) == 0) {

		    char* 	gzcommand;
		    checked_malloc (gzcommand, flen + 12, char); 
		    sprintf (gzcommand, "gzip -cdq %s", filename);

		    html_stream = (FILE*) popen (gzcommand, FOPEN_READ_STRING);
		    was_pipe = 1;
		    checked_free (gzcommand);
		} 
	
	    } else {

		html_stream = fopen (filename, FOPEN_READ_STRING);
		was_pipe = 0;
	    }
    
	    /*
	     *  reading starts here ---------------------------
	     */

	    if (!html_stream) {

		buf = NULL;
		path = NULL;
	    
	    } else {	

		int binary;
	    
		/* if this is no html file, we add the plaintext prefix */
		/* to the buffer  */
		if (!(ftype & file_html)) {

		    /* create a new buffer mit plaintext prefix.  */
		    buf = add_plaintext (filename);

		} else {
		    buf = bf_new ();
		}

		/* and then we read in the contents of the file.  */
		binary = bf_fread (buf, html_stream);
    
		/* we didn't forget to close the stream.  */
		if (was_pipe == 1) pclose (html_stream);
		else fclose (html_stream);

		if (binary == BF_is_binary) {
		    /* first free the buffer.  */
		    bf_free (buf);
		    buf = format_binary (filename);
		}
		path = get_path (filename); /* scope local */
		/* printf ("ref: %s, path: %s\n", file_anchor, path);
		   printf ("html: %s\n", bf_the_buffer (buf));*/
	    
	    } /* if (html_stream)  */

	} else if (ftype & file_dir) {

	    /* directories are the second case.  */
	    buf = format_dir (filename);

	    /* we install the prefered path for directories.  */
	    path = get_path (filename);
	
	    /* directories have NULL filenames.  */
	    file_anchor = NULL;

	} else if (ftype & file_gif) {

	    /* gifs are treated special too.  */
	    buf = format_gif (filename);
	    path = get_path (filename);

	} else  {

	    /* filename was neither a directory nor a regular file.  */
	    buf = NULL;
	}
	
	checked_free (filename);

    } /* if (filename) */
    
    /* we guarantee to return some valid text. */
    if (buf) {
	
	/* we save the buffer into the cache for later use.  */
	bcache_insert (file_anchor, path, buf);
	
    } else {

	/* if we couldn't load the file: we write a warning.  */
	buf = format_warning (file_anchor);
    }
	
    return buf;			/* really != NULL ! */
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	format_dir
**
**  purpose:		reads a directory and formats the files as html
**			references
** -------------------------------------------------------------------- *
**  args:		name of the directory.
**  return type:	buffer_t* (newly allocated)
**			the buffer contains the html version of the
**			directory.
** -------------------------------------------------------------------- */
static buffer_t*
format_dir (/* i  */ char* dirname)
{
    /* directory buffer for sorting (qsort).  */
    static dirfile_t* files = NULL;    /* array of dirs (for qsort).	*/
    static int	      files_len = 0;   /* size of the latter.		*/
    static int	      files_in_use;    /* number of filled slots.	*/

    /* local data.  */
    buffer_t* 		buf;	       /* the buffer to return.		*/
    DIR* 		dir;	       /* directory stream.		*/
    struct dirent* 	next;	       /* one entry in a directory.	*/
    int			i;	       /* counter.			*/
    int			max_len = 0;   /* lenght of the longest filen.	*/
    int			dir_len;       /* lenght of dirname.		*/
    char*		path;	       /* reusable filename buffer.	*/
    char*		add_ptr;       /* position of file in path.	*/
    static char* 	cwd = NULL;    /* the working directory.	*/
    
    execute ("format_dir");
    
    /* initialization  */
    files_in_use = 0;
    if (!files) {
	files_len = 40;
	/* scope: local and static (grows but will not be freed).  */
	checked_malloc (files, files_len, dirfile_t);
    }

    /* reading of directories.  */
    dir = opendir (dirname);
    if (!dir) {
	return NULL;
    }

    /* read the entries into the files array to be sorted.  */
    while ((next = readdir (dir)) /* ! */) {

	if (files_in_use >= files_len) {
	    files_len *= 2;
	    checked_realloc (files, files_len, dirfile_t);
	}
	/* we ignore files beginning with .  */
	if (next->d_name[0] != '.') {
	    /* we remember the maximum length of files.  */
	    int len = c_strlen (next->d_name);
	    if (len > max_len) max_len = len;

	    /* scope: local  */
	    checked_strdup (files[files_in_use].filename, next->d_name);
	    files_in_use++;
	}

    } /* while readdir (dir)  */

    (void) closedir (dir);

    /* ok, we sort the directory.  */
    qsort (files, (size_t) files_in_use, 
	   (size_t) sizeof (dirfile_t), 
	   (int (*)(const void *, const void *)) cmp_files);

    /* we may need the current working directory.  */
    if ((!cwd) && (dirname[0] != '/')) {

	size_t 	size = 128;	/* just start size, can grow. */
	char* 	value;		/* the current working directory. */

	/* there is no limit of pathnames in this implementation:  */
	while (1) {
	    checked_malloc (cwd, size, char); /* local */
	    value = getcwd (cwd, size);
	    if (value) break;
	    size *= 2;
	    checked_free (cwd);
	}
    }

    dir_len = c_strlen (dirname);
    if (cwd) dir_len += c_strlen (cwd);
    max_len += (dir_len + 2);
    checked_malloc (path, max_len, char); /* scope: local. */
    
    /* we copy the dir to the path (but without slashes). */
    if (dirname[0] != '/') {

	/* assertation:  */
	if (!cwd) fatal_error (error_inconsistency);
	
	if (0 == c_strcmp (dirname, ".") || (0 == c_strcmp (dirname, "./"))) {
	    c_strcpy (path, cwd);
	} else  {
	    sprintf (path, "%s/%s", cwd, dirname);
	}
    } else {
	c_strcpy (path, dirname);
    }
    dir_len = c_strlen (path);

    buf = bf_new ();		/* scope: returned */
    
    /* html-buffer is initialized before.  */
    bf_strcat (buf, "<head><title>" str_directory " "); 
    bf_strcat (buf, path); 
    bf_strcat (buf, "</title></head>\n");
    bf_strcat (buf, "<h1><img src=\"icon-dir.gif\"> " str_directory " "); 
    bf_strcat (buf, path); 
    bf_strcat (buf, "</h1>" str_libhelp_bar "<p>\n"); 
    
    /* we eliminate last '/'  */
    if (dir_len > 0) {
	if (path [dir_len - 1] == '/')
	    path [dir_len - 1] = '\0';
    }

    /* we have to compose the name of the parent directory.  */
    if (path[0] != '\0') {

	int 	len;
	char* 	parent = NULL;
	checked_strdup (parent, path); /* scope local */
	len = c_strlen (parent);

	while (len >= 0) {
	    if (parent[len] == '/') {
		parent[len] = '\0';
		break;
	    }
	    len--;
	}

	/* if we have a parent, we add it.  */
	if (c_strlen (parent) == 0) {
	    c_strcpy (parent, "/");
	}
	bf_strcat (buf, "<img src=\"icon-dir.gif\"><a href=\"");
	bf_strcat (buf, parent);
	bf_strcat (buf, "\">" str_parent "</a><br>\n");
	checked_free (parent);
    }

    /*
     *  we set add_ptr to the end of the path. filenames are copied to
     *  this position to  make whole path names. path is big enough to 
     *  hold the path + the longest filename.
     */
    add_ptr = path; 
    while (*add_ptr != '\0') add_ptr++;
    
    /*
     *  the following for loop writes for each directory html text to 
     *  display and access it. each line is of the form:
     *  <dd><img src="icon-.gif"><a href="path"> filename </a>
     *  if the file is not accessible, we don' provide a hypertext link.
     */
    for (i = 0; i < files_in_use; i++) {

	int ftype;
	
	/* we append the filename to the path.  */
	sprintf (add_ptr, "/%s", files[i].filename);
	ftype = file_type (path);
	
	if (ftype & file_dir) {
	    bf_strcat (buf, "<img src=\"icon-dir.gif\">");
	} else if (ftype & file_executable) {
	    bf_strcat (buf, "<img src=\"icon-executable.gif\">");
	} else if (ftype & file_gif) {
	    bf_strcat (buf, "<img src=\"icon-gif.gif\">");
	} else if (ftype & file_html) {
	    bf_strcat (buf, "<img src=\"icon-html.gif\">");
	} else if (ftype & file_plain) {
	    bf_strcat (buf, "<img src=\"icon-plain.gif\">");
	} else {
	    bf_strcat (buf, "<img src=\"icon-other.gif\">");
	}

	if (!(ftype & file_no_access)) {
	    bf_strcat (buf, "<a href=\"");
	    bf_strcat (buf, path); bf_strcat (buf, "\">");
	}
	bf_strcat (buf, files[i].filename);

	if (!(ftype & file_no_access)) {
	    bf_strcat (buf, "</a>");
	} 
	bf_strcat (buf, "<br>\n");
    }
	
    /* clear the files structure.  */
    /* note: files itself is reused.  */
    for (i = 0; i < files_in_use; i++) {
	if (files[i].filename) {
	    checked_free (files[i].filename);
	}
    }

    checked_free (path);
    
    return buf;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	add plaintext
**
**  purpose:		returns a plaintext header in the current buffer
** -------------------------------------------------------------------- *
**  return type:	buffer_t* (newly allocated).
** -------------------------------------------------------------------- */
static buffer_t*
add_plaintext (char* filename)
{
    buffer_t* buf = bf_new ();
    
    execute ("add_plaintext");


    bf_strcat (buf, "<title>Textfile: ");
    bf_strcat (buf, filename);
    bf_strcat (buf, "</title><h1><img src=\"icon-plain.gif\"> "
	       "Textfile ");
    bf_strcat (buf, filename);
    bf_strcat (buf, "</h1>\n" str_libhelp_bar "\n" PLAINTEXT);

    return buf;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	format_gif
**
**  purpose:		returns a html text containing a gif image
** -------------------------------------------------------------------- *
**  args:		name of the gif file
**  return type:	buffer_t*
** -------------------------------------------------------------------- */
static buffer_t*
format_gif (/* i  */ char* giffile)
{
    buffer_t* buf = bf_new ();
    
    execute ("format_gif");
    
    bf_strcat (buf,"<head><title>" str_image " ");
    bf_strcat (buf,giffile);
    bf_strcat (buf,"</title></head>\n<h1>" str_image "</h1><p><code>");
    bf_strcat (buf,giffile); 
    bf_strcat (buf,"<p><img src=\"");
    bf_strcat (buf,giffile); 
    bf_strcat (buf,"\">\n");

    return buf;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	format_binary
**
**  purpose:		returns a html text with a warning about binary 
**			files.
** -------------------------------------------------------------------- *
**  args:		name of the binary file
**  return type:	buffer_t*
** -------------------------------------------------------------------- */
static buffer_t*
format_binary (/* i  */ char* binaryfile)
{
    buffer_t* buf = bf_new ();
    
    execute ("format_binary");
    
    bf_strcat (buf, "<head><title>" str_binary);
    bf_strcat (buf, binaryfile);
    bf_strcat (buf, "</title></head>\n"
	       "<h1><img src=\"icon-warning.gif\" align=middle> "
	       str_binary_cannot "</h1>\n");
    bf_strcat (buf, "<p><code>");
    bf_strcat (buf, binaryfile);
    bf_strcat (buf, "</code>\n");

    return buf;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	format_warning
**
**  purpose:		returns a html text with a warning.
** -------------------------------------------------------------------- *
**  args:	       	filename
**  return type:	buffer_t*
** -------------------------------------------------------------------- */
static buffer_t*
format_warning (/* i  */ char* warningfile)
{
    buffer_t* buf = bf_new ();

    execute ("format_warning");
    
    bf_strcat (buf, str_error_text);
    bf_strcat (buf, warningfile);
    bf_strcat (buf, str_error_end);

    return buf;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	compare_files
**
**  purpose:		compares two filenames and returns an integer 
**			less than, equal to, or greater than zero 
**			corresponding to whether its first argument is
**			considered less than, equal to, or greater than 
**			its second argument.
**			
**			used as a compare function for qsort.
** -------------------------------------------------------------------- */
static int
cmp_files (/* i  */ dirfile_t*	file1,
	   /* i  */ dirfile_t*	file2)
{
    execute ("cmp_files");
    return c_strcmp (file1->filename, file2->filename);
}

/* -------------------------------------------------------------------- *
*g  module test:	compile with -DMODULE_TEST
**			e.g. 
**				gcc -o load -DMODULE_TEST load.c
**			
**			must now be linked to buffer.o, util.o, path.o 
**			and bcache.o
**
**  purpose:		loads the specified (argv[1]) anchor and prints
**			the text to stdout.
** -------------------------------------------------------------------- */
#ifdef MODULE_TEST
void main (int argc, char* argv[])
{
    char* help_file;
    buffer_t* html;
    
    /* check the remaining args.  */
    if (argc == 2) {
	help_file = argv[1];
    } else if (argc > 2) {
	fprintf (stderr, "usage: xmhelp [<help-file>]\n");
	exit (1);
    }
    
    html = read_html_file (help_file);
    
    printf (bf_the_buffer (html));
    fflush (stdout);
    return;
}
#endif /* MODULE_TEST */

/* -------------------------------------------------------------------- *
*l  emacs:
**  local variables:
**  mode:		c
**  outline-regexp:	"\*[HGPLT]"
**  gh-language:	"English"
**  comment-column:	32
**  eval:		(outline-minor-mode t)
**  end:
** -------------------------------------------------------------------- */
