/****************************************************************************
 **
 ** Copyright (C) 2002-2004 Frank Hemer.
 **
 **
 **----------------------------------------------------------------------------
 **
 **----------------------------------------------------------------------------
 **
 ** LinCVS is available under two different licenses:
 **
 ** If LinCVS is linked against the GPLed version of Qt 
 ** LinCVS is released under the terms of GPL also.
 **
 ** If LinCVS is linked against a nonGPLed version of Qt 
 ** LinCVS is released under the terms of the 
 ** LinCVS License for non-Unix platforms (LLNU)
 **
 **
 ** LinCVS License for non-Unix platforms (LLNU):
 **
 ** Redistribution and use in binary form, without modification, 
 ** are permitted provided that the following conditions are met:
 **
 ** 1. Redistributions in binary form must reproduce the above copyright
 **    notice, this list of conditions and the following disclaimer in the
 **    documentation and/or other materials provided with the distribution.
 ** 2. It is not permitted to distribute the binary package under a name
 **    different than LinCVS.
 ** 3. The name of the authors may not be used to endorse or promote
 **    products derived from this software without specific prior written
 **    permission.
 ** 4. The source code is the creative property of the authors.
 **    Extensions and development under the terms of the Gnu Public License
 **    are limited to the Unix platform. Any distribution or compilation of 
 **    the source code against libraries licensed other than gpl requires 
 **    the written permission of the authors.
 **
 **
 ** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
 ** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 ** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 ** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
 ** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 ** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
 ** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 ** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 ** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 ** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 ** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **
 **
 **
 ** LinCVS License for Unix platforms:
 **
 ** This program is free software; you can redistribute it and/or modify 
 ** it under the terms of the GNU General Public License as published by 
 ** the Free Software Foundation; either version 2 of the License, or 
 ** (at your  option) any later version.  This program is distributed in 
 ** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
 ** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 ** PARTICULAR PURPOSE.
 **
 ** See the GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software Foundation,
 ** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **
 *****************************************************************************/

#include "config.h"


#include <qapplication.h>
#include <qgroupbox.h>
#include <qlistbox.h>
#include <qpushbutton.h>
#include <qcombobox.h>
#include <qradiobutton.h>
#include <qcheckbox.h>
#include <qfiledialog.h>
#include <qregexp.h>
#include <qdatetimeedit.h>
#include <qtimer.h>
#include <qwhatsthis.h>

#include "globals.h"
#include "CheckoutDialogImpl.h"
#include "getCvsMod.h"
#include "RevisionWidgetImpl.h"

CheckoutDialogImpl::CheckoutDialogImpl( QString dlgCaption,
      const QIconSet &whatsThisIconSet,
      QString* defaultProfile,
      QStringList* ProfileList,
      QStringList* UserList,
      QStringList* ServerList,
      QStringList* RepositoryList,
      QStringList* WorkdirList,
      QStringList* ModuleList,
      QStringList* CheckoutAsList,
      QWidget* parent,
      const char* name,
      bool modal,
      WFlags fl )
//   : CheckoutDialog( LookAndFeel::g_b0AsParent ? 0 : parent, name, modal, fl ),
  : CheckoutDialog( modal ? parent : (LookAndFeel::g_b0AsParent ? 0 : parent),
		    name, modal, fl ),
     m_parent(parent),
     m_defBrowseModButtonName(BrowseModulesButton->name()),
     m_defBrowseModButtonText(BrowseModulesButton->text()),
     m_ProfileList(*ProfileList),
     m_UserList(*UserList),
     m_ServerList(*ServerList),
     m_RepositoryList(*RepositoryList),
     m_WorkdirList(*WorkdirList),
     m_ModuleList(*ModuleList),
     m_CheckoutAsList(*CheckoutAsList),
     m_running(false)
{
   setCaption(dlgCaption);
   m_defaultProfile = defaultProfile;

   m_pWhatsThis->setIconSet(whatsThisIconSet);
   m_pWhatsThis->setMaximumWidth(m_pWhatsThis->height());
  
   //for now this emulates the enum used in ProfilesDialogImpl
   //this should be replaced by a common usage
   m_AccessDict.insert(tr("Password"),&PSERVER);
   m_AccessDict.insert(tr("Remote Shell"),&RSH);
   m_AccessDict.insert(tr("Local/NFS"),&LOCAL);
   m_AccessList.append(tr("Password"));
   m_AccessList.append(tr("Remote Shell"));
   m_AccessList.append(tr("Local/NFS"));

   m_ProfileList.sort();
   m_ProfileList.prepend(tr("No profile selected"));
   m_ProfileBox->insertStringList(m_ProfileList);
   m_UserList.sort();
   m_UserBox->insertStringList(m_UserList);
   m_ServerList.sort();
   m_ServerBox->insertStringList(m_ServerList);
   m_RepositoryList.sort();
   m_RepositoryBox->insertStringList(m_RepositoryList);
   m_AccessBox->insertStringList(m_AccessList);

   insertSortAndSelectFirst(m_BrowseModules,m_ModuleList);
   simplifyStringList(&m_CheckoutAsList);//need only one!
   m_CheckoutAsList.prepend("");//need an empty top entry
   insertSortAndSelectFirst(m_CheckoutAs,m_CheckoutAsList);
   m_checkoutAsDOption = "";
   insertSortAndSelectFirst(m_ImportDir,m_WorkdirList);

   //read environment(CVSROOT) and set default values
   QString connectMethod;
   QString userName;
   QString passwd;
   QString host;
   int port;
   QString rootDir;
   if (analyzeCVSROOT(connectMethod, userName, passwd, host, port, rootDir)) {
      if (!host.isNull() && connectMethod == "pserver") {
	 host += ":" + QString::number(port);
      }
      m_UserBox->setCurrentText(userName);
      m_ServerBox->setCurrentText(host);
      m_RepositoryBox->setCurrentText(rootDir);
      if (connectMethod.startsWith("local") || connectMethod.startsWith("fork")) {
	 m_AccessBox->setCurrentItem(2);
      } else if (connectMethod.startsWith("pserver")) {
	 m_AccessBox->setCurrentItem(0);
      } else {
	 m_AccessBox->setCurrentItem(1);
      }
   } else {
      m_UserBox->setCurrentText("");
      m_ServerBox->setCurrentText("");
      m_RepositoryBox->setCurrentText("");
      m_AccessBox->setCurrentItem(2);
   }

   m_SshPresetList.append(tr("no ssh"));
   m_SshPresetList.append(tr("use ssh"));
   m_SshPresetList.append(tr("use own ssh-agent"));
   m_SshPresetList.append(tr("use running ssh-agent"));
   m_SshPreset->insertStringList(m_SshPresetList);

   m_ExternalRsh->setChecked(true);
   m_Rsh->setText(ExtApps::g_cvsRsh.path);
   accessActivated(m_AccessBox->currentText());

   m_TagList.clear();
   revisionWidgetImpl->init();
   revisionWidgetImpl->hideStickyTagFrame();
   revisionWidgetImpl->hideRevisionFrame();

   m_Readonly->setChecked(!bRWPermission);

   if (!m_defaultProfile->isEmpty()) {
      setDefaultProfile(*m_defaultProfile);
   }

   resetSize();
}

/*  
 *  Destroys the object and frees any allocated resources
 */
CheckoutDialogImpl::~CheckoutDialogImpl()
{
   // no need to delete child widgets, Qt does it all for us
}

int CheckoutDialogImpl::findlistIndex(QStringList *list, const QString item)
{
   int idx;
   idx = 0;
  
   for ( QStringList::Iterator it = list->begin(); it != list->end(); ++it )
      {
	 if ( (*it).stripWhiteSpace()==item.stripWhiteSpace() )
	    {
	       return idx;
	    }
	 idx++;  
      }
   return 0;
}

void CheckoutDialogImpl::insertSortAndSelectFirst(QComboBox * box,QStringList & list) {
   if (list.isEmpty()) return;
   QString first = list.first();
   list.sort();
   box->insertStringList(list);
   box->setCurrentItem(findlistIndex(&list,first));
}

void CheckoutDialogImpl::setDefaultProfile(const QString& s) {

   /*! search item in combobox profile */
   m_ProfileBox->setCurrentItem( findlistIndex(&m_ProfileList, s) );

   /*! set the profile */
   profileActivated(s);
}

void CheckoutDialogImpl::profileActivated(const QString& s) {

   /*! search index in qlist */
   QValueList<CLincvsProfile>::const_iterator it;
   for (it = cvsProfileContentList.begin(); it != cvsProfileContentList.end(); it++) {
      if ((*it).name().stripWhiteSpace() == s.stripWhiteSpace() ) {
         /*! search item in combobox user */
         m_UserBox->setCurrentItem( findlistIndex(&m_UserList, (*it).user()) );
	
         /*! search item in combobox server */
         m_ServerBox->setCurrentItem( findlistIndex(&m_ServerList, (*it).server()) );
	
         /*! search item in combobox repository */
         m_RepositoryBox->setCurrentItem( findlistIndex(&m_RepositoryList, (*it).repository()) );
	
         /*! set item in combobox method */
         m_AccessBox->setCurrentItem( (*it).method());
         accessActivated( m_AccessBox->currentText());

         m_SshPreset->setCurrentItem( (*it).sshClientPreset());
         
         /*! set default profile */
         *m_defaultProfile = s.stripWhiteSpace();
	
         //and go away
         return;
      }
   }
   *m_defaultProfile = "";
   m_UserBox->setCurrentText("");
   m_ServerBox->setCurrentText("");
   m_RepositoryBox->setCurrentText("");
   m_AccessBox->setCurrentItem(0);
   m_SshPreset->setCurrentItem(0);
   accessActivated( m_AccessBox->currentText());
}

void CheckoutDialogImpl::extRshClicked() {
   m_InternalRsh->setChecked(false);
   m_Rsh->setEnabled(true);
   m_SshPreset->setEnabled(true);
}

void CheckoutDialogImpl::intRshClicked() {
   m_ExternalRsh->setChecked(false);
   m_Rsh->setEnabled(false);
   m_SshPreset->setEnabled(false);
}

void CheckoutDialogImpl::accessActivated(const QString& s) {
   if ((*m_AccessDict.find(s))==RSH) {
      RshFrame->show();
   } else {
      RshFrame->hide();
   }

   if ((*m_AccessDict.find(s))==LOCAL) {
      m_UserBox->setEnabled(false);
      m_ServerBox->setEnabled(false);
   } else {
      m_UserBox->setEnabled(true);
      m_ServerBox->setEnabled(true);
   }

   resetSize();
}

void CheckoutDialogImpl::browseDirClicked() {
   QString fn = QFileDialog::getExistingDirectory (m_ImportDir->currentText(), this, NULL,
	 tr("Choose target directory"),
	 true);

   if (!fn.isEmpty ()) {
      m_ImportDir->setCurrentText(fn);
   }
}

void CheckoutDialogImpl::browseModulesClicked() {

   if (m_running) {
    
      QTimer::singleShot(0,m_parent,SLOT(stopCurAction()));
      BrowseModulesButton->setName(m_defBrowseModButtonName);
      BrowseModulesButton->setText(m_defBrowseModButtonText);

   } else {

      QString connectMethod;
      QString userName;
      QString passwd;
      QString host;
      int port;
      QString rootDir;
    
      analyzeCVSROOT(connectMethod, userName, passwd, host, port, rootDir);
      if ( !m_RepositoryBox->currentText().isEmpty()) {
	 rootDir = m_RepositoryBox->currentText();
      }

      if ( (*m_AccessDict.find(m_AccessBox->currentText()))==LOCAL) {
	 getCvsModule *mBrowser = new getCvsModule(((QString)rootDir)+"/", this);
	 connect (mBrowser, SIGNAL (cvsModuleSelected( const QString & )), 
	       SLOT (insertModule(const QString &)));
      
	 mBrowser->resize( 500, 400 );
	 mBrowser->setCaption(tr("CVS Module Browser"));
	 mBrowser->exec();
      } else {
	 BrowseModulesButton->setName("stop action");
	 BrowseModulesButton->setText(tr("Stop"));
	 callCvs("checkout -p -n CVSROOT/modules",CVS_MBROWSEMODULES_CMD);
      }
   }
}

//unfortunatelly the status cmd requires a sandbox, so it's of no use here
void CheckoutDialogImpl::updateProjectTagList() {
   m_TagList.clear();
   QString command = m_BrowseModules->currentText();
   if (command.isEmpty()) return;
   command = "status -v " + command;
   callCvs(command,CVS_GET_TAG_INFO_PROJECT_CMD);
}

void CheckoutDialogImpl::callCvs(QString strCmd, int cmd) {

   QString connMethod;
   if( (*m_AccessDict.find(m_AccessBox->currentText()))==PSERVER) {
      connMethod = "pserver";
   } else if((*m_AccessDict.find(m_AccessBox->currentText()))==RSH) {
      if(m_ExternalRsh->isChecked()) {
	 connMethod = "ext";
      } else {
	 connMethod = "server";
      }
   }
  
   QString dir = "";
   QString server = m_ServerBox->currentText();
   if (server.find(':') == -1) server += ":";
   m_TmpBrowseModulesRoot =
      ":"+connMethod+
      ":"+m_UserBox->currentText()+
      "@"+server+
      m_RepositoryBox->currentText();
  
   int sshAccess = getSshPreset();
   //check ssh access settings
   bUseSsh = false;
   bUseSshAgent = false;
   bUseSshAgentVars = false;
   switch( sshAccess) {
      case USESSH: {
	 bUseSsh = true;
	 break;
      }
      case USESSHAGENT: {
	 bUseSshAgent = true;
	 break;
      }
      case USESSHAGENTVARS: {
	 bUseSshAgentVars = true;
	 break;
      }
   }
  
   QString topModule = QString::null;
   callInteractive( topModule, dir, m_TmpBrowseModulesRoot,
	 strCmd, cmd, ExtApps::g_cvsRsh.path, //additional options of cvsRsh not supported yet
	 false);
}

void CheckoutDialogImpl::resetSize() {
   QTimer::singleShot(0,this,SLOT(reduceHeight()));
}

void CheckoutDialogImpl::reduceHeight() {
   adjustSize();
}

QSize CheckoutDialogImpl::sizeHint () const {
   QSize s = CheckoutDialog::sizeHint();
   s.setWidth(CheckoutDialog::width());
   return s;
}

void CheckoutDialogImpl::accept() {
   CheckoutDialog::accept();
}

QString CheckoutDialogImpl::user() {
   return m_UserBox->currentText().stripWhiteSpace();
}

QString CheckoutDialogImpl::server() {
   return m_ServerBox->currentText().stripWhiteSpace();
}

QString CheckoutDialogImpl::localDir() {
   return m_ImportDir->currentText().stripWhiteSpace();
}

QString CheckoutDialogImpl::module() {
   return m_BrowseModules->currentText().stripWhiteSpace();
}

QString CheckoutDialogImpl::checkoutAs() {
   return m_CheckoutAs->currentText().stripWhiteSpace();
}

QString CheckoutDialogImpl::checkoutAsDOption() {
   return m_checkoutAsDOption;
}

QString CheckoutDialogImpl::repository() {
   return m_RepositoryBox->currentText().replace (QRegExp ("/+$"), "");
}

QString CheckoutDialogImpl::getCvsRsh() {
   return m_Rsh->displayText();
}

int CheckoutDialogImpl::getSshPreset() {
   switch (m_SshPreset->currentItem()) {
      case 0: return NOSSH;
      case 1: return USESSH;
      case 2: return USESSHAGENT;
      case 3: return USESSHAGENTVARS;
      default: return -1;
   }
}

QString CheckoutDialogImpl::getRevTagDate() {
   QString param = QString::null;
   QString txt;
   txt = revisionWidgetImpl->getRevision();
   if (!txt.isNull()) param += "-r " + txt;
   else {
      txt = revisionWidgetImpl->getTag();
      if ( !txt.isEmpty() ) param += "-r " + txt;
      if (!param.isEmpty()) param += " ";
      txt = revisionWidgetImpl->getDateTime();
      if (!txt.isEmpty()) param += "-D \"" + txt + "\"";
   }
   return param;
}

bool CheckoutDialogImpl::getRWPermission() {
   return m_Readonly->isChecked();
}

int CheckoutDialogImpl::mode() {
   return (*m_AccessDict.find(m_AccessBox->currentText()));
}

int CheckoutDialogImpl::rshMode() {
   if(m_ExternalRsh->isChecked()) {
      return RSH_EXT;
   } else {
      return RSH_SERVER;
   }
}

void CheckoutDialogImpl::getAndSetModule() {
   m_BrowseModules->setCurrentText(m_ModuleBrowser->getModule());
   m_checkoutAsDOption = m_ModuleBrowser->getCheckoutAsDOption();
   if (m_CheckoutAs->currentText().isEmpty()) {
      m_CheckoutAs->setCurrentText(m_checkoutAsDOption);
   }
}

void CheckoutDialogImpl::insertModule(const QString& module) {
   m_BrowseModules->setCurrentText(module);
   m_checkoutAsDOption = "";
}

void CheckoutDialogImpl::setEnabled(bool state) {
   m_ProfileBox->setEnabled(state);
   m_UserBox->setEnabled(state);
   m_ServerBox->setEnabled(state);
   m_RepositoryBox->setEnabled(state);
   m_AccessBox->setEnabled(state);
   RshFrame->setEnabled(state);
   RevisionFrame->setEnabled(state);
   ROFrame->setEnabled(state);
   m_pWhatsThis->setEnabled(state);
   OkButton->setEnabled(state);
   CancelButton->setEnabled(state);
   m_ImportDir->setEnabled(state);
   m_BrowseModules->setEnabled(state);
   m_CheckoutAs->setEnabled(state);
   BrowseDirButton->setEnabled(state);
}

void CheckoutDialogImpl::cvsCallStarted() {
   m_running = true;
   setEnabled(FALSE);
   QApplication::setOverrideCursor(Qt::waitCursor);
}

void CheckoutDialogImpl::cvsCallFinished() {
   QApplication::restoreOverrideCursor();
   setEnabled(TRUE);
   m_running = false;
}

void CheckoutDialogImpl::afterCall( int cmd, CvsBuffer* output, bool failed) {
   cvsCallFinished();

   switch (cmd) {
      case CVS_GET_TAG_INFO_PROJECT_CMD: {

	 if (failed) return;

	 QString line;
	 int pos = 0;
	 unsigned int len = (*output).numLines();
	 unsigned int skipLines = outputLineOffset;
	 for (unsigned int i = skipLines; i<len;++i) {//only two lines if no CVSROOT/modules file
	    line = (*output).textLine(i);
	    if (pos == 0) {
	       if (line.find("Existing Tags:") > -1) pos = 1;
	    }
	    else if ( (line.find("No Tags Exist") > -1)
		  || (line.startsWith("cvs "))
		  || (line.startsWith("===================================================================")) ) {
	       pos = 0;
	    } else {
	       QRegExp rx( "(\\b\\S+\\b)(?:\\s*\\()(\\w+):" );
	       rx.search(line);
	       QString txt;
	       if (rx.cap(2).startsWith("branch")) {
		  txt = "B: "+rx.cap(1);
	       } else if (rx.cap(2).startsWith("revision")) {
		  txt = "T: "+rx.cap(1);
	       } else {
		  qDebug("unknown line on GET_TAG_INFO: "+line);
		  continue;
	       }
	       if (m_TagList.find(txt) == m_TagList.end()) {
		  m_TagList.append(txt);
	       }
	    }
	 }
	 emit tagListFetched();
	 return;
      }
      case CVS_MBROWSEMODULES_CMD: {
	 if (failed) {
	    BrowseModulesButton->setName(m_defBrowseModButtonName);
	    BrowseModulesButton->setText(m_defBrowseModButtonText);
	    return;
	 }
	 m_checkoutAsDOption = "";
	 bool fileOk = false;
	 QString line;
	 unsigned int len = (*output).numLines();
	 unsigned int skipLines = outputLineOffset + 6;
	 for (unsigned int i = skipLines; i<len;i++) {//only two lines if no CVSROOT/modules file
	    line = (*output).textLine(i);
	    if ( (line.at(0)!='#') && ( !(line.isEmpty()) ) ) {
	       fileOk = true;
	       break;
	    }
	 }
	 if (!fileOk) {
	    int sshAccess = getSshPreset();
	    //check ssh access settings
	    bUseSsh = false;
	    bUseSshAgent = false;
	    bUseSshAgentVars = false;
	    switch( sshAccess) {
	       case USESSH: {
		  bUseSsh = true;
		  break;
	       }
	       case USESSHAGENT: {
		  bUseSshAgent = true;
		  break;
	       }
	       case USESSHAGENTVARS: {
		  bUseSshAgentVars = true;
		  break;
	       }
	    }
	    QString dir = "";
	    QString files = "history -xAMR -a -l";
	    QString topModule = QString::null;
	    callInteractive( topModule, dir, m_TmpBrowseModulesRoot,
		  files, CVS_HBROWSEMODULES_CMD,
		  ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
		  false);
	    return;
	 } else {
	    BrowseModulesButton->setName(m_defBrowseModButtonName);
	    BrowseModulesButton->setText(m_defBrowseModButtonText);
	    m_ModuleBrowser = new ModuleBrowserImpl(*(m_pWhatsThis->iconSet()),
		  NULL,
		  m_TmpBrowseModulesRoot,
		  m_parent,
		  this,
		  "Browse Modules",
		  true,
		  LookAndFeel::g_modalF | WDestructiveClose);
	
	    connect(m_ModuleBrowser,SIGNAL(OkClicked()),this,SLOT(getAndSetModule()));
	    m_ModuleBrowser->parseModules (output);
	    m_ModuleBrowser->show();
	 }
	 break;
      }
      case CVS_HBROWSEMODULES_CMD: {
	 if (failed) {
	    BrowseModulesButton->setName(m_defBrowseModButtonName);
	    BrowseModulesButton->setText(m_defBrowseModButtonText);
	    return;
	 }
	 BrowseModulesButton->setName(m_defBrowseModButtonName);
	 BrowseModulesButton->setText(m_defBrowseModButtonText);
	 m_checkoutAsDOption = "";
	 m_ModuleBrowser = new ModuleBrowserImpl(*(m_pWhatsThis->iconSet()),
	       NULL,
	       m_TmpBrowseModulesRoot,
	       m_parent,
	       this,
	       "Browse Modules",
	       true,
	       LookAndFeel::g_modalF | WDestructiveClose);
      
	 connect(m_ModuleBrowser,SIGNAL(OkClicked()),this,SLOT(getAndSetModule()));
	 m_ModuleBrowser->parseHistory (output);
	 m_ModuleBrowser->show();
	 break;
      }
   }
}

void CheckoutDialogImpl::enterWhatsThisMode()
{
   QWhatsThis::enterWhatsThisMode();
}
