###############################################################################
#
# The Plone Content Management System is built on the Content
# Management Framework (CMF) and the Zope Application Server.
# Plone is copyright 2000-2005 Plone Foundation et al.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston,
# MA 02111-1307 USA.
#
###############################################################################

import re
from types import MethodType, FunctionType, UnboundMethodType, ClassType
from types import StringType, UnicodeType

from Interface import Interface
from DateTime import DateTime
from Acquisition import aq_base, aq_parent, aq_inner

from UnicodeNormalizer import normalizeUnicode

from Products.CMFCore.utils import getToolByName
from Products.CMFCore.WorkflowCore import WorkflowException
from Products.CMFPlone.PloneTool import PloneTool as BaseTool

from Products.Archetypes.utils import wrap_method
from Products.Archetypes.utils import isWrapperMethod
from Products.PloneLanguageTool.interfaces import ITranslatable
from Products.LinguaPlone.config import log

# This is just so we don't have to deviate the code too much from
# Plone 2.1. Nothing in Plone 2.0 actually implemented IBrowserDefault
class IBrowserDefault(Interface): pass

# Define and compile static regexes
FILENAME_REGEX = re.compile(r"^(.+)\.(\w{,4})$")
NON_WORD_REGEX = re.compile(r"[\W\-]+")
EXTRA_DASHES_REGEX = re.compile(r"(^\-+)|(\-+$)")

_marker = object()

def patch_base(klass):
    attrs = klass.__dict__.items()
    attrs = filter(lambda item: not item[0].startswith('__'), attrs)
    base = klass.__bases__[-1]
    kname = base.__name__
    for name, attr in attrs:
        old_attr = getattr(base, name, None)
        if (old_attr is not None and name in base.__dict__ and
            isinstance(old_attr, (MethodType, FunctionType, UnboundMethodType))):
            if not isWrapperMethod(old_attr):
                wrap_method(base, name, attr, pattern='__linguaplone_%s__')
                log('Overriding %s.%s' % (kname, name))
            else:
                log('Already overriden %s.%s.' % (kname, name))
        else:
            setattr(base, name, attr)
            log('Enabling compatibility method %s.%s' % (kname, name))


def base_hasattr(obj, name):
    """Like safe_hasattr, but also disables acquisition."""
    return safe_hasattr(aq_base(obj), name)

def safe_hasattr(obj, name, _marker=object()):
    """Make sure we don't mask exceptions like hasattr().

    We don't want exceptions other than AttributeError to be masked,
    since that too often masks other programming errors.
    Three-argument getattr() doesn't mask those, so we use that to
    implement our own hasattr() replacement.
    """
    return getattr(obj, name, _marker) is not _marker

def safe_callable(obj):
    """Make sure our callable checks are ConflictError safe."""
    if getattr(obj, '__class__', None) is not None:
        if getattr(obj, '__call__', None) is not None:
            return True
        else:
            return isinstance(obj, ClassType)
    else:
        return callable(obj)

class PloneTool(BaseTool):

    def pretty_title_or_id(self, obj, empty_value=_marker):
        """Return the best possible title or id of an item, regardless
        of whether obj is a catalog brain or an object, but returning an
        empty title marker if the id is not set (i.e. it's auto-generated).
        """
        title = None
        if base_hasattr(obj, 'Title'):
            title = getattr(obj, 'Title', None)
        if safe_callable(title):
            title = title()
        if title:
            return title
        item_id = getattr(obj, 'getId', None)
        if safe_callable(item_id):
            item_id = item_id()
        if item_id and not self.isIDAutoGenerated(item_id):
            return item_id
        if empty_value is _marker:
            empty_value = self.getEmptyTitle()
        return empty_value

    def isIDAutoGenerated(self, id):
        """Determine if an id is autogenerated"""
        autogenerated = False

        # In 2.1 non-autogenerated is the common case, caught exceptions are
        # expensive, so let's make a cheap check first
        if id.count('.') != 2:
            return autogenerated

        try:
            pt = getToolByName(self, 'portal_types')
            obj_type, date_created, random_number = id.split('.')
            type = ' '.join(obj_type.split('_'))
            portaltypes = pt.objectIds()
            # New autogenerated ids may have a lower case portal type
            if (type in portaltypes or \
               type in [pt.lower() for pt in portaltypes]) \
               and DateTime(date_created) \
               and float(random_number):
                autogenerated = True
        except (ValueError, AttributeError, IndexError, DateTime.DateTimeError):
            pass

        return autogenerated

    def getEmptyTitle(self, translated=True):
        """Returns string to be used for objects with no title or id"""
        empty = self.utf8_portal('\x5b\xc2\xb7\xc2\xb7\xc2\xb7\x5d', 'ignore')
        if translated:
            trans = getToolByName(self, 'translation_service', None)
            if trans is not None:
                empty = trans.utranslate(domain='plone', msgid='title_unset',
                                         default=empty)
        return empty

    def utf8_portal(self, str, errors='strict'):
        """Transforms an utf8 string to portal encoding."""
        charset = self.getSiteEncoding()
        if charset.lower() in ('utf-8', 'utf8'):
            # Test
            unicode(str, 'utf-8', errors)
            return str
        else:
            return unicode(str, 'utf-8', errors).encode(charset, errors)

    def portal_utf8(self, str, errors='strict'):
        """Transforms an string in portal encoding to utf8."""
        charset = self.getSiteEncoding()
        if charset.lower() in ('utf-8', 'utf8'):
            # Test
            unicode(str, 'utf-8', errors)
            return str
        else:
            return unicode(str, charset, errors).encode('utf-8', errors)

    def getSiteEncoding(self):
        """Get the default_charset or fallback to utf8."""
        pprop = getToolByName(self, 'portal_properties', None)
        default = 'utf-8'
        try:
            charset = pprop.site_properties.getProperty('default_charset', default)
        except AttributeError:
            charset = default
        return charset

    def _addToNavTreeResult(self, result, data):
        """Adds a piece of content to the result tree."""
        path = data['path']
        parentpath = '/'.join(path.split('/')[:-1])
        # Tell parent about self
        if result.has_key(parentpath):
            result[parentpath]['children'].append(data)
        else:
            result[parentpath] = {'children':[data]}
        # If we have processed a child already, make sure we register it
        # as a child
        if result.has_key(path):
            data['children'] = result[path]['children']
        result[path] = data

    def createNavTree(self, context, sitemap=None):
        """Returns a structure that can be used by navigation_tree_slot."""
        ct = getToolByName(self, 'portal_catalog')
        ntp = getToolByName(self, 'portal_properties').navtree_properties
        stp = getToolByName(self, 'portal_properties').site_properties
        view_action_types = stp.getProperty('typesUseViewActionInListings', ())
        currentPath = None

        custom_query = getattr(self, 'getCustomNavQuery', None)
        if custom_query is not None and safe_callable(custom_query):
            query = custom_query()
        else:
            query = {}

        # XXX check if isDefaultPage is in the catalogs
        #query['isDefaultPage'] = 0

        if context == self or sitemap:
            currentPath = getToolByName(self, 'portal_url').getPortalPath()
            query['path'] = {'query':currentPath,
                             'depth':ntp.getProperty('sitemapDepth', 2)}
        else:
            currentPath = '/'.join(context.getPhysicalPath())
            query['path'] = {'query':currentPath, 'navtree':1}

        query['portal_type'] = self.typesToList()

        if ntp.getProperty('sortAttribute', False):
            query['sort_on'] = ntp.sortAttribute

        if (ntp.getProperty('sortAttribute', False) and
            ntp.getProperty('sortOrder', False)):
            query['sort_order'] = ntp.sortOrder

        if ntp.getProperty('enable_wf_state_filtering', False):
            query['review_state'] = ntp.wf_states_to_show

        query['is_default_page'] = False

        parentTypesNQ = ntp.getProperty('parentMetaTypesNotToQuery', ())

        # Get ids not to list and make a dict to make the search fast
        ids_not_to_list = ntp.getProperty('idsNotToList', ())
        excluded_ids = {}
        for exc_id in ids_not_to_list:
            excluded_ids[exc_id] = 1

        rawresult = ct(**query)

        # Build result dict
        result = {}
        foundcurrent = False
        for item in rawresult:
            path = item.getPath()
            # Some types may require the 'view' action, respect this
            item_url = (item.portal_type in view_action_types and
                        item.getURL() + '/view') or item.getURL()
            currentItem = path == currentPath
            if currentItem:
                foundcurrent = path
            no_display = (excluded_ids.has_key(item.getId) or
                          not not getattr(item, 'exclude_from_nav', False))
            # XXX This is a workaround for the lack of 'is_folderish'
            # metadata in Plone 2.0
            show_children = (
                (getattr(item, 'is_folderish', False) or
                 'Folder' in item.meta_type or
                 'Container' in item.meta_type) and
                item.portal_type not in parentTypesNQ)
            data = {'Title':self.pretty_title_or_id(item),
                    'currentItem':currentItem,
                    'absolute_url': item_url,
                    'getURL':item_url,
                    'path': path,
                    'icon':item.getIcon,
                    'creation_date': item.CreationDate,
                    'portal_type': item.portal_type,
                    'review_state': item.review_state,
                    'Description':item.Description,
                    'show_children': show_children,
                    'children':[],
                    'no_display': no_display}
            self._addToNavTreeResult(result, data)

        portalpath = getToolByName(self, 'portal_url').getPortalPath()

        if ntp.getProperty('showAllParents', False):
            portal = getToolByName(self, 'portal_url').getPortalObject()
            parent = context
            parents = [parent]
            while not parent is portal:
                parent = parent.aq_parent
                parents.append(parent)

            wf_tool = getToolByName(self, 'portal_workflow')
            for item in parents:
                if getattr(item, 'getPhysicalPath', None) is None:
                    # when Z3-style views are used, the view class will be in
                    # the 'parents' list, but will not support 'getPhysicalPath'
                    # we can just skip it b/c it's not an object in the content
                    # tree that should be showing up in the nav tree (ra)
                    continue
                path = '/'.join(item.getPhysicalPath())
                if not result.has_key(path) or \
                   not result[path].has_key('path'):
                    # item was not returned in catalog search
                    if foundcurrent:
                        currentItem = False
                    else:
                        currentItem = path == currentPath
                        if currentItem:
                            if self.isDefaultPage(item):
                                # don't list folder default page
                                continue
                            else:
                                foundcurrent = path
                    try:
                        review_state = wf_tool.getInfoFor(item, 'review_state')
                    except WorkflowException:
                        review_state = ''
                    # Some types may require the 'view' action, respect this
                    item_url = (item.portal_type in view_action_types and
                         item.absolute_url() + '/view') or item.absolute_url()
                    data = {'Title': self.pretty_title_or_id(item),
                            'currentItem': currentItem,
                            'absolute_url': item_url,
                            'getURL': item_url,
                            'path': path,
                            'icon': item.getIcon(),
                            'creation_date': item.CreationDate(),
                            'review_state': review_state,
                            'Description':item.Description(),
                            'children':[],
                            'show_children': False,
                            'portal_type':item.portal_type,
                            'no_display': 0}
                    self._addToNavTreeResult(result, data)

        if not foundcurrent:
            #    result['/'.join(currentPath.split('/')[:-1])]['currentItem'] = True
            for i in range(1, len(currentPath.split('/')) - len(portalpath.split('/')) + 1):
                p = '/'.join(currentPath.split('/')[:-i])
                if result.has_key(p):
                    foundcurrent = p
                    result[p]['currentItem'] = True
                    break

        if result.has_key(portalpath):
            return result[portalpath]
        else:
            return {}

    def isDefaultPage(self, obj):
        """Finds out if the given obj is the default page in its parent folder.

        Only considers explicitly contained objects, either set as index_html,
        with the default_page property, or using IBrowserDefault.
        """

        parent = aq_parent(aq_inner(obj))
        if not parent:
            return False

        parentDefaultPage = self.getDefaultPage(parent)
        if parentDefaultPage is None or '/' in parentDefaultPage:
            return False
        else:
            return (parentDefaultPage == obj.getId())

    def getDefaultPage(self, obj):
        """Given a folderish item, find out if it has a default-page using
        the following lookup rules:

            1. A content object called 'index_html' wins
            2. If the folder implements IBrowserDefault, query this
            3. Else, look up the property default_page on the object
                - Note that in this case, the returned id may *not* be of an
                  object in the folder, since it could be acquired from a
                  parent folder or skin layer
            4. Else, look up the property default_page in site_properties for
                magic ids and test these

        The id of the first matching item is then used to lookup a translation
        and if found, its id is returned. If no default page is set, None is
        returned. If a non-folderish item is passed in, return None always.
        """

        def lookupTranslationId(obj, page):
            implemented = ITranslatable.isImplementedBy(obj)
            if not implemented or implemented and not obj.isTranslation():
                pageobj = getattr(obj, page, None)
                if pageobj is not None and ITranslatable.isImplementedBy(pageobj):
                    translation = pageobj.getTranslation()
                    if translation is not None and \
                       ids.has_key(translation.getId()):
                        page = translation.getId()
            return page

        # Short circuit if we are not looking at a Folder
        if not obj.isPrincipiaFolderish:
            return None

        # The list of ids where we look for default
        ids = {}

        portal = getToolByName(self, 'portal_url').getPortalObject()
        wftool = getToolByName(self, 'portal_workflow')

        # For BTreeFolders we just use the has_key, otherwise build a dict
        if hasattr(aq_base(obj), 'has_key'):
            ids = obj
        else:
            for id in obj.objectIds():
                ids[id] = 1

        # 1. test for contentish index_html
        if ids.has_key('index_html'):
            return lookupTranslationId(obj, 'index_html')

        # 2. Test for IBrowserDefault
        if IBrowserDefault.isImplementedBy(obj):
            fti = obj.getTypeInfo()
            if fti is not None:
                page = fti.getDefaultPage(obj, check_exists=True)
                if page is not None:
                    return lookupTranslationId(obj, page)

        # 3. Test for default_page property in folder, then skins
        pages = getattr(aq_base(obj), 'default_page', [])
        if type(pages) in (StringType, UnicodeType):
            pages = [pages]
        for page in pages:
            if page and ids.has_key(page):
                return lookupTranslationId(obj, page)
        for page in pages:
            if portal.unrestrictedTraverse(page, None):
                return lookupTranslationId(obj, page)

        # 4. Test for default sitewide default_page setting
        site_properties = portal.portal_properties.site_properties
        for page in site_properties.getProperty('default_page', []):
            if ids.has_key(page):
                return lookupTranslationId(obj, page)

        return None

    def typesToList(self):
        ntp = getToolByName(self, 'portal_properties').navtree_properties
        ttool = getToolByName(self, 'portal_types')
        bl = ntp.getProperty('metaTypesNotToList', ())
        bl_dict = {}
        for t in bl:
            bl_dict[t] = 1
        all_types = ttool.listContentTypes()
        wl = [t for t in all_types if not bl_dict.has_key(t)]
        return wl

    def normalizeString(self, text):
        """Normalizes a title to an id.

        normalizeString() converts a whole string to a normalized form that
        should be safe to use as in a url, as a css id, etc.

        all punctuation and spacing is removed and replaced with a '-':

        >>> normalizeString("a string with spaces")
        'a-string-with-spaces'

        >>> normalizeString("p.u,n;c(t)u!a@t#i$o%n")
        'p-u-n-c-t-u-a-t-i-o-n'

        strings are lowercased:

        >>> normalizeString("UppERcaSE")
        'uppercase'

        punctuation, spaces, etc. are trimmed and multiples are reduced to just
        one:

        >>> normalizeString(" a string    ")
        'a-string'

        >>> normalizeString(">here's another!")
        'here-s-another'

        >>> normalizeString("one with !@#$!@#$ stuff in the middle")
        'one-with-stuff-in-the-middle'

        the exception to all this is that if there is something that looks like a
        filename with an extension at the end, it will preserve the last period.

        >>> normalizeString("this is a file.gif")
        'this-is-a-file.gif'

        >>> normalizeString("this is. also. a file.html")
        'this-is-also-a-file.html'

        normalizeString() uses normalizeUnicode() to convert stray unicode
        characters. it will attempt to transliterate many of the accented
        letters to rough ASCII equivalents:

        >>> normalizeString(u"Eksempel \xe6\xf8\xe5 norsk \xc6\xd8\xc5")
        'eksempel-eoa-norsk-eoa'

        for characters that we can't transliterate, we just return the hex codes of
        the byte(s) in the character. not pretty, but about the best we can do.

        >>> normalizeString(u"\u9ad8\u8054\u5408 Chinese")
        '9ad880545408-chinese'

        >>> normalizeString(u"\uc774\ubbf8\uc9f1 Korean")
        'c774bbf8c9f1-korean'
        """
        # Make sure we are dealing with a stringish type
        if not isinstance(text, basestring):
            # Catch the special None case or we would return 'none' evaluating
            # to True, which is totally unexpected
            # XXX This seems to break the autogenerated ids, reverting
            #if text is None:
            #    return None

            # This most surely ends up in something the user does not expect
            # to see. But at least it does not break.
            text = repr(text)

        # Make sure we are dealing with a unicode string
        if not isinstance(text, unicode):
            text = unicode(text, self.getSiteEncoding())

        text = text.strip()
        text = text.lower()
        text = normalizeUnicode(text)

        base = text
        ext  = ""

        m = FILENAME_REGEX.match(text)
        if m is not None:
            base = m.groups()[0]
            ext  = m.groups()[1]

        base = NON_WORD_REGEX.sub("-", base)
        base = EXTRA_DASHES_REGEX.sub("", base)

        if ext != "":
            base = base + "." + ext
        return base

patch_base(PloneTool)
