##############################################################################
#                                                                            #
#               copyright (c) 2003 ITB, Humboldt-University Berlin           #
#               written by: Raphael Ritz, r.ritz@biologie.hu-berlin.de       #
#                                                                            #
##############################################################################

"""BibliographyRenderer main class"""

# Zope stuff
from Interface import Interface
from Globals import InitializeClass
from AccessControl import ClassSecurityInfo
from OFS.Folder import Folder
from OFS.SimpleItem import SimpleItem
import Products


class IBibliographyRenderer(Interface):
    """
    Interface for the output/export
    renderers of the bibliography tool.
    """
    def getFormatName():
        """
        returns the name of the format
        """

    def getFormatExtension():
        """
        returns the filename extension of the format
        """

    def render(object):
        """
        returns the rendered object
        object may be a bibliography folder or a single bibliography entry
        """

class BibliographyRenderer(SimpleItem):
    """
    Base class for the input parser of the bibliography tool.
    """
    __implements__ = (IBibliographyRenderer ,)

    meta_type = 'Bibliography Renderer'

    format = {'name':'name of the format',
              'extension':'typical filename extension'
              }
    
    security = ClassSecurityInfo()

    def __init__(self, id, title=''):
        """
        minimal initialization
        """
        self.id = id
        self.title = title
        self.format = format

    def getFormatName(self):
        """ returns the name of the format """
        return self.format.get('name', 'No name specified')

    def getFormatExtension(self):
        """ returns the filename extension of the format """
        return self.format.get('extension', 'no extension specified')

    def render(self, object):
        """
        renders a object
        object may be a bibliography folder or a single bibliography item
        """
        pass   #  needs to be provided by the individual renderer

 # Class instanciation
InitializeClass(BibliographyRenderer)


class RendererFolder(Folder):
    """
    A folder that only offers to add objects that implement the
    IBibliographyRenderer interface. 
    """
    meta_type = 'Renderer Folder'

    id = 'Renderers'
    title = "BibliographyTool's renderer folder"

    # we don't want 'View'
    manage_options = ( Folder.manage_options[0], ) \
                     + Folder.manage_options[2:]
    index_html = None

    def __init__(self, id, title=''):
        """
        minimal initialization
        """
        self.id = id
        # self.title = title
    
    def all_meta_types(self):
        product_infos = Products.meta_types
        possibles = []
        for p in product_infos:
            try:
                if IBibliographyRenderer in p.get('interfaces', []):
                    possibles.append(p)
            except TypeError:
                pass
        definites = map(lambda x: x.meta_type, self.objectValues())
        return filter(lambda x,y=definites: x['name'] not in y, possibles)

InitializeClass(RendererFolder) 
   
