/*
 * GtkZclock - A simple but powerful clock widget for GTK/GNOME
 * Copyright (C) 2000  Zack Hobson <zack@malfunktion.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <config.h>
#include <gtk/gtk.h>
#include <time.h>

#include "gtk_zclock.h"

#define GTK_ZCLOCK_STRFTIME_MAX		64

static GtkFrameClass *parent_class;

enum {
	ARG_0,		/* Ignore */
	ARG_FORMAT	/* Display format */
};

static void gtk_zclock_class_init (GtkZclockClass *);
static void gtk_zclock_init (GtkZclock *);
static void gtk_zclock_destroy (GtkObject *);
static gint gtk_zclock_update (gpointer);

guint
gtk_zclock_get_type (void)
{
	static guint zclock_type = 0;

	if (!zclock_type)
	{
		static const GtkTypeInfo zclock_info =
		{
			"GtkZclock",
			sizeof (GtkZclock),
			sizeof (GtkZclockClass),
			(GtkClassInitFunc) gtk_zclock_class_init,
			(GtkObjectInitFunc) gtk_zclock_init,
			(GtkArgSetFunc) gtk_zclock_set_arg,
			(GtkArgGetFunc) gtk_zclock_get_arg,
		};

		zclock_type = gtk_type_unique(gtk_frame_get_type(), &zclock_info);
	}

	return zclock_type;
}

static void
gtk_zclock_class_init (GtkZclockClass *class)
{
	GtkObjectClass		*object_class;
	GtkWidgetClass		*widget_class;
	GtkContainerClass	*container_class;

	gtk_object_add_arg_type("GtkZclock::format",
			GTK_TYPE_STRING,
			GTK_ARG_READWRITE,
			ARG_FORMAT);

	object_class	= (GtkObjectClass *) class;
	widget_class	= (GtkWidgetClass *) class;
	container_class	= (GtkContainerClass *) class;

	parent_class = gtk_type_class(gtk_frame_get_type());

	object_class->set_arg = gtk_zclock_get_arg;
	object_class->get_arg = gtk_zclock_set_arg;

	object_class->destroy = gtk_zclock_destroy;
}

static void
gtk_zclock_init (GtkZclock *zclock)
{
	g_assert(zclock != NULL);

	zclock->format	= NULL;
	zclock->timeout = 0;
	zclock->timeout_interval = 1000; /* 1 second */
}

GtkWidget*
gtk_zclock_new (const gchar *format)
{
	GtkZclock *zclock = gtk_type_new(gtk_zclock_get_type());

	g_return_val_if_fail(format != NULL, NULL);

	/* Set my frame properties */
	gtk_frame_set_label(GTK_FRAME(zclock), NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(zclock), GTK_SHADOW_NONE);

	/* Create and add my label */
	zclock->label = gtk_label_new("GtkZclock");
	gtk_container_add(GTK_CONTAINER(zclock), zclock->label);
	gtk_widget_show(zclock->label);

	/* Set the format string. This also updates the time display */
	gtk_zclock_set_format(zclock, format);

	/* Set up a timeout to continue updating */
	zclock->timeout = gtk_timeout_add(zclock->timeout_interval,
			gtk_zclock_update, zclock);

	return GTK_WIDGET(zclock);
}

static void 
gtk_zclock_destroy (GtkObject *object)
{
	GtkZclock *zclock;

	g_return_if_fail(object != NULL);
	g_return_if_fail(GTK_IS_ZCLOCK(object));

	zclock = GTK_ZCLOCK(object);

	/* remove the timeout first, to avoid a race condition 
	 * with the format */
	if (zclock->timeout)
		gtk_timeout_remove(zclock->timeout);

	/* free and re-initialize the format */
	g_free(zclock->format);
	zclock->format = NULL;

	/* Reset the timeout data, just to be thorough */
	zclock->timeout = 0;
	zclock->timeout_interval = 1000; /* 1 second */

	/* Chain up */
	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		(* GTK_OBJECT_CLASS(parent_class)->destroy) (object);
}

static gint
gtk_zclock_update (gpointer data)
{
	GtkZclock	*zclock;
	char		buf[GTK_ZCLOCK_STRFTIME_MAX] = "";
	time_t		tmp = 0;
	struct tm	*now;
	gint		i;
	gint		max;

	g_assert(data != NULL);
	g_assert(GTK_IS_ZCLOCK(data));

	zclock = GTK_ZCLOCK(data);

	g_assert(zclock->format != NULL);
	g_assert(zclock->label != NULL);

	tmp = time(NULL);
	now = localtime(&tmp);
	if (strftime(buf, GTK_ZCLOCK_STRFTIME_MAX, zclock->format, now) == GTK_ZCLOCK_STRFTIME_MAX)
		buf[GTK_ZCLOCK_STRFTIME_MAX - 1] = '\0';

	/* Remove excess whitespace...
	 * I doubt this is as efficient as it could be.
	 * If anyone has any suggestions to fix that, I'd
	 * love to hear about it.
	 */
	g_strstrip(buf);
	max = strlen(buf);
	for (i = 1; i < max; i++)
	{
		if ((buf[i] == ' ' || buf[i] == '\n') && buf[i + 1] == ' ')
			g_strchug(&(buf[i+1]));
	}

	gtk_label_set_text(GTK_LABEL(zclock->label), buf);

	return TRUE;
}

void
gtk_zclock_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GtkZclock	*zclock;

	g_return_if_fail(object != NULL);
	g_return_if_fail(GTK_IS_ZCLOCK(object));

	zclock = GTK_ZCLOCK(object);

	switch (arg_id)
	{
		case ARG_FORMAT:
			GTK_VALUE_STRING (*arg) = zclock->format;
			break;
		default:
			arg->type = GTK_TYPE_INVALID;
			break;
	}
}

void
gtk_zclock_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GtkZclock *zclock;

	g_return_if_fail(object != NULL);
	g_return_if_fail(GTK_IS_ZCLOCK(object));

	zclock = GTK_ZCLOCK(object);

	switch (arg_id)
	{
		case ARG_FORMAT:
			gtk_zclock_set_format(zclock, GTK_VALUE_STRING (*arg));
			break;
		default:
			/* Invalid enumeration value */
			g_assert_not_reached();
			break;
	}
}

void
gtk_zclock_set_format(GtkZclock *zclock, const gchar *format)
{
	gboolean replace_timeout = FALSE;
	gint i;

	g_return_if_fail(zclock != NULL);
	g_return_if_fail(format != NULL);

	/* if there's a timeout, remove it before changing
	 * the format, to avoid a race condition */
	if (zclock->timeout) {
		gtk_timeout_remove(zclock->timeout);
		zclock->timeout = 0;
		replace_timeout = TRUE;

		/* sanity check: if a timeout was active, there *must*
		 * be a format! */
		g_assert(zclock->format != NULL);
	}

	/* free the existing format, if any */
	if (zclock->format != NULL)
		g_free(zclock->format);

	/* Set the new format, and update immediately */
	zclock->format = g_strdup(format);
	gtk_zclock_update(zclock);

	/*
	 * Calculate the new timeout interval based on the format.
	 * It works like this: Assuming a default of one minute,
	 * reduce the interval to one second if any of the strftime 
	 * formats that might include seconds are found. The formats
	 * chosen are based on the POSIX.2 standard, plus 's', which
	 * (in GNU libc) prints Unix time.
	 */
	zclock->timeout_interval = 60000; /* default to 1 minute */
	for(i = 0; i < strlen(zclock->format); i++) {
		gchar c = (zclock->format)[i+1];
		if ((zclock->format[i] == '%') &&
			(c=='c' || c=='r' || c=='S' || c=='T' || c=='X' || c=='s'))
		{
			/* This format has second granularity */
			zclock->timeout_interval = 1000;
			break;
		}
	}

	/* if an existing timout was removed, replace it */
	if (replace_timeout)
		zclock->timeout = gtk_timeout_add(zclock->timeout_interval,
			gtk_zclock_update,
			zclock);
}

