#include <Python.h>

#include <dirent.h>
#include <sys/stat.h>

#include <uc++/plugin.h>

#include "python-loader.h"
#include "python-script.h"

using namespace SigC;

namespace uC
{

namespace {

void add_paths(PluginManager *mgr)
{
  PyObject *oldpath, *mypath, *newpath, *sysmodule;
  const std::list<std::string>& indep_paths = mgr->arch_indep_paths();
  std::list<std::string>::const_iterator it;
  
  // We add the uC++ arch indep paths to sys.path
  
  sysmodule = PyImport_ImportModule("sys");
  if (!sysmodule)
    throw 0;

  if ((oldpath = PyObject_GetAttrString(sysmodule, "path")) == 0)
    throw 0;
  
  if ((mypath = PyList_New(0)) == 0)
    throw 0;
  
  for (it = indep_paths.begin(); it != indep_paths.end(); ++it)
    PyList_Append(mypath, PyString_FromString((*it + "/python").c_str()));
  
  if ((newpath = PySequence_Concat(mypath, oldpath)) == 0)
  {
    Py_DECREF(oldpath);
    Py_DECREF(mypath);
    throw 0;
  }
  Py_DECREF(oldpath);
  Py_DECREF(mypath);
  
  PyObject_SetAttrString(sysmodule, "path", newpath);
  
  Py_DECREF(newpath);
}

Plugin *load_module(PluginManager  *mgr, PluginLoader *instance, 
                    std::string name)
{
  PyObject *module = PyImport_ImportModule((char *)name.c_str());
  Plugin *plugin = 0;
  
  if (module)
  {
    PyObject *args = 0;
    PyObject *init_func = 0;
    PyObject *result = 0;
    PyObject *pluginmodule = 0;
    PyObject *mgr_constr = 0;
    PyObject *ptr_str = 0;
    PyObject *py_mgr = 0;
    char address[64];
    
    init_func = PyObject_GetAttrString(module,
                                       (char *)(name + "Plugin").c_str());
    
    if (init_func && PyCallable_Check(init_func))
    {
      args = PyTuple_New(1);
      if (!args ||
          !(pluginmodule = PyImport_ImportModule("ucxx")) ||
          !(mgr_constr = PyObject_GetAttrString(pluginmodule,
                                                "PluginManager")))
      {
        instance->emit_error("something wrong with ucxx module");
        goto cleanup;
      }
      
      sprintf(address, "%p", mgr);
      ptr_str = PyString_FromString(address);
      PyTuple_SetItem(args, 0, ptr_str);
      if (!(py_mgr = PyObject_CallObject(mgr_constr, args)))
      {
        instance->emit_error("something wrong with ucxx module");
        goto cleanup;
      }
      PyTuple_SetItem(args, 0, py_mgr);
      result = PyObject_CallObject(init_func, args);
      if (!result)
        goto cleanup;
      ptr_str = PyObject_GetAttrString(result, "__ucxxinstance__");
      if (!ptr_str || !PyString_Check(ptr_str) ||
          sscanf(PyString_AsString(ptr_str), "%p", &plugin) != 1)
        goto cleanup;
    }
 cleanup:
    //Py_XDECREF(ptr_str);
    Py_XDECREF(pluginmodule);
    Py_XDECREF(mgr_constr);
    Py_XDECREF(init_func);
    Py_XDECREF(args);
  }
  else
  {
    PyObject *type, *value, *tracebk, *errstr;
    
    PyErr_Fetch(&type, &value, &tracebk);
    PyErr_NormalizeException(&type, &value, &tracebk);

    if (type && PyErr_GivenExceptionMatches(type, PyExc_Exception) &&
        value != 0 && ((errstr = PyObject_Str(value)) != 0))

    {
      mgr->set_error(PyString_AsString(errstr));
    }
    
    Py_XDECREF(type);
    Py_XDECREF(value);
    Py_XDECREF(tracebk);
  }
  return plugin;
}

}

pythonPluginLoader::pythonPluginLoader(PluginManager *mgr)
    : PluginLoader(mgr), Plugin(mgr)
{
  Script::Language *lang = mgr->language("python");
  
  if (lang)
    pack(slot(add_paths), mgr)->tunnel(lang->tunnel());
}

pythonPluginLoader::~pythonPluginLoader()
{
}


Plugin *pythonPluginLoader::load(PluginManager *mgr, const std::string& name)
{
  Plugin *plugin = 0;
  
  // remove "python." prefix
  if (name.substr(0, 7) != "python.")
    return 0;
  std::string realname = name.substr(7);

  Script::Language *lang = mgr->language("python");
  
  if (lang)
    plugin = pack(slot(load_module), mgr, (PluginLoader *)this, 
		  realname)->tunnel(lang->tunnel(), true);
  return(plugin);
}

namespace
{

void do_scan(PluginManager *mgr, const std::string& path, 
             const std::string& basename)
{
  DIR *dir;
  struct dirent *file;

  if ((dir = opendir(path.c_str())) != NULL)
  {
    while ((file = readdir(dir)) != NULL)
    {
      int len;
      std::string fullname = path + file->d_name;
      struct stat statinfo;
      
      if (file->d_name[0] != '.' &&
          stat(fullname.c_str(), &statinfo) == 0 && S_ISDIR(statinfo.st_mode))
        do_scan(mgr,
                path + file->d_name + "/", basename + file->d_name + ".");
      
      len = strlen(file->d_name);
      if (len > 3 && strcmp(file->d_name + len - 3, ".py") == 0)
        mgr->plugin_found(basename + 
                          std::string(file->d_name).substr(0, len - 3));
      else if (len > 4 && strcmp(file->d_name + len - 4, ".pyc") == 0)
        mgr->plugin_found(basename + 
                          std::string(file->d_name).substr(0, len - 4));
    }
  }
}

}

void pythonPluginLoader::scan(PluginManager *mgr) const
{
  const std::list<std::string>& paths = mgr->arch_indep_paths();
  std::list<std::string>::const_iterator it;

  for (it = paths.begin(); it != paths.end(); ++it)
    do_scan(mgr, (*it) + "/python/", "python.");
}

}
