/*  xxXTideRoot  XTide "root" window
    Last modified 1999-03-15

    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "xtide.hh"

static void
dismissCallback (Widget w, XtPointer client_data, XtPointer call_data) {
  ((xxXTideRoot*)client_data)->hide();
}

static void
applyCallback (Widget w, XtPointer client_data, XtPointer call_data) {
  ((xxXTideRoot*)client_data)->apply();
}

// This code is pretty much duplicated from xxAspect
int
xxXTideRoot::sancheckdouble (char *d) {
  Dstr num (d);
  if (num.length() > 0) {
    if (num.strchr ('\n') != -1 ||
           num.strchr ('\r') != -1 ||
           num.strchr (' ') != -1) {
      Dstr details ("Numbers aren't supposed to contain whitespace.  You entered '");
      details += num;
      details += "'.";
      barf (NOT_A_NUMBER, details, 0);
      return 0;
    } else {
      double temp;
      if (sscanf (num.aschar(), "%lf", &temp) != 1) {
	Dstr details ("The offending input was '");
	details += num;
	details += "'.";
	barf (NOT_A_NUMBER, details, 0);
        return 0;
      } else {
        if (temp <= 0.0) {
  	  Dstr details ("The offending input was '");
	  details += num;
	  details += "'.";
	  barf (NUMBER_RANGE_ERROR, details, 0);
          return 0;
        } else
  	  return 1;
      }
    }
  }
  return 1;
}

void
xxXTideRoot::apply (int saveflag) {

  // First sanity check, then apply if OK.

  // Sanity check all colors
  int a;
  for (a=0; a<numcolors; a++) {
    unsigned char r, g, b;
    Dstr color (colordiags[a]->val());
    if (color.length()) {
      if (!(myTideContext->colors->parseColor (color, r, g, b, 0)))
        return;
    }
  }
  // Sanity check date/time formats -- prevent nasty XML characters
#define datetimebarf { \
    Dstr details ("Please don't use nasty characters like \", >, and < \
in your date/time formats."); \
    barf (XMLPARSE, details, 0); \
    return; \
  }
  if (strchr (dfdiag->val(), '"') ||
      strchr (dfdiag->val(), '>') ||
      strchr (dfdiag->val(), '<'))
    datetimebarf;
  if (strchr (hfdiag->val(), '"') ||
      strchr (hfdiag->val(), '>') ||
      strchr (hfdiag->val(), '<'))
    datetimebarf;
  if (strchr (tfdiag->val(), '"') ||
      strchr (tfdiag->val(), '>') ||
      strchr (tfdiag->val(), '<'))
    datetimebarf;
  // Sanity check doubles.
  if (!(sancheckdouble (gadiag->val())))
    return;
  if (!(sancheckdouble (lwdiag->val())))
    return;

  // Apply changes.
  Settings newset;
  // Colors
  for (a=0; a<numcolors; a++) {
    Dstr color (colordiags[a]->val());
    if (color.length())
      newset.colors[a] = color;
  }
  // Toggles
  switch (elchoice->choice()) {
  case 0:
    newset.el = "y";
    break;
  case 1:
    newset.el = "n";
    break;
  case 2:
    break;
  default:
    assert (0);
  }
  switch (nfchoice->choice()) {
  case 0:
    newset.nf = "y";
    break;
  case 1:
    newset.nf = "n";
    break;
  case 2:
    break;
  default:
    assert (0);
  }
  switch (tlchoice->choice()) {
  case 0:
    newset.tl = "y";
    break;
  case 1:
    newset.tl = "n";
    break;
  case 2:
    break;
  default:
    assert (0);
  }
  switch (zchoice->choice()) {
  case 0:
    newset.z = "y";
    break;
  case 1:
    newset.z = "n";
    break;
  case 2:
    break;
  default:
    assert (0);
  }

  switch (uchoice->choice()) {
  case 0:
    newset.u = "ft";
    break;
  case 1:
    newset.u = "m";
    break;
  case 2:
    newset.u = "x";
    break;
  case 3:
    break;
  default:
    assert (0);
  }

  switch (glchoice->choice()) {
  case 12:
    newset.gl_isnull = 0;
    newset.gl = 360.0;
    break;
  case 13:
    break;
  default:
    newset.gl_isnull = 0;
    newset.gl = (double)(glchoice->choice()) * 30.0 - 180.0;
    break;
  }

  // Date/time formats.
  {
    Dstr fmt (dfdiag->val());
    if (fmt.length())
      newset.df = fmt;
  }
  {
    Dstr fmt (hfdiag->val());
    if (fmt.length())
      newset.hf = fmt;
  }
  {
    Dstr fmt (tfdiag->val());
    if (fmt.length())
      newset.tf = fmt;
  }

  // Unsigned numbers.
  if (gwchoice->choice() != 1000000) {
    newset.gw_isnull = 0;
    newset.gw = gwchoice->choice();
  }
  if (ghchoice->choice() != 1000000) {
    newset.gh_isnull = 0;
    newset.gh = ghchoice->choice();
  }
  if (twchoice->choice() != 1000000) {
    newset.tw_isnull = 0;
    newset.tw = twchoice->choice();
  }
  if (thchoice->choice() != 1000000) {
    newset.th_isnull = 0;
    newset.th = thchoice->choice();
  }
  if (cwchoice->choice() != 1000000) {
    newset.cw_isnull = 0;
    newset.cw = cwchoice->choice();
  }

  // Doubles.
  {
    Dstr num (gadiag->val());
    if (num.length() > 0) {
      double d;
      assert (sscanf (num.aschar(), "%lf", &d) == 1);
      newset.ga_isnull = 0;
      newset.ga = d;
    }
  }
  {
    Dstr num (lwdiag->val());
    if (num.length() > 0) {
      double d;
      assert (sscanf (num.aschar(), "%lf", &d) == 1);
      newset.lw_isnull = 0;
      newset.lw = d;
    }
  }

  mycontext->change_settings (myTideContext, newset);
  global_redraw();

  if (saveflag)
    newset.save();
}

static void
saveCallback (Widget w, XtPointer client_data, XtPointer call_data) {
  xxXTideRoot *f = (xxXTideRoot *)client_data;
  f->apply (1);
}

static void
helpCallback (Widget w, XtPointer client_data, XtPointer call_data) {
  Dstr helpstring ("\
XTide Control Panel\n\
\n\
The Control Panel is used to change global XTide settings.  These settings\n\
take precedence over compiled-in defaults and X application defaults, but\n\
they are overridden by settings made on the command line.  Therefore, any\n\
settings that you make in the Control Panel will not have any visible effect\n\
if you have also made these settings on the command line.\n\
\n\
Some settings have dialog boxes for you to type in; others have pull-down\n\
menus or '+' and '-' controls.  In each case, there is a way to leave the\n\
field blank if you want to keep the inherited settings.  Either delete all\n\
text in the dialog box, or choose the \"(blank)\" option on the pull-down\n\
menu, or lay on the '-' button until the field reads \"(blank)\".\n\
\n\
The settings identified as \"default\" settings (widths, heights, and\n\
aspect ratio) will not affect existing windows.  Widths and heights will\n\
affect all new windows; aspect ratio will only affect new windows that\n\
are created from the location chooser.  (This is because the aspect is\n\
preserved from one tide window to any new windows that it spawns.)\n\
\n\
Use the Apply button to apply the new settings to the current XTide session\n\
without making them permanent.  Use Save to make them permanent.\n\
\n\
If necessary, resize the Control Panel to keep all of the settings visible.\n\
\n\
About colors:  When entering colors, use either standard X11 color names\n\
or 24-bit hex specs of the form rgb:hh/hh/hh.  Do not use more or less bits;\n\
it will not work.\n\
\n\
About date/time formats:  Please refer to the man page for strftime to see\n\
how these formats work.  Here is how to get XTide to use 24-hour time instead\n\
of AM/PM:\n\
   Hour format  %H\n\
   Time format  %H:%M %Z");
  xxXTideRoot *f = (xxXTideRoot *)client_data;
  f->newHelpBox (helpstring);
}

void RootCloseHandler (Widget w, XtPointer client_data,
			   XEvent *event, Boolean *continue_dispatch) {
  xxXTideRoot *f = (xxXTideRoot *)client_data;
  switch (event->type) {
  case ClientMessage:
    {
      XClientMessageEvent *ev = (XClientMessageEvent *) event;
      // Window manager close.
      if (ev->message_type == f->mycontext->protocol_atom &&
	  ev->data.l[0] == f->mycontext->kill_atom)
        f->hide();
    }
    break;
  default:
    ;
  }
}

// This is called by the constructor and also by hide()
void
xxXTideRoot::addTwoButtons() {
  Arg buttonargs[4] =  {
    {XtNvisual, (XtArgVal)mycontext->visual},
    {XtNcolormap, (XtArgVal)mycontext->colormap},
    {XtNbackground, (XtArgVal)mycontext->pixels[Colors::button]},
    {XtNforeground, (XtArgVal)mycontext->pixels[Colors::foreground]}
  };

  {
    Widget buttonwidget = XtCreateManagedWidget ("Dismiss", commandWidgetClass,
      box->manager, buttonargs, 4);
    XtAddCallback (buttonwidget, XtNcallback, dismissCallback,
     (XtPointer)this);
    dismissbutton = new xxContext (box, buttonwidget);
  }
  {
    Widget buttonwidget = XtCreateManagedWidget ("?", commandWidgetClass,
      box->manager, buttonargs, 4);
    XtAddCallback (buttonwidget, XtNcallback, helpCallback,
      (XtPointer)this);
    helpbutton = new xxContext (box, buttonwidget);
  }
}

xxXTideRoot::xxXTideRoot (int in_argc, char **in_argv) {
  numpopups = 0;
  children = NULL;
  Colors *colors;
  Settings *settings;
  mycontext = new xxContext (in_argc, in_argv, colors, settings);
  mycontext->setTitle ("Control Panel");
  myTideContext = new xxTideContext (in_argc, in_argv, mycontext, this,
    colors, settings);
  set_error_context (mycontext, myTideContext);

  Arg args[2] =  {
    {XtNbackground, (XtArgVal)mycontext->pixels[Colors::background]},
    {XtNforeground, (XtArgVal)mycontext->pixels[Colors::foreground]}
  };
  XtSetValues (mycontext->manager, args, 2);
  XtAddEventHandler (mycontext->manager, NoEventMask, True,
    RootCloseHandler, (XtPointer)this);

  {
    Arg boxargs[3] =  {
      {XtNbackground, (XtArgVal)mycontext->pixels[Colors::background]},
      {XtNforeground, (XtArgVal)mycontext->pixels[Colors::foreground]},
      {XtNvSpace, (XtArgVal)0}
    };
    Widget boxwidget = XtCreateManagedWidget ("", boxWidgetClass,
      mycontext->manager, boxargs, 3);
    box = new xxContext (mycontext, boxwidget);
  }

  {
    Widget labelwidget = XtCreateManagedWidget (
      "-------------- XTide Control Panel --------------",
      labelWidgetClass, box->manager, args, 2);
    label = new xxContext (box, labelwidget);
  }

  // Get current settings.  (Another duplication...)
  ud = new UserDefaults();

  // Add dialogs for all of the colors
  int a;
  for (a=0; a<numcolors; a++)
    colordiags[a] = new xxHorizDialog (box, colordesc[a],
      ud->colors[a].aschar());

  Arg buttonargs[4] =  {
    {XtNvisual, (XtArgVal)mycontext->visual},
    {XtNcolormap, (XtArgVal)mycontext->colormap},
    {XtNbackground, (XtArgVal)mycontext->pixels[Colors::button]},
    {XtNforeground, (XtArgVal)mycontext->pixels[Colors::foreground]}
  };

  // Add toggles
  static char *togglechoices[] = {"Yes", "No", "(blank)", NULL};
  {
    Dstr t = ud->el;
    unsigned firstchoice;
    if (t.isNull())
      firstchoice = 2;
    else if (t == "n")
      firstchoice = 1;
    else
      firstchoice = 0;
    elchoice = new xxMultiChoice (box,
      "Draw datum and MSL lines in tide graphs?", togglechoices, firstchoice);
  }
  {
    Dstr t = ud->nf;
    unsigned firstchoice;
    if (t.isNull())
      firstchoice = 2;
    else if (t == "n")
      firstchoice = 1;
    else
      firstchoice = 0;
    nfchoice = new xxMultiChoice (box,
      "Draw tide graphs as line graphs?", togglechoices, firstchoice);
  }
  {
    Dstr t = ud->tl;
    unsigned firstchoice;
    if (t.isNull())
      firstchoice = 2;
    else if (t == "n")
      firstchoice = 1;
    else
      firstchoice = 0;
    tlchoice = new xxMultiChoice (box,
      "Draw depth lines on top of tide graph?", togglechoices, firstchoice);
  }
  {
    Dstr t = ud->z;
    unsigned firstchoice;
    if (t.isNull())
      firstchoice = 2;
    else if (t == "n")
      firstchoice = 1;
    else
      firstchoice = 0;
    zchoice = new xxMultiChoice (box,
      "Coerce all time zones to UTC?", togglechoices, firstchoice);
  }

  static char *unitschoices[] = {"Feet", "Meters", "No preference",
    "(blank)", NULL};
  {
    Dstr t = ud->u;
    unsigned firstchoice;
    if (t.isNull())
      firstchoice = 3;
    else if (t == "ft")
      firstchoice = 0;
    else if (t == "m")
      firstchoice = 1;
    else
      firstchoice = 2;
    uchoice = new xxMultiChoice (box,
      "Preferred units of length:", unitschoices, firstchoice);
  }

  static char *glchoices[] = {"180", "150 W", "120 W", "90 W", "60 W",
    "30 W", "0", "30 E", "60 E", "90 E", "120 E", "150 E",
    "Max stations", "(blank)", NULL};
  {
    unsigned firstchoice = 13;
    if (!(ud->gl_isnull)) {
      if (ud->gl == 360.0)
        firstchoice = 12;
      else
        firstchoice = (unsigned) ((ud->gl + 180.0) / 30.0);
    }
    glchoice = new xxMultiChoice (box,
      "Default center longitude for globe:", glchoices, firstchoice);
  }

  // Date/time format dialogs
  dfdiag = new xxHorizDialog (box,
    "Strftime style format string for printing dates.", ud->df.aschar());
  hfdiag = new xxHorizDialog (box,
    "Strftime style format string for printing hour labels on time axis.",
    ud->hf.aschar());
  tfdiag = new xxHorizDialog (box,
    "Strftime style format string for printing times.", ud->tf.aschar());

  // Unsigned numbers.
  gwchoice = new xxUnsignedChooser (box,
    "Default width for tide graphs (pixels):", ud->gw, ud->gw_isnull, mingwidth);
  ghchoice = new xxUnsignedChooser (box,
    "Default height for tide graphs and clocks (pixels):", ud->gh, ud->gh_isnull,
    mingheight);
  cwchoice = new xxUnsignedChooser (box,
    "Default width for tide clocks (pixels):", ud->cw, ud->cw_isnull, mingwidth);
  twchoice = new xxUnsignedChooser (box,
    "Default width of ASCII graphs and banners (characters):", ud->tw,
     ud->tw_isnull, minttywidth);
  thchoice = new xxUnsignedChooser (box,
    "Default height of ASCII graphs (characters):", ud->th, ud->th_isnull,
    minttyheight);

  // Doubles.
  {
    char temp[80];
    if (ud->ga_isnull)
      strcpy (temp, "");
    else
      sprintf (temp, "%f", ud->ga);
    gadiag = new xxHorizDialog (box,
      "Default aspect for tide graphs.", temp);
  }
  {
    char temp[80];
    if (ud->lw_isnull)
      strcpy (temp, "");
    else
      sprintf (temp, "%f", ud->lw);
    lwdiag = new xxHorizDialog (box,
      "Width for lines in line graphs (pixels, pos. real number).", temp);
  }

  {
    Widget buttonwidget = XtCreateManagedWidget ("Apply", commandWidgetClass,
      box->manager, buttonargs, 4);
    XtAddCallback (buttonwidget, XtNcallback, applyCallback,
     (XtPointer)this);
    applybutton = new xxContext (box, buttonwidget);
  }
  {
    Widget buttonwidget = XtCreateManagedWidget ("Save", commandWidgetClass,
      box->manager, buttonargs, 4);
    XtAddCallback (buttonwidget, XtNcallback, saveCallback,
     (XtPointer)this);
    savebutton = new xxContext (box, buttonwidget);
  }

  addTwoButtons();
}

void
xxXTideRoot::dup() {
  numpopups++;
}

void
xxXTideRoot::dup(xxWindow *child) {
  dup();
  struct cnode *tempc = new cnode;
  tempc->child = child;
  tempc->next = children;
  children = tempc;
}

void xxXTideRoot::release(int is_title_screen) {
  assert (numpopups > 0);
  numpopups--;
  if ((!numpopups) && (!is_title_screen))
    exit (0);
}

void xxXTideRoot::release(xxContext *child, int is_title_screen) {
  XtUnmanageChild (child->manager);
  release(is_title_screen);
}

void xxXTideRoot::release(xxWindow *child) {
  release (child->mypopup, child->is_title_screen);
  struct cnode *tempc = NULL;
  while (children) {
    struct cnode *t2c = children;
    children = children->next;
    if (t2c->child != child) {
      t2c->next = tempc;
      tempc = t2c;
    } else {
      delete t2c;
    }
  }
  children = tempc;
}

void xxXTideRoot::global_redraw() {
  struct cnode *tempc = children;
  while (tempc) {
    tempc->child->global_redraw();
    tempc = tempc->next;
  }

  int a;
  for (a=0; a<numcolors; a++)
    colordiags[a]->global_redraw();

  elchoice->global_redraw();
  nfchoice->global_redraw();
  tlchoice->global_redraw();
  zchoice->global_redraw();
  uchoice->global_redraw();

  dfdiag->global_redraw();
  hfdiag->global_redraw();
  tfdiag->global_redraw();
  gwchoice->global_redraw();
  glchoice->global_redraw();
  ghchoice->global_redraw();
  cwchoice->global_redraw();
  twchoice->global_redraw();
  thchoice->global_redraw();

  gadiag->global_redraw();
  lwdiag->global_redraw();

  Arg buttonargs[2] =  {
    {XtNbackground, (XtArgVal)mycontext->pixels[Colors::button]},
    {XtNforeground, (XtArgVal)mycontext->pixels[Colors::foreground]}
  };
  if (applybutton)
    XtSetValues (applybutton->manager, buttonargs, 2);
  if (savebutton)
    XtSetValues (savebutton->manager, buttonargs, 2);
  if (helpbutton)
    XtSetValues (helpbutton->manager, buttonargs, 2);
  if (dismissbutton)
    XtSetValues (dismissbutton->manager, buttonargs, 2);

  Arg args[2] =  {
    {XtNbackground, (XtArgVal)mycontext->pixels[Colors::background]},
    {XtNforeground, (XtArgVal)mycontext->pixels[Colors::foreground]}
  };
  if (label)
    XtSetValues (label->manager, args, 2);
  if (box)
    XtSetValues (box->manager, args, 2);
  if (mycontext)
    XtSetValues (mycontext->manager, args, 2);
}

xxXTideRoot::~xxXTideRoot () {
  mycontext->unrealize();
  delete applybutton;
  delete savebutton;
  delete helpbutton;
  delete dismissbutton;
  delete label;
  int a;
  for (a=0; a<numcolors; a++)
    delete colordiags[a];
  delete dfdiag;
  delete hfdiag;
  delete tfdiag;
  delete gadiag;
  delete lwdiag;
  delete gwchoice;
  delete ghchoice;
  delete glchoice;
  delete twchoice;
  delete thchoice;
  delete cwchoice;
  delete elchoice;
  delete nfchoice;
  delete tlchoice;
  delete zchoice;
  delete uchoice;
  delete box;
  delete mycontext;
  delete myTideContext;
  delete ud;
}

void xxXTideRoot::mainloop() {
  struct stat buf;
  if (stat (myTideContext->disabledisclaimerfile.aschar(), &buf)) {
    (void) new xxDisclaimer (mycontext, myTideContext);
    xxMainLoop (mycontext);
  }

  if (!(myTideContext->cls->l))
    newChooser (XtGrabNone); // Let the disclaimer scroll.
  else
    newWindows (myTideContext->cls);

  xxMainLoopForever (mycontext);
}

void xxXTideRoot::newHelpBox (const Dstr &help) {
  (void) new xxHelpBox (myTideContext, mycontext, help);
}

void xxXTideRoot::newGraph (Station *s, Timestamp t) {
  (void) new xxGraphMode (myTideContext, mycontext, s, t);
}

void xxXTideRoot::newGraph (StationRef *sr) {
  Timestamp t (time(NULL));
  newGraph (sr->load (myTideContext), t);
}

void xxXTideRoot::newText (Station *s, Timestamp t) {
  (void) new xxTextMode (myTideContext, mycontext, s, t);
}

void xxXTideRoot::newClock (Station *s, int nobuttonsflag) {
  (void) new xxClock (myTideContext, mycontext, s, nobuttonsflag);
}

void xxXTideRoot::newWindows (CommandLineSettings *cls) {
  Timestamp t (time(NULL));
  assert (cls);
  struct CommandLineSettings::lnode *l = cls->l;
  while (l) {
    StationRef *sr;
    if (cls->l_len <= maxfastload)
      sr = myTideContext->fastload (l->name);
    else
      sr = (*(myTideContext->stationIndex()))[l->name];
    if (sr) {
      if (!(cls->b.isNull())) {
        t = Timestamp (cls->b, sr->timezone, myTideContext->settings);
	if (t.isNull()) {
	  Dstr details ("The offending input was ");
	  details += cls->b;
	  details += "\nin the time zone ";
	  if (myTideContext->settings->z == "n")
	    details += sr->timezone;
	  else
	    details += "UTC0";
	  barf (MKTIME_FAILED, details);
	}
      }
      Station *s = sr->load (myTideContext);
      if (cls->m == "g")
        newGraph (s, t);
      else if (cls->m == "p")
        newText (s, t);
      else
        newClock (s, 1);
    } else {
      Dstr details ("Could not find: ");
      details += l->name;
      barf (STATION_NOT_FOUND, details);
    }
    l = l->next;
  }
}

void xxXTideRoot::newChooser (XtGrabKind in_grabkind) {
  (void) new xxGlobe (mycontext, *(myTideContext->stationIndex(in_grabkind)),
    myTideContext);
}

void xxXTideRoot::show() {
  if (!(mycontext->is_realized)) {
    mycontext->realize();
    dup();
  }
}

void xxXTideRoot::hide() {
  if (mycontext->is_realized) {
    mycontext->unrealize();
    release(0);

    // Since the window vanishes, the Dismiss button misses the LeaveWindow
    // event.  The next time the window is realized, Dismiss is still stuck
    // on.  I tried to synthesize a LeaveWindow event, but it didn't work.
    // I don't know a good way to reset the state, so extreme measures are
    // required.  To keep the buttons in order, the help button must also
    // be deleted.
    delete dismissbutton;
    delete helpbutton;
    addTwoButtons();
  }
}
