/*****************************************************************************
 * File: backend_sm.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: backend_sm.c,v 1.16 2005/10/17 03:56:53 chessing Exp $
 * $Date: 2005/10/17 03:56:53 $
 * $Log: backend_sm.c,v $
 * Revision 1.16  2005/10/17 03:56:53  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.15  2005/09/08 16:27:01  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.14  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.13  2005/08/25 03:34:05  chessing
 * Removed a bunch of functions from config.c that could be handled better in other ways.
 *
 * Revision 1.12  2005/08/09 01:39:13  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 ****************************************************************************/

#include <stdio.h>
#include <unistd.h>

#include "xsup_debug.h"
#include "xsup_err.h"
#include "profile.h"
#include "xsupconfig.h"
#include "statemachine.h"
#include "backend_sm.h"
#include "eap.h"
#include "snmp.h"
#include "cardif/cardif.h"

/****************************************************
 *
 * Init any variables that need to be set up, and do anything else that the
 * backend state machine needs.
 *
 ****************************************************/
void backend_sm_init(struct interface_data *thisint)
{
  thisint->statemachine->beCurState = 0xff;     // Unknown state.
  backend_sm_reinit(thisint);

  eap_init(thisint);
}

/****************************************************
 *
 * Re-init any variables that need to be set up, and do anything else that the
 * backend state machine needs.
 *
 ****************************************************/
void backend_sm_reinit(struct interface_data *thisint)
{
  thisint->statemachine->eapNoResp = FALSE;
  thisint->statemachine->eapReq = FALSE;
  thisint->statemachine->eapResp = FALSE;
  backend_sm_change_state(thisint, INITIALIZE);
}

/****************************************************
 *
 * Free any resources that were in use before telling the main state machine
 * that we have aborted this authentication.
 *
 ****************************************************/
void abortSupp()
{
  //XXX Nothing here yet.
}

/****************************************************
 *
 * Wait until the layers above have something to send.  (This is where we
 * get username/passwords.)
 *
 ****************************************************/
void getSuppRsp(struct interface_data *thisint)
{
  switch (eap_process_header(thisint)) 
    {
    case EAP_REQUEST_ID:
      snmp_dot1xSuppEapolReqIdFramesRx();
      if (eap_response_id(thisint) == XENONE)
	{
	  // We have something to send.
	  thisint->statemachine->eapResp = TRUE;
	  snmp_dot1xSuppEapolRespIdFramesTx();
	}
      break;

    case EAP_REQUEST_AUTH:
      snmp_dot1xSuppEapolReqFramesRx();
      if (eap_response_auth(thisint) == XENONE)
	{
	  // We have something to send.
	  thisint->statemachine->eapResp = TRUE;
	  snmp_dot1xSuppEapolRespFramesTx();
	}
      break;

    case EAP_REQUEST_NOTIFY:
      eap_do_notify(thisint);
      break;
    }
}

/****************************************************
 *
 * Transmit the response that should be in the queue.
 *
 ****************************************************/
void txSuppRsp(struct interface_data *thisint)
{
  // Make sure we know that we have sent the frame.
  thisint->statemachine->eapResp = FALSE;
  thisint->statemachine->eapolEap = FALSE;
  cardif_sendframe(thisint);
}

/****************************************************
 *
 * For some reason, the backend state machine timed out.  Depending on the
 * last packet we recieved, we may be able to provide the user with 
 * information about what went wrong.  For now, we are only going to report
 * on things that may be wrong when a Response ID message goes unanswered,
 * but in the future, we should expand to provide information on specific
 * EAP types where possible.
 *
 ****************************************************/
int backend_sm_timeout_display(struct interface_data *thisint)
{
  struct config_globals *globals;

  if (thisint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No valid interface structure available in "
		   "%s!\n", __FUNCTION__);
      return XEGENERROR;
    }

  globals = config_get_globals();

  if (globals == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No valid configuration globals available in "
		   "%s!\n", __FUNCTION__);
      return XEGENERROR;
    }

  switch (thisint->statemachine->lastEapType)
    {
    case EAP_TYPE_IDENTITY:
      // The last EAP message we saw was an Identity request.  We assume that
      // we attempted to send a response to that request.  (If not, it should
      // be reported elsewhere.)  If we get here, then there was a timeout
      // waiting for the authenticator to send us a packet containing the
      // beginning of the actual EAP conversation.

      if (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_NO_FRIENDLY_WARNINGS))
	{
	  debug_printf(DEBUG_NORMAL, "[WARNING]  Timeout waiting for the "
		       "authenticator to begin the EAP conversation.  This "
		       "usually happens when the RADIUS server is "
		       "misconfigured, the authenticator can't talk to the "
		       "RADIUS server, or the username provided is "
		       "invalid.\n");
	}
      break;

    default:
      if (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_NO_FRIENDLY_WARNINGS))
	{
	  debug_printf(DEBUG_NORMAL, "[WARNING] Timeout during the EAP "
		       "conversation!  Please verify that the settings\nin "
		       "your config file are correct and that the "
		       "authenticator and RADIUS servers\nare properly "
		       "configured.  If this error persists, please run "
		       "Xsupplicant in debug\nmode, and e-mail the output, "
		       "along with the config\nfile, and RADIUS config file "
		       "(where possible) to open1x-xsupplicant@"
		       "lists.sourceforge.net.\n");
	}
      break;
    }

  return XENONE;
}

/***************************************************************
 *
 * Display the text string for the backend state we are provided.
 *
 ***************************************************************/
void backend_sm_disp_state(int debuglevel, int state)
{
  switch (state)
    {
    case REQUEST:
      debug_printf_nl(debuglevel, "REQUEST");
      break;

    case RESPONSE:
      debug_printf_nl(debuglevel, "RESPONSE");
      break;

    case SUCCESS:
      debug_printf_nl(debuglevel, "SUCCESS");
      break;

    case FAIL:
      debug_printf_nl(debuglevel, "FAIL");
      break;

    case TIMEOUT:
      debug_printf_nl(debuglevel, "TIMEOUT");
      break;

    case IDLE:
      debug_printf_nl(debuglevel, "IDLE");
      break;

    case INITIALIZE:
      debug_printf_nl(debuglevel, "INITIALIZE");
      break;

    case RECEIVE:
      debug_printf_nl(debuglevel, "RECEIVE");
      break;

    default:
      debug_printf_nl(debuglevel, "UNKNOWN");
      break;
    }
}

/****************************************************
 *
 * Check global states for the backend state machine.
 *
 ****************************************************/
void backend_sm_check_globals(struct interface_data *ctx)
{
  if (ctx->statemachine->initialize || ctx->statemachine->suppAbort)
    {
      debug_printf(DEBUG_STATE, "(global) -> INITIALIZE\n");
      backend_sm_change_state(ctx, INITIALIZE);
    }
  
  if (ctx->statemachine->eapRestart)
    {
      debug_printf(DEBUG_STATE, "Supplicant PAE has issued a restart.\n");
      eap_reset(ctx);
    }
}

/****************************************************
 *
 * Change to initialize state.
 *
 ****************************************************/
void backend_sm_change_to_initialize(struct interface_data *ctx)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      _exit(4);
    }

  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid state machine context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      _exit(5);
    }

  abortSupp();

  ctx->statemachine->suppAbort = FALSE;
  ctx->statemachine->beCurState = INITIALIZE;

  backend_sm_change_state(ctx, IDLE);
}

/****************************************************
 *
 * Do initialize state.
 *
 ****************************************************/
void backend_sm_do_initialize(struct interface_data *ctx)
{
  // Nothing to do here.
}

/****************************************************
 *
 * Change to IDLE state.
 *
 ****************************************************/
void backend_sm_change_to_idle(struct interface_data *ctx)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context!  (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      _exit(4);
    }

  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid state machine structure! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      _exit(4);
    }

  ctx->statemachine->suppStart = FALSE;

  ctx->statemachine->beCurState = IDLE;
}

/****************************************************
 *
 * Process IDLE state.
 *
 ****************************************************/
void backend_sm_do_idle(struct interface_data *ctx)
{
  if (ctx->statemachine->eapFail && ctx->statemachine->suppStart)
    {
      backend_sm_change_state(ctx, FAIL);
    }
  
  if (ctx->statemachine->eapolEap && ctx->statemachine->suppStart)
    {
      backend_sm_change_state(ctx, REQUEST);
    }
  
  if (ctx->statemachine->eapSuccess && ctx->statemachine->suppStart)
    {
      backend_sm_change_state(ctx, SUCCESS);
    }
}

/****************************************************
 *
 * Change to request state.
 *
 ****************************************************/
void backend_sm_change_to_request(struct interface_data *ctx)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      _exit(4);
    }
  
  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid state machine context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      _exit(4);
    }

  ctx->statemachine->authWhile = 0;
  ctx->statemachine->eapReq = TRUE;

  ctx->statemachine->beCurState = REQUEST;

  getSuppRsp(ctx);
}

/****************************************************
 *
 * Process request state.
 *
 ****************************************************/
void backend_sm_do_request(struct interface_data *ctx)
{
  if (ctx->statemachine->eapFail)
    {
      backend_sm_change_state(ctx, FAIL);
    }
  
  if (ctx->statemachine->eapNoResp)
    {
      backend_sm_change_state(ctx, RECEIVE);
    }
  
  if (ctx->statemachine->eapResp)
    {
      backend_sm_change_state(ctx, RESPONSE);
    }
  
  if (ctx->statemachine->eapSuccess)
    {
      backend_sm_change_state(ctx, SUCCESS);
    }
}

/****************************************************
 *
 * Change to response state.
 *
 ****************************************************/
void backend_sm_change_to_response(struct interface_data *ctx)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      _exit(4);
    }

  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid state machine context! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      _exit(4);
    }

  txSuppRsp(ctx);
  ctx->statemachine->eapResp = FALSE;
  ctx->statemachine->beCurState = RESPONSE;
  
  backend_sm_change_state(ctx, RECEIVE);
}

/****************************************************
 *
 * Do response state.
 *
 ****************************************************/
void backend_sm_do_response(struct interface_data *ctx)
{
  debug_printf(DEBUG_NORMAL, "We should have changed state by now!!  Please "
	       "report this to the developers!\n");
  debug_printf(DEBUG_NORMAL, "Moving to RECEIVE like we should have.\n");
  backend_sm_change_state(ctx, RECEIVE);
}

/****************************************************
 *
 * Change to receive state.
 *
 ****************************************************/
void backend_sm_change_to_receive(struct interface_data *ctx)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context!\n");
      _exit(4);
    }

  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid state machine context!\n");
      _exit(4);
    }

  ctx->statemachine->authWhile = ctx->statemachine->authPeriod;
  ctx->statemachine->eapolEap = FALSE;
  ctx->statemachine->eapNoResp = FALSE;
  ctx->statemachine->beCurState = RECEIVE;
}

/****************************************************
 *
 * Do receive state.
 *
 ****************************************************/
void backend_sm_do_receive(struct interface_data *ctx)
{
  if (ctx->statemachine->eapFail)
    {
      backend_sm_change_state(ctx, FAIL);
    }
  
  if (ctx->statemachine->authWhile == 0)
    {
      backend_sm_change_state(ctx, TIMEOUT);
    }
  
  if (ctx->statemachine->eapSuccess)
    {
      backend_sm_change_state(ctx, SUCCESS);
    }
  
  if (ctx->statemachine->eapolEap)
    {
      backend_sm_change_state(ctx, REQUEST);
    }
}

/****************************************************
 *
 * Change to FAIL state.
 *
 ****************************************************/
void backend_sm_change_to_fail(struct interface_data *ctx)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context!\n");
      _exit(4);
    }

  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid state machine context!\n");
      _exit(4);
    }

  ctx->statemachine->suppFail = TRUE;
  ctx->statemachine->beCurState = FAIL;
  backend_sm_change_state(ctx, IDLE);
}

/****************************************************
 *
 * Process FAIL state.
 *
 ****************************************************/
void backend_sm_do_fail(struct interface_data *ctx)
{
  debug_printf(DEBUG_NORMAL, "Invalid state.  We should *NEVER* get here!!\n");
  debug_printf(DEBUG_NORMAL, "Please report this to the developers.\n");
  debug_printf(DEBUG_NORMAL, "Changing to IDLE state like we should have!\n");
  backend_sm_change_state(ctx, IDLE);
}

/****************************************************
 *
 * Change to TIMEOUT state.
 *
 ****************************************************/
void backend_sm_change_to_timeout(struct interface_data *ctx)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context!\n");
      _exit(4);
    }

  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid state machine context!\n");
      _exit(4);
    }

  backend_sm_timeout_display(ctx);
  ctx->statemachine->suppTimeout = TRUE;
  ctx->statemachine->beCurState = TIMEOUT;
  snmp_backend_timeout();
  backend_sm_change_state(ctx, IDLE);
}

/****************************************************
 *
 * Process TIMEOUT state.
 *
 ****************************************************/
void backend_sm_do_timeout(struct interface_data *ctx)
{
  debug_printf(DEBUG_NORMAL, "Invalid state.  We should *NEVER* get here!!\n");
  debug_printf(DEBUG_NORMAL, "Please report this to the developers.\n");
  debug_printf(DEBUG_NORMAL, "Changing to IDLE state like we should have!\n");
  backend_sm_change_state(ctx, IDLE);
}

/****************************************************
 *
 * Change to SUCCESS state.
 *
 ****************************************************/
void backend_sm_change_to_success(struct interface_data *ctx)
{
  if (!ctx)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface context!\n");
      _exit(4);
    }

  if (!ctx->statemachine)
    {
      debug_printf(DEBUG_NORMAL, "Invalid state machine context!\n");
      _exit(4);
    }

  ctx->statemachine->keyRun = TRUE;
  ctx->statemachine->suppSuccess = TRUE;
  ctx->statemachine->beCurState = SUCCESS;
  backend_sm_change_state(ctx, IDLE);
}

/****************************************************
 *
 * Process SUCCESS state.
 *
 ****************************************************/
void backend_sm_do_success(struct interface_data *ctx)
{
  debug_printf(DEBUG_NORMAL, "Invalid state.  We should *NEVER* get here!!\n");
  debug_printf(DEBUG_NORMAL, "Please report this to the developers.\n");
  debug_printf(DEBUG_NORMAL, "Changing to IDLE state like we should have!\n");
  backend_sm_change_state(ctx, IDLE);
}

/****************************************************
 *
 * Change the backend state.
 *
 ****************************************************/
void backend_sm_change_state(struct interface_data *ctx, int newstate)
{
  debug_printf(DEBUG_STATE, "[backend_sm] ");
  backend_sm_disp_state(DEBUG_STATE, ctx->statemachine->beCurState);
  debug_printf_nl(DEBUG_STATE, " -> ");
  backend_sm_disp_state(DEBUG_STATE, newstate);
  debug_printf_nl(DEBUG_STATE, "\n");

  switch (newstate)
    {
    case INITIALIZE:
      backend_sm_change_to_initialize(ctx);
      break;

    case IDLE:
      backend_sm_change_to_idle(ctx);
      break;

    case REQUEST:
      backend_sm_change_to_request(ctx);
      break;

    case RESPONSE:
      backend_sm_change_to_response(ctx);
      break;

    case RECEIVE:
      backend_sm_change_to_receive(ctx);
      break;

    case FAIL:
      backend_sm_change_to_fail(ctx);
      break;

    case TIMEOUT:
      backend_sm_change_to_timeout(ctx);
      break;

    case SUCCESS:
      backend_sm_change_to_success(ctx);
      break;
    }
}

/****************************************************
 *
 * Actually run the backend state machine.
 *
 ****************************************************/
int backend_sm_run(struct interface_data *ctx)
{
  backend_sm_check_globals(ctx);

  switch (ctx->statemachine->beCurState)
    {
    case INITIALIZE:
      // We should *NEVER* get here!
      backend_sm_do_initialize(ctx);
      break;

    case IDLE:
      backend_sm_do_idle(ctx);
      break;

    case REQUEST:
      backend_sm_do_request(ctx);
      break;

    case RESPONSE:
      backend_sm_do_response(ctx);
      break;

    case RECEIVE:
      backend_sm_do_receive(ctx);
      break;

    case FAIL:
      backend_sm_do_fail(ctx);
      break;

    case TIMEOUT:
      backend_sm_do_timeout(ctx);
      break;

    case SUCCESS:
      backend_sm_do_success(ctx);
      break;
    }

  return XENONE;
}

/****************************************************
 *
 * Clean up anything that we set up during the use of the state machine.
 *
 ****************************************************/
void backend_sm_deinit(struct interface_data *thisint)
{
  struct config_network *network_data;

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network data!  (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  eap_cleanup(&network_data->activemethod);
}

