/*

    xpuyopuyo - ppiece.c      Copyright(c) 1999,2000 Justin David Smith
    justins(at)chaos2.org     http://chaos2.org/
    
    Code for controlling individual playing pieces.
    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <stdio.h>
#include <stdlib.h>
#include <config.h>
#include <ppiece.h>


void p_piece_renew(pfield *field, nextpiece *next) {
/* p_piece_renew

   Sets the current piece to what used to be the "next" piece.  
   The contents of nextpiece are not modified by this function,
   but a new nextpiece _should_ be calculated soon afterwards. */

   field->piece->data[0] = next->block[0];   /* Get colored blobs */
   field->piece->data[1] = next->block[1];
   field->piece->data[2] = P_CLEAR;          /* Two cleared blobs too! */
   field->piece->data[3] = P_CLEAR;
   field->piece->x = field->width / 2 - 1;   /* Set x, y position so piece is centered */
   field->piece->y = 1;                      /* at the top of the playing field.  */
   
}


void p_piece_new(pfield *field, nextpiece *next) {
/* p_piece_new

   Creates a new piecefield (field->piece), and then
   sets the piecefield to the specified "next" piece.  */

   field->piece = p_field_new(2, 2, field->number);
   p_piece_renew(field, next);

}


int p_piece_collided(pfield *field) {
/* p_piece_collided

   Returns nonzero if the piece in the playing field has
   actually collided with a blob in the field (a colored
   blob in the piecefield is overlapping a nonclear tile
   in the playing field).  */

   pfield *piece;
   int x;
   int y;
   int i;
   int j;
   int *src;
   
   piece = field->piece;
   x = piece->x;
   y = piece->y;
   
   src = field->piece->data;
   for(j = 0; j < piece->height; j++) {
      for(i = 0; i < piece->width; i++) {
         if(!P_IS_CLEAR(*src) && !P_IS_CLEAR(p_field_get(field, x + i, y + j))) return(1);
         src++;
      }
   }
   return(0);

}


#if USE_NEW_ROTATE
void p_piece_rotate(pfield *field) {
/* p_piece_rotate

   Clockwise rotate the playing piece. */
/*  A B      . A
    . .  ->  . B    x--
    
    . A      . . 
    . B  ->  B A    y--
    
    . .      B .
    B A  ->  A .    x++
    
    B .      A B
    A .  ->  . .    y++   */
    
    
   pfield *piece;
   int i;
   
   piece = field->piece;
   
   i = piece->data[0];
   piece->data[0] = piece->data[1];
   piece->data[1] = piece->data[3];
   piece->data[3] = piece->data[2];
   piece->data[2] = i;

   if(piece->data[0] == P_CLEAR) {
      if(piece->data[2] == P_CLEAR) piece->x--;
      else                          piece->y--;
   } else {
      if(piece->data[1] == P_CLEAR) piece->x++;
      else                          piece->y++;
   }

}
#else
#define p_piece_rotate(f)  p_piece_rotate_old(f)
#endif /* USE_NEW_ROTATE */


void p_piece_rotate_old(pfield *field) {
/* p_piece_rotate_old

   Clockwise rotate the playing piece. */
/* The AI code requires a rotate function that behaves like this! */
    
   pfield *piece;
   int i;
   
   piece = field->piece;
   
   i = piece->data[0];
   piece->data[0] = piece->data[1];
   piece->data[1] = piece->data[3];
   piece->data[3] = piece->data[2];
   piece->data[2] = i;

   if(piece->data[3] != P_CLEAR) {
      if(piece->data[2] != P_CLEAR) {
         piece->data[0] = piece->data[2];
         piece->data[1] = piece->data[3];
         piece->data[2] = P_CLEAR;
         piece->data[3] = P_CLEAR;
      } else {
         piece->data[0] = piece->data[1];
         piece->data[2] = piece->data[3];
         piece->data[1] = P_CLEAR;
         piece->data[3] = P_CLEAR;
      }
   }

}


void p_piece_unrotate(pfield *field) {
/* p_piece_unrotate

   Counter-clockwise-rotate the playing piece.  */

   p_piece_rotate(field);
   p_piece_rotate(field);
   p_piece_rotate(field);

}


int p_piece_rotate_collide(pfield *field) {
/* p_piece_rotate_collide

   Checks for a collision if we rotate the playing piece.
   Returns nonzero if a collision occurred.  */

   int r;
   
   p_piece_rotate(field);
   r = p_piece_collided(field);
   p_piece_rotate(field);
   p_piece_rotate(field);
   p_piece_rotate(field);

   return(r);

}


int p_piece_try_rotate(pfield *field) {
/* p_piece_try_rotate

   Rotates the playing piece if possible, otherwise returns zero.  */

   while(p_piece_rotate_collide(field)) {
      p_piece_rotate(field);
   }
   p_piece_rotate(field);
   return(1);

}


void p_piece_fall(pfield *field) {

   field->piece->y = field->piece->y + 1;

}


void p_piece_unfall(pfield *field) {

   field->piece->y = field->piece->y - 1;

}


int p_piece_fall_collide(pfield *field) {

   int r;
   
   p_piece_fall(field);
   r = p_piece_collided(field);
   p_piece_unfall(field);
   
   return(r);

}


static inline void p_piece_right(pfield *field) {

   field->piece->x = field->piece->x + 1;

}


static inline void p_piece_unright(pfield *field) {

   field->piece->x = field->piece->x - 1;

}


static inline int p_piece_right_collide(pfield *field) {

   int r;
   
   p_piece_right(field);
   r = p_piece_collided(field);
   p_piece_unright(field);
   
   return(r);

}


int p_piece_try_right(pfield *field) {

   if(p_piece_right_collide(field)) return(0);
   p_piece_right(field);
   return(1);

}


static inline void p_piece_left(pfield *field) {

   field->piece->x = field->piece->x - 1;

}


static inline void p_piece_unleft(pfield *field) {

   field->piece->x = field->piece->x + 1;

}


static inline int p_piece_left_collide(pfield *field) {

   int r;
   
   p_piece_left(field);
   r = p_piece_collided(field);
   p_piece_unleft(field);
   
   return(r);

}


int p_piece_try_left(pfield *field) {

   if(p_piece_left_collide(field)) return(0);
   p_piece_left(field);
   return(1);

}


