/*

    xpuyopuyo - pai.h         Copyright(c) 1999 Justin Smith
    justins@chaos2.org        http://chaos2.org/
    
    Header for AI
    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#ifndef __pai_h_defined
#define __pai_h_defined


#include <xpuyopuyo.h>
#include <pfield.h>


/* AI File */
#define P_AI_IO_BUFFER                 0x1000            /* Buffer size for AI I/O */


/* AI Rules */
#if USE_AIBREED
#define P_AI_COUNT                     32    /* Total number of defined AI's */
#else /* Normal list of AI's */
#define P_AI_COUNT                     16    /* Total number of defined AI's */
#endif /* AI breeding? */

#define P_AI_DEFAULT                   0     /* Index for the default AI rule */
#define P_AI_TRAINER                   1     /* Index for the trainer AI rule */
#define P_AIS_HIGH_THRESHOLD           4     /* How many rows from top is the "hard" threshold? */
#define P_AIS_MUTATE_MAX               8     /* Maximum amount of mutation per round */

#define P_AIS_DISTANCE_FACTOR          0     /* If positive, AI prefers placing a blob lower in the field.
                                                Multiplied by height of blob from base of field. */
#define P_AIS_BLOCKED_COLOR_BELOW      1     /* Negative value prevents AI from blocking a blob below */
#define P_AIS_BLOCKED_COLOR_SIDE       2     /* Negative value prevents AI from placing blob between two
                                                others of the same color */
#define P_AIS_SAME_COLOR               3     /* Positive value expresses AI's preference to put two blobs
                                                of same color together */
#define P_AIS_MATCH                    4     /* Large positive value expresses AI's desire to make a match */
#define P_AIS_ROCKS                    5     /* Positive value, AI wants to get rid of rocks if possible.
                                                This may be the deciding factor when an AI is trying to
                                                choose between two possible matches. */
#define P_AIS_HIGH_PENALTY             6     /* Large negative value, AI really does not want to build 
                                                structures in the top $HIGH_THRESHOLD rows of the playfield */
#define P_AIS_INTERFERENCE             7     /* Interference score regards an AI blocking a group or groups
                                                of blobs of the same color (i.e. a three-blob group directly
                                                below).  Negative value, the AI does not want to interfere
                                                with another potential match */
#define P_AIS_BLOCKING_NEXT            8     /* Additional loss to above if the next piece contains a color 
                                                that would complete a match */
#define P_AIS_BOTH_MATCH               9     /* Small bonus if both blobs complete a match */
#define P_AIS_SIMUL_MATCH              10    /* Multiplied by number of simultaneous matches, a bonus score
                                                for completing at least one match. */
#define P_AIS_DIAGONAL_MATCH           11    /* Diagonal matching */
#define P_AIS_SIDE                     12    /* If positive, AI likes pushing to sides */
#define P_AIS_PEAK                     13    /* If positive, AI maximized distance between peak and valley */
#define P_AIS_NUM_RULES                14    /* Total number of rules (1 + above define) */


/* Structure for storing rules for each AI */
/* Positive scores indicate beneficial actions,
   negative scores indicate nonbenificial acts.
   Zero scores indicate AI is indifferent about an action */
typedef struct _airule {
   void *dummy;
} airule;


typedef struct _aiinfo {
   void *dummy;
} aiinfo;


/* AI Decision, cur and next are arrays of int[2] */
void p_ai_decide(pfield *f, airule *a, int *cur, int *next, int mintomatch, int numcolors);


/* Deconstructor functions */
aiinfo *p_ai_load_rules();
void p_ai_save_rules(const aiinfo *ai);
void p_ai_release_rules(aiinfo **ai);


/* Obtain a pointer to a rule from within the aiinfo structure 
   Data in this pointer may be modified if needed.  */
airule *p_ai_get_rule(aiinfo *ai, int index);


/* Mutate an AI rule, based on specified "winner" rule <s>. */
#if USE_AIBREED
void p_ai_mutate(const aiinfo *ai, airule *d, const airule *s, /* TEMP int n, */ int quiet);
#endif


/* AI Informational functions */
const char *p_ai_get_name(const airule *a);
const char *p_ai_get_desc(const airule *a);
int p_ai_get_score(const airule *a);
int p_ai_get_index(const airule *a);
const char *p_ai_name_of(const aiinfo *ai, int rule);
const char *p_ai_desc_of(const aiinfo *ai, int rule);
int p_ai_rule_of(const aiinfo *ai, int aidx, int ridx);
int p_ai_generation_of(const aiinfo *ai, int aidx);
int p_ai_win_count_of(const aiinfo *ai, int aidx);
void p_ai_inc_win_count(const aiinfo *ai, int aidx);
int p_ai_loss_count_of(const aiinfo *ai, int aidx);
void p_ai_inc_loss_count(const aiinfo *ai, int aidx);
int p_ai_default_rule_of(const aiinfo *ai, int aidx, int ridx);
const char *p_ai_rule_name(int ridx);
const char *p_ai_rule_text_of(const aiinfo *ai, int aidx);


#endif
