/*
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Brian W. Barrett, Arun F. Rodrigues, Jeffrey M. Squyres,
 * 	 and Andrew Lumsdaine
 *
 * This file is part of XMPI
 *
 * You should have received a copy of the License Agreement for XMPI 
 * along with the software; see the file LICENSE.  If not, contact 
 * Office of Research, University of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 *
 * Additional copyrights may follow.

 *
 *	$Id: xmpi_nodes.c,v 1.4 1999/11/11 04:47:33 arodrig6 Exp $
 * 
 *	Function:	- node list panel in builder dialog
 *			- only understands host file syntax
 */

#define _NO_PROTO

#include <stdlib.h>
#include <string.h>

#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/List.h>
#include <Xm/PanedW.h>
#include <Xm/TextF.h>
#include <Xm/Xm.h>

#include "args.h"
#include "xmpi.h"

/*
 * global functions
 */
void			xmpi_nodes_build();

/*
 * external functions
 */
extern int		bhostparse();
extern void		itoa();

/*
 * local functions
 */
static void		nodes_fill();
static void		nodes_select_cb();

/*
 * local variables
 */
#define MAXLINE		80
static char		linebuf[MAXLINE];
static Widget		list_w;

/*
 *	xmpi_nodes_build
 *
 *	Function:	- builds node panel
 *	Accepts:	- parent widget
 */
void
xmpi_nodes_build(parent_w)

Widget			parent_w;

{
	Widget		mgr_w;
	XmString	xstr;

	mgr_w = XtVaCreateWidget("nodes_mgr",
		xmPanedWindowWidgetClass, parent_w,
		XmNsashWidth, 1,
		XmNsashHeight, 1,
		XmNseparatorOn, False,
		XmNleftAttachment, XmATTACH_POSITION,
		XmNleftPosition, 2,
		XmNleftOffset, 5,
		XmNrightOffset, 1,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);

	xstr = XmStringCreateSimple("Select Nodes");
	XtVaCreateManagedWidget("banner",
		xmLabelWidgetClass, mgr_w,
		XmNlabelString, xstr,
		NULL);
	XmStringFree(xstr);

        XtVaCreateManagedWidget("nodes_text",
                xmTextFieldWidgetClass, mgr_w,
		XmNvalue, xmpi_info,
		XmNeditable, False,
		XmNcursorPositionVisible, False,
                NULL);

	list_w = XmCreateScrolledList(mgr_w, "nodes_list", NULL, 0);
	XtVaSetValues(list_w,
		XmNscrollBarDisplayPolicy, XmSTATIC,
		XmNscrollHorizontal, False,
		XmNselectionPolicy, XmEXTENDED_SELECT,
		NULL);
	XtAddCallback(list_w, XmNdefaultActionCallback, nodes_select_cb, NULL);
	XtAddCallback(list_w, XmNextendedSelectionCallback,
			nodes_select_cb, NULL);
	XtManageChild(list_w);
/*
 * Fill the list.
 */
	nodes_fill();

	xmpi_nosash(mgr_w);
	XtManageChild(mgr_w);
}

/*
 *	nodes_select_cb
 *
 *	Function:	- builds mnemonic node specification string
 *			  from all selected list members
 *	Accepts:	- widget
 *			- client data
 *			- ptr callback struct
 */
static void
nodes_select_cb(w, cdata, p_callback)

Widget			w;
XtPointer		cdata;
XmListCallbackStruct	*p_callback;

{
	int		i;
	int		*pposn;
	int		n;		/* current mnemonic size */
	int		running;	/* current end of range */
	int		size;		/* current mnemonic buffer size */
	int		start;		/* start of range */
	char		*mnemonic;
	char		*p;
	Arg             arg;

	if (p_callback->selected_item_count == 0) {
		return;
	}

	pposn = p_callback->selected_item_positions;

/*
 * Lesstif seems to have a bug...?
 */
        XtSetArg(arg, XmNitemCount, &size);
        XtGetValues(list_w, &arg, 1);
        if (p_callback->selected_item_count > size) {
          return;
        }

/*
 * MAXLINE must be longer than SAFETY.
 */
	size = MAXLINE;
	mnemonic = malloc((unsigned) size);

	if (!mnemonic) xmpi_fail("xmpi (malloc)");
/*
 * Handle special mnemonics.
 */
	if (pposn[0] < 3) {
		mnemonic[0] = (pposn[0] == 1) ? 'N' : 'h';
		mnemonic[1] = '\0';
		xmpi_run_set_nodes(mnemonic);
		free(mnemonic);
		return;
	}

	mnemonic[0] = 'n';
	pposn[0] -= 3;
	itoa(pposn[0], mnemonic + 1);
	n = strlen(mnemonic);
	p = mnemonic + n;
	start = running = pposn[0];

	for (i = 1; i < p_callback->selected_item_count; ++i) {
		pposn[i] -= 3;

		if (pposn[i] == (running + 1)) {
			running += 1;
			continue;
		}

		if (running > start) {
			*p = '-';
			itoa(running, p + 1);
			n += strlen(p);
			p = mnemonic + n;
		}

		*p = ',';
		itoa(pposn[i], p + 1);
		n += strlen(p);
		p = mnemonic + n;
		start = running = pposn[i];
/*
 * SAFETY must be long enough to fit two decimal numbers plus two commas.
 */
#define SAFETY 30
		if (n >= (size - SAFETY)) {
			size *= 2;
			p = malloc((unsigned) size);

			if (!p) xmpi_fail("xmpi (malloc)");

			strcpy(p, mnemonic);
			free(mnemonic);
			mnemonic = p;
		}
	}

	if (running > start) {
		*p = '-';
		itoa(running, p + 1);
	}

	xmpi_run_set_nodes(mnemonic);
	free(mnemonic);
}

/*
 *	nodes_fill
 *
 *	Function:	- fills list with node names and numbers
 */
static void
nodes_fill()

{
	XmString	xstr;
	char		**nodev;
	int		i;
/*
 * Delete all current list entries.
 */
	XmListDeselectAllItems(list_w);
	XmListDeleteAllItems(list_w);
/*
 * Add special mnemonics.
 */
	xstr = XmStringCreateSimple("ALL NODES");
	XmListAddItem(list_w, xstr, 0);
	XmStringFree(xstr);
	xstr = XmStringCreateSimple("LOCAL NODE");
	XmListAddItem(list_w, xstr, 0);
	XmStringFree(xstr);
/*
 * Establish a node<->hostname correspondence.
 */
	if ((nodev = xmpi_sys_hosts()) == 0) {
		return;
	}
/*
 * Place node entries from the parsed boot schema into the list.
 */
	for (i = 0; nodev[i]; i++) {
		linebuf[0] = 'n';
		linebuf[1] = '\0';
		itoa(i, linebuf + 1);
		strcat(linebuf, "  ");
		strncat(linebuf, nodev[i], (MAXLINE - strlen(linebuf)) - 1);
		xstr = XmStringCreateSimple(linebuf);
		XmListAddItem(list_w, xstr, 0);
		XmStringFree(xstr);
	}

	sfh_argv_free(nodev);
}
