/*  synaesthesia-xmms - port of synaesthesia to an xmms plugin
 *  Copyright (C) 1999-2001 Zinx Verituse
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <pthread.h>
#include "math.h"
#include <xmms/plugin.h>
#include <xmms/util.h>
#include <xmms/configfile.h>
#include <xmms/fullscreen.h>
#include <xmms/xmmsctrl.h>
#include "xmms_logo.xpm"
#include "syna_xmms.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
#define VERSION ""
#endif

#define LOGO_WIDTH 95
#define LOGO_HEIGHT 87
#define LOGO_OWIDTH (LOGO_WIDTH)
#define LOGO_OHEIGHT (LOGO_HEIGHT+15)

pthread_mutex_t synx_res_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t update_signal = PTHREAD_COND_INITIALIZER;
static pthread_mutex_t dummy_mutex = PTHREAD_MUTEX_INITIALIZER;

int synx_img_w = 0, synx_img_h = 0;
guint16 *synx_output[3] = { NULL, NULL, NULL };
gboolean synx_am_fullscreen = FALSE;

static GtkWidget *window = NULL, *area = NULL;
static GtkItemFactory *synx_menu = NULL;
static GdkPixmap *bg_pixmap = NULL;
static gboolean config_read = FALSE;
static gint win_w = 0, win_h = 0;

static gboolean synx_quit = TRUE;
static gboolean synx_playing = FALSE;

static void synx_init(void);
static void synx_cleanup(void);
static void synx_playback_start(void);
static void synx_playback_stop(void);
static void synx_render_pcm(gint16 data[2][512]);
static void synx_about(void);
static void synx_screenShow(guint16 *src, unsigned char *dest);

static gint16 pcm_data[2][synx_numsamples];

SynXConfig synx_cfg;

VisPlugin synx_vp;

VisPlugin *get_vplugin_info(void)
{
	memset(&synx_vp, 0, sizeof(synx_vp));
	synx_vp.description =		"Synaesthesia-XMMS "VERSION;
	synx_vp.num_pcm_chs_wanted =	2;
	synx_vp.num_freq_chs_wanted =	0;

	synx_vp.init =			synx_init;
	synx_vp.cleanup =		synx_cleanup;
	synx_vp.configure =		synx_configure;
	synx_vp.playback_start =	synx_playback_start;
	synx_vp.playback_stop =		synx_playback_stop;
	synx_vp.render_pcm =		synx_render_pcm;
	return &synx_vp;
}

#define SYNX_MENU_AUTOFULL 1
#define SYNX_MENU_FULLSCRN 2
#define SYNX_MENU_CONFIGUR 3
#define SYNX_MENU_ABOUT    4

void synx_menu_cb(gpointer cb_data, guint action, GtkWidget *w);

#define SYNX_MENU_ENTRIES 6
GtkItemFactoryEntry synx_menu_entries[] = {
	{ "/Auto Fullscreen",		NULL, synx_menu_cb,	SYNX_MENU_AUTOFULL,	"<ToggleItem>" },
	{ "/Fullscreen",		NULL, synx_menu_cb,	SYNX_MENU_FULLSCRN,	"<ToggleItem>" },
	{ "/-",				NULL, NULL,		0,			"<Separator>" },
	{ "/Config dialog...",		NULL, synx_menu_cb,	SYNX_MENU_CONFIGUR,	"<Item>" },
	{ "/-",				NULL, NULL,		0,			"<Separator>" },
	{ "/About Synaesthesia-XMMS",	NULL, synx_menu_cb,	SYNX_MENU_ABOUT,	"<Item>" },
};

static GdkRgbCmap *cmap = NULL;

void synx_read_config(void)
{
	ConfigFile *cfg;
	gchar *filename;

	if(!config_read)
	{
		synx_cfg.req_w = 256;
		synx_cfg.req_h = 128;
		synx_cfg.scale = FALSE;

		synx_cfg.fgRed = 0.533333;
		synx_cfg.fgGreen = 0.211111;
		synx_cfg.bgRed = 0.011111;
		synx_cfg.bgGreen = 0.355556;
		synx_cfg.brightnessTwiddler = 0.483333;
		synx_cfg.starSize = 0.106667;
		synx_cfg.fadeMode = Wave;
		synx_cfg.pointsAreDiamonds = TRUE;
		synx_cfg.auto_fullscreen = FALSE;

		filename = g_strconcat(g_get_home_dir(), "/.xmms/config", NULL);
		cfg = xmms_cfg_open_file(filename);

		if (cfg) {
			xmms_cfg_read_int(cfg, "Synaesthesia-XMMS", "width", &synx_cfg.req_w);
			xmms_cfg_read_int(cfg, "Synaesthesia-XMMS", "height", &synx_cfg.req_h);
			xmms_cfg_read_boolean(cfg, "Synaesthesia-XMMS", "do_scale", &synx_cfg.scale);

			xmms_cfg_read_double(cfg, "Synaesthesia-XMMS", "fgRed", &synx_cfg.fgRed);
			xmms_cfg_read_double(cfg, "Synaesthesia-XMMS", "fgGreen", &synx_cfg.fgGreen);
			xmms_cfg_read_double(cfg, "Synaesthesia-XMMS", "bgRed", &synx_cfg.bgRed);
			xmms_cfg_read_double(cfg, "Synaesthesia-XMMS", "bgGreen", &synx_cfg.bgGreen);

			xmms_cfg_read_double(cfg, "Synaesthesia-XMMS", "brightnessTwiddler", &synx_cfg.brightnessTwiddler);
			xmms_cfg_read_double(cfg, "Synaesthesia-XMMS", "starSize", &synx_cfg.starSize);

			xmms_cfg_read_int(cfg, "Synaesthesia-XMMS", "fadeMode", &synx_cfg.fadeMode);
			xmms_cfg_read_boolean(cfg, "Synaesthesia-XMMS", "pointsAreDiamonds", &synx_cfg.pointsAreDiamonds);
			xmms_cfg_read_boolean(cfg, "Synaesthesia-XMMS", "auto_fullscreen", &synx_cfg.auto_fullscreen);
			xmms_cfg_free(cfg);
		}
		g_free(filename);
		config_read = TRUE;
	}
}

void synx_write_cfg(void) {
	ConfigFile *cfg;
	gchar *filename;

	filename = g_strconcat(g_get_home_dir(), "/.xmms/config", NULL);
	cfg = xmms_cfg_open_file(filename);
	if (!cfg) cfg = xmms_cfg_new();

	xmms_cfg_write_int(cfg, "Synaesthesia-XMMS", "width", synx_cfg.req_w);
	xmms_cfg_write_int(cfg, "Synaesthesia-XMMS", "height", synx_cfg.req_h);
	xmms_cfg_write_boolean(cfg, "Synaesthesia-XMMS", "do_scale", synx_cfg.scale);

	xmms_cfg_write_int(cfg, "Synaesthesia-XMMS", "fadeMode", synx_cfg.fadeMode);
	xmms_cfg_write_boolean(cfg, "Synaesthesia-XMMS", "pointsAreDiamonds", synx_cfg.pointsAreDiamonds);
	xmms_cfg_write_boolean(cfg, "Synaesthesia-XMMS", "auto_fullscreen", synx_cfg.auto_fullscreen);

	xmms_cfg_write_double(cfg, "Synaesthesia-XMMS", "brightnessTwiddler", synx_cfg.brightnessTwiddler);
	xmms_cfg_write_double(cfg, "Synaesthesia-XMMS", "starSize", synx_cfg.starSize);

	xmms_cfg_write_double(cfg, "Synaesthesia-XMMS", "fgRed", synx_cfg.fgRed);
	xmms_cfg_write_double(cfg, "Synaesthesia-XMMS", "fgGreen", synx_cfg.fgGreen);
	xmms_cfg_write_double(cfg, "Synaesthesia-XMMS", "bgRed", synx_cfg.bgRed);
	xmms_cfg_write_double(cfg, "Synaesthesia-XMMS", "bgGreen", synx_cfg.bgGreen);

	xmms_cfg_write_file(cfg, filename);
	xmms_cfg_free(cfg);
	g_free(filename);
}

void synx_generate_cmap(void)
{
#define 	SBOUND(x) ((x) > 255 ? 255 : (x))
#define 	SPEAKIFY(x) (SBOUND((x) - (x)*(255-(x))/255/2))
#define 	SMAX(x,y) ((x) > (y) ? (x) : (y))

	guint32 colors[256];
	gint i, f, b, red, green, blue;
	gdouble scale, fgRed, fgGreen, fgBlue, bgRed, bgGreen, bgBlue;

	if (!window) return;

	fgRed = synx_cfg.fgRed;
	fgGreen = synx_cfg.fgGreen;
	fgBlue = 1.0 - SMAX(synx_cfg.fgRed,synx_cfg.fgGreen);
	scale = SMAX(SMAX(fgRed,fgGreen),fgBlue);
	fgRed /= scale;
	fgGreen /= scale;
	fgBlue /= scale;

	bgRed = synx_cfg.bgRed;
	bgGreen = synx_cfg.bgGreen;
	bgBlue = 1.0 - SMAX(synx_cfg.bgRed,synx_cfg.bgGreen);
	scale = SMAX(SMAX(bgRed,bgGreen),bgBlue);
	bgRed /= scale;
	bgGreen /= scale;
	bgBlue /= scale;

	for (i = 0; i < 256; i++) {
		f = i&0xf;
		b = i / 16;
		red = SPEAKIFY(b*bgRed*16+f*fgRed*16);
		green = SPEAKIFY(b*bgGreen*16+f*fgGreen*16);
		blue = SPEAKIFY(b*bgBlue*16+f*fgBlue*16);
		colors[i] = (((guint32)(red) << 16) | ((guint32)(green) << 8) | ((guint32)(blue)));
	}

	if (cmap) gdk_rgb_cmap_free(cmap);
	cmap = gdk_rgb_cmap_new(colors, 256);

	pthread_cond_signal(&update_signal);

#undef SBOUND
#undef SMAX
#undef SPEAKIFY
}

static pthread_t main_thread;

void synx_resize(gint new_img_w, gint new_img_h, gboolean force) {
	if (!force && (new_img_w >= synx_cfg.req_w && new_img_h >= synx_cfg.req_h) && !synx_cfg.scale) return;

	if (synx_output[0]) g_free(synx_output[0]);
	if (synx_output[1]) g_free(synx_output[1]);
	if (synx_output[2]) g_free(synx_output[2]);

	synx_img_w = new_img_w; synx_img_h = new_img_h;

	synx_output[0] = (guint16*)g_malloc(synx_img_w*synx_img_h*2);
	synx_output[1] = (guint16*)g_malloc(synx_img_w*synx_img_h*2);
	synx_output[2] = (guint16*)g_malloc(synx_img_w*synx_img_h*2);

	if (!synx_am_fullscreen && window) {
		win_w = synx_img_w; win_h = synx_img_h;
		gtk_widget_set_usize(window, win_w, win_h);
	}
}

static void synx_fullscreen_toggle() {
	SYNX_LOCK();

	synx_am_fullscreen = !synx_am_fullscreen;
	win_w = synx_cfg.req_w; win_h = synx_cfg.req_h;
	if (synx_am_fullscreen) {
		synx_am_fullscreen = xmms_fullscreen_enter(window, &win_w, &win_h);
	} else {
		xmms_fullscreen_leave(window);
		synx_am_fullscreen = FALSE;
	}
	synx_resize(win_w, win_h, FALSE);
	GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(synx_menu, "/Fullscreen"))->active = synx_am_fullscreen;

	SYNX_UNLOCK();
}

static void synx_destroy_cb(GtkWidget *w, gpointer data) {
	synx_vp.disable_plugin(&synx_vp);
}

void synx_menu_cb(gpointer cb_data, guint action, GtkWidget *w) {
        switch (action) {
                case SYNX_MENU_ABOUT:
                        synx_about(); break;
                case SYNX_MENU_CONFIGUR:
                        synx_configure();
                        break;
                case SYNX_MENU_AUTOFULL:
                        synx_cfg.auto_fullscreen = GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(synx_menu, "/Auto Fullscreen"))->active;
			synx_write_cfg();
			break;
                case SYNX_MENU_FULLSCRN:
			synx_fullscreen_toggle();
                        break;
                default:
                        /* ugh */
	}
}

static void synx_press_cb(GtkWidget *w, GdkEventButton *event, gpointer data) {
	GtkRequisition req;
	gint x, y;

	if (event->button == 3 && event->type == GDK_BUTTON_PRESS) {
                GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(synx_menu, "/Auto Fullscreen"))->active = synx_cfg.auto_fullscreen;
                GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(synx_menu, "/Fullscreen"))->active = synx_am_fullscreen;

		gtk_widget_set_sensitive(GTK_WIDGET(gtk_item_factory_get_widget(synx_menu, "/Config dialog...")), !synx_am_fullscreen);
		gtk_widget_set_sensitive(GTK_WIDGET(gtk_item_factory_get_widget(synx_menu, "/About Synaesthesia-XMMS")), !synx_am_fullscreen);

		gtk_widget_size_request(GTK_WIDGET(synx_menu->widget), &req);
		x = CLAMP(event->x_root-2, 0, MAX(0, gdk_screen_width() - req.width));
		y = CLAMP(event->y_root, 0, MAX(0, gdk_screen_height() - req.height));
                gtk_item_factory_popup(synx_menu, x, y, 3, GDK_CURRENT_TIME);
	}
}

static void synx_release_cb(GtkWidget *w, GdkEventButton *event, gpointer data) {
	if (event->button == 2)
		synx_fullscreen_toggle();
}

static void synx_area_expose_cb(GtkWidget *w, GdkEventExpose *event, gpointer data) {
	if (!synx_playing)
		gdk_draw_pixmap(area->window, area->style->white_gc, bg_pixmap, 0, 0, (win_w-LOGO_OWIDTH)/2, (win_h-LOGO_OHEIGHT)/2, LOGO_WIDTH, LOGO_HEIGHT);
	gtk_signal_emit_stop_by_name(GTK_OBJECT(w), "expose-event");
}

static void synx_configure_cb(GtkWidget *w, GdkEventConfigure *event, gpointer data) {
	gtk_widget_set_usize(GTK_WIDGET(data), event->width, event->height);
}

static void synx_keypress_cb(GtkWidget *w, GdkEventKey *event, gpointer data) {
	switch (event->keyval) {
		case GDK_Z:
		case GDK_z:
			xmms_remote_playlist_prev(synx_vp.xmms_session);
			break;
		case GDK_X:
		case GDK_x:
                        xmms_remote_play(synx_vp.xmms_session);
                        break;
                case GDK_C:
                case GDK_c:
                        xmms_remote_pause(synx_vp.xmms_session);
                        break;
                case GDK_V:
                case GDK_v:
                        xmms_remote_stop(synx_vp.xmms_session);
                        break;
                case GDK_B:
                case GDK_b:
                        xmms_remote_playlist_next(synx_vp.xmms_session);
                        break;
	}
}
			
static void synx_gtk_init() {
	GdkColor black = {0,0,0,0};

	window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_title(GTK_WINDOW(window),"Synaesthesia-XMMS");
	gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
	gtk_widget_set_events(window,GDK_BUTTON_PRESS_MASK|GDK_BUTTON_RELEASE_MASK|GDK_KEY_PRESS_MASK);
	gtk_widget_set_usize(window, win_w, win_h);
	gtk_widget_realize(window);

	gtk_signal_connect(GTK_OBJECT(window),"destroy",GTK_SIGNAL_FUNC(synx_destroy_cb),NULL);
	gtk_signal_connect(GTK_OBJECT(window),"destroy",GTK_SIGNAL_FUNC(gtk_widget_destroyed),&window);
	gtk_signal_connect(GTK_OBJECT(window),"button-press-event",GTK_SIGNAL_FUNC(synx_press_cb),NULL);
	gtk_signal_connect(GTK_OBJECT(window),"button-release-event",GTK_SIGNAL_FUNC(synx_release_cb),NULL);
	gtk_signal_connect(GTK_OBJECT(window),"key-press-event",GTK_SIGNAL_FUNC(synx_keypress_cb),NULL);

	area = gtk_drawing_area_new();
	gtk_container_add(GTK_CONTAINER(window),area);
	gtk_widget_realize(area);

	gtk_signal_connect(GTK_OBJECT(area),"expose-event",GTK_SIGNAL_FUNC(synx_area_expose_cb),NULL);
	gtk_signal_connect(GTK_OBJECT(window),"configure-event",GTK_SIGNAL_FUNC(synx_configure_cb),area);

	bg_pixmap = gdk_pixmap_create_from_xpm_d(window->window,NULL,NULL,xmms_logo_xpm);
	gdk_window_set_background(area->window, &black);

	synx_menu = gtk_item_factory_new(GTK_TYPE_MENU, "<Synaesthesia-XMMS>", NULL);
	gtk_item_factory_create_items(synx_menu, SYNX_MENU_ENTRIES, synx_menu_entries, NULL);

	gtk_widget_show(area);
	gtk_widget_show(window);

	gdk_window_clear(area->window);
	gdk_draw_pixmap(area->window, area->style->white_gc, bg_pixmap, 0, 0, (win_w-LOGO_OWIDTH)/2, (win_h-LOGO_OHEIGHT)/2, LOGO_WIDTH, LOGO_HEIGHT);
}

static void *synx_main_thread(void *arg) {
	/* I don't care about lost conditions, darn it! */
	pthread_mutex_lock(&dummy_mutex);

	for (;;) {
		pthread_cond_wait(&update_signal, &dummy_mutex);
		if (synx_quit) break;

		GDK_THREADS_ENTER();
		SYNX_LOCK();
		if (GTK_WIDGET_REALIZED(window)) {
			guchar tmpscr[synx_img_w*synx_img_h];	/* FIXME: Probably GCC-specific. */

			syna_fade();
			syna_coreGo(pcm_data);

			synx_screenShow(synx_output[0], tmpscr);
			gdk_draw_indexed_image(area->window,area->style->white_gc,(win_w-synx_img_w)/2,(win_h-synx_img_h)/2,synx_img_w,synx_img_h,GDK_RGB_DITHER_NONE,tmpscr,synx_img_w,cmap);
		}
		SYNX_UNLOCK();
		GDK_THREADS_LEAVE();
	}

	pthread_mutex_unlock(&dummy_mutex);

	pthread_exit(NULL);
}

static void synx_init(void)
{
	if (window) return;
	synx_read_config();
	synx_quit = FALSE;

	SYNX_LOCK();
	win_w = synx_cfg.req_w;
	win_h = synx_cfg.req_h;
	synx_resize(win_w, win_h, TRUE);

	synx_gtk_init();

	synx_generate_cmap();

	syna_coreInit();
	syna_setStarSize(synx_cfg.starSize);

	if (!xmms_fullscreen_init(window)) {
		gtk_widget_set_sensitive(GTK_WIDGET(gtk_item_factory_get_widget(synx_menu, "/Auto Fullscreen")), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(gtk_item_factory_get_widget(synx_menu, "/Fullscreen")), FALSE);
	}
	SYNX_UNLOCK();

	pthread_create(&main_thread, NULL, synx_main_thread, NULL);
}

static void synx_cleanup(void)
{
	if (synx_quit) return;

	GDK_THREADS_LEAVE();
	synx_quit = TRUE;

	/* Ok ok, so I _DO_ care if this one is missed.. */
	pthread_mutex_lock(&dummy_mutex);
	pthread_cond_signal(&update_signal);
	pthread_mutex_unlock(&dummy_mutex);

	pthread_join(main_thread, NULL);
	GDK_THREADS_ENTER();

	SYNX_LOCK();
	if (window) {
		xmms_fullscreen_leave(window);
		xmms_fullscreen_cleanup(window);	

		gtk_widget_destroy(window);
		gtk_object_destroy(GTK_OBJECT(synx_menu));
		window = NULL;
		g_free(synx_output[0]); synx_output[0] = NULL;
		g_free(synx_output[1]); synx_output[1] = NULL;
		g_free(synx_output[2]); synx_output[2] = NULL;
	}

	if (bg_pixmap) {
		gdk_pixmap_unref(bg_pixmap);
		bg_pixmap = NULL;
	}

	if (cmap) {
		gdk_rgb_cmap_free(cmap);
		cmap = NULL;
	}
	SYNX_UNLOCK();
}

static void synx_playback_start(void) {
	SYNX_LOCK();
	synx_playing = TRUE;
	if (GTK_WIDGET_REALIZED(window) && synx_cfg.auto_fullscreen) {
		if (!synx_am_fullscreen) synx_fullscreen_toggle();
	}
	SYNX_UNLOCK();
}

static void synx_playback_stop(void) {
	SYNX_LOCK();
	synx_playing = FALSE;

	if (GTK_WIDGET_REALIZED(window)) {
		if (synx_cfg.auto_fullscreen) {
			if (synx_am_fullscreen) synx_fullscreen_toggle();
		}
		gdk_window_clear(area->window);
		gdk_draw_pixmap(area->window, area->style->white_gc, bg_pixmap, 0, 0, (win_w-LOGO_OWIDTH)/2, (win_h-LOGO_OHEIGHT)/2, LOGO_WIDTH, LOGO_HEIGHT);
	}
	SYNX_UNLOCK();
}

static void synx_screenShow(guint16 *src, unsigned char *dest) {
	register unsigned long *ptr2 = (unsigned long*)src;
	unsigned long *ptr1 = (unsigned long*)dest;
	int i = synx_img_w*synx_img_h/4;

	/* Asger Alstrup Nielsen's (alstrup@diku.dk)
	   optimized 32 bit screen loop */
	do {
		register unsigned int const r1 = *(ptr2++);
		register unsigned int const r2 = *(ptr2++);

    /*Fade will continue even after value > 16
      thus black pixel will be written when values just > 0
      thus no need to write true black
      if (r1 || r2) { */
#ifndef WORDS_BIGENDIAN
		register unsigned int const v =
		   ((r1 & 0x000000f0ul) >> 4)
		 | ((r1 & 0x0000f000ul) >> 8)
		 | ((r1 & 0x00f00000ul) >> 12)
		 | ((r1 & 0xf0000000ul) >> 16);
		*(ptr1++) = v |
		 ( ((r2 & 0x000000f0ul) << 12)
		 | ((r2 & 0x0000f000ul) << 8)
		 | ((r2 & 0x00f00000ul) << 4)
		 | ((r2 & 0xf0000000ul)));
#else
		register unsigned int const v =
		   ((r2 & 0x000000f0ul) >> 4)
		 | ((r2 & 0x0000f000ul) >> 8)
		 | ((r2 & 0x00f00000ul) >> 12)
		 | ((r2 & 0xf0000000ul) >> 16);
		*(ptr1++) = v |
		 ( ((r1 & 0x000000f0ul) << 12)
		 | ((r1 & 0x0000f000ul) << 8)
		 | ((r1 & 0x00f00000ul) << 4)
		 | ((r1 & 0xf0000000ul)));
#endif
	} while (--i);
}

/* Better not call this function from two different threads :/ */
static void synx_render_pcm(gint16 data[2][512])
{
	static gint i = 0;
	gint j;

	if (!pthread_mutex_trylock(&synx_res_lock)) {
		for (j = 0; j < 512; i++, j+=2) {
			pcm_data[0][i] = data[0][j];
			pcm_data[1][i] = data[1][j];
		}
		if (i >= synx_numsamples) {
			i = 0;
			pthread_cond_signal(&update_signal);
		}
		SYNX_UNLOCK();
	}
}

static void synx_about(void) {
	GtkWidget *dialog, *button, *label;

	dialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(dialog), "About Synaesthesia-XMMS "VERSION);
	gtk_container_border_width(GTK_CONTAINER(dialog), 5);
	label = gtk_label_new(
"synaesthesia-xmms - port of synaesthesia to an xmms plugin\n\
Copyright (C) 1999-2001 Zinx Verituse\n\
\n\
This program is free software; you can redistribute it and/or modify\n\
it under the terms of the GNU General Public License as published by\n\
the Free Software Foundation; either version 2 of the License, or\n\
(at your option) any later version.\n\
\n\
This program is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n\
GNU General Public License for more details.\n\
\n\
You should have received a copy of the GNU General Public License\n\
along with this program; if not, write to the Free Software\n\
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307  USA"
);

	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	button = gtk_button_new_with_label(" Close ");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	gtk_widget_show(dialog);
	gtk_widget_grab_focus(button);
}
