/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>

#include "singit/macros.h"
#include "singit/control.h"
#include "singit/displayer_plugin.h"
#include "singit/wgt_karaoke.h"

#include "displayer_fullscreen_status.h"

#define FULL_SCREEN_HIDE_MOUSE_DELAY 3000
#define BLACK_BACKGROUND 1

static GtkWidget *displayer_fullscreen_window = NULL;

static void displayer_fullscreen_init(void);
static void displayer_fullscreen_finish(void);
static void displayer_fullscreen_set_song(SingitSong *cur_song);
static void displayer_fullscreen_set_time(guint time, GList *real_next);
static void displayer_fullscreen_config_update(void);

DisplayerPlugin fullscreen_dp =
{
	NULL,
	NULL,
	0,
	0,
	NULL,

	"Fullscreen displayer",

	displayer_fullscreen_init,
	displayer_fullscreen_finish,
	displayer_fullscreen_config_update,

	displayer_fullscreen_set_song,
	displayer_fullscreen_set_time,

	NULL,
	NULL,
	NULL,
	NULL,

	NULL,
	NULL
};

DIPSPLAYER_SYMBOL(libdisplayer_fullscreen, fullscreen_dp)

static void fullscreen_set_jumping_ball()
{
	if (STATUS->song &&
		STATUS->guess_synced_lyrics)
	{
		singit_karaoke_widget_set_jumping_ball(
			SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget),
			(singit_song_guess_sync_lyrics(STATUS->song) &&
			STATUS->show_jumping_ball));
	}
	else {
		singit_karaoke_widget_set_jumping_ball(
			SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget),
			STATUS->show_jumping_ball);
	}
}

static void clear_mouse_cursor(GtkWidget *widget, gint clear)
{
	if (clear) {
		GdkCursor *cursor;
		GdkPixmap *p;

		p = gdk_bitmap_create_from_data(widget->window, "\0\0\0", 1, 1);

		cursor = gdk_cursor_new_from_pixmap(p, p,
			&widget->style->fg[GTK_STATE_ACTIVE],
			&widget->style->bg[GTK_STATE_ACTIVE],
			0, 0);

		gdk_window_set_cursor (widget->window, cursor);

		if (cursor) gdk_cursor_destroy (cursor);
		gdk_pixmap_unref(p);
	}
	else { gdk_window_set_cursor (widget->window, NULL); }
}

static gint fullscreen_hide_mouse_cb(gpointer data)
{
	if (STATUS->hide_mouse_id == -1) { return FALSE; }

	clear_mouse_cursor(displayer_fullscreen_window, TRUE);

	STATUS->hide_mouse_id = -1;

	return FALSE;
}

static void fullscreen_hide_mouse_disable()
{
	if (STATUS->hide_mouse_id != -1) {
		gtk_timeout_remove(STATUS->hide_mouse_id);
		STATUS->hide_mouse_id = -1;
	}
}

static void fullscreen_hide_mouse_reset()
{
	fullscreen_hide_mouse_disable();
	STATUS->hide_mouse_id =
		gtk_timeout_add(FULL_SCREEN_HIDE_MOUSE_DELAY, fullscreen_hide_mouse_cb, NULL);
}

static void fullscreen_mouse_moved(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	clear_mouse_cursor(displayer_fullscreen_window, FALSE);
	fullscreen_hide_mouse_reset();
}

static gboolean displayer_fullscreen_keypress(GtkWidget * w, GdkEventKey * event, gpointer data)
{
	gint volume;

        switch(event->keyval) {
                case GDK_Escape: // exit plugin
			fullscreen_dp.disable(&fullscreen_dp);
                        break;
                case GDK_b:
                        xmms_remote_playlist_next(fullscreen_dp.xmms_session);
                        break;
                case GDK_v:
                        xmms_remote_stop(fullscreen_dp.xmms_session);
                        break;
                case GDK_x:
                        xmms_remote_play(fullscreen_dp.xmms_session);
                        break;
		case GDK_y:
			if (STATUS->toggle_yz)
				{ xmms_remote_playlist_prev(fullscreen_dp.xmms_session); }
			break;
                case GDK_z:
			if (!STATUS->toggle_yz)
				{ xmms_remote_playlist_prev(fullscreen_dp.xmms_session); }
                        break;
		case GDK_KP_Add:
			volume = xmms_remote_get_main_volume(fullscreen_dp.xmms_session);
			if (volume < 100) { volume++; }
			xmms_remote_set_main_volume(fullscreen_dp.xmms_session, volume);
			break;
		case GDK_KP_Subtract:
			volume = xmms_remote_get_main_volume(fullscreen_dp.xmms_session);
			if (volume > 0) { volume--; }
			xmms_remote_set_main_volume(fullscreen_dp.xmms_session, volume);
			break;
		case GDK_KP_Enter:
			if (STATUS->jump_to_song == 0) { break; }
			if (xmms_remote_get_playlist_pos(fullscreen_dp.xmms_session) != (STATUS->jump_to_song-1))
				{ xmms_remote_set_playlist_pos(fullscreen_dp.xmms_session, (STATUS->jump_to_song-1)); }
		case GDK_KP_Decimal:
			STATUS->jump_to_song = 0;
			break;
		case GDK_KP_0:
		case GDK_KP_1:
		case GDK_KP_2:
		case GDK_KP_3:
		case GDK_KP_4:
		case GDK_KP_5:
		case GDK_KP_6:
		case GDK_KP_7:
		case GDK_KP_8:
		case GDK_KP_9:
			STATUS->jump_to_song = (STATUS->jump_to_song*10) + (event->keyval - GDK_KP_0);
			if (STATUS->jump_to_song > xmms_remote_get_playlist_length(fullscreen_dp.xmms_session))
				{ STATUS->jump_to_song = (event->keyval - GDK_KP_0); }
			break;
		case GDK_f:
			if (event->state & GDK_CONTROL_MASK) {
				fullscreen_dp.disable(&fullscreen_dp);
			}
			break;
                default:
                        break;
        }

        return TRUE;
}

static gint displayer_fullscreen_delete_event( GtkWidget *widget, GdkEvent  *event, gpointer data )
{
	fullscreen_dp.disable(&fullscreen_dp);
	return (FALSE);
}

void displayer_fullscreen_init(void)
{
	gint w;
	gint h;
//	GtkStyle *style;

	g_return_if_fail(displayer_fullscreen_window == NULL);
	g_return_if_fail(fullscreen_status_ref() != NULL);

#ifdef CODEDEBUG
	DEBUG(("displayer_fullscreen.c [displayer_fullscreen_init]\n"));
#endif

	w = gdk_screen_width();
	h = gdk_screen_height();

	displayer_fullscreen_window = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_container_border_width(GTK_CONTAINER(displayer_fullscreen_window), 0);
	gtk_signal_connect(GTK_OBJECT(displayer_fullscreen_window), "delete_event", GTK_SIGNAL_FUNC(displayer_fullscreen_delete_event), NULL);
	gtk_signal_connect(GTK_OBJECT(displayer_fullscreen_window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &displayer_fullscreen_window);
	gtk_signal_connect(GTK_OBJECT(displayer_fullscreen_window), "key_press_event", GTK_SIGNAL_FUNC(displayer_fullscreen_keypress), NULL);
	gtk_window_set_title(GTK_WINDOW(displayer_fullscreen_window), _("SingIt full screen mode"));
	gtk_widget_set_usize(displayer_fullscreen_window, w, h);
	gtk_widget_set_uposition(displayer_fullscreen_window, 0, 0);

	STATUS->karaoke_widget = singit_karaoke_widget_new();
	gtk_container_add(GTK_CONTAINER(displayer_fullscreen_window), STATUS->karaoke_widget);
	singit_karaoke_widget_set_show_empty_lines (
		SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), TRUE);
	singit_karaoke_widget_set_lines (SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), 7);
	singit_karaoke_widget_set_toplines (SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), 3);
	gtk_signal_connect(GTK_OBJECT(STATUS->karaoke_widget),
		"key_press_event", GTK_SIGNAL_FUNC(displayer_fullscreen_keypress), NULL);
	gtk_widget_show(STATUS->karaoke_widget);

//	style = gtk_style_copy(gtk_widget_get_style(STATUS->karaoke_widget));
//	gtk_style_ref(style);
//	style->bg[GTK_STATE_NORMAL] = style->black;
//	gtk_widget_set_style(STATUS->karaoke_widget, style);
//	gtk_style_unref(style);

//	gtk_widget_realize(displayer_fullscreen_window);

	singit_karaoke_widget_set_optimal_font(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget));

	gtk_signal_connect(GTK_OBJECT(STATUS->karaoke_widget), "motion_notify_event",
		GTK_SIGNAL_FUNC(fullscreen_mouse_moved), NULL);

	gtk_widget_show(displayer_fullscreen_window);
	clear_mouse_cursor(displayer_fullscreen_window, TRUE);
	gtk_window_set_modal(GTK_WINDOW(displayer_fullscreen_window), TRUE);
	gdk_keyboard_grab(displayer_fullscreen_window->window, TRUE, GDK_CURRENT_TIME);
	gtk_grab_add(displayer_fullscreen_window);
	gtk_widget_grab_focus(displayer_fullscreen_window);
}

void displayer_fullscreen_set_time(guint time, GList *real_next)
{
	g_return_if_fail(GTK_WIDGET_REALIZED(displayer_fullscreen_window));

	GDK_THREADS_ENTER();
	singit_karaoke_widget_set_time
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), time);
	GDK_THREADS_LEAVE();
}

void displayer_fullscreen_finish(void)
{
#ifdef CODEDEBUG
	DEBUG(("displayer_fullscreen.c [displayer_fullscreen_finish]\n"));
#endif

	if (displayer_fullscreen_window) {
		fullscreen_hide_mouse_disable();
		gdk_keyboard_ungrab(GDK_CURRENT_TIME);
		gtk_grab_remove(displayer_fullscreen_window);
		gtk_widget_destroy(displayer_fullscreen_window);
	}

	fullscreen_status_unref(NULL);
}

#define CTRL_SESSION	fullscreen_dp.singit_session
#define CTRL_CFG_DATA	status

void displayer_fullscreen_config_update(void)
{
	gchar *temp_str;
	FullscreenStatus *status = STATUS;

#define wgt_set_color(color) \
	temp_str = singit_remote_cfg_color \
		(fullscreen_dp.singit_session, color); \
	singit_karaoke_widget_set_color \
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), color, temp_str); \
	g_free(temp_str);

	REMOTE_CFG_V(debug_enabled, print_debug);

#ifdef CODEDEBUG
	DEBUG(("displayer_fullscreen.c [displayer_fullscreen_config_update]\n"));
#endif

	REMOTE_CFG(toggle_yz);
	REMOTE_CFG(guess_synced_lyrics);
	REMOTE_CFG(show_jumping_ball);
	REMOTE_CFG(find_optimal_font);

	singit_karaoke_widget_freeze
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget));

	// Update colors
	wgt_set_color(skwc_background);
	wgt_set_color(skwc_normal_text);
	wgt_set_color(skwc_normal_bar);
	wgt_set_color(skwc_progress_text);
	wgt_set_color(skwc_progress_bar);

	temp_str = singit_remote_cfg_lyric_font_name
		(fullscreen_dp.singit_session);
	singit_karaoke_widget_set_font
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget), temp_str);
	g_free(temp_str);

	singit_karaoke_widget_set_time_indicators
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget),
		singit_remote_cfg_show_time_indicator(fullscreen_dp.singit_session));
	fullscreen_set_jumping_ball();
	if (STATUS->find_optimal_font) {
		singit_karaoke_widget_set_optimal_font
			(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget));
	}
	singit_karaoke_widget_thaw
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget));
}

#undef CTRL_CFG_DATA
#undef CTRL_SESSION

static void displayer_fullscreen_set_song(SingitSong *new_song)
{
	SingitSong *tmp_song;

#ifdef CODEDEBUG
	DEBUG(("displayer_fullscreen.c [displayer_fullscreen_set_song]\n"));
#endif

	tmp_song = STATUS->song;
	STATUS->song = singit_song_attach(new_song);
	singit_song_detach(&tmp_song);

	g_return_if_fail(GTK_WIDGET_REALIZED(displayer_fullscreen_window));

	singit_karaoke_widget_freeze
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget));

	singit_karaoke_widget_set_song
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget),
		STATUS->song);
//	if (STATUS->find_optimal_font) {
//		singit_karaoke_widget_set_optimal_font
//			(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget));
//	}
	fullscreen_set_jumping_ball();

	singit_karaoke_widget_thaw
		(SINGIT_KARAOKE_WIDGET(STATUS->karaoke_widget));

	singit_song_attach(new_song);
}
