/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


// Semaphore includes
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <errno.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <ctype.h>

#include "singit_macros.h"
#include "singit_macros_private.h"

#include "singit_song.h"
#include "singit_song_private.h"
#include "singit_sha.h"
#include "singit_tools.h"

// static struct sembuf sops[2] = { {0, -1, SEM_UNDO}, {0, 1, SEM_UNDO} };

LSong* l_song_new(gchar* song_filename)
{
	LSong* result;

#	ifdef CODEDEBUG
	DEBUG(("singit_song.c [l_song_new]: "), 9);
#	endif

	result = g_malloc(sizeof(LSong));

/*	result->song_sema_id = semget(ftok("lib-xmms-singit", 'A'), 1, IPC_CREAT | 0666);
	if (result->song_sema_id == -1) {
#		ifdef CODEDEBUG
		DEBUG(("Error - Semaphore missing - "), 9);
		switch (errno) {
		case EACCES:
			DEBUG(("%i / EACCES\n", errno), 9);
			break;
		case EEXIST:
			DEBUG(("%i / EEXIST\n", errno), 9);
			break;
		case EIDRM:
			DEBUG(("%i / EIDRM\n", errno), 9);
			break;
		case ENOENT:
			DEBUG(("%i / ENOENT\n", errno), 9);
			break;
		case ENOMEM:
			DEBUG(("%i / ENOMEM\n", errno), 9);
			break;
		case ENOSPC:
			DEBUG(("%i / ENOSPC\n", errno), 9);
			break;
		default:
			DEBUG(("Unknown Error\n"), 9);
		}
#		endif
		g_free(result);
		return NULL;
	}*/
//	semctl(result->song_sema_id, 0, SETVAL, 1);

	result->first_token = NULL;
	result->last_token = NULL;
	result->active_token = NULL;
	result->lyrics = NULL;
	result->song_length = 0;
	if (song_filename) {
		result->song_filename = g_strdup(song_filename);
#		ifdef HAVE_ID3
		result->id3tag = ID3Tag_New();
		ID3Tag_Link(result->id3tag, song_filename);
#		endif
	}
	else {
		result->song_filename = song_filename;
#		ifdef HAVE_ID3
		result->id3tag = NULL;
#		endif
	}
	result->lyric_filename = NULL;
	result->delimiter = NULL;
	result->lyric_type = lt_none;
	result->file_size = 0;
	result->attachments = 1;

	memset(&result->sha_digest, 0, 20);

#	ifdef CODEDEBUG
	DEBUG(("Ok\n"), 9);
#	endif

	return result;
}

inline void l_song_no_sem_clear(LSong* song)
{
	GList *token = NULL;

#	ifdef CODEDEBUG
	DEBUG(("singit_song.c [l_song_clear]\n"), 9);
#	endif

	if (song->first_token) {
		token = song->first_token;
		while (token != song->last_token) {
			g_free((LToken*) (token->data));
			token = g_list_next(token);
		}
		g_free((LToken*) (token->data));
		g_list_free(song->first_token);

		song->first_token = NULL;
		song->last_token = NULL;
		song->active_token = NULL;
	}
	if (song->lyrics) {
		g_strfreev(song->lyrics);
		song->lyrics = NULL;
	}
	if (song->lyric_filename) {
		g_free(song->lyric_filename);
		song->lyric_filename = NULL;
	}
	if (song->delimiter) {
		g_free(song->delimiter);
		song->delimiter = NULL;
	}

	memset(&song->sha_digest, 0, 20);

	song->file_size = 0;
	song->song_length = 0;
	song->lyric_type = lt_none;
}

void l_song_clear(LSong* song)
{
	if (!song) { return; }
//	semop(song->song_sema_id, &sops[0], 1);
	l_song_no_sem_clear(song);
//	semop(song->song_sema_id, &sops[1], 1);
}

void l_song_no_sem_free(LSong* song)
{
//	gint sem_id = song->song_sema_id;

#	ifdef CODEDEBUG
	DEBUG(("singit_song.c [l_song_free]\n"), 9);
#	endif

	l_song_no_sem_clear(song);

	if (song->song_filename)
		{ g_free(song->song_filename); }

#ifdef HAVE_ID3
	ID3Tag_Delete(song->id3tag);
#endif
	g_free(song);
	song = NULL;

//	if (sem_id) { semctl(sem_id, 0, IPC_RMID, 0); }
}

void l_song_free(LSong* song)
{
	if (!song) { return; }
//	semop(song->song_sema_id, &sops[0], 1);
	l_song_no_sem_free(song);
}

inline LSong* l_song_attach(LSong *song)
{
	if (!song) { return song; }
//    	semop(song->song_sema_id, &sops[0], 1);
	song->attachments++;
//	semop(song->song_sema_id, &sops[1], 1);
	return song;
}

inline LSong* l_song_detach(LSong *song, gboolean free)
{
	if (!song) { return song; }
//	semop(song->song_sema_id, &sops[0], 1);
	song->attachments--;
	if ((song->attachments == 0) && (free)) {
		l_song_no_sem_free(song);
	}
//	else { semop(song->song_sema_id, &sops[1], 1); }
	return NULL;
}

gboolean l_song_set_song_filename(LSong *song, gchar *filename)
{
	if (!song) { return FALSE; }

	g_free(song->song_filename);
	song->song_filename = g_strdup(filename);

	return TRUE;
}

inline gint inl_l_song_is_time_ok(LSong *song, gint time)
{
	if (!song->first_token) { return 0; }
	if (song->active_token) {
		if (time < tTime(song->active_token)) { return -1; }
		if (!g_list_next(song->active_token)) { return 0; }
		if (time >= tTime(g_list_next(song->active_token))) { return 1; }
	}
	else
		{ if (time >= tTime(song->first_token)) { return 1; } }
	return 0;
}

gint l_song_is_time_ok(LSong *song, gint time)
{
	return inl_l_song_is_time_ok(song, time);
}

gint l_song_find_line(LSong* song, gint line)
{
	GList *item = NULL;

#	ifdef CODEDEBUG
	DEBUG(("singit_song.c [l_song_find_line]\n"), 9);
#	endif

	item = song->first_token;
	while (item != song->last_token) {
		if (tLine(item) == line) { return tLine(item); }
		item = g_list_next(item);
	}
	if (tLine(item) == line) { return tLine(item); }
	return -1;
}

gint l_song_find_time(LSong* song, gint time)
{
	GList *item = NULL;

#	ifdef CODEDEBUG
	DEBUG(("singit_song.c [l_song_find_time]\n"), 9);
#	endif

	item = song->first_token;
	while (item != song->last_token) {
		if (tTime(item) == time) { return tTime(item); }
		item = g_list_next(item);
	}
	if (tTime(item) == time) { return tTime(item); }
	return -1;
}

gint compare_token_by_time(gconstpointer a, gconstpointer b)
{
	if ((a == NULL) || (b == NULL)) return 0;
	if ((((LToken*) a)->time ) < (((LToken*) b)->time )) return -1;
	if ((((LToken*) a)->time ) > (((LToken*) b)->time )) return 1;
	return 0;
}

#ifdef HAVE_ID3
inline void l_song_get_id3_tag(LSong *song, gchar *filename)
{
	if (!song->id3tag) { song->id3tag = ID3Tag_New(); }
	else { ID3Tag_Clear(song->id3tag); }
	if (song->id3tag) { ID3Tag_Link(song->id3tag, filename); }
}
#endif

gboolean l_song_load_lyrics(LSong *song, gchar *filename)
{
	FILE *file;
        struct stat stats;
	gboolean result;

#	ifdef CODEDEBUG
	DEBUG(("singit_song.c [l_song_load_lyrics]\n"), 9);
#	endif

	if (!l_song_attach(song)) { return FALSE; }
	l_song_clear(song);

	if (lstat(filename, &stats) == -1) { l_song_detach(song, TRUE); return FALSE; }
        if (!(file = fopen(filename, "r"))) { l_song_detach(song, TRUE); return FALSE; }
	fclose(file);

#	ifdef CODEDEBUG
	DEBUG(("1: File found\n"), 9);
#	endif

#	ifdef HAVE_ID3
	result = l_song_load_id3v2xx_lyrics(song, filename);
	if (!result) {
#	endif
		result = l_song_load_midi_lyrics(song, filename);
		if (!result)
			result = l_song_load_from_text_file(song, filename);
#	ifdef HAVE_ID3
	}
#	endif
	l_song_detach(song, TRUE);
	return result;
}

GList* l_song_find_current_token(LSong *song, gint time, gint state)
{
	GList *item = NULL;

#	ifdef CODEDEBUG
	DEBUG(("singit_song.c [l_song_find_current_token] : "), 9);
#	endif

	if (state == 0) { state = inl_l_song_is_time_ok(song, time); }
	switch (state) {
	case 0: return song->active_token; break;
	case 1:
		item = inl_l_song_get_next_token(song);
		while (item) {
			if (tTime(item) > time) { return g_list_previous(item); }
			item = g_list_next(item);
		}
		if (!item) { return song->last_token; }
		break;
	case -1:
		item = song->active_token;
		while (item) {
			if (tTime(item) <= time) { return item; }
			item = g_list_previous(item);
		}
		break;
	}
	return item;
}

gboolean l_song_lyrics_changed(LSong *song)
{
	FILE *file;
        gchar *buffer;
        struct stat stats;
	gint i;
	SHA_INFO sha;
	guchar new_sha_digest[SHA_DIGESTSIZE];

	if (!song) { return TRUE; }
	if (lstat(song->lyric_filename, &stats) == -1) { return l_song_text_found(song); }
        if (!(file = fopen(song->lyric_filename, "r"))) { return l_song_text_found(song); }
	if (song->file_size != stats.st_size) { return TRUE; }

	if (stats.st_size > 50000) {
		fclose(file);
		return TRUE;
	}

	buffer = g_malloc(stats.st_size + 1);
	if (fread(buffer, 1, stats.st_size, file) != stats.st_size)
	{
		g_free(buffer);
		fclose(file);
		return TRUE;
	}
	fclose(file);
	buffer[stats.st_size] = '\0';

	sha_mem(new_sha_digest, &sha, buffer, stats.st_size);
	for (i = 0; i < 20; i++)
		if (new_sha_digest[i] != song->sha_digest[i]) {
#			ifdef CODEDEBUG
			DEBUG(("singit_song.c [l_song_lyrics_changed] : SHA Changed\n"), 9);
#			endif
			g_free(buffer);
			return TRUE;
		}

	g_free(buffer);

	return FALSE;
}

inline gint inl_l_song_get_text_length(LSong *song)
{
	gint length = 0;
	GList *next = NULL;

	if (song->active_token == NULL) { return -2; }
	next = g_list_next(song->active_token);
	if (next == NULL) { return -1; }
	if (tLine(song->active_token) != tLine(next)) { return -1; }
	length = (tPos(next) - tPos(song->active_token));
	if (length > 0) { return length; }
	else { return -2; }
}

gint l_song_get_text_length(LSong *song)
{
	return inl_l_song_get_text_length(song);
}

gchar* l_song_create_id3v2_stream(LSong *song)
{
	return NULL;
}

inline gboolean l_song_is_empty_item(LSong *song, GList *item)
{
	if (strlen(song->lyrics[tLine(item)]) == 0) { return TRUE; }
	return FALSE;
}

GList* l_song_find_next_lyric_line(LSong *song, GList *item, gboolean empty, guint *hops)
{
	GList *runner = item;
	guint run_hops = 0;
	if (!item) { runner = song->first_token; }
	if (!empty) {
		while (runner) {
			runner = g_list_next(runner);
			run_hops++;
			while (runner) {
				if (!l_song_is_empty_item(song, runner)) {
					if (!item) {
						if (hops) { *hops = run_hops; }
						return runner;
					}
					if (tLine(runner) != tLine(item)) {
						if (hops) { *hops = run_hops; }
						return runner;
					}
				}
				runner = g_list_next(runner);
				run_hops++;
			}
		}
	}
	else {
		while (runner) {
			runner = g_list_next(runner);
			run_hops++;
			if (runner) {
				if (!item) {
					if (hops) { *hops = run_hops; }
					return runner;
				}
				if (tLine(item) != tLine(runner)) {
					if (hops) { *hops = run_hops; }
					return runner;
				}
			}
		}
	}
	if (hops) { *hops = 0; }
	return NULL;
}

GList* l_song_find_prev_lyric_line(LSong *song, GList *item, gboolean empty, guint *hops)
{
	GList *runner = item;
	guint run_hops = 0;
	if (!item) { return NULL; }
	if (!empty) {
		while (runner) {
			runner = g_list_previous(runner);
			run_hops++;
			while (runner) {
				if (!l_song_is_empty_item(song, runner)) {
					if (!item) {
						if (hops) { *hops = run_hops; }
						return runner;
					}
					if (tLine(runner) != tLine(item)) {
						if (hops) { *hops = run_hops; }
						return runner;
					}
				}
				runner = g_list_previous(runner);
				run_hops++;
			}
		}
	}
	else {
		while (runner) {
			runner = g_list_previous(runner);
			run_hops++;
			if (runner) {
				if (!item) {
					if (hops) { *hops = run_hops; }
					return runner;
				}
				if (tLine(item) != tLine(runner)) {
					if (hops) { *hops = run_hops; }
					return runner;
				}
			}
		}
	}
	if (hops) { *hops = 0; }
	return runner;
}

gint l_song_check_sync_lyric_consistency(LSong *song)
{
	GList *cur_item, *prev_item;
	if (!song) { return -1; }
	if (!song->first_token) { return -1; }
	if (song->first_token == song->last_token) { return -1; }
	if (!l_song_guess_sync_lyrics(song)) { return -1; }

	prev_item = song->first_token;

	// Don't check empty line items
	while ((prev_item) && (!l_song_is_empty_item(song, prev_item)))
		{ prev_item = g_list_next(prev_item); }

	cur_item = prev_item;
	while (cur_item) {
		if (tLine(prev_item) > tLine(cur_item)) { return tLine(cur_item); }
		if (tLine(prev_item) == tLine(cur_item))
			if (tPos(prev_item) > tPos(cur_item)) { return tLine(cur_item); }

		prev_item = cur_item;
		cur_item = g_list_next(cur_item);
		// Don't check empty line items
		while ((cur_item) && (!l_song_is_empty_item(song, cur_item)))
			{ cur_item = g_list_next(cur_item); }
	}
	return -1;
}

void l_song_modify_overall_time(LSong *song, gint time)
{
	GList *cur_item;
	gint song_time;
	if (!song) { return; }

	cur_item = song->first_token;
	while (cur_item) {
		song_time = tTime(cur_item) + time;
		if (song_time < 0) { tTime(cur_item) = 0; }
		else { tTime(cur_item) = song_time; }
		cur_item = g_list_next(cur_item);
	}
}

// If any Item has an offset it's synced
gboolean l_song_guess_sync_lyrics(LSong *song)
{
	GList *cur_item;
	if (!song) { return FALSE; }

	cur_item = song->first_token;
	while (cur_item) {
		if (tPos(cur_item) > 0) { return TRUE; }
		cur_item = g_list_next(cur_item);
	}
	return FALSE;
}

inline GList *inl_l_song_get_next_token(LSong *song)
{
	if (song->active_token) { return g_list_next(song->active_token); }
	return song->first_token;
}

inline GList *inl_l_song_get_prev_token(LSong *song)
{
	if (song->active_token) { return g_list_previous(song->active_token); }
	return NULL;
}
