/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <X11/Xlib.h>
#include <gtk/gtk.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>

#include <stdlib.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>

#include "singit_macros.h"
#include "singit_dialogs.h"
#include "singit_config.h"
#include "singit_displayer_plugin.h"
#include "displayer_clist_config.h"
#include "displayer_clist_config_dlg.h"

static GtkWidget *displayer_clist_win = NULL;
static GtkWidget
		*displayer_clist_vbox,
			*displayer_clist_scrlwnd, *displayer_clist_clst,
			*displayer_clist_hbox,
			*displayer_clist_timestamp_status_bar,
			*displayer_clist_song_gtk_event_box, *displayer_clist_song_status_bar;
static GtkTooltips
		*displayer_clist_tooltips;

extern SingitConfigGen *displayer_clist_config;

void displayer_clist_init(void);
void displayer_clist_show(void);
void displayer_clist_hide(void);
void displayer_clist_toggle(void);
void displayer_clist_finish(void);
void displayer_clist_set_time(guint time, LSong *cur_song, GList *real_next);
void displayer_clist_config_update(const SingitConfigData *singit_config);
void displayer_clist_configure(void);

DisplayerPlugin clist_dp =
{
	NULL,
	NULL,
	0,

	"Navigation displayer",

	displayer_clist_init,
	displayer_clist_show,
	displayer_clist_hide,
	displayer_clist_toggle,
	displayer_clist_finish,
	NULL,
	displayer_clist_config_update,

	displayer_clist_set_time,

	NULL,
	displayer_clist_configure,
	NULL,
	NULL,

	NULL,
	NULL
};

DisplayerPlugin *get_dplugin_info(void)
{
	clist_dp.description = g_strdup("Navigation displayer");
	return &clist_dp;
}

// Changed by program - not restored
typedef struct _CListStatus {

	gint lastRow;
	gint line;
	gint jump_to_song;

	guint timestamp_context_id;
	guint song_context_id;
	guint messageCounter;

	GList *next_lyric_line;
	LSong *song;

	gboolean print_debug;
	gboolean auto_resize;
	gboolean show_empty;
	gboolean toggle_yz;

	gint win_size_y;

} CListStatus;

CListStatus displayer_clist_status =
	{ -1, -1, 0, 0, 0, 0, NULL, NULL, FALSE, TRUE, FALSE, FALSE, -1 };

static void displayer_clist_clst_jumpToTime(GtkWidget *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
	GList *item = NULL;
	gboolean isPlaying, doReturn = FALSE;
	LSong *cur_song = NULL;
	if ((!event) || (!displayer_clist_win) ||
		((cur_song = l_song_attach(displayer_clist_status.song)) == NULL)) { return; }

	// Prevent to show the focus row
	if (GTK_CLIST(clist)->focus_row != -1) {
		GTK_CLIST(clist)->focus_row = -1;
		if (GTK_CLIST(clist)->freeze_count == 0)
			gtk_widget_queue_draw (clist);
	}

	isPlaying = xmms_remote_is_playing(clist_dp.xmms_session);

	if (!isPlaying && cur_song) {
		if (!doReturn) { doReturn = (cur_song->active_token == NULL); }
		if (!doReturn) { doReturn = (tLine(cur_song->active_token) == row); }
		if (doReturn) {
			l_song_detach(cur_song, TRUE);
			return;
		}
	}

#	ifdef CODEDEBUG
	DEBUG(("dlg_singit_navigator.c [clst_jumpToTime] : "));
#	endif

	if (displayer_clist_status.lastRow == row) {
		item = cur_song->first_token;
		while (item != cur_song->last_token) {
			if (tLine(item) == row) {
				if (!isPlaying) { xmms_remote_play(clist_dp.xmms_session); }
				xmms_remote_jump_to_time(clist_dp.xmms_session, tTime(item));
				l_song_detach(cur_song, TRUE);
				return;
			}
			item = g_list_next(item);
		}
		if (tLine(item) == row) {
			if (!isPlaying) { xmms_remote_play(clist_dp.xmms_session); }
			xmms_remote_jump_to_time(clist_dp.xmms_session, tTime(item));
		}
	}
	else {
		displayer_clist_status.lastRow = row;
	}
	l_song_detach(cur_song, TRUE);
#	ifdef CODEDEBUG
        DEBUG(("Ok\n"));
#	endif
}

static void set_statusbar_messages(gint time)
{
	gint new_message_count;
	gchar timeStamp[10];

	new_message_count = (time / 1000);
	if (new_message_count < 0) { new_message_count = 0; }
	else
		if (!(time % 1000 == 0)) { new_message_count++; }
	if (displayer_clist_status.messageCounter == new_message_count) { return; }

	timeStamp[9] = '\0';
	if (displayer_clist_status.timestamp_context_id != 0) {
		while (displayer_clist_status.messageCounter > new_message_count) {
			gtk_statusbar_pop(GTK_STATUSBAR(displayer_clist_timestamp_status_bar),
				displayer_clist_status.timestamp_context_id);
			displayer_clist_status.messageCounter--;
		}
	}
	else {
		displayer_clist_status.timestamp_context_id =
			gtk_statusbar_get_context_id(
				GTK_STATUSBAR(displayer_clist_timestamp_status_bar), "timestamps");
		displayer_clist_status.messageCounter = 0;
	}
	while (displayer_clist_status.messageCounter < new_message_count) {
		displayer_clist_status.messageCounter++;
		sprintf(timeStamp, " - %.2i:%.2i ", displayer_clist_status.messageCounter / 60,
			displayer_clist_status.messageCounter % 60);
		gtk_statusbar_push
			(GTK_STATUSBAR(displayer_clist_timestamp_status_bar),
			displayer_clist_status.timestamp_context_id, timeStamp);
	}
}


static void lyrics_not_found(void)
{
	gchar *fileNotFound, *nlbr[1][1];
	gint width, height;

	gtk_clist_freeze(GTK_CLIST(displayer_clist_clst));
	gtk_clist_clear(GTK_CLIST(displayer_clist_clst));
	fileNotFound = g_strconcat(_("Lyrics not found"), NULL);
	nlbr[0][0] = fileNotFound;
	gtk_clist_append(GTK_CLIST(displayer_clist_clst), nlbr[0]);
	if (displayer_clist_status.auto_resize) {
		gdk_window_get_size (displayer_clist_win->window, &width, &height);
		gtk_window_set_default_size(GTK_WINDOW(displayer_clist_win), 230, 80);
	}
	g_free(fileNotFound);
	gtk_clist_thaw(GTK_CLIST(displayer_clist_clst));

	if (displayer_clist_status.timestamp_context_id != 0) { set_statusbar_messages(0); }
}

static void update_clist(LSong *new_song)
{
        gchar *tmp, *lbr[1][1];
        gint i = 0, strlength = -1;
	LSong *my_song = NULL;
	gint maxLineWidth = 0;

#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [build_clist] : "));
#	endif

	gtk_clist_freeze(GTK_CLIST(displayer_clist_clst));
	gtk_clist_clear(GTK_CLIST(displayer_clist_clst));
	my_song = l_song_attach(new_song);
	if (my_song) {
		if (displayer_clist_status.timestamp_context_id != 0) { set_statusbar_messages(0); }

		i = 0;
		while (my_song->lyrics[i]) {
			strlength = strlen(&my_song->lyrics[i][0]);
			if (strlength > 0) { tmp = g_strndup(&my_song->lyrics[i][0], strlength); }
			else { tmp = strdup("\0"); }
			lbr[0][0] = tmp;
			gtk_clist_append(GTK_CLIST(displayer_clist_clst), lbr[0]);
			if (strlength > 0) {
		  		strlength = gdk_string_width(gtk_widget_get_style(displayer_clist_clst)->font, tmp);
		  		if (strlength > maxLineWidth) { maxLineWidth = strlength; }
			}
			g_free(tmp);
			i++;
		}

		#ifdef CODEDEBUG
		DEBUG((" Most length line: %i\n", maxLineWidth));
		#endif
		if (maxLineWidth < 150) { maxLineWidth = 150; }
		if (displayer_clist_status.auto_resize) {
			gtk_window_set_default_size(GTK_WINDOW(displayer_clist_win),
				maxLineWidth + 45, displayer_clist_status.win_size_y);
		}
		l_song_detach(my_song, TRUE);
	}
	gtk_clist_thaw(GTK_CLIST(displayer_clist_clst));
}

static void set_status_filename(gchar *fileName)
{
	gchar *tmp;
	if (!displayer_clist_win) { return; }
	if (displayer_clist_status.song_context_id) {
		gtk_statusbar_pop(GTK_STATUSBAR(displayer_clist_song_status_bar), displayer_clist_status.song_context_id);
	}
	else {
		displayer_clist_status.song_context_id =
			gtk_statusbar_get_context_id(GTK_STATUSBAR(displayer_clist_song_status_bar), "filename");
	}
	tmp = g_strconcat(" ", fileName, NULL);
	gtk_statusbar_push(GTK_STATUSBAR(displayer_clist_song_status_bar),
		displayer_clist_status.song_context_id, tmp);
	gtk_tooltips_set_tip(displayer_clist_tooltips, displayer_clist_song_gtk_event_box, fileName, NULL);
	g_free(tmp);
}

void select_item(gint new_item, gboolean show)
{
	if (displayer_clist_status.line > -1)
		gtk_clist_unselect_row(GTK_CLIST(displayer_clist_clst),
			displayer_clist_status.line, 0);

	if (show) {
		gtk_clist_select_row(GTK_CLIST(displayer_clist_clst), new_item, 0);
		gtk_clist_moveto(GTK_CLIST(displayer_clist_clst), new_item, -1, 0.5, 0);
	}
	displayer_clist_status.line = new_item;
}

void displayer_clist_set_time(guint time, LSong *cur_song, GList *real_next)
{
	GList *item = NULL;
	LSong* my_song = NULL, *tmp_song;
	gboolean update_song = FALSE;
	gboolean update_selection = FALSE;
	gboolean update_timer = FALSE;
	gboolean show_item = FALSE;
	gint new_item = -1;
	gint timer;

	if (displayer_clist_win && GTK_WIDGET_VISIBLE(displayer_clist_win)) {
		my_song = l_song_attach(cur_song);
		if (!my_song) { return; }

		if ((my_song != displayer_clist_status.song) ||
			(strcmp(my_song->song_filename, displayer_clist_status.song->song_filename) != 0))
		{
			displayer_clist_status.line = -1;
			tmp_song = displayer_clist_status.song;
			displayer_clist_status.song = l_song_attach(my_song);
			l_song_detach(tmp_song, TRUE);
			update_song = TRUE;
		}

		if (l_song_text_found(my_song)) {
			// Has the active token and ???
			if (l_song_lyrics_found(my_song)) {
				new_item = tLine(my_song->active_token);
				if (displayer_clist_status.line != new_item)
				{
					if (!l_song_is_empty_item(my_song, my_song->active_token)
						|| displayer_clist_status.show_empty)
					{
						show_item = TRUE;
					}

					update_selection = TRUE;
				}

				// Has the timer status changed ???
				item = (real_next) ? real_next : my_song->last_token;
				if (item != NULL) {
					timer = (time / 1000);
					if (timer < 0) { timer = 0; }
					else
						if (!(time % 1000 == 0)) { timer++; }
					if (displayer_clist_status.messageCounter != timer)
						{ update_timer = TRUE; }
				}
			}

			// Any updates needed ???
			if (update_song || update_selection || update_timer) {
				// Obtain GDK lock
				GDK_THREADS_ENTER();

				// Run any updates
				if (update_song) {
					set_status_filename(my_song->song_filename);
					update_clist(my_song);
				}
				if (update_selection) { select_item(new_item, show_item); }
				if (update_timer) { set_statusbar_messages(tTime(item) - time); }

				// Release GDK lock
				GDK_THREADS_LEAVE();
			}
		}
		else {
			GDK_THREADS_ENTER();
			lyrics_not_found();
			GDK_THREADS_LEAVE();
		}
		l_song_detach(my_song, TRUE);
	}
}

static gboolean displayer_clist_win_keypress(GtkWidget * w, GdkEventKey * event, gpointer data)
{
	gint volume, value;

        switch(event->keyval) {
                case GDK_Escape: // exit plugin
                        clist_dp.disable(&clist_dp);
                        break;
		case GDK_a:
			if (event->state & GDK_CONTROL_MASK) { singit_about_show(); }
			break;
                case GDK_b:
                        xmms_remote_playlist_next(clist_dp.xmms_session);
                        break;
                case GDK_c:
			if (event->state & GDK_CONTROL_MASK) { displayer_clist_config_show(); }
			else { xmms_remote_pause(clist_dp.xmms_session); }
                        break;
		case GDK_e:
			if (event->state & GDK_CONTROL_MASK) { singit_editor_show(); }
			break;
#		ifdef HAVE_ID3
		case GDK_i:
			if (event->state & GDK_CONTROL_MASK) { singit_id3dlg_show(); }
			break;
#		endif
		case GDK_p:
			if (event->state & GDK_CONTROL_MASK) { singit_config_show(); }
			break;
                case GDK_v:
                        xmms_remote_stop(clist_dp.xmms_session);
                        break;
                case GDK_x:
                        xmms_remote_play(clist_dp.xmms_session);
                        break;
		case GDK_y:
			if (displayer_clist_status.toggle_yz)
				{ xmms_remote_playlist_prev(clist_dp.xmms_session); }
			break;
                case GDK_z:
			if (!displayer_clist_status.toggle_yz)
				{ xmms_remote_playlist_prev(clist_dp.xmms_session); }
                        break;
		case GDK_KP_Add:
			volume = xmms_remote_get_main_volume(clist_dp.xmms_session);
			if (volume < 100) { volume++; }
			xmms_remote_set_main_volume(clist_dp.xmms_session, volume);
			break;
		case GDK_KP_Subtract:
			volume = xmms_remote_get_main_volume(clist_dp.xmms_session);
			if (volume > 0) { volume--; }
			xmms_remote_set_main_volume(clist_dp.xmms_session, volume);
			break;
		case GDK_KP_Enter:
			if (displayer_clist_status.jump_to_song == 0) { break; }
			if (xmms_remote_get_playlist_pos(clist_dp.xmms_session) !=
				(displayer_clist_status.jump_to_song - 1)) {
				xmms_remote_set_playlist_pos
					(clist_dp.xmms_session, (displayer_clist_status.jump_to_song - 1));
			}
		case GDK_KP_Decimal:
			displayer_clist_status.jump_to_song = 0;
			break;
		case GDK_KP_0:
		case GDK_KP_1:
		case GDK_KP_2:
		case GDK_KP_3:
		case GDK_KP_4:
		case GDK_KP_5:
		case GDK_KP_6:
		case GDK_KP_7:
		case GDK_KP_8:
		case GDK_KP_9:
		case GDK_0:
		case GDK_1:
		case GDK_2:
		case GDK_3:
		case GDK_4:
		case GDK_5:
		case GDK_6:
		case GDK_7:
		case GDK_8:
		case GDK_9:
			value = event->keyval - GDK_0;
			if (value < 0 || value > 9)
				value = event->keyval - GDK_KP_0;

			if (event->state & GDK_CONTROL_MASK) {
				enable_plugin(((value + 9) % 10));
			}
			else {
				displayer_clist_status.jump_to_song = (displayer_clist_status.jump_to_song * 10) + value;
				if (displayer_clist_status.jump_to_song > xmms_remote_get_playlist_length
					(clist_dp.xmms_session)) { displayer_clist_status.jump_to_song = value; }
			}
			break;
                default:
			// printf("%x - %x\n", event->state, event->keyval);
                        break;
        }

        return TRUE;
}

void displayer_clist_resize_event (GtkWidget *widget, gpointer user_data)
{
	gint width, height;
	LSong *cur_song = NULL;

	if (!displayer_clist_win) { return; }
	cur_song = l_song_attach(displayer_clist_status.song);
	if (!cur_song) { return; }
	if (l_song_lyrics_found(cur_song) && displayer_clist_win->window)
		if (displayer_clist_status.auto_resize) {
			gdk_window_get_size (displayer_clist_win->window, &width, &height);
			displayer_clist_status.win_size_y = height;
		}
	if (displayer_clist_clst)
		if (cur_song->active_token)
			gtk_clist_moveto(GTK_CLIST(displayer_clist_clst), tLine(cur_song->active_token), -1, 0.5, 0);
	l_song_detach(cur_song, TRUE);
}

void displayer_clist_finish(void)
{
	gint posX, posY;

#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [displayer_clist_finish]\n"));
#	endif

	if (displayer_clist_win) {
//		if (displayer_clist_status.timestamp_context_id != 0) { set_statusbar_messages(0); }
		if (displayer_clist_win->window) {
			gdk_window_get_root_origin(displayer_clist_win->window, &posX, &posY);
			singit_config_save_plugin_position("clist_navigator", posX, posY, -1, -1);
		}
		gtk_widget_destroy(displayer_clist_win);

		displayer_clist_status.messageCounter = 0;
		displayer_clist_status.timestamp_context_id = 0;
	}

	displayer_clist_status.lastRow = -1;
	displayer_clist_status.line = 0;

	displayer_clist_status.song = l_song_detach(displayer_clist_status.song, TRUE);
}

gint displayer_clist_delete_event(GtkWidget *widget, GdkEvent  *event, gpointer data)
{
	clist_dp.disable(&clist_dp);
	return (FALSE);
}

void displayer_clist_init(void)
{
	gchar* title;
	gint posX, posY;
	CListConfigData *ccd;

#	ifdef CODEDEBUG
	DEBUG(("dlg_singit_navigator.c [displayer_clist_init]\n"));
#	endif

	if (displayer_clist_win) { return; }

	if (!singit_config_gen_attach(displayer_clist_config)) {
		displayer_clist_config = displayer_clist_config_new();
		singit_config_gen_load(SINGIT_CONFIG_GEN(displayer_clist_config));
	}

	if (!singit_config_gen_attach(SINGIT_CONFIG_GEN(displayer_clist_config))) {
		displayer_clist_config = displayer_clist_config_new();
		singit_config_gen_load(SINGIT_CONFIG_GEN(displayer_clist_config));
	}
	ccd = getDCC(displayer_clist_config);
	if (!ccd) {
		singit_config_gen_detach(SINGIT_CONFIG_GEN(displayer_clist_config));
		return;
	}

	displayer_clist_tooltips = gtk_tooltips_new ();

	displayer_clist_win = gtk_window_new(GTK_WINDOW_DIALOG);
	title = g_strconcat(SINGIT_VERSION_STRING, " - ", _("Lyrics navigator"), NULL);
	gtk_window_set_title(GTK_WINDOW(displayer_clist_win), title);
	g_free(title);
	gtk_window_set_policy(GTK_WINDOW(displayer_clist_win), TRUE, TRUE, TRUE);
	gtk_widget_set_usize(displayer_clist_win, 180, 200);
	gtk_signal_connect(GTK_OBJECT(displayer_clist_win), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &displayer_clist_win);
	gtk_signal_connect(GTK_OBJECT(displayer_clist_win), "delete_event", GTK_SIGNAL_FUNC(displayer_clist_delete_event), NULL);
	gtk_signal_connect(GTK_OBJECT(displayer_clist_win), "key-press-event", GTK_SIGNAL_FUNC(displayer_clist_win_keypress), NULL);
	gtk_signal_connect(GTK_OBJECT(displayer_clist_win), "size-allocate", GTK_SIGNAL_FUNC(displayer_clist_resize_event), NULL);
	gtk_container_border_width(GTK_CONTAINER(displayer_clist_win), 5);

	displayer_clist_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(displayer_clist_win), displayer_clist_vbox);
	gtk_widget_show(displayer_clist_vbox);

	displayer_clist_scrlwnd = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(displayer_clist_scrlwnd), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(displayer_clist_vbox), displayer_clist_scrlwnd, TRUE, TRUE, 0);
	gtk_widget_show (displayer_clist_scrlwnd);

	displayer_clist_clst = gtk_clist_new(1);
	gtk_signal_connect(GTK_OBJECT(displayer_clist_clst), "select_row",
		GTK_SIGNAL_FUNC(displayer_clist_clst_jumpToTime), NULL);
	gtk_clist_set_selection_mode(GTK_CLIST(displayer_clist_clst), GTK_SELECTION_SINGLE);
	gtk_clist_set_reorderable (GTK_CLIST(displayer_clist_clst), FALSE);
	gtk_container_add(GTK_CONTAINER(displayer_clist_scrlwnd), displayer_clist_clst);
	gtk_clist_set_column_width(GTK_CLIST(displayer_clist_clst), 0, 1);
	gtk_widget_show(displayer_clist_clst);

	displayer_clist_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(displayer_clist_vbox), displayer_clist_hbox, FALSE, FALSE, 1);
	gtk_widget_show(displayer_clist_hbox);

	displayer_clist_timestamp_status_bar = gtk_statusbar_new();
	gtk_box_pack_start (GTK_BOX(displayer_clist_hbox), displayer_clist_timestamp_status_bar, FALSE, FALSE, 0);
	gtk_widget_show(displayer_clist_timestamp_status_bar);

	displayer_clist_song_gtk_event_box = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(displayer_clist_hbox), displayer_clist_song_gtk_event_box);
	gtk_widget_show (displayer_clist_song_gtk_event_box);

	displayer_clist_song_status_bar = gtk_statusbar_new();
	gtk_container_add (GTK_CONTAINER (displayer_clist_song_gtk_event_box), displayer_clist_song_status_bar);
	gtk_widget_show(displayer_clist_song_status_bar);

	gtk_object_set_data (GTK_OBJECT (displayer_clist_win), "tooltips", displayer_clist_tooltips);

	/* Normally a window is created when it's shown first, so if not shown
	we must explicit realize it to get a GDK_WINDOW */
	gtk_widget_realize(displayer_clist_win);

	displayer_clist_status.song_context_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(displayer_clist_song_status_bar), "filename");

	displayer_clist_status.timestamp_context_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(displayer_clist_timestamp_status_bar), "timestamps");
	gtk_statusbar_push(GTK_STATUSBAR(displayer_clist_timestamp_status_bar),
		displayer_clist_status.timestamp_context_id, "         ");

	if (singit_config_load_plugin_position("clist_navigator", &posX, &posY, NULL, NULL))
		gdk_window_move(displayer_clist_win->window, posX, posY);
	else
		gtk_window_set_position
			(GTK_WINDOW(displayer_clist_win), GTK_WIN_POS_CENTER);
}

void displayer_clist_show(void)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [displayer_clist_show]\n"));
#	endif

	if (displayer_clist_win) {
		if (!GTK_WIDGET_VISIBLE(displayer_clist_win))
			{ gtk_widget_show(displayer_clist_win); }
		else { gdk_window_raise(displayer_clist_win->window); }
	}
}

void displayer_clist_hide(void)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [displayer_clist_hide]\n"));
#	endif

	if (displayer_clist_win) {
		if (GTK_WIDGET_VISIBLE(displayer_clist_win) &&
			(!(displayer_clist_config && getDCC(displayer_clist_config)->showOnText &&
			 displayer_clist_status.song &&
			 l_song_text_found(displayer_clist_status.song))
			)
		)
		{
			gtk_widget_hide(displayer_clist_win);
		}
	}
}

void displayer_clist_toggle(void)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [displayer_clist_toggle]\n"));
#	endif

	if (displayer_clist_win) {
		if (GTK_WIDGET_VISIBLE(displayer_clist_win))
			{ gtk_widget_hide(displayer_clist_win); }
		else  { gtk_widget_show(displayer_clist_win); }
	}
}

void displayer_clist_config_update(const SingitConfigData *singit_config)
{
	displayer_clist_status.auto_resize = singit_config->autoResize;
	displayer_clist_status.print_debug = singit_config->debugEnable;
	displayer_clist_status.show_empty = singit_config->showEmpty;
	displayer_clist_status.toggle_yz = singit_config->toggleYZ;
}

void displayer_clist_configure(void)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [displayer_clist_configure]\n"));
#	endif

	displayer_clist_config_show();
}
