/*
 * $Id:$
 *
 * Copyright (C) Adam Feakin <af7567@bris.ac.uk>
 *
 * modified by Daniel Stodden <stodden@in.tum.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 * xmms-alarm.c - a XMMS plugin which allows you to set a time for it to
 * start playing a playlist with the volume fading up and more things
 * the next time I get bored
 */

/* this file really should get split/cleaned up sometime ;) */
#include "config.h"

#if STDC_HEADERS
# include <stdlib.h>
#endif

#include <time.h>
#if TM_IN_SYS_TIME
# include <sys/time.h>
#endif

#include <stdio.h>
#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>
#include <xmms/configfile.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <pthread.h>
#include <assert.h>
#include <math.h>

#include "alarm.h"
#include "interface.h"
#include "callbacks.h"

static pthread_t start_tid;	       /* thread id of alarm loop */
static pthread_t stop_tid;	       /* thread id of stop loop */
//static pthread_mutex_t fader_lock = PTHREAD_MUTEX_INITIALIZER;
static gboolean fader_lock = FALSE;

static GeneralPlugin alarm_plugin;

static struct
{
   GtkSpinButton *alarm_h;
   GtkSpinButton *alarm_m;

   GtkToggleButton *stop_on;
   GtkSpinButton *stop_h;
   GtkSpinButton *stop_m;

   GtkRange *volume;
   GtkRange *quietvol;

   GtkSpinButton *fading;

   GtkEntry *cmdstr;
   GtkToggleButton *cmd_on;

   GtkEntry *playlist;

   int default_hour;
   int default_min;

   alarmday mon;
   alarmday tue;
   alarmday wed;
   alarmday thu;
   alarmday fri;
   alarmday sat;
   alarmday sun;

   GtkEntry *reminder;
   GtkToggleButton *reminder_cb;
   gchar *reminder_msg;
   gboolean reminder_on;
}
alarm_conf;

static gint alarm_h, alarm_m;

static gboolean stop_on;
static gint stop_h, stop_m;

static gint volume, quietvol;

static gint fading;

static gchar *cmdstr = NULL;
static gboolean cmd_on;

static gchar *playlist = NULL;

static GtkWidget *config_dialog = NULL;
static GtkWidget *alarm_dialog = NULL;

static GtkWidget *lookup_widget(GtkWidget *w, const gchar *name)
{
   GtkWidget *widget;

   widget = (GtkWidget*) gtk_object_get_data(GTK_OBJECT(w),
					     name);
   g_return_val_if_fail(widget != NULL, NULL);

   return widget;
}

static void dialog_destroyed(GtkWidget *dialog, gpointer data)
{
   DEBUG("dialog destroyed\n");
   *(GtkObject**)data = NULL;
}

static inline gboolean dialog_visible(GtkWidget *dialog)
{
   return(((dialog != NULL) && GTK_WIDGET_VISIBLE(dialog)));
}

static inline gchar *alarm_config_filename()
{
   return g_strconcat(g_get_home_dir(), "/.xmms/alarmconfig", NULL);
}

/*
 * tell the user about that bug
 */
static void alarm_warning(void)
{

   static GtkWidget *warning_dialog = NULL;

   if(dialog_visible(warning_dialog))
     return;

   warning_dialog = create_warning_dialog();

   gtk_signal_connect(GTK_OBJECT(warning_dialog), "destroy",
		      GTK_SIGNAL_FUNC(dialog_destroyed), &warning_dialog);

   gtk_widget_show_all(warning_dialog);

   return;
}

/*
 * the callback function that is called when the save button is
 * pressed saves configuration to ~/.xmms/alarmconfig
 */
void alarm_save(GtkButton *w, gpointer data)
{
   ConfigFile *conf;
   gchar *filename = alarm_config_filename();

   DEBUG("alarm_save\n");

   /*
    * create a new file if this doesnt exist
    */
   conf = xmms_cfg_open_file(filename);

   if (conf == (ConfigFile *)NULL)
     {
	DEBUG("creating new config file: %s\n", filename);
	conf = xmms_cfg_new();
     }

   /*
    * update the live values and write them out
    */
   alarm_h = alarm_conf.default_hour = 
     gtk_spin_button_get_value_as_int(alarm_conf.alarm_h);
   xmms_cfg_write_int(conf, "alarm", "alarm_h", alarm_h);

   alarm_m = alarm_conf.default_min = 
     gtk_spin_button_get_value_as_int(alarm_conf.alarm_m);
   xmms_cfg_write_int(conf, "alarm", "alarm_m", alarm_m);


   stop_h =
     gtk_spin_button_get_value_as_int( alarm_conf.stop_h);
   //xmms_cfg_write_int(conf, "alarm", "stop_h", stop_h);

   stop_m =
     gtk_spin_button_get_value_as_int(alarm_conf.stop_m);
   //xmms_cfg_write_int(conf, "alarm", "stop_m", stop_m);

   stop_on =
     gtk_toggle_button_get_active(alarm_conf.stop_on);
   //xmms_cfg_write_boolean(conf, "alarm", "stop_on", stop_on);

   /* days of the week */
   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.mon.cb)))
     alarm_conf.mon.flags = 0;
   else
     alarm_conf.mon.flags = ALARM_OFF;
   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.mon.cb_def)))
     alarm_conf.mon.flags|=ALARM_DEFAULT;

   alarm_conf.mon.hour = 
     gtk_spin_button_get_value_as_int(alarm_conf.mon.spin_hr);
   alarm_conf.mon.min = 
     gtk_spin_button_get_value_as_int(alarm_conf.mon.spin_min);

   xmms_cfg_write_int(conf, "alarm", "mon_flags", alarm_conf.mon.flags);
   xmms_cfg_write_int(conf, "alarm", "mon_h", alarm_conf.mon.hour);
   xmms_cfg_write_int(conf, "alarm", "mon_m", alarm_conf.mon.min);

   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.tue.cb)))
     alarm_conf.tue.flags = 0;
   else
     alarm_conf.tue.flags = ALARM_OFF;
   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.tue.cb_def)))
     alarm_conf.tue.flags|=ALARM_DEFAULT;

   alarm_conf.tue.hour = 
     gtk_spin_button_get_value_as_int(alarm_conf.tue.spin_hr);
   alarm_conf.tue.min = 
     gtk_spin_button_get_value_as_int(alarm_conf.tue.spin_min);

   xmms_cfg_write_int(conf, "alarm", "tue_flags", alarm_conf.tue.flags);
   xmms_cfg_write_int(conf, "alarm", "tue_h", alarm_conf.tue.hour);
   xmms_cfg_write_int(conf, "alarm", "tue_m", alarm_conf.tue.min);

   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.wed.cb)))
     alarm_conf.wed.flags = 0;
   else
     alarm_conf.wed.flags = ALARM_OFF;
   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.wed.cb_def)))
     alarm_conf.wed.flags|=ALARM_DEFAULT;

   alarm_conf.wed.hour = 
     gtk_spin_button_get_value_as_int(alarm_conf.wed.spin_hr);
   alarm_conf.wed.min = 
     gtk_spin_button_get_value_as_int(alarm_conf.wed.spin_min);

   xmms_cfg_write_int(conf, "alarm", "wed_flags", alarm_conf.wed.flags);
   xmms_cfg_write_int(conf, "alarm", "wed_h", alarm_conf.wed.hour);
   xmms_cfg_write_int(conf, "alarm", "wed_m", alarm_conf.wed.min);

   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.thu.cb)))
     alarm_conf.thu.flags = 0;
   else
     alarm_conf.thu.flags = ALARM_OFF;
   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.thu.cb_def)))
     alarm_conf.thu.flags|=ALARM_DEFAULT;

   alarm_conf.thu.hour = 
     gtk_spin_button_get_value_as_int(alarm_conf.thu.spin_hr);
   alarm_conf.thu.min = 
     gtk_spin_button_get_value_as_int(alarm_conf.thu.spin_min);

   xmms_cfg_write_int(conf, "alarm", "thu_flags", alarm_conf.thu.flags);
   xmms_cfg_write_int(conf, "alarm", "thu_h", alarm_conf.thu.hour);
   xmms_cfg_write_int(conf, "alarm", "thu_m", alarm_conf.thu.min);

   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.fri.cb)))
     alarm_conf.fri.flags = 0;
   else
     alarm_conf.fri.flags = ALARM_OFF;
   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.fri.cb_def)))
     alarm_conf.fri.flags|=ALARM_DEFAULT;

   alarm_conf.fri.hour = 
     gtk_spin_button_get_value_as_int(alarm_conf.fri.spin_hr);
   alarm_conf.fri.min = 
     gtk_spin_button_get_value_as_int(alarm_conf.fri.spin_min);

   xmms_cfg_write_int(conf, "alarm", "fri_flags", alarm_conf.fri.flags);
   xmms_cfg_write_int(conf, "alarm", "fri_h", alarm_conf.fri.hour);
   xmms_cfg_write_int(conf, "alarm", "fri_m", alarm_conf.fri.min);

   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.sat.cb)))
     alarm_conf.sat.flags = 0;
   else
     alarm_conf.sat.flags = ALARM_OFF;
   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.sat.cb_def)))
     alarm_conf.sat.flags|=ALARM_DEFAULT;

   alarm_conf.sat.hour = 
     gtk_spin_button_get_value_as_int(alarm_conf.sat.spin_hr);
   alarm_conf.sat.min = 
     gtk_spin_button_get_value_as_int(alarm_conf.sat.spin_min);

   xmms_cfg_write_int(conf, "alarm", "sat_flags", alarm_conf.sat.flags);
   xmms_cfg_write_int(conf, "alarm", "sat_h", alarm_conf.sat.hour);
   xmms_cfg_write_int(conf, "alarm", "sat_m", alarm_conf.sat.min);

   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.sun.cb)))
     alarm_conf.sun.flags = 0;
   else
     alarm_conf.sun.flags = ALARM_OFF;
   if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(alarm_conf.sun.cb_def)))
     alarm_conf.sun.flags|=ALARM_DEFAULT;

   alarm_conf.sun.hour = 
     gtk_spin_button_get_value_as_int(alarm_conf.sun.spin_hr);
   alarm_conf.sun.min = 
     gtk_spin_button_get_value_as_int(alarm_conf.sun.spin_min);

   xmms_cfg_write_int(conf, "alarm", "sun_flags", alarm_conf.sun.flags);
   xmms_cfg_write_int(conf, "alarm", "sun_h", alarm_conf.sun.hour);
   xmms_cfg_write_int(conf, "alarm", "sun_m", alarm_conf.sun.min);

   /* END: days of week */

   volume =
     gtk_range_get_adjustment(alarm_conf.volume)->value;
   xmms_cfg_write_int(conf, "alarm", "volume", volume);

   quietvol =
     gtk_range_get_adjustment(alarm_conf.quietvol)->value;
   xmms_cfg_write_int(conf, "alarm", "quietvol", quietvol);

   fading =
     gtk_spin_button_get_value_as_int(alarm_conf.fading);
   //xmms_cfg_write_int(conf, "alarm", "fading", fading);

   /* lets check to see if we need to show the bug warning */
   if((stop_on == TRUE) &&
      ((((stop_h * 60) + stop_m) * 60) < (fading + 65)))
   {
	   DEBUG("Displaying bug warning, stop %dh %dm, fade %d\n",
	         stop_h, stop_m, fading);
	   alarm_warning();
   }
   else
   {
	   /* write the new values */
	   xmms_cfg_write_int(conf, "alarm", "stop_h", stop_h);
	   xmms_cfg_write_int(conf, "alarm", "stop_m", stop_m);
	   xmms_cfg_write_int(conf, "alarm", "fading", fading);
	   xmms_cfg_write_boolean(conf, "alarm", "stop_on", stop_on);
   }


   g_free(cmdstr);
   cmdstr = gtk_editable_get_chars(GTK_EDITABLE(alarm_conf.cmdstr),
				   0, -1);
   xmms_cfg_write_string(conf, "alarm", "cmdstr", cmdstr);

   cmd_on =
     gtk_toggle_button_get_active(alarm_conf.cmd_on);
   xmms_cfg_write_boolean(conf, "alarm", "cmd_on", cmd_on);

   g_free(playlist);
   playlist = gtk_editable_get_chars(GTK_EDITABLE(alarm_conf.playlist),
				     0, -1);
   xmms_cfg_write_string(conf, "alarm", "playlist", playlist);

   /* reminder */
   g_free(alarm_conf.reminder_msg);
   alarm_conf.reminder_msg = gtk_editable_get_chars(GTK_EDITABLE(alarm_conf.reminder),
       0, -1);
   xmms_cfg_write_string(conf, "alarm", "reminder_msg", alarm_conf.reminder_msg);
   
   alarm_conf.reminder_on = 
     gtk_toggle_button_get_active(alarm_conf.reminder_cb);
   xmms_cfg_write_boolean(conf, "alarm", "reminder_on", alarm_conf.reminder_on);


   xmms_cfg_write_file(conf, filename);
   g_free(filename);
   xmms_cfg_free(conf);
}

/*
 * read the current configuration from the file
 */
static void alarm_read_config()
{
   ConfigFile *conf;
   gchar *filename = alarm_config_filename();

   DEBUG("alarm_read_config\n");

   conf = xmms_cfg_open_file(filename);

   g_free(filename);

   /* if theres no config file */
   if(conf == (ConfigFile *)NULL)
   {
	   alarm_h = DEFAULT_ALARM_HOUR;
   	 alarm_m = DEFAULT_ALARM_MIN;

	   stop_h = DEFAULT_STOP_HOURS;
	   stop_m = DEFAULT_STOP_MINS;
	   stop_on = TRUE;

	   volume = DEFAULT_VOLUME;
	   quietvol = DEFAULT_QUIET_VOL;

	   fading = DEFAULT_FADING;

	   cmd_on = FALSE;
	   cmdstr = g_strdup("");

	   playlist = g_strdup("");

     alarm_conf.reminder_msg = g_strdup("");
     alarm_conf.reminder_on = FALSE;

     /* day flags */
     alarm_conf.mon.flags = DEFAULT_FLAGS;
     alarm_conf.tue.flags = DEFAULT_FLAGS;
     alarm_conf.wed.flags = DEFAULT_FLAGS;
     alarm_conf.thu.flags = DEFAULT_FLAGS;
     alarm_conf.fri.flags = DEFAULT_FLAGS;
     alarm_conf.sat.flags = DEFAULT_FLAGS;
     alarm_conf.sun.flags = DEFAULT_FLAGS | ALARM_OFF;

     /* and times */
     alarm_conf.mon.hour = DEFAULT_ALARM_HOUR;
     alarm_conf.mon.min = DEFAULT_ALARM_MIN;

     alarm_conf.tue.hour = DEFAULT_ALARM_HOUR;
     alarm_conf.tue.min = DEFAULT_ALARM_MIN;

     alarm_conf.wed.hour = DEFAULT_ALARM_HOUR;
     alarm_conf.wed.min = DEFAULT_ALARM_MIN;

     alarm_conf.thu.hour = DEFAULT_ALARM_HOUR;
     alarm_conf.thu.min = DEFAULT_ALARM_MIN;

     alarm_conf.fri.hour = DEFAULT_ALARM_HOUR;
     alarm_conf.fri.min = DEFAULT_ALARM_MIN;

     alarm_conf.sat.hour = DEFAULT_ALARM_HOUR;
     alarm_conf.sat.min = DEFAULT_ALARM_MIN;

     alarm_conf.sun.hour = DEFAULT_ALARM_HOUR;
     alarm_conf.sun.min = DEFAULT_ALARM_MIN;
   }
   else
   {
	   if(!xmms_cfg_read_int(conf, "alarm", "alarm_h", &alarm_h))
	     alarm_h = DEFAULT_ALARM_HOUR;
	   if(!xmms_cfg_read_int(conf, "alarm", "alarm_m", &alarm_m))
	     alarm_m = DEFAULT_ALARM_MIN;

     /* save them here too */
     alarm_conf.default_hour = alarm_h;
     alarm_conf.default_min = alarm_m;

	   if(!xmms_cfg_read_int( conf, "alarm", "stop_h", &stop_h))
	     stop_h = DEFAULT_ALARM_HOUR;
	   if(!xmms_cfg_read_int( conf, "alarm", "stop_m", &stop_m))
	     stop_m = DEFAULT_ALARM_MIN;
	   if(!xmms_cfg_read_boolean(conf, "alarm", "stop_on", &stop_on))
	     stop_on = FALSE;

	   if(!xmms_cfg_read_int(conf, "alarm", "volume", &volume))
	     volume = DEFAULT_VOLUME;
	   if(!xmms_cfg_read_int(conf, "alarm", "quietvol", &quietvol))
	     quietvol = DEFAULT_QUIET_VOL;

	   if(!xmms_cfg_read_int(conf, "alarm", "fading", &fading))
	     fading = DEFAULT_FADING;

	   if(!xmms_cfg_read_string(conf, "alarm", "cmdstr", &cmdstr))
	     cmdstr = g_strdup("");
	   if(!xmms_cfg_read_boolean(conf, "alarm", "cmd_on", &cmd_on))
	     cmd_on = FALSE;

	   if(!xmms_cfg_read_string(conf, "alarm", "playlist", &playlist))
	     playlist = g_strdup("");

     if(!xmms_cfg_read_string(conf, "alarm", "reminder_msg", &alarm_conf.reminder_msg))
       alarm_conf.reminder_msg = g_strdup("");
     if(!xmms_cfg_read_boolean(conf, "alarm", "reminder_on", &alarm_conf.reminder_on))
       alarm_conf.reminder_on = FALSE;
         
         
	   /* read the week days */
     /* read the flags */
     if(!xmms_cfg_read_int(conf, "alarm", "mon_flags", &alarm_conf.mon.flags))
       alarm_conf.mon.flags = DEFAULT_FLAGS;
     if(!xmms_cfg_read_int(conf, "alarm", "tue_flags", &alarm_conf.tue.flags))
       alarm_conf.tue.flags = DEFAULT_FLAGS;
     if(!xmms_cfg_read_int(conf, "alarm", "wed_flags", &alarm_conf.wed.flags))
       alarm_conf.wed.flags = DEFAULT_FLAGS;
     if(!xmms_cfg_read_int(conf, "alarm", "thu_flags", &alarm_conf.thu.flags))
       alarm_conf.thu.flags = DEFAULT_FLAGS;
     if(!xmms_cfg_read_int(conf, "alarm", "fri_flags", &alarm_conf.fri.flags))
       alarm_conf.fri.flags = DEFAULT_FLAGS;
     if(!xmms_cfg_read_int(conf, "alarm", "sat_flags", &alarm_conf.sat.flags))
       alarm_conf.sat.flags = DEFAULT_FLAGS;
     if(!xmms_cfg_read_int(conf, "alarm", "sun_flags", &alarm_conf.sun.flags))
       alarm_conf.sun.flags = DEFAULT_FLAGS;

     /* and the times */
     if(!xmms_cfg_read_int(conf, "alarm", "mon_h", &alarm_conf.mon.hour))
       alarm_conf.mon.hour = DEFAULT_ALARM_HOUR;
     if(!xmms_cfg_read_int(conf, "alarm", "mon_m", &alarm_conf.mon.min))
       alarm_conf.mon.min = DEFAULT_ALARM_MIN;
  
     if(!xmms_cfg_read_int(conf, "alarm", "tue_h", &alarm_conf.tue.hour))
      alarm_conf.tue.hour = DEFAULT_ALARM_HOUR;
     if(!xmms_cfg_read_int(conf, "alarm", "tue_m", &alarm_conf.tue.min))
       alarm_conf.tue.min = DEFAULT_ALARM_MIN;
  
     if(!xmms_cfg_read_int(conf, "alarm", "wed_h", &alarm_conf.wed.hour))
       alarm_conf.wed.hour = DEFAULT_ALARM_HOUR;
     if(!xmms_cfg_read_int(conf, "alarm", "wed_m", &alarm_conf.wed.min))
       alarm_conf.wed.min = DEFAULT_ALARM_MIN;
  
     if(!xmms_cfg_read_int(conf, "alarm", "thu_h", &alarm_conf.thu.hour))
       alarm_conf.thu.hour = DEFAULT_ALARM_HOUR;
     if(!xmms_cfg_read_int(conf, "alarm", "thu_m", &alarm_conf.thu.min))
       alarm_conf.thu.min = DEFAULT_ALARM_MIN;
  
     if(!xmms_cfg_read_int(conf, "alarm", "fri_h", &alarm_conf.fri.hour))
       alarm_conf.fri.hour = DEFAULT_ALARM_HOUR;
     if(!xmms_cfg_read_int(conf, "alarm", "fri_m", &alarm_conf.fri.min))
       alarm_conf.fri.min = DEFAULT_ALARM_MIN;
  
     if(!xmms_cfg_read_int(conf, "alarm", "sat_h", &alarm_conf.sat.hour))
       alarm_conf.sat.hour = DEFAULT_ALARM_HOUR;
     if(!xmms_cfg_read_int(conf, "alarm", "sat_m", &alarm_conf.sat.min))
       alarm_conf.sat.min = DEFAULT_ALARM_MIN;
  
     if(!xmms_cfg_read_int(conf, "alarm", "sun_h", &alarm_conf.sun.hour))
       alarm_conf.sun.hour = DEFAULT_ALARM_HOUR;
     if(!xmms_cfg_read_int(conf, "alarm", "sun_m", &alarm_conf.sun.min))
       alarm_conf.sun.min = DEFAULT_ALARM_MIN;
  
	   xmms_cfg_free(conf);
   }
}

/*
 * display an about box
 */
static void alarm_about()
{
   static GtkWidget *about_dialog = NULL;

   DEBUG("alarm_about\n");

   if(dialog_visible(about_dialog))
     return;

   about_dialog = create_about_dialog();

   gtk_signal_connect(GTK_OBJECT(about_dialog), "destroy",
		      GTK_SIGNAL_FUNC(dialog_destroyed), &about_dialog);

   gtk_widget_show_all(about_dialog);

   return;
}

/*
 * create a playlist file selection dialog
 */
static void alarm_playlist_browse(GtkButton *button, gpointer data)
{
   GtkWidget *fs;
   gchar *dirname, *path;

   dirname = g_dirname(playlist);
   DEBUG("dirname = %s\n", dirname);
   path = g_strdup_printf("%s/", dirname);
   DEBUG("path = %s\n", path);
   g_free(dirname);

   fs = create_playlist_fileselection();

   gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), path);
   g_free(path);

   gtk_widget_show_all(fs);
}

/*
 * save selected playlist to the corresponding text entry
 */
void alarm_store_playlistname(GtkButton *button, gpointer data)
{
   GtkFileSelection *fs = GTK_FILE_SELECTION(data);
   gchar *plist;

   DEBUG("alarm_store_playlistname\n");

   plist = gtk_file_selection_get_filename(fs);

   gtk_entry_set_text(alarm_conf.playlist, plist);
}

/*
 * displays the configuration window and opens the config file.
 */
static void alarm_configure(void)
{
   GtkWidget *w;

   DEBUG("alarm_configure\n");

   /*
    * dont want to show more than one config window
    */
   if(dialog_visible(config_dialog))
     return;

   alarm_read_config();

   /*
    * Create the widgets
    */
   config_dialog = create_config_dialog();

   w = lookup_widget(config_dialog, "alarm_h_spin");
   alarm_conf.alarm_h = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.alarm_h, alarm_h);

   w = lookup_widget(config_dialog, "alarm_m_spin");
   alarm_conf.alarm_m =  GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.alarm_m, alarm_m);

   w = lookup_widget(config_dialog, "stop_h_spin");
   alarm_conf.stop_h = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.stop_h, stop_h);

   w = lookup_widget(config_dialog, "stop_m_spin");
   alarm_conf.stop_m = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.stop_m, stop_m);

   w = lookup_widget(config_dialog, "stop_checkb");
   alarm_conf.stop_on = GTK_TOGGLE_BUTTON(w);
   gtk_toggle_button_set_active(alarm_conf.stop_on, stop_on);

   w = lookup_widget(config_dialog, "vol_scale");
   alarm_conf.volume = GTK_RANGE(w);
   gtk_range_set_adjustment(alarm_conf.volume,
			    GTK_ADJUSTMENT(gtk_adjustment_new(volume,
							      0,
							      100, 1,
							      5, 0)));

   w = lookup_widget(config_dialog, "quiet_vol_scale");
   alarm_conf.quietvol = GTK_RANGE(w);
   gtk_range_set_adjustment(alarm_conf.quietvol,
			    GTK_ADJUSTMENT(gtk_adjustment_new(quietvol,
							      0,
							      100, 1,
							      5, 0)));

   /* days of week */
   /* mon */
   w = lookup_widget(config_dialog, "mon_cb");
   alarm_conf.mon.cb = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.mon.cb),
				!(alarm_conf.mon.flags & ALARM_OFF));

   w = lookup_widget(config_dialog, "mon_def");
   alarm_conf.mon.cb_def = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.mon.cb_def),
         alarm_conf.mon.flags & ALARM_DEFAULT);
   
   w = lookup_widget(config_dialog, "mon_h");
   alarm_conf.mon.spin_hr = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.mon.spin_hr, alarm_conf.mon.hour);

   w = lookup_widget(config_dialog, "mon_m");
   alarm_conf.mon.spin_min = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.mon.spin_min, alarm_conf.mon.min);

   if(alarm_conf.mon.flags & ALARM_DEFAULT)
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.mon.spin_hr, FALSE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.mon.spin_min, FALSE);
   }
   else
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.mon.spin_hr, TRUE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.mon.spin_min, TRUE);
   }

   /* tue */
   w = lookup_widget(config_dialog, "tue_cb");
   alarm_conf.tue.cb = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.tue.cb),
				!(alarm_conf.tue.flags & ALARM_OFF));

   w = lookup_widget(config_dialog, "tue_def");
   alarm_conf.tue.cb_def = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.tue.cb_def),
         alarm_conf.tue.flags & ALARM_DEFAULT);
   
   w = lookup_widget(config_dialog, "tue_h");
   alarm_conf.tue.spin_hr = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.tue.spin_hr, alarm_conf.tue.hour);

   w = lookup_widget(config_dialog, "tue_m");
   alarm_conf.tue.spin_min = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.tue.spin_min, alarm_conf.tue.min);

   if(alarm_conf.tue.flags & ALARM_DEFAULT)
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.tue.spin_hr, FALSE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.tue.spin_min, FALSE);
   }
   else
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.tue.spin_hr, TRUE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.tue.spin_min, TRUE);
   }

   
   /* wed */
   w = lookup_widget(config_dialog, "wed_cb");
   alarm_conf.wed.cb = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.wed.cb),
				!(alarm_conf.wed.flags & ALARM_OFF));

   w = lookup_widget(config_dialog, "wed_def");
   alarm_conf.wed.cb_def = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.wed.cb_def),
         alarm_conf.wed.flags & ALARM_DEFAULT);
   
   w = lookup_widget(config_dialog, "wed_h");
   alarm_conf.wed.spin_hr = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.wed.spin_hr, alarm_conf.wed.hour);

   w = lookup_widget(config_dialog, "wed_m");
   alarm_conf.wed.spin_min = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.wed.spin_min, alarm_conf.wed.min);

   if(alarm_conf.wed.flags & ALARM_DEFAULT)
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.wed.spin_hr, FALSE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.wed.spin_min, FALSE);
   }
   else
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.wed.spin_hr, TRUE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.wed.spin_min, TRUE);
   }

   /* thu */
   w = lookup_widget(config_dialog, "thu_cb");
   alarm_conf.thu.cb = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.thu.cb),
				!(alarm_conf.thu.flags & ALARM_OFF));
  
   w = lookup_widget(config_dialog, "thu_def");
   alarm_conf.thu.cb_def = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.thu.cb_def),
         alarm_conf.thu.flags & ALARM_DEFAULT);
   
   w = lookup_widget(config_dialog, "thu_h");
   alarm_conf.thu.spin_hr = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.thu.spin_hr, alarm_conf.thu.hour);

   w = lookup_widget(config_dialog, "thu_m");
   alarm_conf.thu.spin_min = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.thu.spin_min, alarm_conf.thu.min);

   if(alarm_conf.thu.flags & ALARM_DEFAULT)
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.thu.spin_hr, FALSE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.thu.spin_min, FALSE);
   }
   else
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.thu.spin_hr, TRUE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.thu.spin_min, TRUE);
   }

   /* fri */
   w = lookup_widget(config_dialog, "fri_cb");
   alarm_conf.fri.cb = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.fri.cb),
				!(alarm_conf.fri.flags & ALARM_OFF));
  
   w = lookup_widget(config_dialog, "fri_def");
   alarm_conf.fri.cb_def = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.fri.cb_def),
         alarm_conf.fri.flags & ALARM_DEFAULT);
   
   w = lookup_widget(config_dialog, "fri_h");
   alarm_conf.fri.spin_hr = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.fri.spin_hr, alarm_conf.fri.hour);

   w = lookup_widget(config_dialog, "fri_m");
   alarm_conf.fri.spin_min = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.fri.spin_min, alarm_conf.fri.min);

   if(alarm_conf.fri.flags & ALARM_DEFAULT)
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.fri.spin_hr, FALSE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.fri.spin_min, FALSE);
   }
   else
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.fri.spin_hr, TRUE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.fri.spin_min, TRUE);
   }

   /* sat */
   w = lookup_widget(config_dialog, "sat_cb");
   alarm_conf.sat.cb = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.sat.cb),
				!(alarm_conf.sat.flags & ALARM_OFF));

   w = lookup_widget(config_dialog, "sat_def");
   alarm_conf.sat.cb_def = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.sat.cb_def),
         alarm_conf.sat.flags & ALARM_DEFAULT);
   
   w = lookup_widget(config_dialog, "sat_h");
   alarm_conf.sat.spin_hr = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.sat.spin_hr, alarm_conf.sat.hour);

   w = lookup_widget(config_dialog, "sat_m");
   alarm_conf.sat.spin_min = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.sat.spin_min, alarm_conf.sat.min);

   if(alarm_conf.sat.flags & ALARM_DEFAULT)
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.sat.spin_hr, FALSE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.sat.spin_min, FALSE);
   }
   else
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.sat.spin_hr, TRUE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.sat.spin_min, TRUE);
   }

   /* sun */
   w = lookup_widget(config_dialog, "sun_cb");
   alarm_conf.sun.cb = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.sun.cb),
				!(alarm_conf.sun.flags & ALARM_OFF));

   w = lookup_widget(config_dialog, "sun_def");
   alarm_conf.sun.cb_def = GTK_CHECK_BUTTON(w);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(alarm_conf.sun.cb_def),
         alarm_conf.sun.flags & ALARM_DEFAULT);
   
   w = lookup_widget(config_dialog, "sun_h");
   alarm_conf.sun.spin_hr = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.sun.spin_hr, alarm_conf.sun.hour);

   w = lookup_widget(config_dialog, "sun_m");
   alarm_conf.sun.spin_min = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.sun.spin_min, alarm_conf.sun.min);

   if(alarm_conf.sun.flags & ALARM_DEFAULT)
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.sun.spin_hr, FALSE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.sun.spin_min, FALSE);
   }
   else
   {
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.sun.spin_hr, TRUE);
     gtk_widget_set_sensitive((GtkWidget *)alarm_conf.sun.spin_min, TRUE);
   }

   /* END: days of week */

   w = lookup_widget(config_dialog,"fading_spin");
   alarm_conf.fading = GTK_SPIN_BUTTON(w);
   gtk_spin_button_set_value(alarm_conf.fading, fading);

   w = lookup_widget(config_dialog, "cmd_entry");
   alarm_conf.cmdstr = GTK_ENTRY(w);
   gtk_entry_set_text(alarm_conf.cmdstr, cmdstr);

   w = lookup_widget(config_dialog, "cmd_checkb");
   alarm_conf.cmd_on = GTK_TOGGLE_BUTTON(w);
   gtk_toggle_button_set_active(alarm_conf.cmd_on, cmd_on);

   w = lookup_widget(config_dialog, "playlist");
   alarm_conf.playlist = GTK_ENTRY(w);
   gtk_entry_set_text(alarm_conf.playlist, playlist);

   w = lookup_widget(config_dialog, "reminder_text");
   alarm_conf.reminder = GTK_ENTRY(w);
   gtk_entry_set_text(alarm_conf.reminder, alarm_conf.reminder_msg);

   w = lookup_widget(config_dialog, "reminder_cb");
   alarm_conf.reminder_cb = GTK_TOGGLE_BUTTON(w);
   gtk_toggle_button_set_active(alarm_conf.reminder_cb, alarm_conf.reminder_on);

   w = lookup_widget(config_dialog, "playlist_browse_button");
   gtk_signal_connect(GTK_OBJECT(w), "clicked",
		      GTK_SIGNAL_FUNC(alarm_playlist_browse), NULL);

   gtk_signal_connect(GTK_OBJECT(config_dialog), "destroy",
		      GTK_SIGNAL_FUNC(dialog_destroyed), &config_dialog);

   gtk_widget_show_all(config_dialog);
}

/* functions for greying out the time for days */
void on_mon_def_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
   GtkWidget *w;

   w = lookup_widget(config_dialog, "mon_h");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);

   w = lookup_widget(config_dialog, "mon_m");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);
}

void on_tue_def_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
   GtkWidget *w;

   w = lookup_widget(config_dialog, "tue_h");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);

   w = lookup_widget(config_dialog, "tue_m");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);
}

void on_wed_def_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
   GtkWidget *w;

   w = lookup_widget(config_dialog, "wed_h");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);

   w = lookup_widget(config_dialog, "wed_m");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);
}

void on_thu_def_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
   GtkWidget *w;

   w = lookup_widget(config_dialog, "thu_h");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);

   w = lookup_widget(config_dialog, "thu_m");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);
}

void on_fri_def_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
   GtkWidget *w;

   w = lookup_widget(config_dialog, "fri_h");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);

   w = lookup_widget(config_dialog, "fri_m");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);
}

void on_sat_def_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
   GtkWidget *w;

   w = lookup_widget(config_dialog, "sat_h");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);

   w = lookup_widget(config_dialog, "sat_m");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);
}

void on_sun_def_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
   GtkWidget *w;

   w = lookup_widget(config_dialog, "sun_h");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);

   w = lookup_widget(config_dialog, "sun_m");
   if(gtk_toggle_button_get_active(togglebutton) == TRUE)
     gtk_widget_set_sensitive(w, FALSE);
   else
     gtk_widget_set_sensitive(w, TRUE);
}
/* END: greying things */

void alarm_current_volume(GtkButton *button, gpointer data)
{
   gint vol;
   GtkAdjustment *adj;

   DEBUG("on_current_button_clicked\n");

   vol = xmms_remote_get_main_volume(alarm_plugin.xmms_session);

   adj = gtk_range_get_adjustment(alarm_conf.volume);
   gtk_adjustment_set_value(adj, (gfloat)vol);
}

GeneralPlugin *get_gplugin_info()
{
   return &alarm_plugin;
}

/*
 * a thread safe sleeping function -
 * and it even works in solaris (I think)
 */
static void threadsleep(float x)
{
   struct timespec rqtp, rmtp;

   DEBUG("threadsleep: waiting %f seconds (fader_lock %s)\n", x,
       (fader_lock == TRUE) ? "TRUE" : "FALSE");

   rqtp.tv_sec = x;
   rqtp.tv_nsec = (int)((float)(x - (int)x) * 1000000000.0);

   nanosleep(&rqtp, &rmtp);

   return;
}

static inline pthread_t alarm_thread_create(void *(*start_routine)(void *), void *args)
{
   pthread_t tid;
   pthread_attr_t attr;

   pthread_attr_init(&attr);
   pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
   pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED);
   pthread_attr_setschedpolicy(&attr, SCHED_OTHER);
   pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);

   pthread_create(&tid, &attr, start_routine, args);

   return(tid);
}

static void *alarm_fade(void *arg)
{
   gfloat currvol, step, stepabs;
   fader *vols = (fader *)arg;

   currvol = vols->start;

   /* work out increment volume step for each 1/2 sec */
   step = ((gfloat)vols->end - (gfloat)vols->start) / (fading / 0.5);

   /* lock */
   //pthread_mutex_lock(&fader_lock);
   while(fader_lock == TRUE);
   fader_lock = TRUE;

   /* slide volume */
   stepabs = fabs(step);
   while(fabs(currvol - vols->end) > stepabs)
     {
	xmms_remote_set_main_volume(alarm_plugin.xmms_session,
				    (gint)currvol);
	currvol += step;
	threadsleep(0.5);
     }

   xmms_remote_set_main_volume(alarm_plugin.xmms_session,
			       (gint)vols->end);

   /* and */
   //pthread_mutex_unlock(&fader_lock);
   fader_lock = FALSE;

   DEBUG("volume = %f%%\n", (gdouble)vols->end);
   return(0);
}

static void *alarm_stop_thread( void *args )
{
   gint currvol;
   fader fade_vols;
   pthread_t f_tid;

   DEBUG("alarm_stop_thread\n");


   /* sleep for however long we are meant to be sleeping for until
    * its time to shut up
    */
   threadsleep(((stop_h * 60) + stop_m) * 60);

   DEBUG("alarm_stop triggered\n");

   /* lock the fader mutex */
   DEBUG("Stop: attempting to lock mutex...");
   //pthread_mutex_lock(&fader_lock);
   //while(fader_lock == TRUE);
   //fader_lock = TRUE;

   DEBUG("done\n");

   if (dialog_visible(alarm_dialog))
     gtk_widget_destroy(alarm_dialog);

   currvol = xmms_remote_get_main_volume(alarm_plugin.xmms_session),

   /* fade back to zero */
   //alarm_fade(currvol, 0);
   fade_vols.start = currvol;
   fade_vols.end = 0;
   f_tid = alarm_thread_create(alarm_fade, &fade_vols);

   pthread_join(f_tid, NULL);
   xmms_remote_stop(alarm_plugin.xmms_session);

   /* might as well set the volume to something higher than zero so we
    * dont confuse the poor people who just woke up and cant work out why
    * theres no music playing when they press the little play button :)
    */
   xmms_remote_set_main_volume(alarm_plugin.xmms_session, currvol);

   /* let other threads play with volume */
   //pthread_mutex_unlock(&fader_lock);
   //fader_lock = FALSE;

   DEBUG("alarm_stop done\n");
   return NULL;
}

void alarm_stop_cancel(GtkButton *w, gpointer data)
{
   DEBUG("alarm_stop_cancel\n");
   pthread_cancel(stop_tid);
}

/* the main alarm thread */
static void *alarm_start_thread(void *args)
{
   struct tm *currtime;
   time_t timenow;
   guint today;

   while(0 == 0)
     {
       /* sit around and wait for the faders to not be doing anything */
       DEBUG("Waiting for fader to be unlocked..");
       while(fader_lock == TRUE);
       DEBUG("Ok\n");
	DEBUG("Getting time\n");
	timenow = time(NULL);
	currtime = localtime(&timenow);
	today = currtime->tm_wday;
	DEBUG("Today is %d\n", today);

	/* see if its time to do something */
	DEBUG("Checking Day\n");
	switch(today)
	  {
	   case 0:
	     DEBUG("Sunday\n");
	     if(alarm_conf.sun.flags & ALARM_OFF)
	     {
		     threadsleep(10.0);
		     continue;
	     }
       else
       {
         /* set the alarm_h and alarm_m for today, if not default */
         if(!(alarm_conf.sun.flags & ALARM_DEFAULT))
         {
           alarm_h = alarm_conf.sun.hour;
           alarm_m = alarm_conf.sun.min;
         }
         else
         {
           alarm_h = alarm_conf.default_hour;
           alarm_m = alarm_conf.default_min;
         }
       }
	     break;
	   case 1:
	     DEBUG("Monday\n");
	     if(alarm_conf.mon.flags & ALARM_OFF)
	     {
		     threadsleep(10.0);
		     continue;
	     }
       else
       {
         if(!(alarm_conf.mon.flags & ALARM_DEFAULT))
         {
           alarm_h = alarm_conf.mon.hour;
           alarm_m = alarm_conf.mon.min;
         }
         else
         {
           alarm_h = alarm_conf.default_hour;
           alarm_m = alarm_conf.default_min;
         }
       }
	     break;
	   case 2:
	     DEBUG("Tuesday\n");
	     if(alarm_conf.tue.flags & ALARM_OFF)
	     {
		     threadsleep(10.0);
		     continue;
	     }
       else
       {
         if(!(alarm_conf.tue.flags & ALARM_DEFAULT))
         {
           alarm_h = alarm_conf.tue.hour;
           alarm_m = alarm_conf.tue.min;
         }
         else
         {
           alarm_h = alarm_conf.default_hour;
           alarm_m = alarm_conf.default_min;
         }
       }
	     break;
	   case 3:
	     DEBUG("Wednesday\n");
	     if(alarm_conf.wed.flags & ALARM_OFF)
	     {
		     threadsleep(10.0);
		     continue;
	     }
       else
       {
         if(!(alarm_conf.wed.flags & ALARM_DEFAULT))
         {
           alarm_h = alarm_conf.wed.hour;
           alarm_m = alarm_conf.wed.min;
         }
         else
         {
           alarm_h = alarm_conf.default_hour;
           alarm_m = alarm_conf.default_min;
         }
       }
	     break;
	   case 4:
	     DEBUG("Thursday\n");
	     if(alarm_conf.thu.flags & ALARM_OFF)
	     {
		     threadsleep(10.0);
		     continue;
	     }
       else
       {
         if(!(alarm_conf.thu.flags & ALARM_DEFAULT))
         {
           alarm_h = alarm_conf.thu.hour;
           alarm_m = alarm_conf.thu.min;
         }
         else
         {
           alarm_h = alarm_conf.default_hour;
           alarm_m = alarm_conf.default_min;
         }
       }
	     break;
	   case 5:
	     DEBUG("Friday\n");
	     if(alarm_conf.fri.flags & ALARM_OFF)
	     {
		     threadsleep(10.0);
		     continue;
	     }
       else
       {
         if(!(alarm_conf.fri.flags & ALARM_DEFAULT))
         {
           alarm_h = alarm_conf.fri.hour;
           alarm_m = alarm_conf.fri.min;
         }
         else
         {
           alarm_h = alarm_conf.default_hour;
           alarm_m = alarm_conf.default_min;
         }
       }
	     break;
	   case 6:
	     DEBUG("Saturday\n");
	     if(alarm_conf.sat.flags & ALARM_OFF)
	     {
		     threadsleep(10.0);
		     continue;
	     }
       else
       {
         if(!(alarm_conf.sat.flags & ALARM_DEFAULT))
         {
           DEBUG("!!Not default\n");
           alarm_h = alarm_conf.sat.hour;
           alarm_m = alarm_conf.sat.min;
         }
         else
         {
           DEBUG("!!!Using default\n");
           alarm_h = alarm_conf.default_hour;
           alarm_m = alarm_conf.default_min;
         }
       }
	     break;
	   default:
	     g_print("Eeek, someone invented a day without telling me!\n");
	     break;
	  }

  DEBUG("Alarm time is %d:%d (def: %d:%d)\n", alarm_h, alarm_m,
      alarm_conf.default_hour, alarm_conf.default_min);
	DEBUG("Checking time\n");
	if((currtime->tm_hour != alarm_h) ||
	   (currtime->tm_min != alarm_m))
	  {
	     threadsleep(10.0);
	     continue;
	  }

	if(cmd_on == TRUE)
	  {
	     DEBUG("Executing %s, cmd_on is true\n", cmdstr);
	     system(cmdstr);
	  }

	DEBUG("strcmp playlist, playlist is [%s]\n", playlist);
	if(strcmp(playlist, ""))
	  {
	     DEBUG("playlist is not blank, aparently\n");
	     xmms_remote_playlist(alarm_plugin.xmms_session,
				  &playlist, 1, FALSE);
	  }

	if(fading)
	  {
       fader fade_vols;
       
	     DEBUG("Fading is true\n");
	     xmms_remote_set_main_volume(alarm_plugin.xmms_session,
					 quietvol);
	     /* start playing */
	     xmms_remote_play(alarm_plugin.xmms_session);

        /* fade volume */
       fade_vols.start = quietvol;
       fade_vols.end = volume;
	     //alarm_fade(quietvol, volume);
       alarm_thread_create(alarm_fade, &fade_vols);
	  }
	else
	  {
	     /* no fading */

	     /* set volume */
	     xmms_remote_set_main_volume(alarm_plugin.xmms_session,
					 volume);

	     /* start playing */
	     xmms_remote_play(alarm_plugin.xmms_session);
	  }

  if(alarm_conf.reminder_on == TRUE)
  {
    GtkWidget *reminder_dialog;
    DEBUG("Showing reminder '%s'\n", alarm_conf.reminder_msg);
    GDK_THREADS_ENTER();
    reminder_dialog = create_reminder_dialog(alarm_conf.reminder_msg);
    gtk_signal_connect(GTK_OBJECT(reminder_dialog), "destroy",
        GTK_SIGNAL_FUNC(dialog_destroyed), &reminder_dialog);
    gtk_widget_show_all(reminder_dialog);
    GDK_THREADS_LEAVE();
  }

   /* bring up the wakeup call dialog if stop_on is set TRUE, this
    * has been moved to after making xmms play so that it doesnt
    * get in the way for people with manual window placement turned on
    *
    * this means that the dialog doesnt get shown until the volume has
    * finished fading though !, so thats something else to fix
    */
	if(stop_on == TRUE)
	  {
	     /* ok, so when we want to open dialogs in threaded programs
	      * we use this do we?
	      * anyone?
	      */
	     GDK_THREADS_ENTER();
	       {
		  DEBUG("stop_on is true\n");
		  alarm_dialog = create_alarm_dialog();
		  DEBUG("created alarm dialog, %p\n", alarm_dialog);
		  gtk_signal_connect(GTK_OBJECT(alarm_dialog), "destroy",
				     GTK_SIGNAL_FUNC(dialog_destroyed), &alarm_dialog);
		  DEBUG("attached destroy signal to alarm dialog, %p\n", alarm_dialog);
		  gtk_widget_show_all(alarm_dialog);
		  DEBUG("dialog now showing\n");

		  DEBUG("now starting stop thread\n");
		  stop_tid = alarm_thread_create(alarm_stop_thread, NULL);
		  DEBUG("Created wakeup dialog and started stop thread\n");

	       }
	     GDK_THREADS_LEAVE();
      /* now wait for the stop thread */
      DEBUG("Waiting for stop to stop....");
      pthread_join(stop_tid, NULL);
      DEBUG("OK\n");
	  }
  threadsleep(fading);
     }

   return NULL;
}

/*
 * initialization
 * opens the config file and reads the value, creates a new
 * config in memory if the file doesnt exist and sets default vals
 */
static void alarm_init()
{
   DEBUG("alarm_init\n");

   alarm_read_config();

   /* start the main thread running */
   start_tid = alarm_thread_create(alarm_start_thread, NULL);
}

/*
 * kill the main thread
 */
static void alarm_cleanup()
{
   DEBUG("alarm_cleanup\n");

   pthread_cancel(start_tid);
   if(stop_tid)
     pthread_cancel(stop_tid);
}

/*
 * for xmms to get the function names
 */
static GeneralPlugin alarm_plugin =
{
   NULL,
     NULL,
     -1,
     "Alarm "VERSION,
     alarm_init,
     alarm_about,
     alarm_configure,
     alarm_cleanup,
};
