/*  -*- C++ -*-
 *
 * Copyright (C) 2000 Frans Kaashoek (kaashoek@mit.edu)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include <sys/types.h>
#include <grp.h>

#include "xdfssrv.h"
#include "parseopt.h"
#include "rxx.h"

// $Format: "static const char VERSION[] = \"$ReleaseVersion$\";" $
static const char VERSION[] = "2.0-beta9";

static str      config_file;
static bool     option_create  = false;
static bool     option_nuke    = false;
static XdfsSrv *the_server = NULL;

int    __xdfssrv_argc;
char** __xdfssrv_argv;

static void
client_accept (ptr<axprt_crypt> x)
{
    if (!x)
	fatal ("EOF from sfssd\n");
    vNew Proxy (x);
}

XdfsSrv* get_server (void)
{
    assert (the_server);

    return the_server;
}

static void
start_server (XdfsSrv *fsrv)
{
    setgid (sfs_gid);
    setgroups (0, NULL);

    warn ("version %s, pid %d\n", VERSION, getpid ());
    warn << "serving " << sfsroot << "/"
	 << sfs_hostinfo2path (fsrv->_servinfo.host) << "\n";

    assert (the_server == NULL);

    the_server = fsrv;

    sfssd_slave (wrap (client_accept));
}

static int
init_root (DBFS &dbfs)
{
    TXN txn;
    XNFS_STAT  xa;
    nfstime3 now = nfs_now ();
    int ret;
    MAJORC root;

    if ((ret = txn.begin_root (dbfs, DBFS_TXN_SYNC, root))) {
	PROP_ERROR (ret) ("txn_begin");
	return ret;
    }

    xa.mode  = 0777;
    xa.uid   = 0;
    xa.gid   = 0;
    xa.atime = now;
    xa.mtime = now;
    xa.ctime = now;

    if ((ret = xattr_set (root, xa))) {
	PROP_ERROR (ret) ("xattr_set");
	return ret;
    }

    if ((ret = add_dots (root, root))) {
	PROP_ERROR (ret) ("add_dots");
	return ret;
    }

    if ((ret = txn.commit ())) {
	PROP_ERROR (ret) ("txn_commit");
	return ret;
    }

    return 0;
}

static XdfsSrv*
parseconfig (str cf)
{
    parseargs pa (cf);

    XdfsSrv *xsrv = New XdfsSrv;

    int line;
    vec<str> av;
    while (pa.getline (&av, &line)) {
	if (!strcasecmp (av[0], "export")) {

	    int ret;

	    if (option_nuke) {

		str cmd = "rm -rf " << av[1];

		system (cmd);
	    }

	    if ((ret = xsrv->_dbfs.open (av[1], NULL, DBFS_CREATE | DBFS_CLEAR/* | DBFS_ASYNC_INIT | DBFS_MAINT_INIT*/))) {
		PROP_ERROR (ret) ("dbfs_open");
		return NULL;
	    }

	    if ((ret = init_root (xsrv->_dbfs))) {
		PROP_ERROR (ret) ("init_root");
		return NULL;
	    }
	}
	else if (!strcasecmp (av[0], "hostname")) {
	    if (av.size () != 2) {
		warn << cf << ":" << line << ": usage: hostname name\n";
		return NULL;
	    }
	    else if (xsrv->_servinfo.host.hostname) {
		warn << cf << ":" << line << ": hostname already specified\n";
		return NULL;
	    }
	    else
		xsrv->_servinfo.host.hostname = av[1];
	}
	else if (!strcasecmp (av[0], "keyfile")) {
	    if (xsrv->_sk) {
		warn << cf << ":" << line << ": keyfile already specified\n";
		return NULL;
	    }
	    else if (av.size () == 2) {
		str key = file2wstr (av[1]);
		if (!key) {
		    warn << av[1] << ": " << strerror (errno) << "\n";
		    warn << cf << ":" << line << ": could not read keyfile\n";
		    return NULL;
		}
		else if (!(xsrv->_sk = import_rabin_priv (key, NULL))) {
		    warn << cf << ":" << line << ": could not decode keyfile\n";
		    return NULL;
		}
	    }
	    else {
		warn << cf << ":" << line << ": usage: keyfile path\n";
		return NULL;
	    }
	}
    }

    if (! xsrv->_sk) {
	warn << cf << ":" << "no keyfile specified\n";
	return NULL;
    }

    xsrv->_servinfo.release = sfs_release;
    xsrv->_servinfo.host.type = SFS_HOSTINFO;
    if (!xsrv->_servinfo.host.hostname
	&& !(xsrv->_servinfo.host.hostname = myname ()))
	fatal ("could not figure out my host name\n");
    xsrv->_servinfo.host.pubkey = xsrv->_sk->n;
    if (!sfs_mkhostid (&xsrv->_hostid, xsrv->_servinfo.host))
	fatal ("could not marshal my own hostinfo\n");
    xsrv->_servinfo.prog = ex_NFS_PROGRAM;
    xsrv->_servinfo.vers = ex_NFS_V3;

    return xsrv;
}


static void
usage ()
{
    warnx << "usage: " << progname << " -f config_file\n";
    exit (1);
}

int
main (int argc, char **argv)
{
    XdfsSrv *xsrv;

    __xdfssrv_argc = argc;
    __xdfssrv_argv = argv;

    setprogname (argv[0]);

    int ch;

    while ((ch = getopt (argc, argv, "cnf:")) != -1) {
	switch (ch) {
	case 'n':
	    option_nuke = true;
	    break;
	case 'c':
	    option_create = true;
	    break;
	case 'f':
	    config_file = optarg;
	    break;
	case '?':
	default:
	    usage ();
	}
    }

    argc -= optind;
    argv += optind;

    if ( (argc > 0) || (!config_file) ) {
	usage ();
    }

    if (option_nuke && ! option_create) {
	warnx << "error: -n (nuke) requires -c (create)\n";
	exit (1);
    }

    sfsconst_init ();

    if (! (xsrv = parseconfig (config_file))) {
	exit (1);
    }

    start_server (xsrv);
    amain ();
}
