/* -*-Mode: C++;-*-
 * $Id: xdfs_cpp.h 1.34 Tue, 15 May 2001 16:33:56 -0700 jmacd $
 *
 * Copyright (C) 1998, 1999, 2000, Joshua P. MacDonald
 * <jmacd@CS.Berkeley.EDU> and The Regents of the University of
 * California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.  */

#ifndef _XDFS_CPP_H_
#define _XDFS_CPP_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef __cplusplus

extern "C" {
#include <unistd.h>
}

#include <edsio.h>
#include <xdelta.h>
#include <db_cxx.h>

class BASIC_DBENV;
class COMDBT;
class MAJORC;
class NODEC;
class DIRC;
class TXN;
class STXN;
class DBFS;
class SAREA;
class VIEWIMPL;
class VIEWDEF;
class STACKDEF;
class DBCREF;
class DBREF;
class MINOR_DESC;
class MAJOR_DESC;
class SAREA_DESC;
class SHFD_DESC;
class SHDB_DESC;
class SHPG_DESC;
class DKEY;

#include <string>
#include <hash_map>

#include "slp.h"
#include "ehashmap.h"

#define FILETYPE(x) (x)

#define FT_Readable  FILETYPE (1<<1)
#define FT_Container FILETYPE (1<<2)
#define FT_Keyed     FILETYPE (1<<3)

// Bits 4 bits[4..7] remain in this 16-bit bitfield-enum: Symlink, Counter, ?

enum FileType
{
    FT_NotPresent   = (0),
    FT_Reflink      = (1 << 8),
    FT_Arealink     = (1 << 9),
    FT_ShortSeg     = FT_Readable  | (1 << 10),
    FT_LongSeg      = FT_Readable  | (1 << 11),
    FT_ViewSeg      = FT_Readable  | (1 << 12),
    FT_DirSeq       = FT_Container | (1 << 13),
    FT_DirHash      = FT_Container | FT_Keyed | (1 << 14),
    FT_DirBtree     = FT_Container | FT_Keyed | (1 << 15),
};

#undef  FILETYPE
#define FILETYPE(x) ((FileType) (x))

enum DBFS_FLAGS
{
    DBFS_NOFLAG                = 0,
    DBFS_CREATE                = (1 << 0),
    DBFS_CLEAR                 = (1 << 1),
    DBFS_RECOVER               = (1 << 2),
    DBFS_RECOVER_ADMIN         = (1 << 3),
    DBFS_ADMIN_TOOL            = (1 << 4),

    DBFS_CREATEROOT_HASH       = (1 << 5),
    DBFS_NOROOT_SPECIAL        = (1 << 6),

    DBFS_NOOVERWRITE           = (1 << 7),
    DBFS_OVERWRITE             = (1 << 8),

    DBFS_STAYOPEN              = (1 << 9),
    DBFS_LINK_RMW              = (1 << 10),
    DBFS_LINK_REVERSIBLE       = (1 << 11),

    DBFS_TXN_SYNC              = (1 << 12),
    DBFS_TXN_NOSYNC_INTERNAL   = (1 << 13),
    DBFS_TXN_ASYNC             = (1 << 14),
    DBFS_TXN_NOWAIT            = (1 << 15),

    DBFS_ASYNC_INIT            = (1 << 16),
    DBFS_MAINT_INIT            = (1 << 17),
    DBFS_DEADLOCK_INIT         = (1 << 18),

    DBFS_ALLOC_RANDOM          = (1 << 19),
    DBFS_ALLOC_ASCEND          = (1 << 20),
    DBFS_ALLOC_DEFAULT         = DBFS_ALLOC_ASCEND,
    DBFS_ALLOC_MASK            = (DBFS_ALLOC_RANDOM | DBFS_ALLOC_ASCEND),
};

enum DBFS_RMWFLAG
{
    // These are for internal use only
    DBFS_RMW             = 17,
    DBFS_NORMW           = 31
};

enum DBFS_ERRORS
{
    DBFS_MAX_MAJOR            = -18503,
    DBFS_MAX_MINOR            = -18504,
    DBFS_NOT_IMPL             = -18505,
    DBFS_EXISTS               = -18506,
    DBFS_NOT_READABLE         = -18507,
    DBFS_VIEW_UNDEF           = -18508,
    DBFS_INVAL                = -18509,
    DBFS_NEXISTS              = -18510,
    DBFS_LINK_INCONSISTENT    = -18512,
    DBFS_NOSPACE              = -18513,
    DBFS_NOTFOUND             = -18514,
    DBFS_NOPOSITION           = -18515,
    DBFS_NOTDIR               = -18516,
    DBFS_NOTSEQ               = -18517,
    DBFS_STACK_UNDEF          = -18518,
    DBFS_NEEDS_ATTENTION      = -18519,
};

enum DBFS_ADMIN_CODES
{
    DBFS_ADMIN_LISTPROC = (1 << 0),
};

template <class T>
class SAFEINT
{
public:

    T&          key  ()       { return _key; }
    const T&    key  () const { return _key; }
    uint        hash () const { return (uint) _key; }

protected:

    SAFEINT ()
	: _key (0) { }

    SAFEINT (T key)
	: _key (key) { }

    SAFEINT (const SAFEINT& num)
	: _key (num._key) { }

    T _key;
};

template <class T>
inline ostream&
operator<< (ostream &os, const SAFEINT<T>& i)
{
    os << i.key ();
    return os;
}

#define MAKE_INT(TYPE,ITYPE)                    \
	class TYPE : public SAFEINT<ITYPE> {    \
        public: typedef SAFEINT<ITYPE> P;       \
        TYPE () : P (0) { }                     \
        explicit TYPE (ITYPE key) : P (key) { } \
        TYPE (const TYPE& copy) : P (copy) { }  \
        bool     operator== (const TYPE&  num) const { return _key == num._key; } \
        bool     operator>  (const TYPE&  num) const { return _key > num._key; } \
        bool     operator>  (const ITYPE& num) const { return _key > num; } \
        bool     operator<  (const TYPE&  num) const { return _key < num._key; } \
        bool     operator<  (const ITYPE& num) const { return _key < num; } \
        TYPE&    operator=  (const TYPE&  num)       { _key =  num._key; return *this; } \
        TYPE     operator+  (const ITYPE& num) const { return TYPE (_key + num); } \
        TYPE     operator+  (const TYPE&  num) const { return TYPE (_key + num._key); } \
        TYPE     operator-  (const ITYPE& num) const { return TYPE (_key - num); } \
        TYPE     operator-  (const TYPE&  num) const { return TYPE (_key - num._key); } \
        TYPE     operator*  (const ITYPE& num) const { return TYPE (_key * num); } \
        TYPE     operator*  (const TYPE&  num) const { return TYPE (_key * num._key); } \
        TYPE&    operator+= (const ITYPE& num)       { _key += num; return *this; } \
        TYPE&    operator+= (const TYPE&  num)       { _key += num._key; return *this; } \
        TYPE&    operator-= (const ITYPE& num)       { _key -= num; return *this; } \
        TYPE&    operator-= (const TYPE&  num)       { _key -= num._key; return *this; } \
        TYPE&    operator&= (const ITYPE& num)       { _key &= num; return *this; } \
        TYPE&    operator&= (const TYPE&  num)       { _key &= num._key; return *this; } \
        TYPE     operator<< (guint        shl) const { return TYPE (_key << shl); } \
        };

MAKE_INT(XLNK,guint16);
MAKE_INT(XSIZ,guint32);
MAKE_INT(XSEQNO,guint32); // @@ separate cont_seq from seqdir_key
MAKE_INT(XTYPE,guint16);
MAKE_INT(XSTCK,guint16);
MAKE_INT(XVID,guint32);
MAKE_INT(XFID,guint32);
MAKE_INT(XPGNO,guint32);
MAKE_INT(XSID,guint32);
MAKE_INT(XAREA,guint32);

class XCONT
{
public:

    XCONT (const XAREA &area_id, const XSEQNO &cont_seq)
	: _area_id (area_id),
	  _cont_seq (cont_seq.key ()) { }

    XCONT (const XAREA &area_id, const XVID &view_id)
	: _area_id (area_id),
	  _cont_seq (view_id.key ()) { }

    XCONT (const XAREA &area_id, const XFID &file_id)
	: _area_id (area_id),
	  _cont_seq (file_id.key ()) { }

    XCONT (const XAREA &area_id, const XSID &short_id)
	: _area_id (area_id),
	  _cont_seq (short_id.key ()) { }

    explicit XCONT (guint64 num)
	: _area_id  (num >> 32),
	  _cont_seq (num & 0xffffffff) { }

    XCONT () { }

    const XAREA&  area_id  () const { return _area_id; }
    const XSEQNO& cont_seq  () const { return _cont_seq; }

    guint64       key      () const { return (((guint64) _area_id.key ()) << 32) | (guint64) _cont_seq.key (); }

    XVID          view_id  () const { return XVID (_cont_seq.key ()); }
    XFID          file_id  () const { return XFID (_cont_seq.key ()); }
    XSID          short_id () const { return XSID (_cont_seq.key ()); }

    bool          operator== (const XCONT& cont) const { return _area_id == cont._area_id && _cont_seq == cont._cont_seq; }

private:

    XAREA   _area_id;
    XSEQNO  _cont_seq;
};

class XFREEELT
{
public:

    XFREEELT () { }

    XFREEELT (const XFID& fid, const XTYPE &type)
	: fid  (fid),
	  type (type) { }

    XFID  fid;
    XTYPE type;
};

typedef XCONT XNUM; // Major node numbers

extern const guint   DBFS_ENV_OPEN_FLAGS;
extern const guint   DBFS_FS_SHORT_THRESH;
extern const guint   DBFS_PAGE_SIZE;
extern const guint   DBFS_LOGBSIZE;
extern const guint   DBFS_CACHESIZE;
extern const guint   DBFS_TX_MAX;
extern const guint   DBFS_LK_MAX;

extern const guint   DBFS_MINOR_TABLE_SIZE;
extern const guint   DBFS_MAJOR_TABLE_SIZE;
extern const guint   DBFS_SAREA_TABLE_SIZE;
extern const guint   DBFS_SHARED_DB_TABLE_SIZE;

extern const guint   DBFS_CONT_ID_INTERVAL;
extern const guint   DBFS_CONT_ID_MASK;

extern const guint   DBFS_FID_BASE_BITS;
extern const guint   DBFS_FID_BASE_MASK;
extern const guint   DBFS_FID_BASE_ALLOC;

#define DBFS_DEF_STACK_VAL 1

extern /*const*/ XSTCK   DBFS_DEF_STACK;
extern /*const*/ XSTCK   DBFS_INV_STACK;

extern const XSEQNO  DBFS_ROOT_CONT_ID;

extern const XLNK    DBFS_ZERO_REFS;
extern const XLNK    DBFS_ONE_REF;

extern const XAREA   DBFS_ROOT_AREA;
extern const XNUM    DBFS_ROOT_DIRECTORY;

extern const XFID    XFID_NONE;

extern const XTYPE   XTYPE_NOT_PRESENT;
extern const XTYPE   XTYPE_REFLINK;
extern const XTYPE   XTYPE_AREALINK;
extern const XTYPE   XTYPE_DIRHASH;
extern const XTYPE   XTYPE_DIRBTREE;
extern const XTYPE   XTYPE_DIRSEQ;
extern const XTYPE   XTYPE_SHORTSEG;
extern const XTYPE   XTYPE_LONGSEG;
extern const XTYPE   XTYPE_VIEWSEG;

extern /*const*/ XSTCK   __CLL_PREV_STCK;
extern /*const*/ XSTCK   __CLL_NEXT_STCK;

extern /*const*/ XSTCK   __COMP_SRC_STCK;
extern /*const*/ XSTCK   __COMP_CONTROL_STCK;
extern /*const*/ XSTCK   __COMP_INDEX_STCK;
extern /*const*/ XSTCK   __COMP_STATE_STCK;
extern /*const*/ XSTCK   __COMP_SHAREDLNK_STCK;

extern const XVID    __COMP_RECONSTRUCT_VIEW;

int         dbfs_viewdef_find (const XVID  &id,
			       VIEWDEF    **vdefp);

int         dbfs_stackdef_find (const XSTCK &id,
				STACKDEF   **sdefp);

FileHandle* dbfs_seg_replace  (const NODEC &repl,
			       int          flags);

int         dbfs_seg_read     (const NODEC  &read,
			       FileHandle  **fhp,
			       int           flags);

int         dbfs_check_ow_flags (int flags);

const char* dbfs_xtype_to_string (XTYPE    type);
const char* dbfs_type_to_string  (FileType type);


int     dbfs_async_start        (int oflags);
int     dbfs_async_txn_chkflags (int beginflags);

#include "dbfs_errs.h"
#include "seqv.h"
#include "shared.h"
#include "curs.h"

//////////////////////////////////////////////////////////////////////
//				 DBFS
//////////////////////////////////////////////////////////////////////

enum KILLPROC_STATUS {
    KILLPROC_SIGTERM,
    KILLPROC_SIGKILL,
    KILLPROC_NOTFOUND,
    KILLPROC_PERMISSION,
    KILLPROC_NODEATH,
};

class KILLPROC_DATA
{
public:

    class KILLPROC_ITEM
    {
    public:
	KILLPROC_ITEM (pid_t            pid,
		       const string    &desc,
		       KILLPROC_STATUS  status)
	    : _pid    (pid),
	      _desc   (desc),
	      _status (status) {}

	pid_t  _pid;
	string _desc;
	KILLPROC_STATUS   _status;
    };

    void add (pid_t pid, const string& desc, KILLPROC_STATUS status)
    {
	_vec.push_back (KILLPROC_ITEM (pid, desc, status));
    }

    int             size   ()      const { return _vec.size (); }
    pid_t           pid    (int i) const { return _vec[i]._pid; }
    const string&   desc   (int i) const { return _vec[i]._desc; }
    KILLPROC_STATUS status (int i) const { return _vec[i]._status; }

    vector<KILLPROC_ITEM> _vec;
};

class LISTPROC_DATA
{
public:

    class LISTPROC_ITEM
    {
    public:
	LISTPROC_ITEM (pid_t         pid,
		       const string &desc,
		       bool          exists)
	    : _pid  (pid),
	      _desc (desc),
	      _exists (exists) {}

	pid_t  _pid;
	string _desc;
	bool   _exists;
    };

    void add (pid_t pid, const string& desc, bool exists)
    {
	_vec.push_back (LISTPROC_ITEM (pid, desc, exists));
    }

    int           size   ()      const { return _vec.size (); }
    pid_t         pid    (int i) const { return _vec[i]._pid; }
    const string& desc   (int i) const { return _vec[i]._desc; }
    bool          exists (int i) const { return _vec[i]._exists; }

    vector<LISTPROC_ITEM> _vec;
};

class BASIC_DBENV
{
    friend class STATIC_DB;
    friend class STXN;

protected:

    BASIC_DBENV (int dbtabsize,
		 int fdtabsize,
		 int pgtabsize);

    DbEnv     *_env;
    int        _envopen_ret;
    STATIC_DB *_static_list;

    int          basic_db_open (const char* name, int oflags, int env_open_flags);
    virtual bool is_dbfs () const = 0;

public:

    virtual ~BASIC_DBENV ();

    int         checkpoint  ();
    virtual int close       ();

    SHDB_DSET _shared_dbs;
    SHFD_DSET _shared_fds;
    SHPG_DSET _shared_pgs;
};

class ADMIN_DB
    : public BASIC_DBENV
{
    friend class DBFS;

public:

    ADMIN_DB ();

    virtual bool is_dbfs () const { return false; }
    int    open          (const char* name, int oflags, int env_open_flags);

private:

    STATIC_DB  _pids;
    STATIC_DB  _settings;
};

class DBFS
    : public BASIC_DBENV
{
private:

    friend class STXN;
    friend class STATIC_DB;
    friend class DXN;
    friend class TXN;
    friend class SEQV;
    friend class SAREA_DESC;
    friend class DYNAMIC_DB;
    friend class NODEC;

    DBFS (const DBFS& nocopy);
    DBFS& operator= (const DBFS& noassign);

public:

    DBFS  (int argc, char const* const* argv);
    ~DBFS ();

    virtual bool is_dbfs () const { return true; }

    int  open          (const char* fsdir, const char* logdir, int flag);
    int  close         ();

    int  catalog_stack_id  (TXN        &txn,
			    const char *module,
			    const char *name,
			    const char *abbrev,
			    XSTCK      &stckval);

    int  admin_listproc    (LISTPROC_DATA &data);
    int  admin_killproc    (KILLPROC_DATA &data);

    void relative_fname (const XFID    &fid, string        &fname);
    void absolute_fname (const XFID    &fid, string        &fname);

protected:

    int  create_root   (int flags);
    int  create_base   (XFID &fid_base);

private:

    int  catalog_read     ();
    int  catalog_printfs  (int flags);
    int  catalog_builtins (TXN &txn);

    int  deadlock      ();
    int  clean_log     ();
    int  trickle       ();

    void  maint_start    (int flags);
    void  deadlock_start (int flags);

    int   maint_thread    ();
    int   deadlock_thread ();

    static void* maint_thread_start    (DBFS *dbfs);
    static void* deadlock_thread_start (DBFS *dbfs);

    void append_fname   (const XFID    &fid,
			 string        &fname,
			 bool           only_dir);
    void relative_dname (const XFID    &fid,
			 string        &fname);

    void absolute_dname (const XFID    &fid,
			 string        &fname);

    int  clear_dir     (const string &dir);
    int  create_dir    (const string &dir);
    int  check_dir     (const string &dir);
    int  create_config ();

    int  check_recovery  (int flags, int &dbenv_oflags);
    int  finish_recovery ();
    int  close_recovery  ();

    int  absolute_path (const char *maybe_rel, string &abs);

    bool has_logdir    () { return _logdir.size () > 0; }

    STXN       _admintxn;
    ADMIN_DB   _admindb;

    int        _openflags;

    STATIC_DB  _area_db;
    STATIC_DB  _seqv_db;

    STATIC_DB  _alloc_fids;
    STATIC_DB  _clean_fids;

    STATIC_DB  _stackcat_db;

    SEQV       _fid_seqv; /* FILE ID */
    SEQV       _sid_seqv; /* STCKCAT ID */

    pthread_t  _maint_tid;
    pthread_t  _deadlock_tid;

    string     _fsdir;
    string     _logdir;
    string     _filesdir;
    string     _admindir;
    string     _cmdline;
};

//////////////////////////////////////////////////////////////////////
//			       INCLUDES
//////////////////////////////////////////////////////////////////////

#include "dxn.h"
#include "dirc.h"

extern const MKEY    DBFS_DEF_MKEY; // Warning: Do not use in an initializer!
extern const DKEY    DBFS_NULL_DKEY;

//////////////////////////////////////////////////////////////////////
//			       NODEBASE
//////////////////////////////////////////////////////////////////////

template <class T>
class NODEBASE
{
protected:

    NODEBASE (const NODEBASE &copy)
	: _ref (copy._ref),
	  _txn (copy._txn) { }

    NODEBASE (TXN &txn, const T &ref)
	: _ref (& ref),
	  _txn (& txn) { }

    bool       equals  (const NODEBASE& node) const { g_assert (_txn == node._txn); return _ref == node._ref; }
    void       set     (const NODEBASE& node)       { _txn = node._txn; _ref = node._ref; }

private:

    const T *_ref;
    TXN     *_txn;

public:

    NODEBASE ()
	: _ref (NULL),
	  _txn (NULL) { }

    void       assert_valid () const { g_assert (_ref && _txn); }
    bool       is_valid     () const { return _ref != NULL && _txn != NULL; }
    const T&   ref          () const { g_assert (_ref); return *_ref; }
    TXN&       txn          () const { g_assert (_txn); return *_txn; }
    void       reset        () { _ref = NULL; _txn = NULL; }
};

//////////////////////////////////////////////////////////////////////
//			     FILENODEBASE
//////////////////////////////////////////////////////////////////////

template <class T>
class FILENODEBASE
    : public NODEBASE<T>
{
protected:

    FILENODEBASE (const NODEBASE<T> &copy)
	: NODEBASE<T> (copy) { }

    FILENODEBASE (TXN &txn, const T &ref)
	: NODEBASE<T> (txn, ref) { }

    FILENODEBASE ()
	: NODEBASE<T> () { }

public:

    // Basic access
    virtual FileType    type           (void) const = 0;
    virtual XTYPE       xtype          (void) const = 0;
    virtual XCONT       cont_id        (void) const = 0;
    virtual XSIZ        length         (void) const = 0;
    virtual SAREA       sarea          (void) const = 0;

    bool is_container () const { return (type () & FT_Container) != 0; }
    bool is_keyed     () const { return (type () & FT_Keyed) != 0; }
    bool is_sequence  () const { return (type () & FT_DirSeq) != 0; }
    bool is_segment   () const { return (type () & FT_Readable) != 0; }
    bool is_reflink   () const { return (type () & FT_Reflink) != 0; }
    bool is_arealink  () const { return (type () & FT_Arealink) != 0; }
    bool is_present   () const { return (type ()) != 0; }

    // Directory/Sequence-specific operations
    virtual int dir_inverse_node        (const MAJORC &node,
					 NODEC        &attr)  const = 0;
    virtual int dir_open               (DIRC         &curs)  const = 0;

    virtual int dir_link_invert        (const MAJORC &major,
					DKEY         &dname) const = 0;
    virtual int dir_link_invert_seqno  (const MAJORC &major,
					XSEQNO       &seqno) const = 0;

    virtual int dir_link_insert        (const DKEY   &key,
					const MAJORC &node,
					int           flags) const = 0;
    virtual int dir_link_insert_seqno  (const MAJORC &node,
					XSEQNO       &seqno,
					int           flags) const = 0;

    virtual int dir_link_lookup        (const DKEY   &key,
					MAJORC       &node,
					int           flags) const = 0;
    virtual int dir_link_lookup_seqno  (const XSEQNO &key,
					MAJORC       &node,
					int           flags) const = 0;

    virtual int dir_link_remove        (const DKEY   &key)   const = 0;
    virtual int dir_link_remove_seqno  (const XSEQNO &key)   const = 0;

    // Get/put short
    virtual int put_segment            (const guint8 *buf,
					int           size,
					int           flags) const = 0;
    virtual int get_segment            (guint8       *buf,
					int           flags) const = 0;

    // Unsafe longs
    virtual int unsafe_create          (int           flags) const = 0;
    virtual int unsafe_truncate        (const XSIZ   &size) const = 0;
    virtual int unsafe_write           (const guint8 *buf,
					const XSIZ   &nbytes,
					const XSIZ   &offset) const = 0;
    virtual int unsafe_read            (guint8       *buf,
					const XSIZ   &nbytes,
					const XSIZ   &offset,
					XSIZ         &nbytes_out) const = 0;
    virtual int unsafe_sync            (void) const = 0;

    // Short/long read/write handle
    virtual int repl_segment           (FileHandle  **fhp,
					int           flags) const = 0;
    virtual int read_segment           (FileHandle  **fhp,
					int           flags) const = 0;

    // Other file types
    virtual int mk_view                (XSIZ          length,
					VIEWDEF      &vd,
					int           flags) const = 0;
    virtual int mk_reflink             (const MAJORC &linkto,
					int           flags) const = 0;
    virtual int read_reflink           (MAJORC       &node,
					int           flags)  const = 0;
    virtual int mk_directory           (XTYPE         type,
					int           flags) const = 0;

    // Areas
    virtual int mk_arealink            (const SAREA  &area,
					int           flags) const = 0;
    virtual int read_arealink          (SAREA        &area,
					int           flags) const = 0;
};

//////////////////////////////////////////////////////////////////////
//			     STORAGE AREA
//////////////////////////////////////////////////////////////////////

class SAREA
    : public NODEBASE<SAREA_DESC>
{
    friend class MAJORC;
    friend class DXN;
    friend class TXN;

private:

    SAREA (TXN &txn, const SAREA_DESC &ref)
	: NODEBASE<SAREA_DESC> (txn, ref) { }

    int    printfs       (int flags) const;
    int    printfs_minor (int flags) const;

public:

    SAREA (const SAREA& copy)
	: NODEBASE<SAREA_DESC> (copy) { }
    SAREA () { }

    bool        operator==   (const SAREA& node) const { return equals (node); }
    SAREA&      operator=    (const SAREA& node)       { set (node); return *this; }

    int    allocate   (MAJORC &major,
		       int     flags) const;
    XAREA  area_id    (void) const;
};

//////////////////////////////////////////////////////////////////////
//				MAJORC
//////////////////////////////////////////////////////////////////////

class MAJORC
    : public FILENODEBASE<MAJOR_DESC>
{
private:

    friend class NODEC;
    friend class DIRC;
    friend class DXN;
    friend class TXN;
    friend class DBFS;

    MAJORC (TXN &txn, const MAJOR_DESC& ref)
	: FILENODEBASE<MAJOR_DESC> (txn, ref) { }

public:

    MAJORC (const MAJORC& copy)
	: FILENODEBASE<MAJOR_DESC> (copy) { }
    MAJORC () { }

private:

    int         dir_link_to            (const XFID    &db_id,
					const COMDBT  &dkey_dbt,
					int            flags) const;
    int         del_link_to            (const XFID    &db_id)    const;

    DBCREF&     locc                   () const;
    void        incr_refcount          (int change) const;

public:

    int  compare_to   (const MAJORC &node) const;

    bool        operator==   (const MAJORC& node) const { return equals (node); }
    MAJORC&     operator=    (const MAJORC& node)       { set (node); return *this; }

    // Sub-nodes
    int         node_mkey              (const MKEY   &mkey,
				       NODEC        &node) const;
    int         node_exists            (const MKEY   &mkey,
				       NODEC        &node) const;

    // Fields
    NODEC       def                    (void) const;
    XNUM        number                 (void) const;
    XCONT       cont_id                (void) const;
    FileType    type                   (void) const;
    XTYPE       xtype                  (void) const;
    XSIZ        length                 (void) const;
    SAREA       sarea                  (void) const;
    XLNK        nlinks                 (void) const;

    // Major-specific
    int         del_major              (void) const;
    int         del_minor              (void) const;
    int         unlink                 (void) const;

    // Directory/Sequence-specific operations
    int         dir_inverse_node        (const MAJORC &node,
					 NODEC        &attr)  const;
    int         dir_open               (DIRC         &curs)  const;

    int         dir_link_invert        (const MAJORC &major,
					DKEY         &dname) const;
    int         dir_link_invert_seqno  (const MAJORC &major,
					XSEQNO       &seqno) const;

    int         dir_link_insert        (const DKEY   &key,
					const MAJORC &node,
					int           flags) const;
    int         dir_link_insert_seqno  (const MAJORC &node,
					XSEQNO       &seqno,
					int           flags) const;

    int         dir_link_lookup        (const DKEY   &key,
					MAJORC       &node,
					int           flags) const;
    int         dir_link_lookup_seqno  (const XSEQNO &key,
					MAJORC       &node,
					int           flags) const;

    int         dir_link_remove        (const DKEY   &key)   const;
    int         dir_link_remove_seqno  (const XSEQNO &key)   const;

    int         put_segment            (const guint8 *buf,
					int           size,
					int           flags) const;
    int         get_segment            (guint8       *buf,
					int           flags) const;

    int         unsafe_create          (int           flags) const;
    int         unsafe_truncate        (const XSIZ   &size) const;
    int         unsafe_write           (const guint8 *buf,
					const XSIZ   &nbytes,
					const XSIZ   &offset) const;
    int         unsafe_read            (guint8       *buf,
					const XSIZ   &nbytes,
					const XSIZ   &offset,
					XSIZ         &nbytes_out) const;
    int         unsafe_sync            (void) const;

    int         repl_segment           (FileHandle  **fhp,
					int           flags) const;
    int         read_segment           (FileHandle  **fhp,
					int           flags) const;

    int         mk_view                (XSIZ          length,
					VIEWDEF      &vd,
					int           flags) const;
    int         mk_reflink             (const MAJORC &linkto,
					int           flags) const;
    int         read_reflink           (MAJORC       &node,
					int           flags) const;

    int         mk_directory           (XTYPE         type,
					int           flags) const;

    int         mk_arealink            (const SAREA  &area,
					int           flags) const;
    int         read_arealink          (SAREA        &area,
					int           flags) const;
};

//////////////////////////////////////////////////////////////////////
//				NODEC
//////////////////////////////////////////////////////////////////////

class NODEC
    : public FILENODEBASE<MINOR_DESC>
{
private:
    friend class MAJORC;
    friend class WRH;
    friend class DXN;
    friend class MAJOR_DESC;

    NODEC (TXN &txn, const MINOR_DESC &ref)
	: FILENODEBASE<MINOR_DESC> (txn, ref) { }

public:

    NODEC (const NODEC& copy)
	: FILENODEBASE<MINOR_DESC> (copy) { }
    NODEC () { }

private:

    int         pre_delete             () const;
    int         pre_delete_cont        () const;
    int         pre_overwrite_later    (int           flags) const { return pre_overwrite_when (flags, false); }
    int         pre_overwrite_now      (int           flags) const { return pre_overwrite_when (flags, true); }
    int         pre_overwrite_when     (int           flags,
					bool          now) const;

    DBCREF&     locc                   () const;

public:

    bool        operator== (const NODEC& node) const { return equals (node); }
    NODEC&      operator=  (const NODEC& node)       { set (node); return *this; }

    // Fields
    MAJORC      majorc                 (void) const;
    const MKEY& mkey                   (void) const;
    FileType    type                   (void) const;
    XTYPE       xtype                  (void) const;
    XCONT       cont_id                (void) const;
    XSIZ        length                 (void) const;
    SAREA       sarea                  (void) const;

    // Minor-specific
    int         del_minor               (void) const;

    // Directory/Sequence-specific operations
    int         dir_inverse_node        (const MAJORC &node,
					 NODEC        &attr)  const;
    int         dir_open               (DIRC         &curs)  const;

    int         dir_link_invert        (const MAJORC &major,
					DKEY         &dname) const;
    int         dir_link_invert_seqno  (const MAJORC &major,
					XSEQNO       &seqno) const;

    int         dir_link_insert        (const DKEY   &key,
					const MAJORC &node,
					int           flags) const;
    int         dir_link_insert_seqno  (const MAJORC &node,
					XSEQNO       &seqno,
					int           flags) const;

    int         dir_link_lookup        (const DKEY   &key,
					MAJORC       &node,
					int           flags) const;
    int         dir_link_lookup_seqno  (const XSEQNO &key,
					MAJORC       &node,
					int           flags) const;

    int         dir_link_remove        (const DKEY   &key)   const;
    int         dir_link_remove_seqno  (const XSEQNO &key)   const;

    int         put_segment            (const guint8 *buf,
					int           size,
					int           flags) const;
    int         get_segment            (guint8       *buf,
					int           flags) const;

    int         unsafe_create          (int           flags) const;
    int         unsafe_truncate        (const XSIZ   &size) const;
    int         unsafe_write           (const guint8 *buf,
					const XSIZ   &nbytes,
					const XSIZ   &offset) const;
    int         unsafe_read            (guint8       *buf,
					const XSIZ   &nbytes,
					const XSIZ   &offset,
					XSIZ         &nbytes_out) const;
    int         unsafe_sync            (void) const;

    int         repl_segment           (FileHandle  **fhp,
					int           flags) const;
    int         read_segment           (FileHandle  **fhp,
					int           flags) const;

    int         mk_view                (XSIZ          length,
					VIEWDEF      &vd,
					int           flags) const;
    int         mk_reflink             (const MAJORC &linkto,
					int           flags) const;
    int         read_reflink           (MAJORC       &node,
					int           flags) const;

    int         mk_directory           (XTYPE         type,
					int           flags) const;

    int         mk_arealink            (const SAREA  &area,
					int           flags) const;
    int         read_arealink          (SAREA        &area,
					int           flags) const;
};

//////////////////////////////////////////////////////////////////////
//				 TXN
//////////////////////////////////////////////////////////////////////

class TXN
    : public STXN
{
    friend class DXN;
    friend class DYNAMIC_DB;
    friend class MAJORC;

public:

    TXN  ();
    ~TXN ();

    int   begin_root   (DBFS &dbfs, int flags, MAJORC &root);
    int   begin        (DBFS &dbfs, int flags);

    int   find_major  (MAJORC &major, const XNUM &num);
    int   create_area (SAREA  &area);

    int   printfs     (int flags);

    const MAJORC& root () { return _root; }

protected:

    int   pre_commit   ();
    int   post_commit  ();
    void  pre_abort    ();
    void  post_abort   ();
    void  common_abort ();

private:

    int   private_find_root     (MAJORC &root);

    MAJORC            _root;
    MAJOR_ORDER       _major_order;
    MAJOR_MAP         _major_map;
    SAREA_MAP         _area_map;

    DSET<MAJOR_DESC>  _majors;
    DSET<MINOR_DESC>  _minors;
    DSET<SAREA_DESC>  _areas;

    TXN_DEALLOC_LIST  _dealloc;
};

//////////////////////////////////////////////////////////////////////
//			     MISC INLINE
//////////////////////////////////////////////////////////////////////

inline NODEC       MAJORC::def    () const { return NODEC (txn (), *ref ().def_minor); }
inline XNUM        MAJORC::number () const { return ref ().num; }
inline FileType    MAJORC::type   () const { return def ().type (); }
inline XTYPE       MAJORC::xtype  () const { return def ().xtype (); }
inline XSIZ        MAJORC::length () const { return def ().length (); }
inline XCONT       MAJORC::cont_id() const { return def ().cont_id (); }
inline XLNK        MAJORC::nlinks () const { return ref ().rec.refs; }
inline SAREA       MAJORC::sarea  () const { return SAREA (txn (), *ref ().area); }

inline MAJORC      NODEC::majorc  () const { return MAJORC (txn (), *ref ().major); }
inline const MKEY& NODEC::mkey    () const { return ref ().mkey; }
inline FileType    NODEC::type    () const { return (FileType) ref ().rec.type.key (); }
inline XTYPE       NODEC::xtype   () const { return ref ().rec.type; }
inline XCONT       NODEC::cont_id () const { return ref ().rec.cont; }
inline XSIZ        NODEC::length  () const { return ref ().rec.length; }
inline SAREA       NODEC::sarea   () const { return majorc ().sarea (); }

inline XAREA       SAREA::area_id () const { return ref ().area_id; }

inline DbEnv*      STXN::env      () const { return dbfs()._env; }

inline MESSAGE_CLOSURE&  MESSAGE_CLOSURE::operator () (const NODEC  *node) { return (*this) (*node); }
inline MESSAGE_CLOSURE&  MESSAGE_CLOSURE::operator () (const NODEC  &node) { return (*this) (node.ref ()); }

inline MESSAGE_CLOSURE&  MESSAGE_CLOSURE::operator () (const MAJORC *node) { return (*this) (*node); }
inline MESSAGE_CLOSURE&  MESSAGE_CLOSURE::operator () (const MAJORC &node) { return (*this) (node.ref ()); }

inline MESSAGE_CLOSURE&  MESSAGE_CLOSURE::operator () (const SAREA *node) { return (*this) (*node); }
inline MESSAGE_CLOSURE&  MESSAGE_CLOSURE::operator () (const SAREA &node) { return (*this) (node.ref ()); }

inline MESSAGE_CLOSURE&  MESSAGE_CLOSURE::operator () (const MINOR_DESC *node) { return (*this) (*node); }
inline MESSAGE_CLOSURE&  MESSAGE_CLOSURE::operator () (const MAJOR_DESC *node) { return (*this) (*node); }
inline MESSAGE_CLOSURE&  MESSAGE_CLOSURE::operator () (const SAREA_DESC *node) { return (*this) (*node); }

inline DBCREF&           NODEC::locc  () const { return ref ().major->area->locc; }
inline DBCREF&           MAJORC::locc () const { return ref ().area->locc; }

inline DBFS&             DXN::dbfs    () const { return _txn.dbfs (); }
inline DBFS&             STXN::dbfs   () const { g_assert (_dbfs && _dbfs->is_dbfs ()); return (DBFS&) *_dbfs; }

//////////////////////////////////////////////////////////////////////
//				 VIEW
//////////////////////////////////////////////////////////////////////

typedef int (ViewBegin) (const NODEC &node);

typedef int (ViewPgin)  (guint8      *pgbuf,
			 const XSIZ  &offset,
			 size_t       len);

class VIEWIMPL
{
public:
    virtual ~VIEWIMPL () { }

    virtual ViewBegin begin = 0;
    virtual ViewPgin  pgin  = 0;

protected:

    VIEWIMPL () { }

private:

    VIEWIMPL (const VIEWIMPL& copy);
};

class VIEWDEF
{
public:

    VIEWDEF (XVID view_id);

    virtual VIEWIMPL*   make () = 0;
    virtual const char* name () = 0;

    const XVID& id () const { return _view_id; }

private:

    VIEWDEF (const VIEWDEF& copy);

    XVID _view_id;
};

class STACKDEF
{
public:

    const char*  abbrev () const { return _abbrev.c_str (); }
    const char*  name   () const { return _name.c_str (); }
    const char*  module () const { return _module.c_str (); }
    const XSTCK& id     () const { return _stck_id; }

private:

    friend class DBFS;

    STACKDEF (const XSTCK &stck_id, const CKEY &key);

    XSTCK  _stck_id;
    string _name;
    string _module;
    string _abbrev;
};

XSEQNO XHTONL (const XSEQNO& x);
XSEQNO XNTOHL (const XSEQNO& x);

#define DEBUG_ME() \
{static int x = 1; INFO_DBFS ("debug me at %s: %d", __FILE__, getpid ()); while (x) { }}

#endif /* __cplusplus */

#endif /* _XDFS_CPP_H_ */
