/* -*-Mode: C++;-*-
 * $Id: dxn.h 1.20 Tue, 15 May 2001 16:33:56 -0700 jmacd $
 *
 * Copyright (C) 1998, 1999, 2000, Joshua P. MacDonald
 * <jmacd@CS.Berkeley.EDU> and The Regents of the University of
 * California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _DXN_H_
#define _DXN_H_

class SAREA_DESC;
class MAJOR_DESC;
class MINOR_DESC;

//////////////////////////////////////////////////////////////////////
//			      COMMON KEY
//////////////////////////////////////////////////////////////////////

class CKEY
    : public COMDBT
{
public:

    CKEY ()
    {
    }
    CKEY (size_t len)
    {
	_vec.resize (len);
	reset_dbt (data (), len);
    }

    template <class T>
    void set_rec    (const T      &key) { clear (); append_rec (key); }

    void set_buf    (const guint8* d,
		     size_t        len) { clear (); append_buf (d, len); }

    void set_ckey   (const CKEY   &ckey) { clear (); append_ckey (ckey); }

    template <class T>
    void append_rec (const T      &key) { append_buf ((guint8*) & key, sizeof (key)); }

    void append_ckey (const CKEY   &ckey) { append_buf (ckey.data (), ckey.size ()); }

    void append_str (const char* d)
    {
	append_buf ((const guint8*) d, strlen (d));
    }

    void append_buf (const guint8* d,
		     size_t        len)
    {
	size_t off = size ();
	_vec.resize (off + len);
	for (size_t i = 0; i < len; i += 1) {
	    _vec[off+i] = d[i];
	}
	reset_dbt (data (), off + len);
    }

    void append_byte (guint8 b)
    {
	size_t off = size ();
	_vec.resize (off + 1);
	_vec[off] = b;
	reset_dbt (data (), off + 1);
    }

    bool equals (const CKEY&   key) const
    {
	if (size () != key.size ()) {
	    return false;
	}

	for (size_t i = 0; i < size (); i += 1) {
	    if (data ()[i] != key.data ()[i]) {
		return false;
	    }
	}

	return true;
    }

    template<class T>
    void get_rec (T &val) const
    {
	g_assert (size () == sizeof (T));

	memcpy (& val, data (), sizeof (T));
    }

    template<class T>
    void get_rec_prefix (T &val) const
    {
	get_rec_at (0, val);
    }

    template<class T>
    void get_rec_at (size_t off, T &val) const
    {
	g_assert (size () >= (sizeof (T) + off));

	memcpy (& val, data () + off, sizeof (T));
    }

    void get_string (string &val)
    {
	val = "";
	val.append ((const char*) data (), size ());
    }

    void clear ()
    {
	_vec.clear ();
	reset_dbt (data (), 0);
    }

    size_t        size  () const { return get_size (); }
    const guint8* data  () const { return & _vec.front (); }

    void enlarge ()
    {
	_vec.resize (size ());
	reset_ulen (data (), size ());
    }

    void post () { }

    void ckey_write (ostream &os, size_t offset) const;

protected:

    CKEY (CKEY& ckey);

private:

    vector<guint8> _vec;
};

//////////////////////////////////////////////////////////////////////
//			    DIRECTORY KEY
//////////////////////////////////////////////////////////////////////

class DKEY
    : public CKEY
{
public:

    explicit DKEY () { }

             DKEY (const DKEY&   key) { set_ckey (key); }
             DKEY (const char   *key) { set_buf ((guint8*) key, strlen (key)); } // non-explicit
             DKEY (const char   *key,
		   size_t        len) { set_buf ((guint8*) key, len); }
             DKEY (const guint8 *key,
		   size_t        len) { set_buf (key, len); }

    explicit DKEY (guint16       key) { this->template set_rec<guint16> (key); }
    explicit DKEY (guint32       key) { this->template set_rec<guint32> (key); }
    explicit DKEY (guint64       key) { this->template set_rec<guint64> (key); }
    explicit DKEY (const XFID&   key) { this->template set_rec<XFID>    (key); }
    explicit DKEY (const XCONT&  key) { this->template set_rec<XCONT>   (key); }

    bool  operator== (const DKEY& dkey) const { return equals (dkey); }
    DKEY& operator=  (const DKEY& dkey)       { set_ckey (dkey); return *this; }
};

//////////////////////////////////////////////////////////////////////
//			    MINOR NODE KEY
//////////////////////////////////////////////////////////////////////

class MKEY
    : public CKEY
{
public:

    MKEY () { }
    MKEY (const MKEY&   key) { set_ckey (key); }

    MKEY (const DKEY&   dkey,
	  XSTCK         stck = XSTCK (DBFS_DEF_STACK_VAL))
    {
	this->template set_rec<XSTCK> (stck);

	append_ckey (dkey);
    }

    MKEY (XSTCK stck)
    {
	this->template set_rec<XSTCK> (stck);
    }

    MKEY (guint16       key,
	  XSTCK         stck = XSTCK (DBFS_DEF_STACK_VAL))
    {
	this->template set_rec<XSTCK>      (stck);
	this->template append_rec<guint16> (key);
    }

    MKEY (const char   *key,
	  XSTCK         stck = XSTCK (DBFS_DEF_STACK_VAL))
    {
	this->template set_rec<XSTCK> (stck);
	append_buf ((guint8*) key, strlen (key));
    }

    MKEY (const MKEY&   dkey,
	  XSTCK         stck)
    {
	this->template set_rec<XSTCK> (stck);

	append_buf (dkey.data () + sizeof (stck), dkey.size () - sizeof (stck));
    }

    bool  operator== (const MKEY& mkey) const { return equals (mkey); }
    MKEY& operator=  (const MKEY& mkey)       { set_ckey (mkey); return *this; }

    XSTCK get_stck () const
    {
	XSTCK stck;
	this->template get_rec_prefix<XSTCK> (stck);
	return stck;
    }
};

ostream& operator<< (ostream &os, const DKEY& key);
ostream& operator<< (ostream &os, const MKEY& key);

//////////////////////////////////////////////////////////////////////
//			    MINOR NODE DBT
//////////////////////////////////////////////////////////////////////

class MINOR_DB_KEY
    : public CKEY
{
public:

    MINOR_DB_KEY (const XNUM &major, const MKEY& mkey)
    {
	this->template set_rec<XSEQNO> (major.cont_seq ());

	append_ckey (mkey);
    }
};

class REVERSE_LINK_MKEY
    : public MKEY
{
public:
    REVERSE_LINK_MKEY (const XFID &db_id)
	: MKEY (DKEY (db_id), DBFS_INV_STACK) { }
};

class REVERSE_LINK_KEY
    : public MINOR_DB_KEY
{
public:

    REVERSE_LINK_KEY (const XNUM  &major_num,
		      const XFID  &db_id)
	: MINOR_DB_KEY (major_num, REVERSE_LINK_MKEY (db_id)) {}
};

//////////////////////////////////////////////////////////////////////
//		       PERSISTENT NODE RECORDS
//////////////////////////////////////////////////////////////////////

struct MINOR_REC // 14 bytes
{
    XTYPE    type;   // 2
    XSIZ     length; // 4
    XCONT    cont;   // 8
};

struct MAJOR_REC
{
    XLNK    refs; // 2 bytes
};

struct SAREA_REC
{
    XFID    container_fid;
};

//////////////////////////////////////////////////////////////////////
//			   NODE DESCRIPTORS
//////////////////////////////////////////////////////////////////////

template <class T>
class DSET
{
public:

    DSET (size_t size)
	: _size     (size),
	  _array    (new T[size]),
	  _free     (NULL),
	  _inuse    (NULL)
    {
	for (size_t i = 0; i < size; i += 1) {
	    _array[i]._next = _free;
	    _free           = & _array[i];
	}
    }

    ~DSET ()
    {
	delete [] _array;
    }

    int alloc (T*& descp)
    {
	if (_free == NULL) {
	    return DBFS_NOSPACE;
	}

	descp = _free;
	_free = _free->_next;

	descp->_next = _inuse;
	_inuse       = descp;

	return 0;
    }

    T*     inuse () const { return _inuse; }
    size_t size  () const { return _size; }

protected:

    size_t  _size;
    T      *_array;
    T      *_free;
    T      *_inuse;
};

enum COMMON_DESC_MODE
{
    COMMON_DESC_CLEAN      = 0,
    COMMON_DESC_DIRTY      = (1 << 0),
    COMMON_DESC_DELETED    = (1 << 1),
    COMMON_DESC_NOTPRESENT = (1 << 2),
};

template <class T>
class COMMON_DESC
{
    friend class DSET<T>;

public:

    COMMON_DESC ()
	: _mode (COMMON_DESC_CLEAN),
	  _next (NULL) { }

    bool get_deleted    () const { return _mode & COMMON_DESC_DELETED; }
    bool get_dirty      () const { return _mode & COMMON_DESC_DIRTY; }
    bool get_notpresent () const { return _mode & COMMON_DESC_NOTPRESENT; }

    void set_dirty ()
    {
	_mode |=  COMMON_DESC_DIRTY;
	_mode &= ~COMMON_DESC_DELETED;
    }

    void set_deleted ()
    {
	int both = COMMON_DESC_DELETED & COMMON_DESC_DIRTY;

	if (get_notpresent ()) {
	    _mode &= ~both;
	} else {
	    _mode |= both;
	}
    }

    void set_clean ()
    {
	_mode = COMMON_DESC_CLEAN;
    }

    void set_clean_notpresent ()
    {
	_mode = COMMON_DESC_NOTPRESENT;
    }

    T* next () const { return _next; }

private:

    int    _mode;
    T     *_next;
};

class SAREA_DESC
    : public COMMON_DESC<SAREA_DESC>
{
public:

    SAREA_DESC ()
	: container_dbr ()
    { }

    int      create (STXN   &txn);
    int      get    (STXN   &txn);
    int      get    (DBCREF &curs);
    int      put    (DBCREF &curs);
    int      open_cont (TXN  &txn,
			int  flag);

    int      close  ();

    XAREA       area_id;
    SAREA_REC   rec;

    DYNAMIC_DB  container_dbr;
    DBCREF      locc;
};

class MAJOR_DESC
    : public COMMON_DESC<MAJOR_DESC>
{
public:

    int  minor_final             (TXN &txn);
    int  minor_delete_notpresent (TXN &txn);
    int  minor_delete_present    (TXN &txn);
    int  final                   (TXN &txn);
    int  get                     ();
    int  find_minor              (TXN        &txn,
				  const MKEY &mkey,
				  NODEC      &node);

    XNUM              num;
    MAJOR_REC         rec;

    MINOR_DESC       *def_minor;
    SAREA_DESC       *area;
    MINOR_DESC       *all_minors;
};

class MINOR_DESC
    : public COMMON_DESC<MINOR_DESC>
{
public:

    MINOR_DESC () { }

    int  final   (TXN &txn);
    int  get     ();

    MKEY        mkey;
    MINOR_REC   rec;

    MAJOR_DESC *major;
    MINOR_DESC *next_in_major;
};

//////////////////////////////////////////////////////////////////////
//			    FILE SYSTEM TXNS
//////////////////////////////////////////////////////////////////////

typedef hash_map<guint64, MAJOR_DESC*, int64_hash, int64_equals> MAJOR_MAP;
typedef hash_map<guint32, SAREA_DESC*>                           SAREA_MAP;
typedef Slp<guint64,MAJOR_DESC*,UINT64_MAX,8,0>                  MAJOR_ORDER;

//////////////////////////////////////////////////////////////////////
//			 NODE DESCRIPTOR TXNS
//////////////////////////////////////////////////////////////////////

enum NODEKEY_TYPE {
    NODEKEY_MAJOR,
    NODEKEY_MINOR,
    NODEKEY_SHORT,
    NODEKEY_INVLNK
};

int dbfs_area_key (const SAREA  &area,
		   const CKEY   &key,
		   MAJORC       *majorp,
		   NODEC        *minorp,
		   XNUM         &number,
		   XSTCK        &stck,
		   MKEY         &mkey,
		   XFID         &inv_dbid,
		   NODEKEY_TYPE &type);

// This class doesn't really need to exist, but it encapsulates a lot
// of complex node operations
class DXN
{
public:

    DXN (TXN &txn, int& scope_ret)
	: _txn       (txn),
	  _scope_ret (scope_ret)
    {
	_scope_ret = 0;
    }

    DBFS&       dbfs           () const;

    int         get_area       (const XAREA   &area_id,
				SAREA         &area);
    int         get_major      (const XNUM    &majnum,
				MAJORC        &major);
    int         get_minor      (const MAJORC  &majnum,
				const MKEY    &mkey,
				NODEC         &node);

    int         create_area    (SAREA         &area);

    int         create_major   (const SAREA   &area,
				MAJORC        &node,
				int            flags);

    int         create_index   (const NODEC   &node,
				const XTYPE   &type,
				int            owf);
    int         create_view    (const NODEC   &node,
				const XSIZ    &length,
				const VIEWDEF &vdef,
				int            owf);
    int         create_reflink (const NODEC   &node,
				const MAJORC  &linkto,
				int            owf);

    int         create_arealink (const NODEC   &node,
				 const SAREA   &area,
				 int            owf);

    int         create_long    (const NODEC   &node,
			        const XFID    &fid,
			        const XSIZ    &length,
				int            owf);

    int         create_short   (const NODEC   &node,
			        const guint8  *buf,
			        const XSIZ    &length,
				int            owf);

    int         allocate_fid   (XFID          &nfid);
    int         deallocate_fid (const XFID    &nfid,
				const XTYPE   &type);

    int         get_short      (const NODEC   &node,
			        guint8        *buf);
    int         del_short      (const NODEC   &node);

    int         get_index      (const NODEC   &node,
			        DBREF         &dbref);

private:

    int         allocate_major (const SAREA   &area,
				XNUM          &next,
				int            flags);

    void        clean_refs     (MAJOR_DESC    &mdesc);

private:

    TXN &_txn;
    int &_scope_ret;
};

class TXN_DEALLOC
{
public:

    TXN_DEALLOC (const XFID &fid)
	: _fid (fid) { }

    XFID               _fid;
    elink<TXN_DEALLOC> _link;
};

typedef ELIST     (TXN_DEALLOC,_link) TXN_DEALLOC_LIST;

#endif /* _DXN_H_ */
