/*
 *   Copyright (C) 2002-2004 by Jonathan Naylor G4KLX
 *   Copyright (C) 2001 by Volker Fischer, TU-Darmstadt
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "SoundCard.h"
#include "Exception.h"

#include <sys/types.h>

#ifdef __WINDOWS__

CSoundCard::CSoundCard(const wxString& device, int sampleRate, int rxBufferSize, int txBufferSize) :
m_device(device),
m_sampleRate(sampleRate),
m_rxBufferSize(rxBufferSize),
m_txBufferSize(txBufferSize),
m_devIn(WAVE_MAPPER),
m_waveIn(NULL),
m_waveInEvent(NULL),
m_waveInBuffer(),
m_waveInHeader(),
m_bufferInNum(0),
m_devOut(WAVE_MAPPER),
m_waveOut(NULL),
m_waveOutEvent(NULL),
m_waveOutBuffer(),
m_waveOutHeader()
{
	wxASSERT(m_rxBufferSize > 0);
	wxASSERT(m_txBufferSize > 0);
	wxASSERT(m_sampleRate > 0);

	/* Init buffer pointer to zero */
	for (int i = 0; i < NUM_SOUND_BUFFERS_IN; i++) {
		memset(m_waveInHeader + i, 0x00, sizeof(WAVEHDR));
		m_waveInBuffer[i] = new wxInt16[rxBufferSize];
	}

	for (int i = 0; i < NUM_SOUND_BUFFERS_OUT; i++) {
		memset(m_waveOutHeader + i, 0x00, sizeof(WAVEHDR));
		m_waveOutBuffer[i] = new wxInt16[txBufferSize];
	}

	/* We use an event controlled wave-in (wave-out) structure */
	/* Create events */
	m_waveInEvent  = ::CreateEvent(NULL, FALSE, FALSE, NULL);
	m_waveOutEvent = ::CreateEvent(NULL, FALSE, FALSE, NULL);

	int n = ::waveInGetNumDevs();

	for (int i = 0; i < n; i++) {
		WAVEINCAPS waveInCaps;

		if (::waveInGetDevCaps(i, &waveInCaps, sizeof(WAVEINCAPS)) == MMSYSERR_NOERROR)
			if (::strcmp(waveInCaps.szPname, m_device.c_str()) == 0)
				m_devIn = i;
	}

	n = ::waveOutGetNumDevs();

	for (int i = 0; i < n; i++) {
		WAVEOUTCAPS waveOutCaps;

		if (::waveOutGetDevCaps(i, &waveOutCaps, sizeof(WAVEOUTCAPS)) == MMSYSERR_NOERROR)
			if (::strcmp(waveOutCaps.szPname, m_device.c_str()) == 0)
				m_devOut = i;
	}
}

CSoundCard::~CSoundCard()
{
	/* Delete allocated memory */
	for (int i = 0; i < NUM_SOUND_BUFFERS_IN; i++)
		delete[] m_waveInBuffer[i];

	for (int i = 0; i < NUM_SOUND_BUFFERS_OUT; i++)
		delete[] m_waveOutBuffer[i];

	/* Close the handle for the events */
	::CloseHandle(m_waveInEvent);
	::CloseHandle(m_waveOutEvent);
}

void CSoundCard::openWrite()
{
	WAVEFORMATEX waveFormat;

	waveFormat.wFormatTag      = WAVE_FORMAT_PCM;
	waveFormat.nChannels       = NUM_IN_OUT_CHANNELS;
	waveFormat.wBitsPerSample  = BITS_PER_SAMPLE;
	waveFormat.nSamplesPerSec  = m_sampleRate;
	waveFormat.nBlockAlign     = waveFormat.nChannels * waveFormat.wBitsPerSample / 8;
	waveFormat.nAvgBytesPerSec = waveFormat.nBlockAlign * waveFormat.nSamplesPerSec;
	waveFormat.cbSize          = 0;

	MMRESULT result = ::waveOutOpen(&m_waveOut, m_devOut, &waveFormat, (DWORD)m_waveOutEvent, (DWORD)0, CALLBACK_EVENT);

	if (result != MMSYSERR_NOERROR) {
		wxString text;
		text.Printf(wxT("Couldn't open the sound card for writing, error = %u"), result);
		throw CException(text);
	}

	/* Reset interface */
	::waveOutReset(m_waveOut);

	for (int j = 0; j < NUM_SOUND_BUFFERS_OUT; j++) {
		/* Clear buffer */
		for (int i = 0; i < m_txBufferSize; i++)
			m_waveOutBuffer[j][i] = 0;

		/* Initially, send all buffers to the interface */
		addOutBuffer(j);
	}
}

void CSoundCard::openRead()
{
	WAVEFORMATEX waveFormat;

	waveFormat.wFormatTag      = WAVE_FORMAT_PCM;
	waveFormat.nChannels       = NUM_IN_OUT_CHANNELS;
	waveFormat.wBitsPerSample  = BITS_PER_SAMPLE;
	waveFormat.nSamplesPerSec  = m_sampleRate;
	waveFormat.nBlockAlign     = waveFormat.nChannels * waveFormat.wBitsPerSample / 8;
	waveFormat.nAvgBytesPerSec = waveFormat.nBlockAlign * waveFormat.nSamplesPerSec;
	waveFormat.cbSize          = 0;

	MMRESULT result = ::waveInOpen(&m_waveIn, m_devIn, &waveFormat, (DWORD)m_waveInEvent, (DWORD)0, CALLBACK_EVENT);

	if (result != MMSYSERR_NOERROR) {
		wxString text;
		text.Printf(wxT("Couldn't open the sound card for reading, error = %u"), result);
		throw CException(text);
	}

	/* Reset interface so that all buffers are returned from the interface */
	::waveInReset(m_waveIn);
	::waveInStop(m_waveIn);

	/* Reset current buffer ID (it is important to do this BEFORE calling "addInBuffer()" */
	m_bufferInNum = 0;

	for (int i = 0; i < NUM_SOUND_BUFFERS_IN; i++)
		addInBuffer();

	/* Notify that sound capturing can start now */
	::waveInStart(m_waveIn);

	/* This reset event is very important for initialization, otherwise we will
	   get errors! */
	::ResetEvent(m_waveInEvent);
}

void CSoundCard::read(double* sample, int& len)
{
	if (len < m_rxBufferSize)
		throw CException(wxT("Read buffer is too short"));

	/* Wait until data is available */
	if (!(m_waveInHeader[m_bufferInNum].dwFlags & WHDR_DONE))
		::WaitForSingleObject(m_waveInEvent, INFINITE);

	for (int i = 0; i < m_rxBufferSize; i++)
		sample[i] = double(m_waveInBuffer[m_bufferInNum][i]) / 32768.0;

	/* Add the buffer so that it can be filled with new samples */
	addInBuffer();

	/* In case more than one buffer was ready, reset event */
	::ResetEvent(m_waveInEvent);

	len = m_rxBufferSize;
}

void CSoundCard::addInBuffer()
{
	/* Unprepare old wave-header */
	::waveInUnprepareHeader(m_waveIn, m_waveInHeader + m_bufferInNum, sizeof(WAVEHDR));

	/* Prepare buffers for sending to sound interface */
	m_waveInHeader[m_bufferInNum].lpData         = (LPSTR)m_waveInBuffer[m_bufferInNum];
	m_waveInHeader[m_bufferInNum].dwBufferLength = m_rxBufferSize * BYTES_PER_SAMPLE;
	m_waveInHeader[m_bufferInNum].dwFlags        = 0;

	/* Prepare wave-header */
	::waveInPrepareHeader(m_waveIn, m_waveInHeader + m_bufferInNum, sizeof(WAVEHDR));

	/* Send buffer to driver for filling with new data */
	::waveInAddBuffer(m_waveIn, m_waveInHeader + m_bufferInNum, sizeof(WAVEHDR));

	/* Toggle buffers */
	m_bufferInNum++;

	if (m_bufferInNum == NUM_SOUND_BUFFERS_IN)
		m_bufferInNum = 0;
}

void CSoundCard::write(double* sample, int len)
{
	if (len != m_txBufferSize)
		throw CException(wxT("Write buffer is not the correct length"));

	int iCntPrepBuf;
	int iIndexDoneBuf;

	/* Get number of "done"-buffers and position of one of them */
	getDoneBuffer(iCntPrepBuf, iIndexDoneBuf);

	/* Now check special cases (Buffer is full or empty) */
	if (iCntPrepBuf == 0) {
		/* Blocking wave out routine. Needed for transmitter. Always
		   ensure that the buffer is completely filled to avoid buffer
		   underruns */
		::WaitForSingleObject(m_waveOutEvent, INFINITE);

		getDoneBuffer(iCntPrepBuf, iIndexDoneBuf);

		if (iCntPrepBuf == 0)
			wxLogWarning(wxT("Write buffers still full even after event"));
	} else if (iCntPrepBuf == NUM_SOUND_BUFFERS_OUT) {
		/* ---------------------------------------------------------------------
		   Buffer is empty -> send as many cleared blocks to the sound-
		   interface until half of the buffer size is reached */
		/* Send half of the buffer size blocks to the sound-interface */
		for (int j = 0; j < NUM_SOUND_BUFFERS_OUT / 2; j++) {
			/* First, clear these buffers */
			for (int i = 0; i < m_txBufferSize; i++)
				m_waveOutBuffer[j][i] = 0;

			/* Then send them to the interface */
			addOutBuffer(j);
		}

		/* Set index for done buffer */
		iIndexDoneBuf = NUM_SOUND_BUFFERS_OUT / 2;
	}

	/* Copy audio data from input in soundcard buffer */
	for (int i = 0; i < m_txBufferSize; i++)
		m_waveOutBuffer[iIndexDoneBuf][i] = short(sample[i] * 32768.0);

	/* Now, send the current block */
	addOutBuffer(iIndexDoneBuf);
}

void CSoundCard::getDoneBuffer(int& iCntPrepBuf, int& iIndexDoneBuf) const
{
	/* Get number of "done"-buffers and position of one of them */
	iCntPrepBuf   = 0;
	iIndexDoneBuf = 0;

	for (int i = 0; i < NUM_SOUND_BUFFERS_OUT; i++) {
		if (m_waveOutHeader[i].dwFlags & WHDR_DONE) {
			iCntPrepBuf++;
			iIndexDoneBuf = i;
		}
	}
}

void CSoundCard::addOutBuffer(int n)
{
	/* Unprepare old wave-header */
	::waveOutUnprepareHeader(m_waveOut, m_waveOutHeader + n, sizeof(WAVEHDR));

	/* Prepare buffers for sending to sound interface */
	m_waveOutHeader[n].lpData         = (LPSTR)m_waveOutBuffer[n];
	m_waveOutHeader[n].dwBufferLength = m_txBufferSize * BYTES_PER_SAMPLE;
	m_waveOutHeader[n].dwFlags        = 0;

	/* Prepare wave-header */
	::waveOutPrepareHeader(m_waveOut, m_waveOutHeader + n, sizeof(WAVEHDR));

	/* Send buffer to driver for filling with new data */
	::waveOutWrite(m_waveOut, m_waveOutHeader + n, sizeof(WAVEHDR));
}

void CSoundCard::close()
{
	if (m_waveIn != NULL) {
		::waveInReset(m_waveIn);

		/* Set event to ensure that thread leaves the waiting function */
		::SetEvent(m_waveInEvent);

		/* Wait for the thread to terminate */
		::wxUsleep(250L);

		/* Unprepare wave-headers */
		for (int i = 0; i < NUM_SOUND_BUFFERS_IN; i++)
			::waveInUnprepareHeader(m_waveIn, &m_waveInHeader[i], sizeof(WAVEHDR));

		/* Close the sound in device */
		::waveInClose(m_waveIn);
		m_waveIn = NULL;
	}

	if (m_waveOut != NULL) {
		::waveOutReset(m_waveOut);

		/* Set event to ensure that thread leaves the waiting function */
		::SetEvent(m_waveOutEvent);

		/* Wait for the thread to terminate */
		::wxUsleep(250L);

		for (int i = 0; i < NUM_SOUND_BUFFERS_OUT; i++)
			::waveOutUnprepareHeader(m_waveOut, &m_waveOutHeader[i], sizeof(WAVEHDR));

		/* Close the sound out device */
		::waveOutClose(m_waveOut);
		m_waveOut = NULL;
	}
}

wxArrayString CSoundCard::getDevices()
{
	wxArrayString devices;

	int n = ::waveInGetNumDevs();

	devices.Alloc(n + 1);

	/* Get info about the devices and store the names */
	for (int i = 0; i < n; i++) {
		WAVEINCAPS waveInCaps;

		if (!::waveInGetDevCaps(i, &waveInCaps, sizeof(WAVEINCAPS)))
			devices.Add(waveInCaps.szPname);
	}

	devices.Add(wxT("Wave Mapper"));

	return devices;
}

#else

#include <sys/stat.h>
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>

#include <wx/debug.h>
#include <wx/log.h>

CSoundCard::CSoundCard(const wxString& device, int sampleRate, int rxBufferSize, int txBufferSize) :
m_device(device),
m_sampleRate(sampleRate),
m_rxBufferSize(rxBufferSize),
m_txBufferSize(txBufferSize),
m_fd(-1)
{
	wxASSERT(m_rxBufferSize > 0);
	wxASSERT(m_txBufferSize > 0);
	wxASSERT(m_sampleRate > 0);
}

CSoundCard::~CSoundCard()
{
}

void CSoundCard::openRead()
{
	wxASSERT(m_fd == -1);
	wxASSERT(m_sampleRate > 0);

	m_fd = ::open(m_device.mb_str(), O_RDONLY, 0);
	if (m_fd == -1) {
		wxString text;
		text.Printf(wxT("Error opening the sound card: %u %s"), errno, ::strerror(errno));
		throw CException(text);
	}

	unsigned int param, wanted;

	wanted = param = AFMT_S16_LE;
	if (::ioctl(m_fd, SNDCTL_DSP_SETFMT, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_SETFMT"));
	}

	if (param != AFMT_S16_LE) {
		close();
		throw CException(wxT("Error opening the sound card: Format not supported"));
	}

	wanted = param = 1;
	if (::ioctl(m_fd, SNDCTL_DSP_CHANNELS, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_CHANNELS"));
	}

	if (param != wanted) {
		close();
		throw CException(wxT("Error opening the sound card: Cannot set Mono"));
	}

	wanted = param = m_sampleRate;
	if (::ioctl(m_fd, SNDCTL_DSP_SPEED, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_SPEED"));
	}

	if (param != wanted) {
		close();
		wxString text;
		text.Printf(wxT("Error opening the sound card: Sample Rate, wanted %u returned %u"), wanted, param);
		throw CException(text);
	}

	wanted = param = 0x7FFF000A;
	if (::ioctl(m_fd, SNDCTL_DSP_SETFRAGMENT, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_SETFRAGMENT"));
	}

	if (param != wanted) {
		close();
		wxString text;
		text.Printf(wxT("Error opening the sound card: Fragment Size, wanted %u returned %u"), wanted, param);
		throw CException(text);
	}
}

void CSoundCard::openWrite()
{
	wxASSERT(m_fd == -1);
	wxASSERT(m_sampleRate > 0);

	m_fd = ::open(m_device.mb_str(), O_WRONLY, 0);
	if (m_fd == -1) {
		wxString text;
		text.Printf(wxT("Error opening the sound card: %s"), ::strerror(errno));
		throw CException(text);
	}

	unsigned int param, wanted;

	param  = wanted = AFMT_S16_LE;
	if (::ioctl(m_fd, SNDCTL_DSP_SETFMT, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_SETFMT"));
	}

	if (param != wanted) {
		close();
		throw CException(wxT("Error opening the sound card: Format not supported"));
	}

	wanted = param = 1;
	if (::ioctl(m_fd, SNDCTL_DSP_CHANNELS, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_CHANNELS"));
	}

	if (param != wanted) {
		close();
		throw CException(wxT("Error opening the sound card: Cannot set Mono"));
	}

	wanted = param = m_sampleRate;
	if (::ioctl(m_fd, SNDCTL_DSP_SPEED, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_SPEED"));
	}

	if (param != wanted) {
		close();
		throw CException(wxT("Error opening the sound card: Sample Rate"));
	}

	param = wanted = 0x0004000A;
	if (::ioctl(m_fd, SNDCTL_DSP_SETFRAGMENT, &param) < 0) {
		close();
		throw CException(wxT("Error opening the sound card: SNDCTL_DSP_SETFRAGMENT"));
	}

	if (param != wanted) {
		close();
		throw CException(wxT("Error opening the sound card: Fragment Size"));
	}
}

void CSoundCard::read(double* sample, int& len)
{
	wxASSERT(m_fd != -1);
	wxASSERT(len > 0);
	wxASSERT(sample != NULL);

	if (len < m_rxBufferSize)
		throw CException(wxT("Read buffer is too short"));

	wxInt16* data = new wxInt16[len];

	len = ::read(m_fd, data, len * sizeof(wxInt16));

	if (len <= 0) {
		delete[] data;
		return;
	}

	len /= sizeof(wxInt16);

	for (int i = 0; i < len; i++)
		sample[i] = double(data[i]) / 32767.0;

	delete[] data;
}

void CSoundCard::write(double* sample, int len)
{
	wxASSERT(m_fd != -1);
	wxASSERT(len > 0);
	wxASSERT(sample != NULL);

	if (len != m_txBufferSize)
		throw CException(wxT("Write buffer is not the correct length"));

	wxInt16* data = new wxInt16[len];

	for (int i = 0; i < len; i++)
		data[i] = (wxInt16)(sample[i] * 32767.0);

	int n = ::write(m_fd, data, len * sizeof(wxInt16));

	delete[] data;

	if (n <= 0)
		throw CException(wxT("Error writing to the sound card"));

	n /= sizeof(wxInt16);

	if (n != len)
		throw CException(wxT("Error writing to the sound card"));
}

void CSoundCard::close()
{
	wxASSERT(m_fd != -1);

	if (::ioctl(m_fd, SNDCTL_DSP_SYNC, 0) < 0)
		throw CException(wxT("Cannot set SNDCTL_DSP_SYNC"));

	::close(m_fd);
	m_fd = -1;
}

wxArrayString CSoundCard::getDevices()
{
	wxArrayString devices;

	devices.Alloc(5);

	devices.Add(wxT("/dev/dsp"));
	devices.Add(wxT("/dev/dsp0"));
	devices.Add(wxT("/dev/dsp1"));
	devices.Add(wxT("/dev/dsp2"));
	devices.Add(wxT("/dev/dsp3"));

	return devices;
}

#endif
