///////////////////////////////////////////////////////////////////////////////
//
// Wall Paper Changer for GNOME
// Copyright (C) 2000-2001  Kenichi Shibata <ken_s@cat.zero.ad.jp>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// Please read the file "COPYING" for more information.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////
//
// ken_s@cat.zero.ad.jp
// http://www3.might.co.jp/~sibata/top/
//
///////////////////////////////////////////////////////////////////////////////

#define __CALLBACKS_H
#include "global.h"

// static 
static GtkWidget *sFileselDialog;
static guint      sTimeoutHandler = 0;
static gint       sPageNum = 0;
static gboolean   sF_select = FALSE;
static GtkWidget *sAboutDialog = NULL;

/// Wall Paper Change Timer ///////////////////////////////////////////////////

////////////////////////////////////////////////////////
//                                                    //
//  Initiale Timer                                    //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
void initial_wpac_timer(void)
{
  guint32 time;
  
  if (sTimeoutHandler) {
    gtk_timeout_remove(sTimeoutHandler);
    sTimeoutHandler = 0;
  }
  if (wpac_get_changeMode() == TRUE) {
    time = (guint32)wpac_get_timeout();
    if (time < 1)
      time = 1;
    else if (time > 999)
      time = 999;
    time = time * 60 * 1000;
    sTimeoutHandler = gtk_timeout_add(time, (GtkFunction)timeout_wallpaper_change, NULL);
  }
}

////////////////////////////////////////////////////////
//                                                    //
//  Time out event                                    //
//                                                    //
////////////////////////////////////////////////////////
void timeout_wallpaper_change(gpointer data)
{
  wpac_change_wallpaper();
}

/// Main Dialog ///////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////
//                                                    //
//  Initiale Main Dialog                              //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
void main_dialog_init(void)
{
  GtkWidget *image_file_list, *radio, *entry;
  GList *glist;
  gchar *text[1], minute[1024];
  WALL_PAPER *data;
  gulong time;
  
  sPageNum = 0;

  wpac_init_editData();


  entry = get_mainDialogChild("minute_entry");

  if (wpac_get_changeMode() == TRUE) {
    enable_mainDialogChild("minute_entry", TRUE);
    gtk_entry_set_editable(GTK_ENTRY(entry), TRUE);
    radio = get_mainDialogChild("timeliness_auto_radio");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio), TRUE);
  } else {
    enable_mainDialogChild("minute_entry", FALSE);
    gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
    radio = get_mainDialogChild("timeliness_manual_radio");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio), TRUE);
  }
  
  if (wpac_get_random() == TRUE) {
    radio = get_mainDialogChild("mode_random_radiobutton");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio), TRUE);
  } else {
    radio = get_mainDialogChild("mode_sequential_radiobutton");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio), TRUE);
  }
  
  gtk_entry_set_max_length(GTK_ENTRY(entry), 3);
  time = wpac_get_timeout();
  if (time < 1) {
    time = 1;
    wpac_set_timeout(time);
  }

  sprintf(minute, "%ld", time);
  gtk_entry_set_text(GTK_ENTRY(entry), minute);

  image_file_list = get_mainDialogChild("image_file_clist");
  gtk_clist_clear(GTK_CLIST(image_file_list));
  gtk_clist_set_auto_sort(GTK_CLIST(image_file_list), TRUE);
  gtk_clist_set_column_auto_resize(GTK_CLIST(image_file_list), 0, TRUE);

  glist = wpac_get_wallPaper();
  while (glist) {
    data = (WALL_PAPER *)(glist->data);
    text[0] = data->fileName;
    gtk_clist_append(GTK_CLIST(image_file_list), text);
    glist = g_list_next(glist);
  }
  change_sensitive_SelectButton(GTK_CLIST(image_file_list));
  enable_mainDialogChild("save_button", FALSE);
}

////////////////////////////////////////////////////////
//                                                    //
//  Main Dialog "delete" event                        //
//                                                    //
////////////////////////////////////////////////////////
gboolean on_main_dialog_delete_event (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
  mainDialog_destroy();
  return (FALSE);
}

////////////////////////////////////////////////////////
//                                                    //
//  Change Wall Paper button "clicked" event          //
//                                                    //
////////////////////////////////////////////////////////
gboolean on_main_dialog_destroy_event (GtkWidget *widget, GdkEvent *event,gpointer user_data)
{
  return (FALSE);
}


//--- Wall Paper Files CList -------------------------------------

////////////////////////////////////////////////////////
//                                                    //
//  Wall Paper Files CList "select_row" event         //
//                                                    //
////////////////////////////////////////////////////////
void on_image_file_clist_select_row (GtkCList *clist, gint row, gint column, GdkEvent *event, gpointer user_data)
{
  if (sPageNum == 0)
    change_sensitive_deleteButton(clist);
  else {
    // Enable All Widget
    individual_page_widget_enable(TRUE);
    sF_select = TRUE;
    individual_page_data_load();
    sF_select = FALSE;
  }
  change_sensitive_SelectButton(clist);
}

////////////////////////////////////////////////////////
//                                                    //
//  Wall Paper Files CList "unselect_row" event       //
//                                                    //
////////////////////////////////////////////////////////
void on_image_file_clist_unselect_row (GtkCList *clist, gint row, gint column, GdkEvent *event, gpointer user_data)
{
  if (sPageNum == 0)
    change_sensitive_deleteButton(clist);
  else {
    individual_page_data_save(row);
    // Disable All Widget
    individual_page_widget_enable(FALSE);
  }
  enable_mainDialogChild("all_select_button", TRUE);
  change_sensitive_SelectButton(clist);
}

////////////////////////////////////////////////////////
//                                                    //
//  Wall Paper CList find filename                    //
//                                                    //
//    Parameter                                       //
//       find_filename : filename                     //
//    Return                                          //
//       gboolean : TRUE  = Found                     //
//                  FALSE = Not found                 //
//                                                    //
////////////////////////////////////////////////////////
static gboolean find_imagefile_clist(gchar *find_filename)
{
  GtkWidget *image_file_list;
  gint row;
  gchar *filename;

  image_file_list = get_mainDialogChild("image_file_clist");
  for (row=0; gtk_clist_get_text(GTK_CLIST(image_file_list), row, 0, &filename); row++) {
    if (strcmp(find_filename, filename) == 0)
      return (TRUE);
  }
  return (FALSE);
}

//--- Notebook "Common" page -------------------------------------

////////////////////////////////////////////////////////
//                                                    //
//  Sequential radio button "toggled" event           //
//                                                    //
////////////////////////////////////////////////////////
void on_mode_sequential_radiobutton_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
  check_notebook_data();
}


////////////////////////////////////////////////////////
//                                                    //
//  Random radio button "toggled" event               //
//                                                    //
////////////////////////////////////////////////////////
void on_mode_random_radiobutton_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  Timeliness manual radio "toggled" event           //
//                                                    //
////////////////////////////////////////////////////////
void on_timeliness_manual_radio_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  Timeliness auto radio "toggled" event             //
//                                                    //
////////////////////////////////////////////////////////
void on_timeliness_auto_radio_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
  GtkWidget *entry;
  
  entry = get_mainDialogChild("minute_entry");
  if (GTK_TOGGLE_BUTTON(togglebutton)->active) {
    // Auto ON
    enable_mainDialogChild("minute_entry", TRUE);
    gtk_entry_set_editable(GTK_ENTRY(entry), TRUE);
  } else {
    // Auto OFF
    enable_mainDialogChild("minute_entry", FALSE);
    gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
  }
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  Minete entry "insert_text" event                  //
//                                                    //
////////////////////////////////////////////////////////
void on_minute_entry_insert_text (GtkEditable *editable, gchar *new_text, gint new_text_length, gint *position, gpointer user_data)
{
  gchar buff[256];


  if (new_text_length > 255)
    new_text_length = 255;
  
  strncpy(buff, new_text, new_text_length);
  buff[new_text_length] = (gchar)NULL;

  if (!isDigitStr(buff)) {
    *new_text = (gchar)NULL;
    return;
  }
  toDigitStr(buff);
  strcpy(new_text, buff);
}

////////////////////////////////////////////////////////
//                                                    //
//  Minute entry "change" event                       //
//                                                    //
////////////////////////////////////////////////////////
void on_minute_entry_changed (GtkEditable *editable, gpointer user_data)
{
  check_notebook_data();

}

////////////////////////////////////////////////////////
//                                                    //
//  Wall Paper Addition button "clicked" event        //
//                                                    //
////////////////////////////////////////////////////////
void on_add_button_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *file_list, *selection_entry, *selection_text;

  sFileselDialog = create_fileselection();
  file_list = GTK_FILE_SELECTION(sFileselDialog)->file_list;
  gtk_clist_set_selection_mode (GTK_CLIST(file_list), GTK_SELECTION_EXTENDED);

  gtk_signal_handlers_destroy(GTK_OBJECT (file_list));

  selection_entry = GTK_FILE_SELECTION(sFileselDialog)->selection_entry;
  gtk_widget_hide(selection_entry);
  selection_text = GTK_FILE_SELECTION(sFileselDialog)->selection_text;
  gtk_widget_hide(selection_text);
  gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(sFileselDialog));
  gtk_grab_add(sFileselDialog);

  gtk_widget_show(sFileselDialog);
}

////////////////////////////////////////////////////////
//                                                    //
//  Wall Paper Deletes button "clicked" event         //
//                                                    //
////////////////////////////////////////////////////////
void on_del_button_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *image_file_list;
  GList *selection;
  gint  row;
  gchar *filename;
  
  image_file_list = get_mainDialogChild("image_file_clist");
  while ((selection = GTK_CLIST(image_file_list)->selection) != (GList *)NULL) {
    row = GPOINTER_TO_INT(selection->data);
    gtk_clist_get_text(GTK_CLIST(image_file_list), row, 0, &filename);
    wpac_del_image(filename);
    gtk_clist_remove(GTK_CLIST(image_file_list), row);
  }
  enable_mainDialogChild("save_button", TRUE);
}

////////////////////////////////////////////////////////
//                                                    //
//  Wall Paper select button "clicked" event          //
//                                                    //
////////////////////////////////////////////////////////
void on_all_select_button_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *widget;

  widget = get_mainDialogChild("image_file_clist");
  gtk_clist_select_all(GTK_CLIST(widget));
}

////////////////////////////////////////////////////////
//                                                    //
//  Wall Paper unselect button "clicked" event        //
//                                                    //
////////////////////////////////////////////////////////
void on_all_unselect_button_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *widget;

  widget = get_mainDialogChild("image_file_clist");
  gtk_clist_unselect_all(GTK_CLIST(widget));
}

////////////////////////////////////////////////////////
//                                                    //
//  Check Deletes button Sensitive                    //
//                                                    //
//    Parameter                                       //
//       image_file_list : Wall Paper files CList     //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
static void change_sensitive_deleteButton(GtkCList *image_file_list)
{
  if (image_file_list->selection) {
    enable_mainDialogChild("del_button", TRUE);
  } else {
    enable_mainDialogChild("del_button", FALSE);
  }
}

////////////////////////////////////////////////////////
//                                                    //
//  Notebook common page data save                    //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
static void common_page_data_save(void)
{
  GtkWidget *image_file_list, *radio;
  gchar *minute;

  image_file_list = get_mainDialogChild("image_file_clist");

  minute = gtk_entry_get_text(GTK_ENTRY(get_mainDialogChild("minute_entry")));

  wpac_set_timeout(d_to_gulong(minute));

  radio = get_mainDialogChild("timeliness_auto_radio");
  if (GTK_TOGGLE_BUTTON(radio)->active)
    wpac_set_changeMode(TRUE);
  else
    wpac_set_changeMode(FALSE);

  radio = get_mainDialogChild("mode_random_radiobutton");
  if (GTK_TOGGLE_BUTTON(radio)->active)
    wpac_set_random(TRUE);
  else
    wpac_set_random(FALSE);
}

////////////////////////////////////////////////////////
//                                                    //
//  Notebook common page data compare                 //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       gboolean : TRUE  = changed                   //
//                  FALSE = not changed               //
//                                                    //
////////////////////////////////////////////////////////
static gboolean common_page_data_comp(void)
{
  GtkWidget *radio;
  gchar *minute;
  gboolean change;

  change = FALSE;
  minute = gtk_entry_get_text(GTK_ENTRY(get_mainDialogChild("minute_entry")));

  if (d_to_gulong(minute) != wpac_get_timeout())
    change = TRUE;

  radio = get_mainDialogChild("timeliness_auto_radio");
  if ((GTK_TOGGLE_BUTTON(radio)->active) != wpac_get_changeMode())
    change = TRUE;

  radio = get_mainDialogChild("mode_random_radiobutton");
  if ((GTK_TOGGLE_BUTTON(radio)->active) != wpac_get_random())
    change = TRUE;

  return (change);
}

////////////////////////////////////////////////////////
//                                                    //
//  Change Select button sensitive                    //
//                                                    //
//    Parameter                                       //
//       clist : Wall Paper files CList               //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
static void change_sensitive_SelectButton(GtkCList *clist)
{
  GList *selection;
  gint  selrow, allrow;
  gchar *text;

  selection = clist->selection;
  for (selrow=0; selection; selrow++)
    selection = g_list_next(selection);
  for (allrow=0; gtk_clist_get_text(clist, allrow, 0, &text); allrow++);

  if (selrow == allrow)
    enable_mainDialogChild("all_select_button", FALSE);
  else
    enable_mainDialogChild("all_select_button", TRUE);
  if (clist->selection == NULL)
    enable_mainDialogChild("all_unselect_button", FALSE);
  else
    enable_mainDialogChild("all_unselect_button", TRUE);
}



//--- Notebook "Individual" page ---------------------------------

////////////////////////////////////////////////////////
//                                                    //
//  Layout Center radio button "toggled" event        //
//                                                    //
////////////////////////////////////////////////////////
void on_layout_center_radiobutton_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  Layout fullscreen radio button "toggled" event    //
//                                                    //
////////////////////////////////////////////////////////
void on_layout_full_radiobutton_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
  if (GTK_TOGGLE_BUTTON(togglebutton)->active) {
    enable_mainDialogChild("xy_checkbutton", FALSE);
    enable_mainDialogChild("xzoom_hscale", FALSE);
    enable_mainDialogChild("yzoom_hscale", FALSE);
    enable_mainDialogChild("xzoom_entry", FALSE);
    enable_mainDialogChild("yzoom_entry", FALSE);
    enable_mainDialogChild("zoom_label", FALSE);
    enable_mainDialogChild("xzoom_label", FALSE);
    enable_mainDialogChild("yzoom_label", FALSE);

  } else {
    enable_mainDialogChild("xy_checkbutton", TRUE);
    enable_mainDialogChild("xzoom_hscale", TRUE);
    enable_mainDialogChild("yzoom_hscale", TRUE);
    enable_mainDialogChild("xzoom_entry", TRUE);
    enable_mainDialogChild("yzoom_entry", TRUE);
    enable_mainDialogChild("zoom_label", TRUE);
    enable_mainDialogChild("xzoom_label", TRUE);
    enable_mainDialogChild("yzoom_label", TRUE);
  }
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  Layout Tile radio button "toggled" event          //
//                                                    //
////////////////////////////////////////////////////////
void on_layout_tile_radiobutton_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
  if (GTK_TOGGLE_BUTTON(togglebutton)->active) {
    enable_mainDialogChild("bg_colorselection", FALSE);
    enable_mainDialogChild("bgcolor_label", FALSE);
  } else {
    enable_mainDialogChild("bg_colorselection", TRUE);
    enable_mainDialogChild("bgcolor_label", TRUE);
  }
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  Dither check button "toggled" event               //
//                                                    //
////////////////////////////////////////////////////////
void on_dither_checkbutton_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  Gray check button "toggled" event                 //
//                                                    //
////////////////////////////////////////////////////////
void on_gray_checkbutton_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  Rotate hscale "value_changed" event               //
//                                                    //
////////////////////////////////////////////////////////
void on_rotate_hscale_value_changed (GtkWidget *widget, gpointer user_data)
{
  guint  value;
  gchar  buff[256];

  value = get_value_individual_page_scale("rotate_hscale") * 90;
  sprintf(buff, "%d", value);
  set_text_individual_page_entry("rotate_entry", buff);
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  Gamma hscale "value_changed" event                //
//                                                    //
////////////////////////////////////////////////////////
void on_gamma_hscale_value_changed (GtkWidget *widget, gpointer user_data)
{
  guint  value;
  gchar  buff[256];

  value = get_value_individual_page_scale("gamma_hscale");
  sprintf(buff, "%d.%01d", value / 10, value % 10);
  set_text_individual_page_entry("gamma_entry", buff);
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  Brighten hscale "value_changed" event             //
//                                                    //
////////////////////////////////////////////////////////
void on_brighten_hscale_value_changed (GtkWidget *widget, gpointer user_data)
{
  guint  value;
  gchar  buff[256];

  value = get_value_individual_page_scale("brighten_hscale");
  sprintf(buff, "%d%%", value);
  set_text_individual_page_entry("brighten_entry", buff);
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  X equale Y check button "toggled" event           //
//                                                    //
////////////////////////////////////////////////////////
void on_xy_checkbutton_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
  GtkWidget *xhscale, *yhscale;
  GtkAdjustment *adjustment;

  if (GTK_TOGGLE_BUTTON(togglebutton)->active) {
    xhscale = get_mainDialogChild("xzoom_hscale");
    yhscale = get_mainDialogChild("yzoom_hscale");
    adjustment = gtk_range_get_adjustment(GTK_RANGE(xhscale));
    gtk_range_set_adjustment(GTK_RANGE(yhscale), adjustment);
    gtk_range_slider_update (GTK_RANGE(yhscale));
    on_yzoom_hscale_value_changed(NULL, NULL);
  } else {
    yhscale = get_mainDialogChild("yzoom_hscale");
    adjustment = GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 201, 1, 1, 1));
    gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed", GTK_SIGNAL_FUNC(on_yzoom_hscale_value_changed), NULL);
    gtk_range_set_adjustment(GTK_RANGE(yhscale), adjustment);
  }
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  X Zoom hscale "value_changed" event               //
//                                                    //
////////////////////////////////////////////////////////
void on_xzoom_hscale_value_changed (GtkWidget *widget, gpointer user_data)
{
  guint  value;
  gchar  buff[256];
  GtkWidget *xyCheckButton;

  value = get_value_individual_page_scale("xzoom_hscale");
  sprintf(buff, "%d%%", value);
  set_text_individual_page_entry("xzoom_entry", buff);
  
  xyCheckButton = get_mainDialogChild("xy_checkbutton");
  if (GTK_TOGGLE_BUTTON(xyCheckButton)->active) {
    on_yzoom_hscale_value_changed(widget, user_data);
  }
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  Y Zoom hscale "value_changed" event               //
//                                                    //
////////////////////////////////////////////////////////
void on_yzoom_hscale_value_changed (GtkWidget *widget, gpointer user_data)
{
  guint  value;
  gchar  buff[256];

  value = get_value_individual_page_scale("yzoom_hscale");
  sprintf(buff, "%d%%", value);
  set_text_individual_page_entry("yzoom_entry", buff);
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  Background color selection "color_changed" event  //
//                                                    //
////////////////////////////////////////////////////////
void on_bg_colorselection_color_changed (GtkColorSelection *colorselection, gpointer user_data)
{
  check_notebook_data();
}

////////////////////////////////////////////////////////
//                                                    //
//  Option entry "change" event                       //
//                                                    //
////////////////////////////////////////////////////////
void on_option_entry_changed (GtkEditable *editable, gpointer user_data)
{
  check_notebook_data();

}

////////////////////////////////////////////////////////
//                                                    //
//  Test button "clicked" event                       //
//                                                    //
////////////////////////////////////////////////////////
void on_test_button_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *widget;
  GList *selection;
  gchar  *filename;
  gint row;
  WALL_PAPER *wallPaper;

  widget = get_mainDialogChild("image_file_clist");
  selection = GTK_CLIST(widget)->selection;
  if (selection) {
    row = GPOINTER_TO_INT(selection->data);
    individual_page_data_save(row);
    gtk_clist_get_text(GTK_CLIST(widget), row, 0, &filename);
    wallPaper = wpac_find_wallPaper(filename);
    wpac_xloadimage(wallPaper);
  }
}

////////////////////////////////////////////////////////
//                                                    //
//  Notebook individual page data load                //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
static void individual_page_data_load(void)
{
  GtkWidget *hscale, *widget, *image_file_list;
  GtkAdjustment *adjustment;
  GList *selection;
  gint  row;
  gchar *filename, *buffp[1];
  WALL_PAPER *wallpaper;

  hscale = get_mainDialogChild("rotate_hscale");
  adjustment = gtk_range_get_adjustment(GTK_RANGE(hscale));
  gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed", GTK_SIGNAL_FUNC(on_rotate_hscale_value_changed), NULL);

  hscale = get_mainDialogChild("xzoom_hscale");
  adjustment = gtk_range_get_adjustment(GTK_RANGE(hscale));
  gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed", GTK_SIGNAL_FUNC(on_xzoom_hscale_value_changed), NULL);

  hscale = get_mainDialogChild("yzoom_hscale");
  adjustment = gtk_range_get_adjustment(GTK_RANGE(hscale));
  gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed", GTK_SIGNAL_FUNC(on_yzoom_hscale_value_changed), NULL);

  hscale = get_mainDialogChild("gamma_hscale");
  adjustment = gtk_range_get_adjustment(GTK_RANGE(hscale));
  gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed", GTK_SIGNAL_FUNC(on_gamma_hscale_value_changed), NULL);

  hscale = get_mainDialogChild("brighten_hscale");
  adjustment = gtk_range_get_adjustment(GTK_RANGE(hscale));
  gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed", GTK_SIGNAL_FUNC(on_brighten_hscale_value_changed), NULL);


  image_file_list = get_mainDialogChild("image_file_clist");
  selection = GTK_CLIST(image_file_list)->selection;
  if (!selection) {
    return;
  }
  row = GPOINTER_TO_INT(selection->data);
  gtk_clist_get_text(GTK_CLIST(image_file_list), row, 0, &filename);
  buffp[0] = filename;
  wallpaper = wpac_find_wallPaper(filename);
  // Layout
  switch (wallpaper->layout) {
  case LAYOUT_CENTER:
    widget = get_mainDialogChild("layout_center_radiobutton");
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    break;
  case LAYOUT_FULLSCREEN:
    widget = get_mainDialogChild("layout_full_radiobutton");
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    break;
  case LAYOUT_TILE:
  default:
    widget = get_mainDialogChild("layout_tile_radiobutton");
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    break;
  }
  // Dither
  widget = get_mainDialogChild("dither_checkbutton");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), wallpaper->dither);
  // Gray
  widget = get_mainDialogChild("gray_checkbutton");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), wallpaper->gray);
  // X == Y
  widget = get_mainDialogChild("xy_checkbutton");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), wallpaper->zoom_x_eq_y);
  // X zoom
  widget = get_mainDialogChild("xzoom_hscale");
  adjustment = gtk_range_get_adjustment (GTK_RANGE(widget));
  gtk_adjustment_set_value(adjustment, (gfloat)(wallpaper->zoom_x));
  gtk_range_slider_update (GTK_RANGE(widget));
  // Y zoom
  widget = get_mainDialogChild("yzoom_hscale");
  adjustment = gtk_range_get_adjustment (GTK_RANGE(widget));
  gtk_adjustment_set_value(adjustment, (gfloat)(wallpaper->zoom_y));
  gtk_range_slider_update (GTK_RANGE(widget));
  // Rotate
  widget = get_mainDialogChild("rotate_hscale");
  adjustment = gtk_range_get_adjustment (GTK_RANGE(widget));
  gtk_adjustment_set_value(adjustment, (gfloat)(wallpaper->rotate / 90));
  gtk_range_slider_update (GTK_RANGE(widget));
  // Gamma
  widget = get_mainDialogChild("gamma_hscale");
  adjustment = gtk_range_get_adjustment (GTK_RANGE(widget));
  gtk_adjustment_set_value(adjustment, (gfloat)(wallpaper->gamma));
  gtk_range_slider_update (GTK_RANGE(widget));
  // Brighten
  widget = get_mainDialogChild("brighten_hscale");
  adjustment = gtk_range_get_adjustment (GTK_RANGE(widget));
  gtk_adjustment_set_value(adjustment, (gfloat)(wallpaper->brighten));
  gtk_range_slider_update (GTK_RANGE(widget));
  // Background color
  widget = get_mainDialogChild("bg_colorselection");
  gtk_color_selection_set_color(GTK_COLOR_SELECTION(widget), wallpaper->bg_color);
  // Option
  widget = get_mainDialogChild("option_entry");
  if (wallpaper->option)
    gtk_entry_set_text(GTK_ENTRY(widget), wallpaper->option);
  else
    gtk_entry_set_text(GTK_ENTRY(widget), "");

  on_rotate_hscale_value_changed(NULL, (gpointer)NULL);
  on_xzoom_hscale_value_changed(NULL, (gpointer)NULL);
  on_yzoom_hscale_value_changed(NULL, (gpointer)NULL);
  on_gamma_hscale_value_changed(NULL, (gpointer)NULL);
  on_gamma_hscale_value_changed(NULL, (gpointer)NULL);
  on_brighten_hscale_value_changed(NULL, (gpointer)NULL);
  widget = get_mainDialogChild("layout_center_radiobutton");
  on_layout_center_radiobutton_toggled(GTK_TOGGLE_BUTTON(widget), NULL);
  widget = get_mainDialogChild("layout_full_radiobutton");
  on_layout_full_radiobutton_toggled(GTK_TOGGLE_BUTTON(widget), NULL);

}

////////////////////////////////////////////////////////
//                                                    //
//  Notebook individual page data save                //
//                                                    //
//    Parameter                                       //
//       row : Wall Paper files CList row             //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
static void individual_page_data_save(gint row)
{
  GtkWidget *clist, *widget;
  guint value;
  WALL_PAPER data;

  // Layout
  if (GTK_TOGGLE_BUTTON(get_mainDialogChild("layout_center_radiobutton"))->active) {
    data.layout = LAYOUT_CENTER;
  } else if (GTK_TOGGLE_BUTTON(get_mainDialogChild("layout_full_radiobutton"))->active) {
    data.layout = LAYOUT_FULLSCREEN;
  } else {
    data.layout = LAYOUT_TILE;
  }

  // Dither
  if (GTK_TOGGLE_BUTTON(get_mainDialogChild("dither_checkbutton"))->active) {
    data.dither = TRUE;
  } else {
    data.dither = FALSE;
  }

  // Gray
  if (GTK_TOGGLE_BUTTON(get_mainDialogChild("gray_checkbutton"))->active) {
    data.gray = TRUE;
  } else {
    data.gray = FALSE;
  }

  // Zoom X = Y
  if (GTK_TOGGLE_BUTTON(get_mainDialogChild("xy_checkbutton"))->active) {
    data.zoom_x_eq_y = TRUE;
  } else {
    data.zoom_x_eq_y = FALSE;
  }

  // Zoom X
  value = get_value_individual_page_scale("xzoom_hscale");
  data.zoom_x = (guchar)value;

  // Zoom Y
  value = get_value_individual_page_scale("yzoom_hscale");
  data.zoom_y = (guchar)value;

  // Rotate
  value = get_value_individual_page_scale("rotate_hscale");
  switch (value) {
  case 0:
    data.rotate = 0;
    break;
  case 1:
    data.rotate = 90;
    break;
  case 2:
    data.rotate = 180;
    break;
  case 3:
  default:
    data.rotate = 270;
    break;
  }

  // Gamma
  value = get_value_individual_page_scale("gamma_hscale");
  data.gamma = (guchar)value;

  // Brighten
  value = get_value_individual_page_scale("brighten_hscale");
  data.brighten = (gushort)value;

  // Color
  widget = get_mainDialogChild("bg_colorselection");
  gtk_color_selection_get_color(GTK_COLOR_SELECTION(widget), data.bg_color);

  // Option
  widget = get_mainDialogChild("option_entry");
  data.option = gtk_entry_get_text(GTK_ENTRY(widget));
  if (strlen(data.option) <= 0)
    data.option = (gchar *)NULL;

  clist = get_mainDialogChild("image_file_clist");
  gtk_clist_get_text(GTK_CLIST(clist), row, 0, &(data.fileName));
  wpac_add_image(&data);
}

////////////////////////////////////////////////////////
//                                                    //
//  Notebook individual page data compare             //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       gboolean : TRUE  = changed                   //
//                  FALSE = not changed               //
//                                                    //
////////////////////////////////////////////////////////
static gboolean individual_page_data_comp(void)
{
  GList *selection;
  GtkWidget *widget;
  guint value;
  WALL_PAPER *data;
  gint row, i;
  gboolean change;
  gchar *filename, *option;
  gdouble color[3];
  
  if (sF_select == TRUE)
    return (FALSE);

  change = FALSE;
  widget = get_mainDialogChild("image_file_clist");
  selection = GTK_CLIST(widget)->selection;
  if (selection) {
    row = GPOINTER_TO_INT(selection->data);
    gtk_clist_get_text(GTK_CLIST(widget), row, 0, &filename);
    data = wpac_find_wallPaper(filename);

    // Layout
    if (GTK_TOGGLE_BUTTON(get_mainDialogChild("layout_center_radiobutton"))->active) {
      if (data->layout != LAYOUT_CENTER)
	change = TRUE;
    } else if (GTK_TOGGLE_BUTTON(get_mainDialogChild("layout_full_radiobutton"))->active) {
      if (data->layout != LAYOUT_FULLSCREEN)
	change = TRUE;
    } else {
      if (data->layout != LAYOUT_TILE)
	change = TRUE;
    }
    
    // Dither
    if (GTK_TOGGLE_BUTTON(get_mainDialogChild("dither_checkbutton"))->active) {
      if (data->dither != TRUE)
	change = TRUE;
    } else {
      if (data->dither != FALSE)
	change = TRUE;
    }
    
    // Gray
    if (GTK_TOGGLE_BUTTON(get_mainDialogChild("gray_checkbutton"))->active) {
      if (data->gray != TRUE)
	change = TRUE;
    } else {
      if (data->gray != FALSE)
	change = TRUE;
    }
    
    // Zoom X = Y
    if (GTK_TOGGLE_BUTTON(get_mainDialogChild("xy_checkbutton"))->active) {
      if (data->zoom_x_eq_y != TRUE)
	change = TRUE;
    } else {
      if (data->zoom_x_eq_y != FALSE)
	change = TRUE;
    }
    
    // Zoom X
    value = get_value_individual_page_scale("xzoom_hscale");
    if (data->zoom_x != (guchar)value)
      change = TRUE;
    
    // Zoom Y
    value = get_value_individual_page_scale("yzoom_hscale");
    if (data->zoom_y != (guchar)value)
      change = TRUE;
    
    // Rotate
    value = get_value_individual_page_scale("rotate_hscale");
    switch (value) {
    case 0:
      if (data->rotate != 0)
	change = TRUE;
      break;
    case 1:
      if (data->rotate != 90)
	change = TRUE;
      break;
    case 2:
      if (data->rotate != 180)
	change = TRUE;
      break;
    case 3:
    default:
      if (data->rotate != 270)
	change = TRUE;
      break;
    }
    
    // Gamma
    value = get_value_individual_page_scale("gamma_hscale");
    if (data->gamma != (guchar)value)
      change = TRUE;
    
    // Brighten
    value = get_value_individual_page_scale("brighten_hscale");
    if (data->brighten != (gushort)value)
      change = TRUE;
    
    // Color
    widget = get_mainDialogChild("bg_colorselection");
    gtk_color_selection_get_color(GTK_COLOR_SELECTION(widget), color);
    for (i=0; i<3; i++) {
      if (data->bg_color[i] != color[i])
	change = TRUE;
    }

    // Option
    if (data->option) {
      widget = get_mainDialogChild("option_entry");
      if (strcmp(data->option, gtk_entry_get_text(GTK_ENTRY(widget))) != 0)
	change = TRUE;
    } else {
      widget = get_mainDialogChild("option_entry");
      option = gtk_entry_get_text(GTK_ENTRY(widget));
      if (option)
	change = TRUE;
    }
  }
  return (change);
}

////////////////////////////////////////////////////////
//                                                    //
//  Notebook individual page widget enable            //
//                                                    //
//    Parameter                                       //
//       enable : TRUE  = all widget enable           //
//                FALSE = all widget disable          //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
static void individual_page_widget_enable(gboolean enable)
{
  static gchar widgetName[27][30] = {
    "layout_label",            "layout_center_radiobutton", "layout_full_radiobutton", 
    "layout_tile_radiobutton", "effect_label",              "dither_checkbutton", 
    "gray_checkbutton",        "rotate_label",              "rotate_hscale",
    "rotate_entry",            "gamma_label",               "gamma_hscale",
    "gamma_entry",             "brighten_label",            "brighten_hscale",
    "brighten_entry",          "zoom_label",                "xy_checkbutton",
    "xzoom_label",             "xzoom_hscale",              "xzoom_entry", 
    "yzoom_label",             "yzoom_hscale",              "yzoom_entry", 
    "bgcolor_label",           "bg_colorselection",         "test_button"
  };
  gint i;

  for (i=0; i<27; i++)
    enable_mainDialogChild(widgetName[i], enable);
}

////////////////////////////////////////////////////////
//                                                    //
//  Get Notebook individual page scale VALUE          //
//                                                    //
//    Parameter                                       //
//       name : scale widget name                     //
//    Return                                          //
//       guint : value                                //
//                                                    //
////////////////////////////////////////////////////////
static guint get_value_individual_page_scale(gchar *name)
{
  GtkWidget *hscale;
  GtkAdjustment *adjustment;

  hscale = get_mainDialogChild(name);
  adjustment = gtk_range_get_adjustment(GTK_RANGE(hscale));
  return ((guint)adjustment->value);
}

////////////////////////////////////////////////////////
//                                                    //
//  Set Notebook individual page entry TEXT           //
//                                                    //
//    Parameter                                       //
//       clist : Wall Paper files CList               //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
static void set_text_individual_page_entry(gchar *name, gchar *str)
{
  GtkWidget *entry;

  entry = get_mainDialogChild(name);
  gtk_entry_set_text(GTK_ENTRY(entry), str);
}

//--- Common -----------------------------------------------------

////////////////////////////////////////////////////////
//                                                    //
//  Notebook "switch_page" event                      //
//                                                    //
////////////////////////////////////////////////////////
void on_wpac_notebook_switch_page (GtkNotebook *notebook, GtkNotebookPage *page, gint page_num, gpointer user_data)
{
  GtkWidget *widget;
  GList *selection;
  gint row;

  switch (sPageNum) {
  case 0:
    break;
  case 1:
    widget = get_mainDialogChild("image_file_clist");
    selection = GTK_CLIST(widget)->selection;
    if (selection) {
      row = GPOINTER_TO_INT(selection->data);
      individual_page_data_save(row);
    }
    break;
  default:
    break;
  }
  switch (page_num) {
  case 0:
    widget = get_mainDialogChild("image_file_clist");
    gtk_clist_set_selection_mode (GTK_CLIST(widget), GTK_SELECTION_EXTENDED);
    break;
  case 1:
    widget = get_mainDialogChild("image_file_clist");
    gtk_clist_set_selection_mode (GTK_CLIST(widget), GTK_SELECTION_SINGLE);
    individual_page_widget_enable(FALSE);
    individual_page_data_load();
    break;
  default:
    break;
  }
  sPageNum = page_num;
}

////////////////////////////////////////////////////////
//                                                    //
//  Save button "clicked" event                       //
//                                                    //
////////////////////////////////////////////////////////
void on_save_button_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *widget;
  GList *selection;
  gint row;

  switch (sPageNum) {
  case 0:
    common_page_data_save();
    break;
  case 1:
    widget = get_mainDialogChild("image_file_clist");
    selection = GTK_CLIST(widget)->selection;
    if (selection) {
      row = GPOINTER_TO_INT(selection->data);
      individual_page_data_save(row);
    }
    break;
  default:
    break;
  }

  wpac_save_file((gboolean)user_data);

  initial_wpac_timer();
  enable_mainDialogChild("save_button", FALSE);
}

////////////////////////////////////////////////////////
//                                                    //
//  OK button "clicked" event                         //
//                                                    //
////////////////////////////////////////////////////////
void on_ok_button_clicked (GtkButton *button, gpointer user_data)
{
  on_save_button_clicked(button, user_data);
  mainDialog_destroy();
}

////////////////////////////////////////////////////////
//                                                    //
//  Cancel button "clicked" event                     //
//                                                    //
////////////////////////////////////////////////////////
void on_cancel_button_clicked (GtkButton *button, gpointer user_data)
{
  mainDialog_destroy();
}

////////////////////////////////////////////////////////
//                                                    //
//  About  button "clicked" event                     //
//                                                    //
////////////////////////////////////////////////////////
void on_about_button_clicked (GtkButton *button, gpointer user_data)
{
  if (sAboutDialog == NULL) {
    sAboutDialog = create_about_dialog();
    gtk_grab_add(sAboutDialog);
    gtk_widget_show(sAboutDialog);
  }
}

////////////////////////////////////////////////////////
//                                                    //
//  Exit button "clicked" event                       //
//                                                    //
////////////////////////////////////////////////////////
void on_exit_button_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *save_dialog, *save_button;

  save_button = get_mainDialogChild("save_button");
  if (GTK_WIDGET_SENSITIVE(save_button)) {
    save_dialog = create_save_dialog();
    gtk_grab_add(save_dialog);
    gtk_widget_show(save_dialog);
  } else {
    applet_widget_gtk_main_quit();
  }
}

////////////////////////////////////////////////////////
//                                                    //
//  Check Notebook data changed                       //
//                                                    //
//    Parameter                                       //
//       void                                         //
//    Return                                          //
//       void                                         //
//                                                    //
////////////////////////////////////////////////////////
static void check_notebook_data(void)
{
  gboolean change;

  change = FALSE;
  switch (sPageNum) {
  case 0:
    change = common_page_data_comp();
    break;
  case 1:
    change = individual_page_data_comp();
    break;
  default:
    break;
  }
  
  if (change == TRUE) {
    enable_mainDialogChild("save_button", TRUE);
  }
}


/// File Selection ////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////
//                                                    //
//  File selection dialog "delete" event              //
//                                                    //
////////////////////////////////////////////////////////
gboolean on_fileselection_delete_event (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
  return (FALSE);
}

////////////////////////////////////////////////////////
//                                                    //
//  File selection dialog "destroy" event             //
//                                                    //
////////////////////////////////////////////////////////
gboolean on_fileselection_destroy_event (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
  return (FALSE);
}

////////////////////////////////////////////////////////
//                                                    //
//  File selection dialog OK button "clicked" event   //
//                                                    //
////////////////////////////////////////////////////////
void on_fileselect_ok_button_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *file_list, *image_file_list;
  gchar *filename, *pathp, path[1024], *buffp[1], buff[1024];
  GList *selection, *file_glist;
  gint  row, i;
  WALL_PAPER data;
  gboolean change;

  change = FALSE;
  file_list = GTK_FILE_SELECTION(sFileselDialog)->file_list;
  selection = GTK_CLIST(file_list)->selection;
  pathp = gtk_file_selection_get_filename(GTK_FILE_SELECTION(sFileselDialog));
  strcpy(path, pathp);
  for (i=strlen(path)-1; i > 0; i--) {
    if (path[i] == '/') {
      path[i+1] = (gchar)NULL;
      break;
    }
  }

  file_glist = NULL;
  image_file_list = get_mainDialogChild("image_file_clist");
  while (selection) {
    row = GPOINTER_TO_INT(selection->data);
    gtk_clist_get_text (GTK_CLIST(file_list), row, 0, &filename);
    strcpy(buff, path);
    strcat(buff, filename);
    if (find_imagefile_clist(buff) == FALSE) {
      buffp[0] = buff;
      gtk_clist_append(GTK_CLIST(image_file_list), buffp);
      data.fileName = buff;
      data.layout = LAYOUT_TILE;
      data.dither = FALSE;
      data.gray = FALSE;
      data.zoom_x_eq_y = TRUE;
      data.zoom_x = 100;
      data.zoom_y = 100;
      data.rotate = 0;
      data.gamma = 10;
      data.brighten = 100;
      data.bg_color[0] = 1.0;
      data.bg_color[1] = 1.0 ;
      data.bg_color[2] = 1,0;
      wpac_add_image(&data);
      change = TRUE;
    }
    selection = g_list_next(selection);
  }
  gtk_clist_set_column_auto_resize(GTK_CLIST(image_file_list), 0, TRUE);
  gtk_widget_destroy(sFileselDialog);
  if (change == TRUE)
    enable_mainDialogChild("save_button", TRUE);
  change_sensitive_SelectButton(GTK_CLIST(image_file_list));

}

////////////////////////////////////////////////////////
//                                                    //
//  File selection dialog                             //
//                    Cancel button "clicked" event   //
//                                                    //
////////////////////////////////////////////////////////
void on_fileselect_cancel_button_clicked (GtkButton *button, gpointer user_data)
{
  gtk_widget_destroy(sFileselDialog);
}

/// About Dialog //////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////
//                                                    //
//  About Dialog "delete" event                       //
//                                                    //
////////////////////////////////////////////////////////
gboolean on_about_dialog_delete_event (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
  gtk_widget_destroy(sAboutDialog);
  sAboutDialog = NULL;
  return (FALSE);
}

////////////////////////////////////////////////////////
//                                                    //
//  About Dialog "destroy" event                      //
//                                                    //
////////////////////////////////////////////////////////
gboolean on_about_dialog_destroy (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
  return (FALSE);
}

////////////////////////////////////////////////////////
//                                                    //
//  About Dialog OK button "clicked" event            //
//                                                    //
////////////////////////////////////////////////////////
void on_about_ok_button_clicked (GtkButton *button, gpointer user_data)
{
  gtk_widget_destroy(sAboutDialog);
  sAboutDialog = NULL;
}

/// Applet ////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////
//                                                    //
//  Applet popup menu Property "clicked" event        //
//                                                    //
////////////////////////////////////////////////////////
void on_applet_property_button(AppletWidget *applet, gpointer data)
{
  mainDialog_create();
}

/// Save Dialog ///////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////
//                                                    //
//  Save Dialog Yes button "clicked" event            //
//                                                    //
////////////////////////////////////////////////////////
void on_save_yes_button_clicked (GtkButton *button, gpointer user_data)
{
  on_save_button_clicked (NULL, NULL);
  applet_widget_gtk_main_quit();
}


////////////////////////////////////////////////////////
//                                                    //
//  Save Dialog No button "clicked" event             //
//                                                    //
////////////////////////////////////////////////////////
void on_save_no_button_clicked (GtkButton *button, gpointer user_data)
{
  applet_widget_gtk_main_quit();
}

