/*
 *	wmnut-0.09
 *
 *		A WindowMaker dockable application that allows UPS users
 *		to graphically monitor the status of their power source
 *              using NUT framework (www.exploits.org/nut). 
 *		(I.e. whether or not AC or battery is in use as well as
 *		 how long it will take to drain or charge the battery).
 *
 *              Written (and copyrighted under GPL) by
 *              Arnaud Quette <arnaud.quette@free.fr>
 *              Sponsored by MGE UPS SYSTEMS <www.mgeups.com>
 *
 *              widely based on wmapm originally written by (huge thanks) 
 * 		Chris D. Faulhaber <jedgar@speck.ml.org>. Version 3.0
 * 		and extensively modified version of version 2.0 
 *		by Michael G. Henderson <mghenderson@lanl.gov>.
 *
 *
 * 	This program is free software; you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation; either version 2, or (at your option)
 * 	any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program (see the file COPYING); if not, write to the
 * 	Free Software Foundation, Inc.,
 * 	59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 */


/*
 * 	Changes:
 *
 * 	See CHANGES file in parent directory.
 *
 */


/* standard system includes */
#ifdef FreeBSD
#include <err.h>
#include <sys/file.h>
#endif 

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>

/* X11 includes */
#include <X11/X.h>
#include <X11/xpm.h>

/* network includes */
#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>

/* nut and wmnut includes */
#include <upsfetch.h>
#include "wmnut.h"
#include "../wmgeneral/wmgeneral.h"

/* pixmaps */
#include "wmnut_master.xpm"
#include "wmnut_master_LowColor.xpm"
#include "wmnut_mask.xbm"

/* defines */
#define LARGEBUF      1024

void ParseCMDLine(int argc, char *argv[]);

/* base parameters */
char	*upshost = NULL;
int	Verbose 		= 0;	/* 1 for verbose mode : displays NUT available features and base values */
int	CriticalLevel 		= 10;
int	LowLevel      		= 40;
float   BlinkRate     		= 3.0;	 /* blinks per second */
float   UpdateRate    		= 0.8;   /* Number of updates per second */
int	Beep 	      		= 0;	 /* Controls beeping when you get to CriticalLevel: Off by default */
int	Volume	      		= 50;	 /* ring bell at 50% volume */
int	Alert	      		= 0;  	 /* Controls whether alert is sent to all users via wall: Off by default  */
int	UseLowColorPixmap 	= 0; 	 /* Use a lower number of colors for the poor saps on 8-bit displays */
float  	LAlertRate     		= 300.0; /* send alert every 5 minutes when Low */
float  	CAlertRate     		= 120.0; /* send alert every 2 minutes when Critical */
int     WithDrawn     		= 1;	 /* start in withdrawn shape (for WindowMaker) */

ups_info *CurHost; /* current ups host monitored */
my_nut_info *Hosts;
int nbHosts;
struct ups_info 	my_cur_info;
rckeys	wmnut_keys[13];


/* Get a variable from the UPS using UDP transfer mode */
int get_ups_var_udp (char *variable, char *value)
{
  if (getupsvar (CurHost->hostname, variable, value, 128) < 0 ) 
    {
#ifdef DEBUG
      fprintf(stderr, "Error reading from ups: %s\n", upsstrerror (upserror));
#endif
      
      switch(upserror) 
	{
	case UPSF_DATASTALE :
	  /* CurHost->comm_status = COM_LOST; */
	  my_cur_info.comm_status = COM_LOST;
	  break;
	case UPSF_RECVTIMEOUT : /* try TCP mode */
	case UPSF_RECVFAILURE :
	  /* preferences_udp_set (FALSE); */
	  /* printf("Trying TCP connection instead...\n"); */	
	  /* connect_ups_tcp (CurHost->hostname); */      
	  /* start_fetching (); */
	  break;
	case UPSF_VARNOTSUPP:
	default:
	  break;
	}
      return NOK;
    } 
  else
    {
      my_cur_info.comm_status = COM_OK;
      return OK;
    }
}

/* Get a variable from the UPS using TCP transfer mode */
/* char * */
/* get_ups_var_tcp (char *variable) */
/* { */
/* 	char *uerror; */
	
/* 	if (getupsvarfd (hostfd, upshost, variable, value, sizeof (value)) < 0 ) { */
/* 		uerror = upsstrerror (upserror); */
/* 		free (uerror); */
/* 		return NULL; */
/* 	} else { */
/* 		return value; */
/* 	} */
/* } */




/*  
 *   main  
 */
int main(int argc, char *argv[]) {

  int           time_left, 
		hour_left, 
                min_left;
  int		m, mMax, n, nMax, k, Toggle = OFF;
  long int	r, rMax, s, sMax;
  char		vars[LARGEBUF], *v, *ptr;
  char  	*value;
 
  /*  
   *  Default values and init.
   */

  value = (char *)malloc(128);  
  upshost = (char *)malloc(50);

  if ((value == NULL) || (upshost == NULL)) {
    fprintf(stderr, "malloc failed\n");
    exit(0);
  }

  strcpy(upshost, "localhost");  
  my_cur_info.battery_runtime = -1;
  nbHosts = 0;
  CurHost = NULL;
  BlinkRate = 3.0;
  UpdateRate = 1.0/1.25;

  AddRcKey(&wmnut_keys[0], "UPS", TYPE_STRING, upshost);
  AddRcKey(&wmnut_keys[1], "LAlertRate", TYPE_FLOAT, &LAlertRate);
  AddRcKey(&wmnut_keys[2], "CAlertRate", TYPE_FLOAT, &CAlertRate);
  AddRcKey(&wmnut_keys[3], "Alert", TYPE_BOOL, &Alert);
  AddRcKey(&wmnut_keys[4], "BlinkRate", TYPE_FLOAT, &BlinkRate); 
  AddRcKey(&wmnut_keys[5], "Beep", TYPE_BOOL, &Beep);
  AddRcKey(&wmnut_keys[6], "Volume", TYPE_INT, &Volume);
  AddRcKey(&wmnut_keys[7], "LowLevel", TYPE_INT, &LowLevel);
  AddRcKey(&wmnut_keys[8], "CriticalLevel", TYPE_INT, &CriticalLevel);
  AddRcKey(&wmnut_keys[9], "UseLowColorPixmap", TYPE_BOOL, &UseLowColorPixmap);
  AddRcKey(&wmnut_keys[10], "Verbose", TYPE_BOOL, &Verbose);
  AddRcKey(&wmnut_keys[11], "WithDrawn", TYPE_BOOL, &WithDrawn);
  AddRcKey(&wmnut_keys[12], NULL, TYPE_NULL, NULL);

  /*
   * Setup signal handlers
   */

  signal(SIGUSR1, ReloadRCFile);

  /*
   *  Parse rcfile command arguments.
   *  First, try with /etc/wmnutrc else 
   *  (if not exists), try ~/.wmnutrc 
   *  Note that the 2nd override the 1st
   */
  
  LoadRCFile(wmnut_keys);

#ifdef DEBUG
  if(upshost != NULL)
    printf("upshost %s\n", upshost);
  else
    printf("upshost == NULL\n");

  printf("Alert = %d\n", Alert);
  printf("LAlertRate = %f\n", LAlertRate);
  printf("CAlertRate = %f\n", CAlertRate);
  printf("BlinkRate = %f\n", BlinkRate);
  printf("Beep = %d\n", Beep);
  printf("Volume = %d\n", Volume);
  printf("LowLevel = %d\n", LowLevel);
  printf("CriticalLevel = %d\n", CriticalLevel);
  printf("UseLowColorPixmap = %d\n", UseLowColorPixmap);
  printf("Verbose = %d\n", Verbose);
  printf("WithDrawn = %d\n", WithDrawn);
#endif

  /*
   *  Parse any command line arguments.
   */
  ParseCMDLine(argc, argv);
  //CurHost = Hosts->Ups_list;
  
  if (CurHost == NULL) {
	CurHost = (ups_info *) malloc(sizeof(ups_info));
	CurHost->hostname = (char *) malloc(10);
	strcpy(CurHost->hostname, "localhost");
	CurHost->hostnumber = 1;
  }

  if (Verbose)
    fprintf(stdout, "UPS : %s\n", CurHost->hostname);

  BlinkRate = (BlinkRate >= 0.0) ? BlinkRate : -1.0*BlinkRate;
  UpdateRate = (UpdateRate >= 0.0) ? UpdateRate : -1.0*UpdateRate;
  
  
  nMax = (int)( 1.0e6/(2.0*UpdateRate*DELAY)  );
  mMax = (BlinkRate > 0.0) ? (int)( 1.0e6/(2.0*BlinkRate*DELAY)  ) : nMax;
  rMax = (int)( LAlertRate*1.0e6/(2.0*DELAY)  );
  sMax = (int)( CAlertRate*1.0e6/(2.0*DELAY)  );
  
  
  /*  
   *  Check NUT daemon availability on host(s) 
   */   
  if (getupsvarlist(CurHost->hostname, vars, sizeof(vars)) < 0) {
    fprintf (stderr, "Unable to get variable list - %s\n", 
	    upsstrerror(upserror));
    exit (1);
  }
    
    
  if (strlen(vars) == 0) {
    fprintf (stderr, "No variables available!  Check your configuration (ups.conf and upsd.conf)\n");
    exit (1);
  }
       
  if (UseLowColorPixmap)
    openXwindow(argc, argv, wmnut_master_LowColor, wmnut_mask_bits, wmnut_mask_width, wmnut_mask_height, WithDrawn);
  else
    openXwindow(argc, argv, wmnut_master, wmnut_mask_bits, wmnut_mask_width, wmnut_mask_height, WithDrawn);
  
  /* try to access all variables */
  v = vars;
  while (v != NULL) {
    ptr = strchr (v, ' ');
    if (ptr)
      *ptr++ = '\0';

    if (Verbose)
      fprintf (stdout, "%s: ", v);
    
    if (getupsvar(CurHost->hostname, v, value, 128) < 0)
      fprintf (stderr, "Error: %s\n", upsstrerror(upserror));
    else if (Verbose)
	fprintf (stdout, "%s\n", value);
    
    v = ptr;
  }
  free(value);
  
  /*
   *     Loop until we die...
   */
  n = 32000;
  m = 32000;
  r = rMax+1;
  s = sMax+1;

  while(1) {
            
    /*
     *  Only process nut info only every nMax cycles of this
     *  loop. We run it faster to catch the xevents like button 
     *  presses and expose events, etc...
     *  
     *  DELAY is set at 0.00625 seconds, so process nut info
     *  every 1.25 seconds...
     *
     */
    if (n>nMax){
      
      /* invert toggle */
      if (Toggle)
	Toggle = OFF;
      else
	Toggle = ON;
      
      n = 0;
      
      /* Get battery charge level */
      if (get_ups_var_udp ("BATTPCT", value) != NOK) {
	my_cur_info.battery_percentage = atoi(value);
      }	  

      /* Get UPS status */
      if (get_ups_var_udp ("STATUS", value) != NOK) {
	if (!strncmp(value, "OL", 2))
	  my_cur_info.ups_status = UPS_ONLINE;
	else if (!strncmp(value, "OB", 2))
	  my_cur_info.ups_status = UPS_ONBATT;
	else if (!strncmp(value, "LB", 2))
	  my_cur_info.ups_status = UPS_LOWBATT;
	else if (!strncmp(value, "OVER", 4))
	  my_cur_info.ups_status = UPS_OVERLOAD;
      }

      /* Get runtime to empty */
      if (get_ups_var_udp ("RUNTIME", value) != NOK) {
	my_cur_info.battery_runtime = atoi(value);
      }
      else {
	my_cur_info.battery_runtime = -1;
      }

      /* Get Battery load level */
      if (get_ups_var_udp ("LOADPCT", value) != NOK) {
	my_cur_info.battery_load = atoi(value);
      }
      else {
	my_cur_info.battery_load = -1;
      }

      /*
       *  Check communication status.
       */
      if ( (int)(my_cur_info.comm_status) != COM_LOST){
	
	/*
	 *   Communication Status: COM_OK.
	 */
	copyXPMArea(104,  6, 5, 7,  6,  7);
      }
      else {
	/*
	 *  Communication Status: COM_LOST.
	 *  Blink red "C" on/off...
	 */
#ifdef DEBUG
	fprintf(stderr, "Communication lost with ups\n");
#endif
	if (Toggle||(BlinkRate == 0.0)){
	  if (Beep) XBell(display, Volume);
	  {
/*  	    Toggle = OFF; */
	    copyXPMArea(110,  6, 5, 7,  6,  7);
	  }
	} else{
/*  	  Toggle = ON; */
	  copyXPMArea(98,  6, 5, 7,  6,  7);
	}
      }
      
      /* 
       *   Check UPS status.
       */    
      
      switch (my_cur_info.ups_status) {
	
	/*       case UPS_LOWBATT:  	 */
	
      case UPS_ONBATT:
	{	    
	  if(my_cur_info.battery_percentage <= CriticalLevel)
	    {
	      /*
	       *  Battery Status: Critical.
	       *  Blink red battery [TODO : and digital %age] on/off...
	       */
	      if (Toggle||(BlinkRate == 0.0)){
		if (Beep) XBell(display, Volume);
/*  		Toggle = OFF; */
		copyXPMArea(99, 20, 12, 7, 30, 50);
	      } else{
/*  		Toggle = ON; */
		copyXPMArea(83, 20, 12, 7, 30, 50);
	      }
	    } else if(my_cur_info.battery_percentage <= LowLevel) {
	      /*
	       *  Battery Status: Low.
	       *  Blink the yellow battery [TODO : and digital %age] on/off...
	       */
	      if (Toggle||(BlinkRate == 0.0)){
		if (Beep) XBell(display, Volume);
/*  		Toggle = OFF; */
		copyXPMArea(99, 20, 12, 7, 30, 50);
	      } else{
/*  		Toggle = ON; */
		copyXPMArea(69, 20, 12, 7, 30, 50);
	      }
	    } else{
	      /*
	       *  Battery Status: High but charging.
	       *  Blink the green battery [TODO : and digital %age] on/off...
	       */
	      if (Toggle||(BlinkRate == 0.0)){
		if (Beep) XBell(display, Volume);
/*  		Toggle = OFF; */
		copyXPMArea(99, 20, 12, 7, 30, 50);
	      } else{
/*  		Toggle = ON; */
		copyXPMArea(83, 6, 12, 7, 30, 50);
	      }
	    }
	}
	break;
	
      case UPS_OVERLOAD:  
	{	
	  /*
	   *  UPS is overloaded.
	   *  [TODO : Blink the battery load %age on/off...]
	   */
	  /* Hide digits and '%' */
	  copyXPMArea(37, 34, 19, 7,37, 34);						
	}
	break;
	
      case UPS_ONLINE:  
	{	
	  /*
	   *   UPS on-line. I.e. we are "plugged-in".
	   */
	  copyXPMArea(68, 6, 12, 7, 30, 50);
	  /* copyXPMArea(S.x, S.y, w, h, D.x, D.y); */
	}
	break;
	
      default: 	
	break;
	
      }
      
      /* 
       *    Paste up the default communication status and runtime
       */
      copyXPMArea(83, 93, 41, 9, 15, 7);
                 
      /*
       *   Repaint buttons.
       */
      copyXPMArea(42, 106, 13, 11, 5, 48);
      copyXPMArea(57, 106, 13, 11, 46, 48);

      /* 
       *   Repaint host number.
       */
      copyXPMArea((CurHost->hostnumber) * 7 + 5, 93, 7, 9, 22, 49); 

      /* 
       *  Paste up the "Time Left". This time means (format HH:MM) :
       *  
       *         If not charging: Time left before battery drains to 0% 
       *         If charging:     Time left before battery gets to maximum
       *         If not supported (RUNTIME feature) --:--
       */	      
      if(my_cur_info.battery_runtime != -1) {
	time_left = (my_cur_info.using_minutes) ? my_cur_info.battery_runtime : my_cur_info.battery_runtime / 60;
	
	hour_left = time_left / 60;
	min_left  = time_left % 60;
      
	copyXPMArea( (hour_left / 10) * 7 + 5, 93, 7, 9, 21, 7); 	/* Show 10's (hour) */
	copyXPMArea((hour_left % 10) * 7 + 5, 93, 7, 9, 29, 7); 	/* Show 1's (hour)  */
	copyXPMArea(76, 93, 2, 9, 38, 7);		 		/* colon  	    */
	copyXPMArea((min_left / 10) * 7 + 5, 93, 7, 9, 42, 7); 		/* Show 10's (min)  */
	copyXPMArea((min_left % 10) * 7 + 5, 93, 7, 9, 50, 7); 		/* Show 1's (min)   */
      }
      else
	copyXPMArea(83, 106, 41, 9, 15, 7); 		 		/* Show --:--       */
  
      /* 
       *   Do Battery Load.
       */

      if (my_cur_info.battery_load != -1) {      

	if ((my_cur_info.ups_status != UPS_OVERLOAD)){
	  if (my_cur_info.battery_load >= 10)
	    copyXPMArea((my_cur_info.battery_load / 10) * 6 + 4, 81, 6, 7, 39, 34);  	/* Show 10's */
	  copyXPMArea((my_cur_info.battery_load % 10) * 6 + 4, 81, 6, 7, 45, 34);    	/* Show 1's */
	  copyXPMArea(64, 81, 7, 7, 52, 34);						/* Show '%' */
	}
      }
     
      /* 
       *   Do Battery Percentage.
       */
      
      copyXPMArea(76, 81, 19, 7, 7, 34);            		/* Show Default % */
      copyXPMArea(66, 31, 49, 9, 7, 21);           		/* Show Default Meter */
      
      /* displays battery percent bis */
      if (my_cur_info.battery_percentage == 100){
	copyXPMArea(15, 81, 1, 7,  7, 34);             	/* If 100%, show 100% */
	copyXPMArea( 5, 81, 6, 7,  9, 34);
	copyXPMArea( 5, 81, 6, 7, 15, 34);
	copyXPMArea(64, 81, 7, 7, 21, 34);            	/* Show '%' */
	copyXPMArea(66, 42, 49, 9, 7, 21);           	/* Show Meter */
      }
      else {	  
	if (my_cur_info.battery_percentage >= 10)
	  copyXPMArea((my_cur_info.battery_percentage / 10) * 6 + 4, 81, 6, 7,  9, 34);  	/* Show 10's */
	copyXPMArea((my_cur_info.battery_percentage % 10) * 6 + 4, 81, 6, 7, 15, 34);		/* Show 1's */
	copyXPMArea(64, 81, 7, 7, 21, 34);							/* Show '%' */
      }
      
      /*
       *  Show Meter
       */
      k = my_cur_info.battery_percentage * 49 / 100;
      copyXPMArea(66, 42, k, 9, 7, 21);
      if (k%2) 
	copyXPMArea(66+k-1, 52, 1, 9, 7+k-1, 21);
      else     
	copyXPMArea(66+k, 52, 1, 9, 7+k, 21);
    } else {
      
      /*
       *  Update the counter. When it hits nMax, we will 
       *  process /proc/nut information again.
       */
      ++n;           
    }
    
    /*
     *  This controls Critical Alerts [TODO : move it to switch(ups_status)]
     */
/*     if (Alert){ */
/*       if (( (int)(my_cur_info.battery_status) == 2) */
/* 	  ||( (int)(my_cur_info.battery_percentage) <= CriticalLevel )){ */
	
/* 	if (s>sMax){ */
	  
/* 	  s = 0; */
/* 	  fp = popen("wall", "w"); */
/* 	  fprintf(fp, "UPS Battery is critical!. Percent: %d\n", (int)(my_cur_info.battery_percentage)); */
/* 	  pclose(fp); */
	  
/* 	} else { */
	  
	  /*
	   *  Update the counter. 
	   */
/* 	  ++s; */
/* 	}	 */
/*       } else if (( (int)(my_cur_info.battery_status) == 1) */
/* 		 ||( (int)(my_cur_info.battery_percentage) <= LowLevel )){ */
	
/* 	if (r>rMax){ */
	  
/* 	  r = 0; */
/* 	  fp = popen("wall", "w"); */
/* 	  fprintf(fp, "UPS Battery is low. Percent: %d\n", (int)(my_cur_info.battery_percentage)); */
/* 	  pclose(fp); */
	  
/* 	} else { */
	  
	  /*
	   *  Update the counter. 
	   */
/* 	  ++r; */
/* 	} */
/*       } */
/*     } */
    
    /* 
     *   Process any pending X events.
     */
    CheckX11Events();
	
    /* 
     *  Redraw and wait for next update 
     */
    RedrawWindow();
    usleep(DELAY);
  }
}

/* 
 *   ParseCMDLine()  
 */
void ParseCMDLine(int argc, char *argv[]) {
  char *cmdline;
  int  i, k;

  k = 1;

  for (i = 1; i < argc; i++) {
    cmdline = argv[i];
    
    if (cmdline[0] == '-') {
      switch(cmdline[1]) {
      case 'd': 
	++i;
	break;
      case 'A': 
	Alert = 1;
	LAlertRate = atof(argv[++i]);
	CAlertRate = atof(argv[++i]);
	break;
      case 'b': 
	BlinkRate = atof(argv[++i]);
	break;
      case 'C': 
	CriticalLevel = atoi(argv[++i]);
	break;
      case 'L': 
	LowLevel = atoi(argv[++i]);
	break;
      case 'l': 
	UseLowColorPixmap = 1;
	break;
      case 'V': 
	Verbose = 1;
	break;
      case 'v':   
	printf("\nThis is wmnut version: %s\n", WMNUT_VERSION);
	printf("\nCopyright 2001,2002 Arnaud Quette\n");
      	printf("This program is Sponsored by MGE UPS SYSTEMS (www.mgeups.com)\n");
      	printf("\nComplete documentation for WMNUT should be found on this system using\n");
      	printf("`man wmnut' or `wmnut -h'.  If you have access to the Internet, point\n");
      	printf("your browser at http://wmnut.tuxfamily.org, the WMNUT Home Page.\n\n");
	exit(1);
      case 'w': 
	WithDrawn = 0; /* not in default withdrawn mode, so in windowed mode */
	break;
      case 'B': 
	Beep = 1;
	Volume = atoi(argv[++i]);
	break;
      case 'U': 
	CurHost = (ups_info *) malloc(sizeof(ups_info));
	CurHost->hostname = (char *) malloc(strlen(argv[++i]));
	strcpy(CurHost->hostname, argv[i]);
	CurHost->hostnumber = ++nbHosts;
	//Hosts->Ups_list[nbHosts]= CurHost;
	break;
      default:  
	printf("\nwmnut version: %s\n", WMNUT_VERSION);
	printf("Usage: wmnut [arguments]\n\n");
	printf("-A <T1 T2>\tSend messages to users terminals when Low and critical.\n");
	printf("             \tT1 is seconds between messages when Low.\n"); 
	printf("             \tT2 is seconds between messages when Critical.\n");
	printf("-b <BlinkRate>\tBlink rate for red LED. (0 for no blinking.)\n");
	printf("-B <Volume>\tBeep at Critical Level with Volume (between -100%% to 100%%).\n");
	printf("-C <CriticaLvl>\tDefine level at which red LED turns on (CriticalLevel).\n");
	printf("-d <display>\tUse alternate display.\n");
	printf("-h\t\tDisplay this help screen.\n");
	printf("-l\t\tUse a low-color pixmap to conserve colors on 8-bit displays.\n");
	printf("-L <LowLevel>\tDefine level at which yellow LED turns on.\n");
	printf("             \tCriticalLevel takes precedence if LowLevel < CriticalLevel.\n");
	printf("-U <upsname>\tDefine upsname ([upsnmame@]hostname, default is localhost)\n");
	printf("-v \t\tPrint version (includes important WMNUT info).\n");
	printf("-V \t\tVerbose mode : display NUT available features and base value.\n");
	printf("-w \t\tWindowed mode (opposite to native Window Maker withdrawn mode).\n\n");
	exit(1);
      }
    }
  }    
}




