/*
 * Copyright (C) 2014 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#ifndef DFGValueStrength_h
#define DFGValueStrength_h

#if ENABLE(DFG_JIT)

#include <wtf/PrintStream.h>

namespace JSC { namespace DFG {

enum ValueStrength {
    // The value is known to the DFG but no optimizations have been performed that require the
    // value to be kept alive. All OSR entry values are fragile until we do some optimization that
    // uses them, like actually constant folding a variable to that value. By convention we say
    // that all non-cells are fragile.
    FragileValue,
    
    // The value has been used for optimization and it arose through inference. We don't want the
    // fact that we optimized the code to result in the GC keeping this value alive unnecessarily,
    // so we'd rather kill the code and recompile than keep the object alive longer.
    WeakValue,
    
    // The code will keep this value alive. This is true of constants that were present in the
    // source. String constants tend to be strong.
    StrongValue
};

inline ValueStrength merge(ValueStrength a, ValueStrength b)
{
    switch (a) {
    case FragileValue:
        return b;
    case WeakValue:
        if (b == StrongValue)
            return StrongValue;
        return WeakValue;
    case StrongValue:
        return StrongValue;
    }
    RELEASE_ASSERT_NOT_REACHED();

    return FragileValue;
}

} } // namespace JSC::DFG

namespace WTF {

void printInternal(PrintStream&, JSC::DFG::ValueStrength);

} // namespace WTF

#endif // ENABLE(DFG_JIT)

#endif // DFGValueStrength_h

