/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "issuehistorywriter.h"

#include "linklocator.h"

IssueHistoryWriter::IssueHistoryWriter( Format format ) :
    m_format( format )
{
}

IssueHistoryWriter::~IssueHistoryWriter()
{
}

void IssueHistoryWriter::begin()
{
    if ( m_format == Report )
        m_lines.append( "<table class=\"th\" cellspacing=\"0\" cellpadding=\"5\" border=\"2\" width=\"100%\">" );
    else
        m_lines.append( "<table cellspacing=\"5\" cellpadding=\"3\" width=\"100%\">" );
}

void IssueHistoryWriter::end()
{
    m_lines.append( "</table>" );
}

void IssueHistoryWriter::writeChanges( const QString& header, const QStringList& items )
{
    m_lines.append( "<tr>" );
    if ( m_format == Report )
        m_lines.append( "<td bgcolor=\"#eeeeee\">" + header + "</td>" );
    else
        m_lines.append( "<td bgcolor=\"#ccccff\">" + header + "</td>" );
    m_lines.append( "</tr>" );
    m_lines.append( "<tr>" );
    m_lines.append( "<td><ul>" );
    m_lines.append( "<li>" + items.join( "</li><li>" ) + "</li>" );
    m_lines.append( "</ul></td>" );
    m_lines.append( "</tr>" );
}

void IssueHistoryWriter::writeComment( const QString& header, const QString& text, int commentId )
{
    QString anchor = QString( "<a name=\"id%1\"></a>" ).arg( commentId );

    m_lines.append( "<tr>" );
    if ( m_format == Report )
        m_lines.append( "<td bgcolor=\"#eeeeee\">" + anchor + header + "</td>" );
    else
        m_lines.append( "<td bgcolor=\"#ffff88\">" + anchor + header + "</td>" );
    m_lines.append( "</tr>" );
    m_lines.append( "<tr>" );
    m_lines.append( "<td>" );
    m_lines.append( "<pre style=\"white-space: pre-wrap\">" + text + "</pre>" );
    m_lines.append( "</td>" );
    m_lines.append( "</tr>" );
}

void IssueHistoryWriter::writeAttachment( const QString& header, const QString& item, int attachmentId )
{
    QString anchor = QString( "<a name=\"id%1\"></a>" ).arg( attachmentId );

    m_lines.append( "<tr>" );
    if ( m_format == Report )
        m_lines.append( "<td bgcolor=\"#eeeeee\">" + anchor + header + "</td>" );
    else
        m_lines.append( "<td bgcolor=\"#99ff99\">" + anchor + header + "</td>" );
    m_lines.append( "</tr>" );
    m_lines.append( "<tr>" );
    m_lines.append( "<td><ul>" );
    m_lines.append( "<li>" + item + "</li>" );
    m_lines.append( "</ul></td>" );
    m_lines.append( "</tr>" );
}

QString IssueHistoryWriter::name( const QString& text )
{
    return QString ( "<strong>%1</strong>" ).arg( LinkLocator::escape( text ) );
}

QString IssueHistoryWriter::date( const QString& text )
{
    return QString ( "<strong>%1</strong>" ).arg( text );
}

QString IssueHistoryWriter::value( const QString& text )
{
    if ( m_format == Report )
        return LinkLocator::convertToHtml( text, LinkLocator::NoInternalLinks );
    else
        return LinkLocator::convertToHtml( text, 0 );
}

QString IssueHistoryWriter::description( const QString& text )
{
    if ( m_format == Report )
        return LinkLocator::convertToHtml( text, LinkLocator::NoInternalLinks );
    else
        return LinkLocator::convertToHtml( text, 0 );
}

QString IssueHistoryWriter::comment( const QString& text )
{
    if ( m_format == Report )
        return LinkLocator::convertToHtml( text, LinkLocator::NoInternalLinks );
    else
        return LinkLocator::convertToHtml( text, 0 );
}

QString IssueHistoryWriter::attachment( int attachmentId, const QString& name )
{
    QString href = QString( ( m_format == Report ) ? "#id%1" : "attachment://%1" ).arg( attachmentId );
    return QString( "<a href=\"%1\">%2</a>" ).arg( href, LinkLocator::escape( name ) );
}

QString IssueHistoryWriter::result() const
{
    return m_lines.join( "\r\n" );
}
