/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "reportdialog.h"

#include <QLayout>
#include <QLabel>
#include <QGroupBox>
#include <QRadioButton>
#include <QPushButton>
#include <QDialogButtonBox>
#include <QTextDocument>
#include <QPrintDialog>
#include <QFileDialog>
#include <QMessageBox>
#include <QTextStream>
#include <QTextCodec>
#include <QSettings>

#include "previewdialog.h"
#include "data/datamanager.h"
#include "models/csvreportgenerator.h"
#include "models/csvreportwriter.h"
#include "models/htmlreportgenerator.h"
#include "models/htmlreportwriter.h"
#include "models/treeviewsettings.h"
#include "widgets/gradientwidget.h"
#include "widgets/previewwidget.h"
#include "configdata.h"
#include "iconloader.h"

ReportDialog::ReportDialog( HtmlReportGenerator::Source source, ReportMode mode, QWidget* parent ) : QDialog( parent ),
    m_source( source ),
    m_mode( mode ),
    m_folderId( 0 ),
    m_printer( NULL )
{
    QVBoxLayout* topLayout = new QVBoxLayout( this );
    topLayout->setMargin( 0 );
    topLayout->setSpacing( 0 );

    QWidget* promptWidget = new GradientWidget( this );
    topLayout->addWidget( promptWidget );

    QHBoxLayout* promptLayout = new QHBoxLayout( promptWidget );
    promptLayout->setSpacing( 10 );

    QLabel* promptPixmap = new QLabel( promptWidget );
    promptLayout->addWidget( promptPixmap, 0 );

    QLabel* promptLabel = new QLabel( promptWidget );
    promptLabel->setWordWrap( true );
    promptLabel->setMinimumWidth( 300 );
    promptLayout->addWidget( promptLabel, 1 );

    QFrame* separator = new QFrame( this );
    separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    topLayout->addWidget( separator );

    QVBoxLayout* mainLayout = new QVBoxLayout();
    mainLayout->setMargin( 9 );
    mainLayout->setSpacing( 6 );
    topLayout->addLayout( mainLayout );

    QGroupBox* typeGroup = new QGroupBox( tr( "Report Type" ), this );
    QVBoxLayout* typeLayout = new QVBoxLayout( typeGroup );
    mainLayout->addWidget( typeGroup );

    if ( source == HtmlReportGenerator::Folder ) {
        m_tableButton = new QRadioButton( tr( "Table with visible columns only" ), typeGroup );
        typeLayout->addWidget( m_tableButton );
        m_fullTableButton = new QRadioButton( tr( "Table with all system and user columns" ), typeGroup );
        typeLayout->addWidget( m_fullTableButton );
    } else {
        m_tableButton = NULL;
        m_fullTableButton = NULL;
    }

    if ( mode != ExportCsv ) {
        m_summaryButton = new QRadioButton( tr( "Summary report including issue details" ), typeGroup );
        typeLayout->addWidget( m_summaryButton );
    } else {
        m_summaryButton = NULL;
    }

    if ( mode != ExportCsv && source == HtmlReportGenerator::Issue ) {
        m_fullReportButton = new QRadioButton( tr( "Full report including issue details and history" ), typeGroup );
        typeLayout->addWidget( m_fullReportButton );
    } else {
        m_fullReportButton = NULL;
    }

    QHBoxLayout* buttonsLayout = new QHBoxLayout();
    mainLayout->addLayout( buttonsLayout );

    if ( mode == Print ) {
        QPushButton* previewButton = new QPushButton( tr( "&Preview..." ), this );
        buttonsLayout->addWidget( previewButton );

        connect( previewButton, SIGNAL( clicked() ), this, SLOT( showPreview() ) );
    }

    QDialogButtonBox* buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Cancel, Qt::Horizontal, this );
    buttonsLayout->addWidget( buttonBox );

    buttonBox->button( QDialogButtonBox::Ok )->setText( tr( "&OK" ) );
    buttonBox->button( QDialogButtonBox::Cancel )->setText( tr( "&Cancel" ) );

    connect( buttonBox, SIGNAL( accepted() ), this, SLOT( accept() ) );
    connect( buttonBox, SIGNAL( rejected() ), this, SLOT( reject() ) );

    if ( m_tableButton )
        m_tableButton->setChecked( true );
    else if ( m_summaryButton )
        m_summaryButton->setChecked( true );

    QString sourceName;
    switch ( source ) {
        case HtmlReportGenerator::Folder:
            sourceName = tr( "list of issues" );
            break;
        case HtmlReportGenerator::Issue:
            sourceName = tr( "issue details" );
            break;
    }

    switch ( mode ) {
        case Print:
            setWindowTitle( tr( "Print" ) );
            promptPixmap->setPixmap( IconLoader::pixmap( "file-print", 22 ) );
            promptLabel->setText( tr( "Print %1" ).arg( sourceName ) );
            break;
        case ExportCsv:
            setWindowTitle( tr( "Export To CSV" ) );
            promptPixmap->setPixmap( IconLoader::pixmap( "export-csv", 22 ) );
            promptLabel->setText( tr( "Export %1 to CSV file" ).arg( sourceName ) );
            break;
        case ExportHtml:
            setWindowTitle( tr( "Export To HTML" ) );
            promptPixmap->setPixmap( IconLoader::pixmap( "export-html", 22 ) );
            promptLabel->setText( tr( "Export %1 to HTML file" ).arg( sourceName ) );
            break;
        case ExportPdf:
            setWindowTitle( tr( "Export To PDF" ) );
            promptPixmap->setPixmap( IconLoader::pixmap( "export-pdf", 22 ) );
            promptLabel->setText( tr( "Export %1 to PDF file" ).arg( sourceName ) );
            break;
    }

    promptLabel->setFixedHeight( promptLabel->heightForWidth( promptLabel->minimumWidth() ) );

    setFixedSize( sizeHint() );
}

ReportDialog::~ReportDialog()
{
    delete m_printer;
}

void ReportDialog::setIssue( int issueId )
{
    m_folderId = 0;
    m_issues.clear();
    m_issues.append( issueId );
    m_filterName.clear();
}

void ReportDialog::setFolder( int folderId, const QList<int>& issues, const QString& filterName )
{
    m_folderId = folderId;
    m_issues = issues;
    m_filterName = filterName;
}

void ReportDialog::accept()
{
    bool result = false;

    switch ( m_mode ) {
        case Print:
            result = print();
            break;
        case ExportCsv:
            result = exportCsv();
            break;
        case ExportHtml:
            result = exportHtml();
            break;
        case ExportPdf:
            result = exportPdf();
            break;
    }

    if ( result )
        QDialog::accept();
}

bool ReportDialog::print()
{
    if ( !m_printer )
        m_printer = new PreviewPrinter();

    QPrintDialog dialog( m_printer, this );

    if ( dialog.exec() != QDialog::Accepted )
        return false;

    QTextDocument document;
    document.setHtml( generateHtmlReport() );

    document.print( m_printer );

    return true;
}

bool ReportDialog::exportCsv()
{
    QString fileName = getReportFileName( "report.csv", tr( "CSV Files (*.csv)" ) );

    if ( fileName.isEmpty() )
        return false;

    QFile file( fileName );
    if ( !file.open( QIODevice::WriteOnly ) ) {
        QMessageBox::warning( this, tr( "Warning" ), tr( "Could not save file" ) );
        return false;
    }

    QTextStream stream( &file );
    stream.setCodec( QTextCodec::codecForName( "UTF-8" ) );

    stream << generateCsvReport();

    return true;
}

bool ReportDialog::exportHtml()
{
    QString fileName = getReportFileName( "report.html", tr( "HTML Files (*.html)" ) );

    if ( fileName.isEmpty() )
        return false;

    QFile file( fileName );
    if ( !file.open( QIODevice::WriteOnly ) ) {
        QMessageBox::warning( this, tr( "Warning" ), tr( "Could not save file" ) );
        return false;
    }

    QTextStream stream( &file );
    stream.setCodec( QTextCodec::codecForName( "UTF-8" ) );

    stream << generateHtmlReport();

    return true;
}

bool ReportDialog::exportPdf()
{
    QString fileName = getReportFileName( "report.pdf", tr( "PDF Files (*.pdf)" ) );

    if ( fileName.isEmpty() )
        return false;

    QTextDocument document;
    document.setHtml( generateHtmlReport() );

    QPrinter printer( QPrinter::HighResolution );
    printer.setOutputFormat( QPrinter::PdfFormat );
    printer.setOutputFileName( fileName );

    document.print( &printer );

    return true;
}

void ReportDialog::showPreview()
{
    if ( !m_printer )
        m_printer = new PreviewPrinter();

    QTextDocument document;
    document.setHtml( generateHtmlReport() );

    PreviewDialog dialog( m_printer, &document, this );

    dialog.exec();
}

QString ReportDialog::generateCsvReport()
{
    CsvReportGenerator generator;

    generator.setFolder( m_folderId, m_issues );
    if ( m_tableButton->isChecked() )
        generator.setTable( CsvReportGenerator::CurrentTable );
    else if ( m_fullTableButton->isChecked() )
        generator.setTable( CsvReportGenerator::FullTable );

    CsvReportWriter writer;
    generator.write( &writer );

    return writer.result();
}

QString ReportDialog::generateHtmlReport()
{
    HtmlReportGenerator generator( m_source );

    if ( m_source == HtmlReportGenerator::Folder ) {
        generator.setFolder( m_folderId, m_issues, m_filterName );
        if ( m_tableButton->isChecked() )
            generator.setTable( HtmlReportGenerator::CurrentTable );
        else if ( m_fullTableButton->isChecked() )
            generator.setTable( HtmlReportGenerator::FullTable );
        if ( m_summaryButton->isChecked() )
            generator.setSummaryEnabled( true );
    } else if ( m_source == HtmlReportGenerator::Issue && !m_issues.isEmpty() ) {
        generator.setIssue( m_issues.first() );
        generator.setSummaryEnabled( true );
        if ( m_fullReportButton->isChecked() )
            generator.setHistoryEnabled( true );
    }

    HtmlReportWriter writer;
    generator.write( &writer );

    return writer.result();
}

QString ReportDialog::getReportFileName( const QString& name, const QString& filter )
{
    QSettings* settings = configData->settings();
    QString dir = settings->value( "Paths/SaveReport", QDir::homePath() ).toString();

    QFileInfo fileInfo( QDir( dir ), name );

    QString path = QFileDialog::getSaveFileName( this, tr( "Save As" ), fileInfo.absoluteFilePath(), filter );

    if ( !path.isEmpty() ) {
        fileInfo.setFile( path );
        settings->setValue( "Paths/SaveReport", fileInfo.absoluteDir().path() );
    }

    return path;
}
