/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "preferencesbatch.h"

#include "commands/command.h"
#include "commands/commandmanager.h"
#include "connectionmanager.h"
#include "connectioninfo.h"
#include "datamanager.h"

LoadPreferencesBatch::LoadPreferencesBatch( int userId ) : AbstractBatch( 0 )
{
    Job job( &LoadPreferencesBatch::listPreferencesJob );
    job.addArg( userId );
    m_queue.addJob( job );
}

LoadPreferencesBatch::~LoadPreferencesBatch()
{
}

PreferencesData* LoadPreferencesBatch::preferencesData() const
{
    return new PreferencesData( *this );
}

Command* LoadPreferencesBatch::fetchNext()
{
    if ( m_queue.moreJobs() )
        return m_queue.callJob( this );

    return NULL;
}

Command* LoadPreferencesBatch::listPreferencesJob( const Job& job )
{
    Command* command = new Command();

    command->setKeyword( "LIST PREFERENCES" );
    command->addArg( job.argInt( 0 ) );

    command->setAcceptNullReply( true );
    command->addRule( "P ss", ReplyRule::ZeroOrMore );

    connect( command, SIGNAL( commandReply( const Reply& ) ), this, SLOT( listPreferencesReply( const Reply& ) ) );

    return command;
}

void LoadPreferencesBatch::listPreferencesReply( const Reply& reply )
{
    for ( int i = 0; i < reply.lines().count(); i++ ) {
        ReplyLine line = reply.lines().at( i );
        m_map.insert( line.argString( 0 ), line.argString( 1 ) );
    }
}

SavePreferencesBatch::SavePreferencesBatch( int userId, const PreferencesData& data ) : AbstractBatch( 0 ),
    PreferencesData( data ),
    m_userId( userId ),
    m_iterator( m_map ),
    m_update( false )
{
}

SavePreferencesBatch::~SavePreferencesBatch()
{
}

Command* SavePreferencesBatch::fetchNext()
{
    if ( m_iterator.hasNext() ) {
        m_iterator.next();
        return createSetCommand();
    }

    if ( m_update ) {
        m_update = false;
        return dataManager->updatePreferences();
    }

    return NULL;
}

Command* SavePreferencesBatch::createSetCommand()
{
    Command* command = new Command();

    command->setKeyword( "SET PREFERENCE" );
    command->addArg( m_userId );
    command->addArg( m_iterator.key() );
    command->addArg( m_iterator.value() );

    command->setAcceptNullReply( true );
    command->addRule( "OK", ReplyRule::One );

    if ( m_userId == connectionManager->connectionInfo()->userId() )
        connect( command, SIGNAL( commandReply( const Reply& ) ), this, SLOT( setUpdate() ) );

    return command;
}

void SavePreferencesBatch::setUpdate()
{
    m_update = true;
}
