#include "DefaultTimeWarpEventSet.h"
#include "Event.h"
#include "SimulationObject.h"
#include "TimeWarpSimulationManager.h"

using std::cerr;
using std::endl;

class reclaimEvent : public std::unary_function< const Event *, void> {
public:
  reclaimEvent( SimulationObject *initObject, 
		const VTime &initCollectTime ) 
    : reclaimer( initObject ),
      collectTime( initCollectTime ){}
  void operator()( const Event *& toReclaim ){
    if( toReclaim->getReceiveTime() <= collectTime ){
      reclaimer->reclaimEvent( toReclaim );
      toReclaim = 0;      
    }
  }
private:
  SimulationObject *reclaimer;
  const VTime &collectTime;
};

class reclaimEventId : public std::unary_function< const EventId, void> {
public:
  reclaimEventId( SimulationObject *initObject, 
		  const EventId initReclaimId ) 
    : reclaimer( initObject ),
      reclaimId( initReclaimId ){}
  void operator()( const Event *& toCheck ){
    if( toCheck->getEventId() == reclaimId ){
      reclaimer->reclaimEvent( toCheck );
      toCheck = 0;      
    }
  }
private:
  SimulationObject *reclaimer;
  const EventId reclaimId;
};


void
DefaultTimeWarpEventSet::EventContainer::remove( SimulationObject *reclaimer,
						 const EventId toRemoveId ){
  ASSERT( reclaimer != 0 );

  for_each( unprocessedEvents.begin(),
	    unprocessedEvents.end(),
	    reclaimEventId( reclaimer, toRemoveId ) );
  unprocessedEvents.erase( std::remove( unprocessedEvents.begin(),
					unprocessedEvents.end(),
					static_cast<Event *>(0) ),
			   unprocessedEvents.end() );
}

void
DefaultTimeWarpEventSet::EventContainer::rollback( const VTime &rollbackTime ){
  vector<const Event *>::iterator newEnd = remove_if( processedEvents.begin(),
						      processedEvents.end(),
						      receiveTimeGreaterThanEqual( rollbackTime ));
  unprocessedEvents.insert( unprocessedEvents.end(),
			    newEnd,
			    processedEvents.end() );
  processedEvents.erase( newEnd, processedEvents.end() );
  sortStatus = UNSORTED;
}

void
DefaultTimeWarpEventSet::EventContainer::debugDump( ostream &os ) const {
  os << "-------------------------\n";
  os << "EventContainer::debugDump\n";
  os << "Processed Events: ";
  for( vector<const Event *>::const_iterator i = processedEvents.begin();
       i < processedEvents.end();
       i++ ){
    os << (*i)->getReceiveTime() << " ";
  }
  os << "\n";
  os << "sortStatus: " << sortStatus << "\n";
  os << "Unprocessed Events: ";
  for( deque<const Event *>::const_iterator i = unprocessedEvents.begin();
       i < unprocessedEvents.end();
       i++ ){
    os << (*i)->getReceiveTime() << " ";
  }
  os << "-------------------------" << endl;
}


void
DefaultTimeWarpEventSet::EventContainer::garbageCollect( SimulationObject *object,
							 const VTime &collectTime ){
  ASSERT( object != 0 );
  for_each( processedEvents.begin(),
	    processedEvents.end(),
	    reclaimEvent( object, collectTime ) );
  processedEvents.erase( std::remove( processedEvents.begin(),
				      processedEvents.end(),
				      static_cast<Event *>(0) ),
			 processedEvents.end() );
}

const Event *
DefaultTimeWarpEventSet::EventContainer::nextEvent(){
  const Event *retval = 0;
  if( unprocessedEvents.size() > 0 ){
    if( sortStatus == UNSORTED ){
      sortByReceiveTimes();
    }
    retval = unprocessedEvents.front();
  }
  return retval;
}

const Event *
DefaultTimeWarpEventSet::EventContainer::getNextEvent(){
  const Event *retval = nextEvent();
  if( retval != 0 ){
    unprocessedEvents.pop_front();
    processedEvents.push_back( retval );
  }
  return retval;
}

DefaultTimeWarpEventSet::DefaultTimeWarpEventSet( TimeWarpSimulationManager *initSimManager ) :
  mySimulationManager( initSimManager ){}

DefaultTimeWarpEventSet::EventContainer &
DefaultTimeWarpEventSet::getEventContainer( const string &objectName ){
  EventContainer *retval = events.find( objectName );
  if( retval == 0 ){
    retval = new EventContainer();
    events.insert( objectName, retval );
  }
  return *retval;
}

bool
DefaultTimeWarpEventSet::insert( const Event *event ){
  bool retval = false;
  ASSERT( event != 0 );
  SimulationObject *object = mySimulationManager->getObjectHandle( event->getReceiver() );
  ASSERT( object != 0 );
  if( event->getReceiveTime() < object->getSimulationTime() ){
    retval = true;
  }

  getEventContainer( event->getReceiver() ).insert( event );

  return retval;
}

void
DefaultTimeWarpEventSet::handleAntiMessage( SimulationObject *eventFor,
					    const EventId cancelEventId ){
  getEventContainer( eventFor->getName() ).remove( eventFor, cancelEventId );
}

void
DefaultTimeWarpEventSet::remove( const Event *, findMode ){
  abort();
}

const Event *
DefaultTimeWarpEventSet::getEvent( SimulationObject *object ){
  ASSERT( object != 0 );
  return getEventContainer( object->getName() ).getNextEvent();
}

const Event *
DefaultTimeWarpEventSet::getEvent(SimulationObject *object, 
				  const VTime &minimumTime ){
  const Event *retval = 0;
  ASSERT( object != 0 );
  const Event *peeked = peekEvent( object, minimumTime );
  if( peeked != 0 ){
    retval = getEvent( object );
    ASSERT( peeked == retval );
  }
  return retval;
}

const Event *
DefaultTimeWarpEventSet::peekEvent(SimulationObject *object ){
  ASSERT( object != 0 );
  return getEventContainer( object->getName() ).nextEvent();
}

const Event *
DefaultTimeWarpEventSet::peekEvent( SimulationObject *object, 
				    const VTime &minimumTime ){
  ASSERT( object != 0 );
  EventContainer &objectContainer = getEventContainer( object->getName() );
  const Event *retval = objectContainer.nextEvent();
  if( retval != 0 ){
    if( !(retval->getReceiveTime() < minimumTime) ){
      retval = 0;
    }
  }
  return retval;
}

void
DefaultTimeWarpEventSet::garbageCollect( SimulationObject *object,
					 const VTime &collectTime ){
  getEventContainer( object->getName() ).garbageCollect( object, collectTime );
}

void
DefaultTimeWarpEventSet::rollback( SimulationObject *object,
				   const VTime &rollbackTime ){
  getEventContainer( object->getName() ).rollback( rollbackTime );  
}
