//=============================================================================
//   This file is part of VTKEdge. See vtkedge.org for more information.
//
//   Copyright (c) 2008 Kitware, Inc.
//
//   VTKEdge may be used under the terms of the GNU General Public License 
//   version 3 as published by the Free Software Foundation and appearing in 
//   the file LICENSE.txt included in the top level directory of this source
//   code distribution. Alternatively you may (at your option) use any later 
//   version of the GNU General Public License if such license has been 
//   publicly approved by Kitware, Inc. (or its successors, if any).
//
//   VTKEdge is distributed "AS IS" with NO WARRANTY OF ANY KIND, INCLUDING
//   THE WARRANTIES OF DESIGN, MERCHANTABILITY, AND FITNESS FOR A PARTICULAR
//   PURPOSE. See LICENSE.txt for additional details.
//
//   VTKEdge is available under alternative license terms. Please visit
//   vtkedge.org or contact us at kitware@kitware.com for further information.
//
//=============================================================================

#include "vtkKWEClipConvexPolyData.h"

#include "vtkCellArray.h"
#include "vtkMath.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkObjectFactory.h"
#include "vtkPlane.h"
#include "vtkPlaneCollection.h"
#include "vtkPolyData.h"

#include <vtkstd/vector>
#include <vtkstd/set>
#include <vtkstd/algorithm>
#include <vtkstd/iterator>

vtkCxxRevisionMacro(vtkKWEClipConvexPolyData, "$Revision: 385 $");
vtkStandardNewMacro(vtkKWEClipConvexPolyData);

vtkCxxSetObjectMacro(vtkKWEClipConvexPolyData, Planes, vtkPlaneCollection);

class vtkKWECCPDVertex
{
public:
  double Point[3];
};

class vtkKWECCPDPolygon
{
public:
  vtkstd::vector<vtkKWECCPDVertex*> Vertices;
  vtkstd::vector<vtkKWECCPDVertex*> NewVertices;
};


class vtkKWEClipConvexPolyDataInternals
{
public:
  vtkstd::vector<vtkKWECCPDPolygon*> Polygons;
};



// Construct an the hull object with no planes
vtkKWEClipConvexPolyData::vtkKWEClipConvexPolyData()
{
  this->Planes             = NULL;
  
  this->Internal = new vtkKWEClipConvexPolyDataInternals;
  
}

// Destructor for a hull object - remove the planes if necessary
vtkKWEClipConvexPolyData::~vtkKWEClipConvexPolyData()
{
  this->SetPlanes(NULL);
  this->ClearInternals();
  delete this->Internal;
}

// ----------------------------------------------------------------------------
// Description:
// Redefines this method, as this filter depends on time of its components
// (planes)
unsigned long int vtkKWEClipConvexPolyData::GetMTime()
{
  unsigned long int result=Superclass::GetMTime();
  if(this->Planes!=0)
    {
    unsigned long int planesTime=this->Planes->GetMTime();
    if(planesTime>result)
      {
      result=planesTime;
      }
    }
  return result;
}

void vtkKWEClipConvexPolyData::ClearInternals()
{
  unsigned int j;
  for(unsigned int i=0; i<this->Internal->Polygons.size(); i++)
    {
    for (j=0; j<this->Internal->Polygons[i]->Vertices.size(); j++)
      {
      delete this->Internal->Polygons[i]->Vertices[j];
      }
    this->Internal->Polygons[i]->Vertices.clear();

    for (j=0; j<this->Internal->Polygons[i]->NewVertices.size(); j++)
      {
      delete this->Internal->Polygons[i]->NewVertices[j];
      }
    this->Internal->Polygons[i]->NewVertices.clear();

    
    delete this->Internal->Polygons[i];
    }
  this->Internal->Polygons.clear(); 
}

void vtkKWEClipConvexPolyData::ClearNewVertices()
{
  for(unsigned int i=0; i<this->Internal->Polygons.size(); i++)
    {
    for (unsigned int j=0; j<this->Internal->Polygons[i]->NewVertices.size(); j++)
      {
      delete this->Internal->Polygons[i]->NewVertices[j];
      }
    this->Internal->Polygons[i]->NewVertices.clear();
    }
}

void vtkKWEClipConvexPolyData::RemoveEmptyPolygons()
{
  int done = 0;
  
  while (!done )
    {
    done = 1;
    for(unsigned int i=0; i<this->Internal->Polygons.size(); i++)
      {
      if ( this->Internal->Polygons[i]->Vertices.size() == 0 )
        {
        vtkstd::vector<vtkKWECCPDPolygon*>::iterator where =
          vtkstd::find(this->Internal->Polygons.begin(),
               this->Internal->Polygons.end(),
               this->Internal->Polygons[i]);
        if ( where != this->Internal->Polygons.end() )
          {
          delete this->Internal->Polygons[i];
          this->Internal->Polygons.erase(where);
          done = 0;
          break;
          }
        }
      }
    }
}


// Create the n-sided convex hull from the input geometry according to the
// set of planes.
int vtkKWEClipConvexPolyData::RequestData(
  vtkInformation *vtkNotUsed(request),
  vtkInformationVector **inputVector,
  vtkInformationVector *outputVector)
{
  // Pre-conditions
  if(this->Planes==0)
    {
    vtkErrorMacro("plane collection is null");
    return 0;
    }
  if(this->Planes->GetNumberOfItems()==0)
    {
    vtkErrorMacro("plane collection is empty");
    return 0;
    }
  
  // get the info objects
  vtkInformation *inInfo = inputVector[0]->GetInformationObject(0);
  vtkInformation *outInfo = outputVector->GetInformationObject(0);

  // get the input and ouptut
  vtkPolyData *input = vtkPolyData::SafeDownCast(
    inInfo->Get(vtkDataObject::DATA_OBJECT()));
  vtkPolyData *output = vtkPolyData::SafeDownCast(
    outInfo->Get(vtkDataObject::DATA_OBJECT()));
  
  vtkCellArray *polys = input->GetPolys();
  vtkPoints *points = input->GetPoints();

  // Compute tolerance to be 0.00001 of diagonal
  double min[3] = {VTK_FLOAT_MAX, VTK_FLOAT_MAX, VTK_FLOAT_MAX};
  double max[3] = {VTK_FLOAT_MIN, VTK_FLOAT_MIN, VTK_FLOAT_MIN};
  
  int i, j;
  double tolerance;
  for ( i = 0; i < points->GetNumberOfPoints(); i++ )
    {
    double pt[3];
    points->GetPoint(i, pt);
    for ( j = 0; j < 3; j++ )
      {
      min[j] = (pt[j] < min[j])?(pt[j]):(min[j]);
      max[j] = (pt[j] > max[j])?(pt[j]):(max[j]);
      }
    }
  tolerance = sqrt(vtkMath::Distance2BetweenPoints(min,max))*0.00001;

  vtkIdType npts;
  vtkIdType *pts;
  polys->InitTraversal();
  while ( polys->GetNextCell(npts, pts) )
    {
    vtkKWECCPDPolygon *polygon = new vtkKWECCPDPolygon;
    for ( i = 0; i < npts; i++ )
      {
      vtkKWECCPDVertex *v = new vtkKWECCPDVertex;
      points->GetPoint(pts[i], v->Point);
      polygon->Vertices.push_back(v);
      }
    this->Internal->Polygons.push_back(polygon);
    }

  this->Planes->InitTraversal();
  vtkPlane *plane;

  while ( (plane = this->Planes->GetNextItem()) )
    {
    if ( !this->HasDegeneracies( plane ) )
      {
      this->ClipWithPlane( plane, tolerance );
      }
    }

  // Create a new set of points and polygons into which the results will
  // be stored
  vtkPoints *outPoints = vtkPoints::New();
  vtkCellArray *outPolys  = vtkCellArray::New();

  for ( i = 0; i < static_cast<int>(this->Internal->Polygons.size()); i++ )
    {
    vtkIdType numPoints = this->Internal->Polygons[i]->Vertices.size();
    vtkIdType *polyPts = new vtkIdType[numPoints];
    for ( j = 0; j < numPoints; j++ )
      {
      polyPts[j] = outPoints->InsertNextPoint(
        this->Internal->Polygons[i]->Vertices[j]->Point );
      }
    outPolys->InsertNextCell( numPoints, polyPts );
    delete [] polyPts;
    }

  // Set the output vertices and polygons
  output->SetPoints( outPoints );
  output->SetPolys( outPolys );

  // Delete the temporary storage
  outPoints->Delete();
  outPolys->Delete();

  this->ClearInternals();
  
  return 1;
}

void vtkKWEClipConvexPolyData::ClipWithPlane( vtkPlane *plane, double tolerance )
{
  double origin[3];
  double normal[4];
  
  plane->GetOrigin( origin );
  plane->GetNormal( normal );
  
  vtkMath::Normalize(normal);

  normal[3] = -(origin[0]*normal[0] +
                origin[1]*normal[1] +
                origin[2]*normal[2]);
  
  int numNewPoints = 0;
  unsigned int i;
  size_t j;
  // For each polygon
  for ( i = 0; i < this->Internal->Polygons.size(); i++ )
    {
    // the new polygons
    vtkKWECCPDPolygon *newPoly = new vtkKWECCPDPolygon;
    
    // For each vertex
    size_t numVertices = this->Internal->Polygons[i]->Vertices.size();
    for ( j = 0; j < numVertices; j++ )
      {
      double *p1 = this->Internal->Polygons[i]->Vertices[j]->Point;
      double *p2 = this->Internal->Polygons[i]->Vertices[(j+1)%numVertices]->Point;
      
      double p1D = p1[0]*normal[0] + p1[1]*normal[1] + p1[2]*normal[2] + normal[3];
      double p2D = p2[0]*normal[0] + p2[1]*normal[1] + p2[2]*normal[2] + normal[3];

      // Add p1 in if it is not clipped
      if ( p1D > 0 )
        {
        vtkKWECCPDVertex *v = new vtkKWECCPDVertex;
        v->Point[0] = p1[0];
        v->Point[1] = p1[1];
        v->Point[2] = p1[2];
        newPoly->Vertices.push_back(v);
        }
      
      // If the plane clips this edge - find the crossing point
      if ( p1D*p2D <= 0 )
        {
        double w = -p1D / (p2D - p1D);
        
        vtkKWECCPDVertex *v = new vtkKWECCPDVertex;
        v->Point[0] = p1[0] + w * (p2[0]-p1[0]);
        v->Point[1] = p1[1] + w * (p2[1]-p1[1]);
        v->Point[2] = p1[2] + w * (p2[2]-p1[2]);
        newPoly->Vertices.push_back(v);

        v = new vtkKWECCPDVertex;
        v->Point[0] = p1[0] + w * (p2[0]-p1[0]);
        v->Point[1] = p1[1] + w * (p2[1]-p1[1]);
        v->Point[2] = p1[2] + w * (p2[2]-p1[2]);
        this->Internal->Polygons[i]->NewVertices.push_back(v);
        numNewPoints++;
        }
      }
    
    // Remove the current polygon
    for ( j = 0; j < numVertices; j++ )
      {
      delete this->Internal->Polygons[i]->Vertices[j];
      }
    this->Internal->Polygons[i]->Vertices.clear();
    
    // copy in the new polygon
    numVertices = newPoly->Vertices.size();
    
    if ( numVertices > 0 )
      {
      for ( j = 0; j < numVertices; j++ )
        {
        this->Internal->Polygons[i]->Vertices.push_back( newPoly->Vertices[j] );
        }
      newPoly->Vertices.clear();
      }
    delete newPoly;
    }
  
  if ( numNewPoints )
    {
    if ( numNewPoints < 6 )
      {
      cout << "Failure - not enough new points" << endl;
      return;
      }
  
    // Check that all new arrays contain exactly 0 or 2 points
    for ( i = 0; i < this->Internal->Polygons.size(); i++ )
      {
      if ( this->Internal->Polygons[i]->NewVertices.size() != 0 &&
           this->Internal->Polygons[i]->NewVertices.size() != 2 )
        {
        cout << "Horrible error - we have " <<
          this->Internal->Polygons[i]->NewVertices.size() << " crossing points" << endl;
        return;
        }
      }
    
    // Find the first polygon with a new point
    int idx = -1;
    
    for ( i = 0; i < this->Internal->Polygons.size(); i++ )
      {
      if ( this->Internal->Polygons[i]->NewVertices.size() > 0 )
        {
        idx = i;
        break;
        }
      }
    
    if ( idx < 0 )
      {
      cout << "Couldn't find any new vertices!" << endl;
      return;
      }
    
    // the new polygon
    vtkKWECCPDPolygon *newPoly = new vtkKWECCPDPolygon;
    vtkKWECCPDVertex *v = new vtkKWECCPDVertex;
    v->Point[0] = this->Internal->Polygons[idx]->NewVertices[0]->Point[0];
    v->Point[1] = this->Internal->Polygons[idx]->NewVertices[0]->Point[1];
    v->Point[2] = this->Internal->Polygons[idx]->NewVertices[0]->Point[2];
    newPoly->Vertices.push_back(v);
    
    v = new vtkKWECCPDVertex;
    v->Point[0] = this->Internal->Polygons[idx]->NewVertices[1]->Point[0];
    v->Point[1] = this->Internal->Polygons[idx]->NewVertices[1]->Point[1];
    v->Point[2] = this->Internal->Polygons[idx]->NewVertices[1]->Point[2];
    newPoly->Vertices.push_back(v);
    
    double lastPoint[3];
    lastPoint[0] = this->Internal->Polygons[idx]->NewVertices[1]->Point[0];
    lastPoint[1] = this->Internal->Polygons[idx]->NewVertices[1]->Point[1];
    lastPoint[2] = this->Internal->Polygons[idx]->NewVertices[1]->Point[2];
    
    int lastPointIdx = idx;
    int subIdx;
    
    while ( static_cast<int>(newPoly->Vertices.size()) < numNewPoints / 2 )
      {
      // Find the index of a polygon match the lastPoint but
      // not the lastPointIdx
      subIdx = -1;
      for ( i = 0; i < this->Internal->Polygons.size(); i++ )
        {
          if ( static_cast<int>(i) != lastPointIdx && this->Internal->Polygons[i]->NewVertices.size() > 0 )
          {
          for ( j = 0; j < 2; j++ )
            {
            if ( vtkMath::Distance2BetweenPoints(lastPoint,
                   this->Internal->Polygons[i]->NewVertices[j]->Point) < tolerance )
              {
              idx = i;
              subIdx = static_cast<int>(j);
              break;
              }
            }
          if ( subIdx >= 0 )
            {
            break;
            }
          }
        }
      
      v = new vtkKWECCPDVertex;
      v->Point[0] = this->Internal->Polygons[idx]->NewVertices[(subIdx+1)%2]->Point[0];
      v->Point[1] = this->Internal->Polygons[idx]->NewVertices[(subIdx+1)%2]->Point[1];
      v->Point[2] = this->Internal->Polygons[idx]->NewVertices[(subIdx+1)%2]->Point[2];
      newPoly->Vertices.push_back(v);

      lastPoint[0] = this->Internal->Polygons[idx]->NewVertices[(subIdx+1)%2]->Point[0];
      lastPoint[1] = this->Internal->Polygons[idx]->NewVertices[(subIdx+1)%2]->Point[1];
      lastPoint[2] = this->Internal->Polygons[idx]->NewVertices[(subIdx+1)%2]->Point[2];
      lastPointIdx = idx;
      }
    // check to see that the polygon vertices are in the right order.
    // cross product of p1p2 and p3p2 should point in the same direction
    // as the plane normal. Otherwise, reverse the order
    double *p1 = newPoly->Vertices[0]->Point;
    double *p2 = newPoly->Vertices[1]->Point;
    double *p3 = newPoly->Vertices[2]->Point;
    double v1[3];
    double v2[3];
    double cross[3];
    v1[0] = p1[0] - p2[0];
    v1[1] = p1[1] - p2[1];
    v1[2] = p1[2] - p2[2];
    v2[0] = p3[0] - p2[0];
    v2[1] = p3[1] - p2[1];
    v2[2] = p3[2] - p2[2];
    vtkMath::Cross(v1,v2,cross);
    vtkMath::Normalize(cross);
    if ( vtkMath::Dot(cross,normal) < 0 )
      {
      vtkstd::reverse(newPoly->Vertices.begin(), newPoly->Vertices.end());
      }
    this->Internal->Polygons.push_back(newPoly);
    } 
  this->RemoveEmptyPolygons();
  this->ClearNewVertices();
}

int vtkKWEClipConvexPolyData::HasDegeneracies( vtkPlane *plane )
{
  double origin[3];
  double normal[4];
  
  plane->GetOrigin( origin );
  plane->GetNormal( normal );
  
  normal[3] = -(origin[0]*normal[0] +
                origin[1]*normal[1] +
                origin[2]*normal[2]);
  
  unsigned int i;
  size_t j;
  // For each polygon
  int totalNumNewVertices = 0;
  for ( i = 0; i < this->Internal->Polygons.size(); i++ )
    {
    // For each vertex
    size_t numVertices = this->Internal->Polygons[i]->Vertices.size();
    int numNewVertices = 0;
    for ( j = 0; j < numVertices; j++ )
      {
      double *p1 = this->Internal->Polygons[i]->Vertices[j]->Point;
      double *p2 = this->Internal->Polygons[i]->Vertices[(j+1)%numVertices]->Point;
      
      double p1D = p1[0]*normal[0] + p1[1]*normal[1] + p1[2]*normal[2] + normal[3];
      double p2D = p2[0]*normal[0] + p2[1]*normal[1] + p2[2]*normal[2] + normal[3];

      // If the plane clips this edge - find the crossing point
      if ( p1D*p2D <= 0 )
        {
        numNewVertices++;
        }
      }
    if ( numNewVertices != 0 && numNewVertices != 2)
      {
      return 1;
      }    
    totalNumNewVertices += numNewVertices;
    }  
  
  if ( totalNumNewVertices < 6 )
    {
    return 1;
    }
  return 0;
}

// Print the object
void vtkKWEClipConvexPolyData::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
}
