
// permstart.cpp
// Copyright (c) 1998-2010 by The VoxBo Development Team

// This file is part of VoxBo
// 
// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
// 
// original version written by Kosh Banerjee

#include <iostream>
#include "perm.h"

using namespace std;

void usage(const unsigned short exitValue, char *progName);

gsl_rng *theRNG = NULL;

int main(int argc, char *argv[]) {
  SEGV_HANDLER
  GSL_ERROR_HANDLER_OFF

  if (argc == 1)
    usage(1, argv[0]);

  string matrixStemName;
  string permDir;
  short method = -1;

/*********************************************************************
* Now processing the command line options.                           *
*                                                                    *
* -h ==> Display usage information.                                  *
* -m ==> Specifies the matrix stem name.                             *
* -d ==> Permutation directory.                                      *
* -t ==> Permutation type.                                           *
* -v ==> Print out the gobal VoxBo version number.                   *
*                                                                    *
* VARIABLES:                                                         *
* printHelp - a flag, used to determine if the "-h" command line     *
*             option was used or not.                                *
* printVersion - a flag, used to determine if the "-v" command line  *
*                option was used or not.                             *
*********************************************************************/
  GLMInfo glmi;
  arghandler a;
  a.setArgs("-h", "--help", 0);
  a.setArgs("-m", "--matrixstemname", 1);
  a.setArgs("-d", "--permdir", 1);
  a.setArgs("-t", "--method", 1);
  a.setArgs("-v", "--version", 0);
  a.parseArgs(argc, argv);
  string errstring = a.badArg();
  if (errstring.size()) {
     errstring = "[E] unknown flag: " + errstring;
     printErrorMsg(VB_ERROR, errstring.c_str());
     exit(-1);
  }
  if (a.flagPresent("-h"))
     usage(0, argv[0]);
  matrixStemName = a.getFlaggedArgs("-m")[0];
  permDir = a.getFlaggedArgs("-d")[0];
  method = atoi(a.getFlaggedArgs("-t")[0].c_str());
  if (a.flagPresent("-v"))
    printf("\nVoxBo v%s\n",vbversion.c_str());
  if (matrixStemName.size() == 0) {
    ostringstream errorMsg;
    errorMsg << "Must specify the matrix stem name, using the \"-m\" option.";
    printErrorMsgAndExit(VB_ERROR, errorMsg, 1);
  } 
  if (permDir.size() == 0) {
    ostringstream errorMsg;
    errorMsg << "Must specify the permutation directory name, using the \"-d\" option.";
    printErrorMsgAndExit(VB_ERROR, errorMsg, 1);
  } 
  if ((method < 0) || (method > 2)) {
    ostringstream errorMsg;
    errorMsg << "The \"-t\" argument must be 1 or 2, not [" << method << "].";
    printErrorMsgAndExit(VB_ERROR, errorMsg, 1);
  }
  permclass pc;
  glmi.setup(matrixStemName);
  pc.AddMatrixStemName(glmi.stemname);
  pc.AddPermDir(permDir);
  pc.AddMethod(method);
  int err = 0;
  err = permStart(pc);
  if (err)
     switch(err) {
         case 200:
          printErrorMsg(VB_ERROR, "permstart: no stem name specified.\n");
          break;
         case 201:
          printErrorMsg(VB_ERROR, "permstart: no perm directory specified.\n");
          return -1;
         case 202:
          printErrorMsg(VB_ERROR, "permstart: prm file not valid.\n");
          return -1;
         case 203:
          printErrorMsg(VB_ERROR, "permstart: g matrix file not readable.\n");
          return -1;
         case 204:
          printErrorMsg(VB_ERROR, "permstart: MAT1 header not valid.\n");
          break;
         case 205:
          printErrorMsg(VB_ERROR, "permstart: failed to allocate permutation.\n");
          return -1;
         case 206:
          printErrorMsg(VB_ERROR, "permstart: bad permuation type.\n");
          return -1;
         case 207:
          printErrorMsg(VB_ERROR, "permstart: failed to make subdirectory.\n");
          return -1;
         case 208:
          printErrorMsg(VB_ERROR, "permstart: failed to make permutation matrix.\n");
          break;
         case 209:
          printErrorMsg(VB_ERROR, "permstart: permutation matrix is not valid.\n");
          return -1;
         case 210:
          printErrorMsg(VB_ERROR, "permstart: failed to write permutation matrix file.\n");
          return -1;
         default:
          printErrorMsg(VB_ERROR, "permstart: unknown error.\n");
          return -1;
     }
  return 0;
} 

void usage(const unsigned short exitValue, char *progName) {
  printf("\nVoxBo permstart (v%s)\n",vbversion.c_str());
  printf("summary: ");
  printf(" Permutation start.\n");
  printf("usage:\n");
  printf(" permstart -h -m[matrix stem name] -d[permutation directory] -t[permutation type] -v\n");
  printf("flags:\n");
  printf(" -h                        Print usage information. Optional.\n");
  printf(" -m <matrix stem name>                       Specify the matrix stem name. Required.\n");
  printf(" -d                        Permutation directory name. Required.\n");
  printf(" -t                        Permutation type. 1 means regular permutation.\n");
  printf("                           2 means sign permutation. Required.\n");
  printf(" -v                        Global VoxBo version number. Optional.\n\n");
  exit(-1);
} 
