(**
   Implements horizontal and vertical scrollers.

  TODO
  * Support for prior, next, home, end
**)

MODULE VOScroller;

(*
    Implements a scroll gadget.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT B  := VOButton,
       D  := VODisplay,
       E  := VOEvent,
       F  := VOFrame,
       G  := VOGUIObject,
       O  := VOObject,
       P  := VOPrefs,
       U  := VOUtil,
       V  := VOVecImage,
       VM := VOValue;


CONST
(*  repeatTimeOut = 75000;*) (* Time between button repeat *)

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the cycle is stored.
  **)

  PrefsDesc* = RECORD (P.PrefsDesc)
                 frame*,
                 knob*,
                 up*,
                 down*,
                 left*,
                 right* : LONGINT;
               END;

  Scroller*     = POINTER TO ScrollerDesc;
  ScrollerDesc* = RECORD (G.GadgetDesc)
                    prefs      : Prefs;
                    box        : F.Frame;
                    minus,plus : B.Button;
                    knob       : V.VecImage;
                    offset     : LONGINT;
                    corr       : LONGINT;
                    top,
                    vis,
                    total      : VM.ValueModel;
                    vert,
                    selected   : BOOLEAN;
                    timeOut    : D.TimeOut;
                  END;


CONST
  scrollPatWidth  = 16;
  scrollPatHeight = 4;

VAR
  prefs*          : Prefs;
  scrollerPattern : ARRAY 8 OF CHAR;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DIn;
    p.knob:=V.knob;
    p.up:=V.arrowUp;
    p.down:=V.arrowDown;
    p.left:=V.arrowLeft;
    p.right:=V.arrowRight;
  END Init;

  PROCEDURE (s : Scroller) Init*;

  BEGIN
    s.Init^;

    s.prefs:=prefs;

    INCL(s.flags,G.canFocus);

    s.timeOut:=NIL;
    s.vert:=TRUE;

    s.top:=NIL;
    s.vis:=NIL;
    s.total:=NIL;

    s.selected:=FALSE;

    s.offset:=0;

    s.corr:=0;

    NEW(s.box);
    s.box.Init;
    s.box.SetFlags({G.horizontalFlex,G.verticalFlex});

    NEW(s.minus);
    s.minus.Init;
    s.minus.SetType(B.image);
    s.minus.SetFlags({G.horizontalFlex,G.verticalFlex});
    s.minus.SetPulse(TRUE);
    s.minus.Forward(B.pressedMsg,s);

    NEW(s.plus);
    s.plus.Init;
    s.plus.SetType(B.image);
    s.plus.SetFlags({G.horizontalFlex,G.verticalFlex});
    s.plus.SetPulse(TRUE);
    s.plus.Forward(B.pressedMsg,s);

    NEW(s.knob);
    s.knob.Init;
    s.knob.SetFlags({G.horizontalFlex,G.verticalFlex});
  END Init;

  PROCEDURE (s : Scroller) Set*(vert : BOOLEAN);

  BEGIN
    s.vert:=vert;
  END Set;

  PROCEDURE (s : Scroller) SetOffset*(offset : LONGINT);

  BEGIN
    s.corr:=offset;
  END SetOffset;

  PROCEDURE (s : Scroller) SetModel*(top, visible, total : VM.ValueModel);

  BEGIN
    IF s.top#NIL THEN
      s.UnattachModel(s.top);
    END;
    s.top:=top;
    s.AttachModel(top);

    IF s.vis#NIL THEN
      s.UnattachModel(s.vis);
    END;
    s.vis:=visible;
    s.AttachModel(s.vis);

    IF s.total#NIL THEN
      s.UnattachModel(s.total);
    END;
    s.total:=total;
    s.AttachModel(s.total);
  END SetModel;

  PROCEDURE (s : Scroller) CalcSize*(display : D.Display);

  VAR
    innerWidth : LONGINT;
    image      : V.VecImage;

  BEGIN
    NEW(image);
    image.Init;
    image.SetFlags({G.horizontalFlex,G.verticalFlex});
    IF s.vert THEN
      image.Set(s.prefs.up);
    ELSE
      image.Set(s.prefs.left);
    END;
    s.minus.SetImage(image);

    NEW(image);
    image.Init;
    image.SetFlags({G.horizontalFlex,G.verticalFlex});
    IF s.vert THEN
      image.Set(s.prefs.down);
    ELSE
      image.Set(s.prefs.right);
    END;
    s.plus.SetImage(image);

    s.box.SetInternalFrame(s.prefs.frame);
    s.box.CalcSize(display);

    s.knob.Set(s.prefs.knob);
    s.knob.CalcSize(display);

    s.minus.CalcSize(display);
    s.plus.CalcSize(display);

    IF s.vert THEN
      innerWidth:=G.MaxLong(s.knob.oWidth,s.minus.oWidth); (* Assume: minus.width=plus.width *)
      s.width:=s.box.leftBorder+innerWidth+s.box.rightBorder;
      s.height:=s.box.topBorder+innerWidth+s.box.bottomBorder;
      INC(s.height,s.minus.oHeight+s.plus.oHeight);
    ELSE
      innerWidth:=G.MaxLong(s.knob.oHeight,s.minus.oHeight); (* Assume: minus.height=minus.height *)
      s.width:=s.box.leftBorder+innerWidth+s.box.rightBorder;
      s.height:=s.box.topBorder+innerWidth+s.box.bottomBorder;
      INC(s.width,s.minus.oWidth+s.plus.oWidth);
    END;

    s.minWidth:=s.width;
    s.minHeight:=s.height;

    s.CalcSize^(display);
  END CalcSize;

  PROCEDURE (s : Scroller) GetFocus*(event : E.Event):G.Object;

  BEGIN
    IF ~s.visible OR s.disabled THEN
      RETURN NIL;
    END;

    IF (s.top=NIL) OR s.top.IsNull()
    OR (s.vis=NIL) OR s.vis.IsNull()
    OR (s.total=NIL) OR s.total.IsNull()
    OR (s.vis.GetLongint()>=s.total.GetLongint()) THEN
      RETURN NIL;
    END;


    WITH event : E.MouseEvent DO

      (* Hmm, there could be other events, that trigger the scroller buttons *)

      IF s.minus.PointIsIn(event.x,event.y) THEN
        RETURN s.minus.GetFocus(event);
      ELSIF s.plus.PointIsIn(event.x,event.y) THEN
        RETURN s.plus.GetFocus(event);
      END;

      IF (event.type=E.mouseDown) & s.box.PointIsIn(event.x,event.y)
      & (event.qualifier={}) & (event.button=E.button1) THEN
        IF s.knob.PointIsIn(event.x,event.y) THEN
          IF s.vert THEN
            s.offset:=event.y-s.knob.y;
          ELSE
            s.offset:=event.x-s.knob.x;
          END;
          s.selected:=TRUE;
          s.Redraw;
          RETURN s;
        ELSE
          IF s.vert THEN
            IF event.y<s.knob.y THEN
              IF s.top.GetLongint()+s.corr>s.vis.GetLongint() THEN
                s.top.Sub(s.vis.GetLongint());
              ELSE
                s.top.SetLongint(1-s.corr);
              END;
              RETURN s;
            ELSIF event.y>s.knob.y+s.knob.height THEN
              IF s.top.GetLongint()<s.total.GetLongint()-s.vis.GetLongint()+1 THEN
                s.top.Add(s.vis.GetLongint());
              ELSE
                s.top.SetLongint(s.total.GetLongint()+s.corr-s.vis.GetLongint()+1);
              END;
              RETURN s;
              END;
          ELSE
            IF event.x<s.knob.x THEN
              IF s.top.GetLongint()+s.corr>s.vis.GetLongint() THEN
                s.top.Sub(s.vis.GetLongint());
              ELSE
                s.top.SetLongint(1-s.corr);
              END;
              RETURN s;
            ELSIF event.x>s.knob.x+s.knob.width THEN
              IF s.top.GetLongint()<s.total.GetLongint()-s.vis.GetLongint()+1 THEN
                s.top.Add(s.vis.GetLongint());
              ELSE
                s.top.SetLongint(s.total.GetLongint()+s.corr-s.vis.GetLongint()+1);
              END;
              RETURN s;
            END;
          END;
        END;
      END;
    ELSE
    END;
    RETURN NIL;
  END GetFocus;

  PROCEDURE (s : Scroller) HandleEvent*(event : E.Event):BOOLEAN;

  VAR
    new      : LONGINT;

  BEGIN
    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        s.selected:=FALSE;
        IF s.timeOut#NIL THEN
          s.display.RemoveTimeOut(s.timeOut);
        END;
        RETURN TRUE;
      END;
    | event : E.MotionEvent DO
      IF s.selected THEN
        IF s.vert THEN
          new:=s.top.GetLongint()+s.corr
                 +((event.y-s.knob.y-s.offset)*(s.total.GetLongint()+s.corr))
                    DIV (s.box.height-s.box.topBorder-s.box.bottomBorder)
                 +1;
          IF (new>0) & (new<=s.total.GetLongint()+s.corr-s.vis.GetLongint()+1) THEN
            s.top.SetLongint(new);
          ELSIF new<=0 THEN
            s.top.SetLongint(1-s.corr);
          ELSE
            s.top.SetLongint(s.total.GetLongint()+s.corr-s.vis.GetLongint()+1);
          END;
        ELSE
          new:=s.top.GetLongint()+s.corr
                 +((event.x-s.knob.x-s.offset)*(s.total.GetLongint()+s.corr))
                    DIV (s.box.width-s.box.leftBorder-s.box.rightBorder)
                 +1;
          IF (new>0) & (new<=s.total.GetLongint()+s.corr-s.vis.GetLongint()+1) THEN
            s.top.SetLongint(new);
          ELSIF new<=0 THEN
            s.top.SetLongint(1-s.corr);
          ELSE
            s.top.SetLongint(s.total.GetLongint()+s.corr-s.vis.GetLongint()+1);
          END;
        END;
      END;
    ELSE
    END;

    RETURN FALSE;
  END HandleEvent;

  PROCEDURE (s : Scroller) HandleFocusEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    keysym   : LONGINT;

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      IF s.vert & (keysym=E.up) THEN
        IF s.top.GetLongint()+s.corr>1 THEN
          s.top.Dec;
        END;
      ELSIF s.vert & (keysym=E.down) THEN
        IF s.top.GetLongint()+s.corr<=s.total.GetLongint()+s.corr-s.vis.GetLongint() THEN
          s.top.Inc;
        END;
      ELSIF ~s.vert & (keysym=E.left) THEN
        IF s.top.GetLongint()+s.corr>1 THEN
          s.top.Dec;
        END;
      ELSIF ~s.vert & (keysym=E.left) THEN
        IF s.top.GetLongint()+s.corr<=s.total.GetLongint()+s.corr-s.vis.GetLongint() THEN
          s.top.Inc;
        END;
      ELSE
        RETURN FALSE;
      END;
      RETURN TRUE;
    END;
    RETURN FALSE;
  END HandleFocusEvent;

  PROCEDURE (s : Scroller) Receive*(message : O.Message);

  BEGIN
    WITH
      message : B.PressedMsg DO
      IF message.source=s.minus THEN
        IF s.top.GetLongint()+s.corr>1 THEN
          s.top.Dec;
        END;
      ELSE
        IF s.top.GetLongint()+s.corr<=s.total.GetLongint()+s.corr-s.vis.GetLongint() THEN
          s.top.Inc;
        END;
      END;
    ELSE
      s.Receive^(message);
    END;
  END Receive;

  PROCEDURE (s : Scroller) DrawKnob;

  VAR
    kSize,
    kStart,
    bSize   : LONGINT;

  BEGIN
    IF s.vert THEN
      bSize:=s.box.height-s.box.topBorder-s.box.bottomBorder-s.minus.oHeight-s.plus.oHeight;
    ELSE
      bSize:=s.box.width-s.box.leftBorder-s.box.rightBorder-s.minus.oWidth-s.plus.oWidth;
    END;

    IF (s.top#NIL) & ~s.top.IsNull()
     & (s.vis#NIL) & ~s.vis.IsNull()
     & (s.total#NIL) & ~s.total.IsNull()
     & ~(s.vis.GetLongint()>=s.total.GetLongint()) THEN

      IF s.total.GetLongint()+s.corr=0 THEN
        kSize:=bSize;
        kStart:=0;
      ELSE
        kSize:=U.RoundDiv((bSize*s.vis.GetLongint()),s.total.GetLongint()+s.corr);
        kStart:=U.RoundDiv(bSize*(s.top.GetLongint()+s.corr-1),s.total.GetLongint()+s.corr);
      END;

      IF s.vert THEN
        IF kSize<s.knob.minWidth THEN
          kSize:=s.knob.minWidth;
        END;
      ELSE
        IF kSize<s.knob.minHeight THEN
          kSize:=s.knob.minHeight;
        END;
      END;

      IF kSize>bSize THEN
        kSize:=bSize;
      END;

      IF kStart+kSize>bSize THEN
        kStart:=bSize-kSize;
      END;

(*      s.knob.Hide; (* Does not work :-| *)*)

      IF s.vert THEN
        s.knob.Resize(s.width-s.box.leftBorder-s.box.rightBorder,kSize);
        s.knob.Draw(s.box.x+s.box.leftBorder,s.box.y+s.box.topBorder+s.minus.oHeight+kStart,s.draw);
      ELSE
        s.knob.Resize(kSize,s.height-s.box.topBorder-s.box.bottomBorder);
        s.knob.Draw(s.box.x+s.box.leftBorder+s.minus.oWidth+kStart,s.box.y+s.box.topBorder,s.draw);
      END;

      s.draw.PushForeground(D.halfShineColor);
      s.draw.PushBackground(D.shineColor);
      s.draw.PushPattern(scrollerPattern,scrollPatWidth,scrollPatHeight,D.fbPattern);
      IF s.vert THEN
        s.draw.FillRectangle(s.box.x+s.box.leftBorder,
                            s.box.y+s.box.topBorder+s.minus.oHeight,
                            s.box.oWidth-s.box.leftBorder-s.box.rightBorder,
                            s.knob.y-(s.box.y+s.box.topBorder+s.minus.oHeight));

        s.draw.FillRectangle(s.box.x+s.box.leftBorder,
                            s.knob.y+s.knob.oHeight,
                            s.box.oWidth-s.box.leftBorder-s.box.rightBorder,
                            bSize-s.knob.oHeight-kStart);
      ELSE
        s.draw.FillRectangle(s.box.x+s.box.leftBorder+s.minus.oWidth,
                            s.box.y+s.box.topBorder,
                            s.knob.x-(s.box.x+s.box.leftBorder+s.minus.oWidth),
                            s.box.oHeight-s.box.topBorder-s.box.bottomBorder);

        s.draw.FillRectangle(s.knob.x+s.knob.oWidth,
                            s.box.y+s.box.topBorder,
                            bSize-s.knob.oWidth-kStart,
                            s.box.oHeight-s.box.topBorder-s.box.bottomBorder);
      END;
      s.draw.PopPattern;
      s.draw.PopForeground;
      s.draw.PopBackground;

      IF s.top.GetLongint()+s.corr<=1 THEN
        s.minus.Disable(TRUE);
      ELSE
        s.minus.Disable(FALSE);
      END;
      IF s.top.GetLongint()+s.corr+s.vis.GetLongint()-1>=s.total.GetLongint() THEN
        s.plus.Disable(TRUE);
      ELSE
        s.plus.Disable(FALSE);
      END;

    ELSE (* no models *)
      s.draw.PushForeground(D.halfShineColor);
      s.draw.PushBackground(D.shineColor);
      s.draw.PushPattern(scrollerPattern,scrollPatWidth,scrollPatHeight,D.fbPattern);
      IF s.vert THEN
        s.draw.FillRectangle(s.box.x+s.box.leftBorder,
                            s.box.y+s.box.topBorder+s.plus.oHeight,
                            s.box.width-s.box.leftBorder-s.box.rightBorder,
                            bSize);
      ELSE
        s.draw.FillRectangle(s.box.x+s.box.leftBorder+s.plus.oWidth,
                            s.box.y+s.box.topBorder,
                            bSize,
                            s.box.height-s.box.topBorder-s.box.bottomBorder);
      END;
      s.draw.PopPattern;
      s.draw.PopForeground;
      s.draw.PopBackground;

      s.plus.Disable(TRUE);
      s.minus.Disable(TRUE);
    END;
  END DrawKnob;

  PROCEDURE (s : Scroller) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    s.Draw^(x,y,draw);

    s.box.Resize(s.width,s.height);
    s.box.Draw(s.x,s.y,draw);

    IF s.vert THEN

      s.minus.Resize(s.width-s.box.leftBorder-s.box.rightBorder,-1);
      s.plus.Resize(s.width-s.box.leftBorder-s.box.rightBorder,-1);

      s.minus.Draw(s.x+(s.width-s.minus.oWidth) DIV 2,
                   s.y+s.box.topBorder,draw);
      s.plus.Draw(s.x+(s.width-s.plus.oWidth) DIV 2,
                  s.y+s.height-s.box.bottomBorder-s.plus.oHeight,draw);
    ELSE
      s.minus.Resize(-1,s.height-s.box.topBorder-s.box.bottomBorder);
      s.plus.Resize(-1,s.height-s.box.topBorder-s.box.bottomBorder);

      s.minus.Draw(s.x+s.box.leftBorder,
                   s.y+(s.height-s.minus.oHeight) DIV 2, draw);
      s.plus.Draw(s.x+s.width-s.box.rightBorder-s.plus.oWidth,
                  s.y+(s.height-s.plus.oHeight) DIV 2,draw);
    END;

    s.DrawKnob;

    IF s.disabled THEN
      s.DrawDisabled;
    END;
  END Draw;

  PROCEDURE (s : Scroller) Hide*;

  BEGIN
    IF s.visible THEN
      s.plus.Hide;
      s.minus.Hide;
      s.knob.Hide;
      s.box.Hide;
      s.DrawHide;
      s.Hide^;
    END;
  END Hide;

  PROCEDURE (s : Scroller) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF s.visible & ~s.disabled THEN
      s.DrawKnob;
    END;
  END Resync;

BEGIN
  scrollerPattern[0]:=CHR(204); (* 11001100 *)
  scrollerPattern[1]:=CHR(204); (* 11001100 *)
  scrollerPattern[2]:=CHR(204); (* 11001100 *)
  scrollerPattern[3]:=CHR(204); (* 11001100 *)
  scrollerPattern[4]:=CHR(51);  (* 00110011 *)
  scrollerPattern[5]:=CHR(51);  (* 00110011 *)
  scrollerPattern[6]:=CHR(51);  (* 00110011 *)
  scrollerPattern[7]:=CHR(51);  (* 00110011 *)

  NEW(prefs);
  prefs.Init;
END VOScroller.