(**
  A scale for numerical intervals.
**)

MODULE VOScale;

(*
  A scale for numerical intervals.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT  D := VODisplay,
        G := VOGUIObject,
        P := VOPrefs;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (P.PrefsDesc)
               END;

  Scale*     = POINTER TO ScaleDesc;
  ScaleDesc* = RECORD (G.ImageDesc)
                 prefs : Prefs;
                 from,
                 to    : LONGINT;
                 vert  : BOOLEAN;
               END;

VAR
  prefs* : Prefs;

  PROCEDURE (s : Scale) Init*;

  BEGIN
    s.Init^;

    s.prefs:=prefs;

    s.vert:=FALSE;

    s.from:=0;
    s.to:=100;
  END Init;

  (**
    Define, if you want a vertical or horizontal scale.
  **)

  PROCEDURE (s : Scale) SetDirection*(vertical : BOOLEAN);

  BEGIN
    s.vert:=vertical;
  END SetDirection;

  (**
    Set the range the scale should go. For example 0,100 for a procentual
    scale.
  **)

  PROCEDURE (s : Scale) SetInterval*(from, to : LONGINT);

  BEGIN
    s.from:=from;
    s.to:=to;
  END SetInterval;

  PROCEDURE (s : Scale) CalcSize*(display : D.Display);

  BEGIN
    s.width:=2*display.spaceWidth;
    s.height:=2*display.spaceHeight;

    s.minWidth:=s.width;
    s.minHeight:=s.height;

    s.CalcSize^(display);
  END CalcSize;

  (**
    Calculates log10 number.
  **)

  PROCEDURE (s : Scale) GetExp(number : LONGINT):LONGINT;

  VAR
    exp : LONGINT;

  BEGIN
    exp:=0;

    WHILE number DIV 10 > 0 DO
      number:=number DIV 10;
      INC(exp);
    END;

    RETURN exp;
  END GetExp;

  (**
    Calculates 10^exp
  **)

  PROCEDURE (s : Scale) Pow10(exp : LONGINT):LONGINT;

  VAR
    x : LONGINT;

  BEGIN
    x:=1;
    WHILE exp>0 DO
      x:=x*10;
      DEC(exp);
    END;
    RETURN x;
  END Pow10;

  PROCEDURE (s : Scale) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    interval,
    exp,min,
    max,pos,
    mark,int,
    width     : LONGINT;

  BEGIN
    s.Draw^(x,y,draw);

    interval:=s.to-s.from;

    exp:=s.GetExp(interval);
    max:=exp;

    draw.PushForeground(D.textColor);
    IF s.vert THEN
      min:=s.display.spaceHeight DIV 2;
      width:=s.height;
    ELSE
      min:=s.display.spaceWidth DIV 2;
      width:=s.width;
    END;

    WHILE (exp>=0) & ((width*s.Pow10(exp)) DIV interval>min) DO
      int:=s.Pow10(exp);
      mark:=(s.from DIV int)*int;

      WHILE mark<=s.to DO
        IF (mark>=s.from) & ( (exp=max) OR (mark MOD s.Pow10(exp+1)#0) ) THEN
          IF s.vert THEN
            pos:=s.height*(mark-s.from) DIV interval;
            draw.PushForeground(D.shadowColor);
            draw.DrawLine(s.x+s.width-(s.width DIV (max-exp+1)),s.y+pos,
                          s.x+s.width-1,s.y+pos);
            draw.PopForeground;
(*            draw.PushForeground(D.shineColor);
            draw.DrawLine(s.x+s.width-(s.width DIV (max-exp+1)),s.y+pos+1,
                          s.x+s.width-1,s.y+pos+1);
            draw.PopForeground;*)
          ELSE
            pos:=s.width*(mark-s.from) DIV interval;
            draw.PushForeground(D.shadowColor);
            draw.DrawLine(s.x+pos,s.y+s.height-(s.height DIV (max-exp+1)),
                          s.x+pos,s.y+s.height-1);
            draw.PopForeground;
(*            draw.PushForeground(D.shineColor);
            draw.DrawLine(s.x+pos+1,s.y+s.height-(s.height DIV (max-exp+1)),
                          s.x+pos+1,s.y+s.height-1);
            draw.PopForeground;*)
          END;

        END;
        INC(mark,int);
      END;
      DEC(exp);
    END;

    draw.PopForeground;
  END Draw;

  PROCEDURE (s : Scale) Hide*;

  BEGIN
    IF s.visible THEN
      s.DrawHide;
      s.Hide^;
    END;
  END Hide;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOScale.