/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse m�l :
	BILLARD, non joignable par m�l ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant � visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est r�gi par la licence CeCILL soumise au droit fran�ais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffus�e par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez acc�der � cet en-t�te signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept� les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <glib.h>
#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h> /* For the access markers R_OK, W_OK ... */
#include <sys/stat.h>
#include <sys/types.h>

#include "interface.h"
#include "support.h"
#include "gtk_main.h"
#include "gtk_interactive.h"
#include "gtk_pick.h"
#include "gtk_pairs.h"
#include "gtk_about.h"
#include "gtk_save.h"
#include "gtk_renderingWindowWidget.h"
#include "extraGtkFunctions/gtk_colorComboBoxWidget.h"
#include "extraGtkFunctions/gtk_shadeComboBoxWidget.h"

#include "panelModules/externalModules.h"
#include "panelModules/panelBrowser.h"
#include "panelModules/panelDataFile.h"
#include "panelModules/panelPlanes.h"
#include "panelModules/gtkAtomic.h"
#include "panelModules/gtkSpin.h"

#include "visu_gtk.h"
#include "opengl.h"
#include "visu_object.h"
#include "visu_data.h"
#include "visu_configFile.h"
#include "visu_basic.h"            /* To have loadDataFromFile */
#include "visu_pairs.h"            /* To have visuPairSet_status */
#include "visu_commandLine.h"      /* To have getArgFilename and getXWindowGeometry*/
#include "visu_extension.h"
#include "visu_pickMesure.h"
#include "visu_plugins.h"
#include "renderingBackend/visu_windowInterface.h"
#include "coreTools/toolOptions.h"
#include "coreTools/toolConfigFile.h"
#include "extraFunctions/dataFile.h"

/* Functions used to set some methods, related to gtkMain, such as a
 * custom load file chooser for each rendering method.
 */
VisuToolsInitFunc listInitRendenringGtkFunc[] = {
  panelElementAtomicInit_gtkMain,
  panelElementSpinInit_gtkMain,
  (VisuToolsInitFunc)0};

enum {
  SPIN_BOUNDS_CHANGED_SIGNAL,
  LAST_SIGNAL
};

struct _GtkMainClass
{
  GtkWindowClass parent;

  guint signal_ids[LAST_SIGNAL];

  /* Gestion of the main window positioning. */
  gboolean rememberWindowPosition;
  GHashTable *windowPosition;

  /* Alert or not when quiting. */
  gboolean warningWhenQuit;
};

struct GtkMain_private_struct
{
  gboolean dispose_has_run;

  /* Pointers on different inside widgets. */
  GtkWidget *loadButton;
  GtkWidget *checkPairs;
  GtkWidget *pairsButton;
  GtkWidget *mouseActions;
  GtkWidget *saveButton;
  GtkWidget *quitButton;
  GtkWidget *aboutButton;
  GtkWidget *vboxMain;

  /* Variables. */
  gboolean oneWindow;
};

/* Local object method. */
static void gtkMainBase_init     (GtkMainClass *klass);
static void gtkMainBase_finalise (GtkMainClass *klass);
static void gtkMainClass_init (GtkMainClass *klass);
static void gtkMain_init      (GtkMain *obj);
static void gtkMain_dispose   (GObject *obj);
static void gtkMain_finalize  (GObject *obj);

/* Local variables. */
static GtkWindowClass *parent_class = NULL;
static GtkMainClass   *my_class     = NULL;
static GtkMain *currentGtkMain      = NULL;

/* Miscelaneous functions with gtk */
static void setRenderingButtonSensitive(GtkMain *main, gboolean bool);
static void gtkMainHide_window(GtkWindow *win);
static void gtkMainShow_window(GtkWindow *win);


/* Local callbacks */
static gboolean onHomePressed(GtkWidget *widget _U_, GdkEventKey *event,
			       gpointer data);
static void onLoadButtonClicked(GtkMain *main, GtkButton *button);
static void onPairsCheckToggled(GtkMain *main, GtkToggleButton *toggle);
static void onPairsButtonClicked(GtkMain *main, GtkButton *button);
static void onMouseActionsClicked(GtkMain *main, GtkButton *button);
static void onSaveButtonClicked(GtkMain *main, GtkButton *button);
static void onQuitButtonClicked(GtkMain *main, GtkButton *button);
static void onAboutButtonClicked(GtkMain *main, GtkButton *button);
static void onResourcesLoaded(GtkMain *main, VisuData *dataObj, gpointer data);
static void onRenderingChanged(GtkMain *main, VisuRendering *method, gpointer data);
static void onFileChange(GtkMain *main, VisuData *dataObj, gpointer data);
static gboolean onKillPairsDialog(GtkMain *main, GdkEvent *event, gpointer data);
static gboolean onKillAboutDialog(GtkMain *main, GdkEvent *event, gpointer data);
static gboolean onKillInteractiveDialog(GtkMain *main, GdkEvent *event,
					gpointer data);
static gboolean onKillMainWindowEvent(GtkWidget *widget, GdkEvent *event,
				      gpointer user_data);

void onHideNextTime(GtkToggleButton *button, gpointer data);

/* Parameter to store the position of windows. */
#define PARAMETER_GTKMAIN_REMEMBER_DEFAULT TRUE

/* Parameter to change the policy of the warning quit dialog. */
#define FLAG_PARAMETER_GTKMAIN_QUIT    "main_confirmQuit"
#define DESC_PARAMETER_GTKMAIN_QUIT    "Show up a dialog to confirm when quit button is clicked ; boolean 0 or 1"
#define PARAMETER_GTKMAIN_QUIT_DEFAULT TRUE
static gboolean readMainQuitPolicy(gchar **lines, int nbLines, int position,
				   VisuData *dataObj, GError **error);
#define FLAG_PARAMETER_GTKMAIN_PANEL   "main_panelStatus"
#define DESC_PARAMETER_GTKMAIN_PANEL   "Attach a panel to a tool window ; panel_name window_name (or None or Main)"
static gboolean readMainPanelStatus(gchar **lines, int nbLines, int position,
				    VisuData *dataObj, GError **error);
#define FLAG_PARAMETER_GTKMAIN_DOCK   "main_dock"
#define DESC_PARAMETER_GTKMAIN_DOCK   "Define the characteristic of a dock window ; visibility size(x,y) position(w,h) window_name"
static gboolean readMainDock(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error);
static void exportParametersGtkMain(GString *data, VisuData *dataObj);


/* Number of time V_Sim try to attach the OpenGL context. */
#define GTK_MAIN_N_MAX_OPENGL 10

/*******************/
/* Object methods. */
/*******************/
GType gtkMain_get_type(void)
{
  static GType gtkMain_type = 0;

  if (!gtkMain_type)
    {
      static const GTypeInfo gtkMain_info =
      {
        sizeof (GtkMainClass),
        (GBaseInitFunc)gtkMainBase_init,
        (GBaseFinalizeFunc)gtkMainBase_finalise,
        (GClassInitFunc)gtkMainClass_init,
        NULL, /* class_finalise */
        NULL, /* class_data */
        sizeof (GtkMain),
        0, /* number of prealloc instances */
        (GInstanceInitFunc)gtkMain_init,
	NULL
      };
      gtkMain_type = g_type_register_static(GTK_TYPE_WINDOW, "GtkMain",
					    &gtkMain_info, 0);
      DBG_fprintf(stderr, "Gtk Main: creating the type GtkMain %d.\n",
		  (int)gtkMain_type);
    }

  return gtkMain_type;
}

static void gtkMainBase_init     (GtkMainClass *klass)
{
  DBG_fprintf(stderr, "Gtk Main: creating the class of the object.\n");

  DBG_fprintf(stderr, "                - allocate dynamic pointers ;\n");

  /* Create the hashtable to store windows and their position. */
  klass->windowPosition = g_hash_table_new_full(g_direct_hash,
						g_direct_equal,
						NULL, g_free);
}
static void gtkMainBase_finalise (GtkMainClass *klass)
{
  g_hash_table_destroy(klass->windowPosition);
}

static void gtkMainClass_init(GtkMainClass *klass)
{
  int i;
  VisuConfigFileEntry *resourceEntry;

  parent_class = g_type_class_peek_parent(klass);
  my_class     = klass;

  /* Connect freeing methods. */
  G_OBJECT_CLASS(klass)->dispose  = gtkMain_dispose;
  G_OBJECT_CLASS(klass)->finalize = gtkMain_finalize;

  /* Initialise internal variables. */
  for (i = 0; listInitRendenringGtkFunc[i]; i++)
    listInitRendenringGtkFunc[i]();

  /* Add the config file entries. */
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_GTKMAIN_QUIT,
					  DESC_PARAMETER_GTKMAIN_QUIT,
					  1, readMainQuitPolicy);
  visuConfigFileSet_version(resourceEntry, 3.3f);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_GTKMAIN_PANEL,
					  DESC_PARAMETER_GTKMAIN_PANEL,
					  1, readMainPanelStatus);
  visuConfigFileSet_version(resourceEntry, 3.3f);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_GTKMAIN_DOCK,
					  DESC_PARAMETER_GTKMAIN_DOCK,
					  1, readMainDock);
  visuConfigFileSet_version(resourceEntry, 3.3f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParametersGtkMain);

  /* Set static parameters. */
  klass->rememberWindowPosition = PARAMETER_GTKMAIN_REMEMBER_DEFAULT;
  klass->warningWhenQuit        = PARAMETER_GTKMAIN_QUIT_DEFAULT;

  /* Set the pixmap directory for the Glade stuff. */
  add_pixmap_directory(V_SIM_PIXMAPS_DIR);

  /* Force the creation of the ColorComboBoxClass. */
  g_type_class_ref(colorComboBox_get_type());
}

static void _buildWidgets(GtkMain *obj, gboolean oneWindow)
{
  int width, height;
  int i;
  GdkPixbuf *obj_icon_pixbuf;
  GtkWidget *vbox, *hbox, *hbox2, *wd, *align, *image, *label, *pane;
  ToolPanel *panel;
  DockWindow *dockMain;
  gpointer window;
#if GTK_MINOR_VERSION < 12
  GtkTooltips *tooltips;
  tooltips = gtk_tooltips_new ();
#endif

  obj->private->oneWindow = oneWindow;
  pane = (GtkWidget*)0;

  DBG_fprintf(stderr,"--- Initialising the rendering window ---\n");
  commandLineGet_XWindowGeometry(&width, &height);
  if (oneWindow)
    {
      obj->renderingWindow = renderingWindow_new(width, height, TRUE, TRUE);
      pane = gtk_hpaned_new();
      gtk_widget_show(pane);
      gtk_paned_pack2(GTK_PANED(pane), obj->renderingWindow, TRUE, TRUE);
    }
  else
    {
      obj->renderingWindow =
	renderingWindow_newEmbedded(width, height);
      window = g_object_get_data(G_OBJECT(obj->renderingWindow), RENDERING_WINDOW_ID);
      g_signal_connect(G_OBJECT(window), "delete-event",
		       G_CALLBACK(onKillMainWindowEvent), (gpointer)obj);
      g_signal_connect(G_OBJECT(window), "destroy-event",
		       G_CALLBACK(onKillMainWindowEvent), (gpointer)obj);
      renderingWindowSet_panel(RENDERING_WINDOW(obj->renderingWindow),
			       GTK_WIDGET(obj));
    }
  visuRenderingWindowSet_default((GenericRenderingWindow)obj->renderingWindow);
  currentGtkMain = obj;

  /* Private data. */
  DBG_fprintf(stderr, "--- Initialising the GTK interface ---\n");
  DBG_fprintf(stderr, " | Gtk Main");

/*   main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL); */

  gtk_window_set_title(GTK_WINDOW(obj), _("Command panel"));
  gtk_window_set_wmclass(GTK_WINDOW(obj), "v_sim_commandPanel", "V_Sim");
  obj_icon_pixbuf = create_pixbuf("icone-panel.png");
  if(obj_icon_pixbuf)
    {
      gtk_window_set_icon(GTK_WINDOW(obj), obj_icon_pixbuf);
      gdk_pixbuf_unref(obj_icon_pixbuf);
    }

  obj->private->vboxMain = gtk_vbox_new(FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(obj->private->vboxMain), 7);
  if (oneWindow)
    {
      gtk_paned_pack1(GTK_PANED(pane), obj->private->vboxMain, FALSE, FALSE);
      gtk_container_add(GTK_CONTAINER(obj), pane);
    }
  else
    gtk_container_add(GTK_CONTAINER(obj), obj->private->vboxMain);
  gtk_widget_set_size_request(obj->private->vboxMain, 350, -1);

  wd = gtk_frame_new(_("Actions"));
  gtk_box_pack_end(GTK_BOX(obj->private->vboxMain), wd, FALSE, FALSE, 0);

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(wd), vbox);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);

  obj->private->loadButton = gtk_button_new_from_stock(GTK_STOCK_OPEN);
  gtk_box_pack_start(GTK_BOX(hbox), obj->private->loadButton, TRUE, TRUE, 0);
  gtk_widget_set_sensitive(obj->private->loadButton, FALSE);
  gtk_widget_set_tooltip_text(obj->private->loadButton,
		       _("Select a file to render."));

  align = gtk_alignment_new(0.5, 0.5, 1., 0.);
  gtk_box_pack_start(GTK_BOX(hbox), align, TRUE, TRUE, 10);

  hbox2 = gtk_hbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(align), hbox2);

  obj->private->checkPairs = gtk_check_button_new();
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(obj->private->checkPairs),
			       visuPairGet_status());
  gtk_box_pack_start(GTK_BOX(hbox2), obj->private->checkPairs, FALSE, FALSE, 0);
  gtk_widget_set_tooltip_text(obj->private->checkPairs,
		       _("Check to draw pairs between elements."));

  obj->private->pairsButton = gtk_button_new();
  gtk_widget_set_sensitive(obj->private->pairsButton, FALSE);
  gtk_box_pack_start(GTK_BOX(hbox2), obj->private->pairsButton, TRUE, TRUE, 0);
  gtk_widget_set_tooltip_text(obj->private->pairsButton,
		       _("Configure parameters for bindings such as color, thickness..."));

  align = gtk_alignment_new(0.5, 0.5, 0., 0.);
  gtk_container_add(GTK_CONTAINER(obj->private->pairsButton), align);

  wd = gtk_hbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(align), wd);

  image = gtk_image_new_from_stock(GTK_STOCK_CONVERT, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start(GTK_BOX(wd), image, FALSE, FALSE, 0);

  label = gtk_label_new_with_mnemonic(_("_Pairs"));
  gtk_box_pack_start(GTK_BOX(wd), label, FALSE, FALSE, 0);

  obj->private->mouseActions = gtk_button_new();
  gtk_widget_set_sensitive(obj->private->mouseActions, FALSE);
  gtk_box_pack_start(GTK_BOX(hbox), obj->private->mouseActions, TRUE, TRUE, 0);
  gtk_widget_set_tooltip_text(obj->private->mouseActions,
		       _("Use the mouse to change the view and get position informations."));

  align = gtk_alignment_new(0.5, 0.5, 0., 0.);
  gtk_container_add(GTK_CONTAINER(obj->private->mouseActions), align);

  wd = gtk_hbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(align), wd);

  image = gtk_image_new_from_stock(GTK_STOCK_ZOOM_FIT, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start(GTK_BOX(wd), image, FALSE, FALSE, 0);

  label = gtk_label_new_with_mnemonic(_("Mouse _actions"));
  gtk_box_pack_start(GTK_BOX(wd), label, FALSE, FALSE, 0);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);

  obj->private->saveButton = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(hbox), obj->private->saveButton, TRUE, TRUE, 5);
  gtk_widget_set_tooltip_text(obj->private->saveButton,
		       _("Click to save the parameters or the resources."));

  align = gtk_alignment_new(0.5, 0.5, 0., 0.);
  gtk_container_add(GTK_CONTAINER(obj->private->saveButton), align);

  wd = gtk_hbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(align), wd);

  image = gtk_image_new_from_stock(GTK_STOCK_EDIT, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start(GTK_BOX(wd), image, FALSE, FALSE, 0);

  label = gtk_label_new_with_mnemonic(_("_Config. files"));
  gtk_box_pack_start(GTK_BOX(wd), label, FALSE, FALSE, 0);

  obj->private->quitButton = gtk_button_new_from_stock(GTK_STOCK_QUIT);
  gtk_box_pack_start(GTK_BOX(hbox), obj->private->quitButton, TRUE, TRUE, 5);

  obj->private->aboutButton = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(hbox), obj->private->aboutButton, FALSE, FALSE, 15);
  gtk_widget_set_tooltip_text(obj->private->aboutButton,
		       _("V_Sim program. Written by L. Billard, modified by D. Caliste."));
  gtk_button_set_relief(GTK_BUTTON(obj->private->aboutButton), GTK_RELIEF_NONE);

  image = create_pixmap(GTK_WIDGET(obj), "logo_petit.png");
  gtk_container_add(GTK_CONTAINER(obj->private->aboutButton), image);

  g_signal_connect_swapped(G_OBJECT(obj->private->loadButton), "clicked",
			   G_CALLBACK(onLoadButtonClicked), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(obj->private->checkPairs), "toggled",
			   G_CALLBACK(onPairsCheckToggled), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(obj->private->pairsButton), "clicked",
			   G_CALLBACK(onPairsButtonClicked), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(obj->private->mouseActions), "clicked",
			   G_CALLBACK(onMouseActionsClicked), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(obj->private->saveButton), "clicked",
			   G_CALLBACK(onSaveButtonClicked), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(obj->private->quitButton), "clicked",
			   G_CALLBACK(onQuitButtonClicked), (gpointer)obj);
  g_signal_connect_swapped(G_OBJECT(obj->private->aboutButton), "clicked",
			   G_CALLBACK(onAboutButtonClicked), (gpointer)obj);
  g_signal_connect(G_OBJECT(obj), "delete-event",
		   G_CALLBACK(onKillMainWindowEvent), (gpointer)obj);
  g_signal_connect(G_OBJECT(obj), "destroy-event",
		   G_CALLBACK(onKillMainWindowEvent), (gpointer)obj);
  g_signal_connect_swapped(VISU_INSTANCE, "resourcesLoaded",
			   G_CALLBACK(onResourcesLoaded), (gpointer)obj);
  g_signal_connect_swapped(VISU_INSTANCE, "dataReadyForRendering", 
			   G_CALLBACK(onFileChange), (gpointer)obj);
  g_signal_connect_swapped(VISU_INSTANCE, "renderingChanged",
			   G_CALLBACK(onRenderingChanged), (gpointer)obj);

  DBG_fprintf(stderr, " ... OK.\n");

  /* init the sub panel contains. */
  dockMain = toolPanelClassGet_commandPanel();
  wd = dockWindowGet_container(dockMain);
  gtk_box_pack_start(GTK_BOX(obj->private->vboxMain), wd, TRUE, TRUE, 0);

  /* Show all. */
  gtk_widget_show_all(obj->private->vboxMain);

  for (i = 0; panelListAll[i]; i++)
    {
      panel = panelListAll[i]();
      if (!panel)
	{
	  g_error("Can't initialise subpanel number %d.\n", i);
	}

      toolPanelAttach(panel, dockMain);
      if (i == 0)
	gtk_notebook_set_current_page(GTK_NOTEBOOK(dockWindowGet_notebook(dockMain)), 0);
      
      DBG_fprintf(stderr, "Gtk Main: initialise '%s' subpanel OK.\n",
		  toolPanelGet_label(panel));
    }
  DBG_fprintf(stderr, "Gtk Main: build widgets OK.\n");
}

static void gtkMain_init(GtkMain *obj)
{
  GList *pnt;

  DBG_fprintf(stderr, "Gtk Main: initializing a new object (%p).\n",
	      (gpointer)obj);

  obj->private = g_malloc(sizeof(GtkMain_private));
  obj->private->dispose_has_run = FALSE;

  /* Public data. */
  obj->renderingWindow    = (GtkWidget*)0;
  obj->pairsDialog        = (GtkWidget*)0;
  obj->interactiveDialog  = (GtkWidget*)0;
  obj->aboutDialog        = (GtkWidget*)0;

  /* Private variables. */
  obj->private->oneWindow = FALSE;

  g_signal_connect(G_OBJECT(obj), "key-press-event",
		   G_CALLBACK(onHomePressed), (gpointer)obj);

  /* Others elements linked to the main window. */
  DBG_fprintf(stderr, " | Gtk Interactive\n");
  gtkInteractiveInit();
  DBG_fprintf(stderr, " | Gtk Pairs\n");
  gtkPairsInit();
  DBG_fprintf(stderr, " | Plugins Gtk\n");
  for (pnt = visuPluginsGet_listLoaded(); pnt; pnt = g_list_next(pnt))
    if (((VisuPlugin*)pnt->data)->initGtk)
      ((VisuPlugin*)pnt->data)->initGtk();
}

/* This method can be called several times.
   It should unref all of its reference to
   GObjects. */
static void gtkMain_dispose(GObject* obj)
{
  DBG_fprintf(stderr, "Gtk Main: dispose object %p.\n", (gpointer)obj);

  if (GTK_MAIN(obj)->private->dispose_has_run)
    return;

  GTK_MAIN(obj)->private->dispose_has_run = TRUE;

  /* Chain up to the parent class */
  G_OBJECT_CLASS(parent_class)->dispose(obj);
}
/* This method is called once only. */
static void gtkMain_finalize(GObject* obj)
{
  g_return_if_fail(obj);

  DBG_fprintf(stderr, "Gtk Main: finalize object %p.\n", (gpointer)obj);

  /* Chain up to the parent class */
  G_OBJECT_CLASS(parent_class)->finalize(obj);

  DBG_fprintf(stderr, "Gtk Main: freeing ... OK.\n");
}

GtkWidget* gtkMainNew(gboolean oneWindow)
{
  GtkMain *commandPanel;

  commandPanel = GTK_MAIN(g_object_new(GTK_MAIN_TYPE, NULL));
  if (!commandPanel)
    return (GtkWidget*)0;

  _buildWidgets(commandPanel, oneWindow);

  return GTK_WIDGET(commandPanel);
}


/*******************/
/* Local callbacks */
/*******************/
static gboolean onHomePressed(GtkWidget *widget _U_, GdkEventKey *event,
			       gpointer data)
{
  DBG_fprintf(stderr, "Gtk Main: get key pressed.\n");
  if(event->keyval == GDK_Home && !GTK_MAIN(data)->private->oneWindow)
    {
      gtk_window_present(GTK_WINDOW(g_object_get_data
				    (G_OBJECT(GTK_MAIN(data)->renderingWindow),
				     RENDERING_WINDOW_ID)));
      return TRUE;
    }
  return FALSE;
}
static void onLoadButtonClicked(GtkMain *main, GtkButton *button _U_)
{
  openNewFile(); /* from panel_vibration */
  renderingWindowLoad_file(RENDERING_WINDOW(main->renderingWindow), GTK_WINDOW(main));
}
static void onPairsCheckToggled(GtkMain *main, GtkToggleButton *toggle)
{
  gboolean res;
  VisuData *dataObj;

  res = gtk_toggle_button_get_active(toggle);
  visuPairSet_status(res);
  if (res)
    {
      dataObj = renderingWindowGet_visuData(RENDERING_WINDOW(main->renderingWindow));
      if (dataObj && visuPairBuild(dataObj))
	g_idle_add(visuObjectRedraw, (gpointer)0);
    }
  else
    g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onPairsCloseClicked(GtkMain *main, GtkButton *button _U_)
{
  gtkMainHide_window(GTK_WINDOW(main->pairsDialog));
}
static gboolean onKillPairsDialog(GtkMain *main, GdkEvent *event _U_,
				  gpointer data _U_)
{
  gtkMainHide_window(GTK_WINDOW(main->pairsDialog));

  return TRUE;
}
static void onPairsButtonClicked(GtkMain *main, GtkButton *button _U_)
{
  GtkWidget *wd;

  if (!main->pairsDialog)
    {
      gtkPairsBuild_window(main);
      wd = lookup_widget(main->pairsDialog, "closebutton3");
      g_signal_connect_swapped(G_OBJECT(wd), "clicked",
			       G_CALLBACK(onPairsCloseClicked),
			       (gpointer)main);
      g_signal_connect_swapped(G_OBJECT(main->pairsDialog), "delete-event",
			       G_CALLBACK(onKillPairsDialog),
			       (gpointer)main);
      g_signal_connect_swapped(G_OBJECT(main->pairsDialog), "destroy-event",
			       G_CALLBACK(onKillPairsDialog),
			       (gpointer)main);
    }

  gtkPairsBuild_pairs
    (main,
     renderingWindowGet_visuData(RENDERING_WINDOW(main->renderingWindow)), TRUE);

  gtkMainShow_window(GTK_WINDOW(main->pairsDialog));
}
static void onActionsCloseClicked(GtkMain *main, GtkButton *button _U_)
{
  gtkMainHide_window(GTK_WINDOW(main->interactiveDialog));
  if (!main->private->oneWindow)
    {
      renderingWindowSet_panel(RENDERING_WINDOW(main->renderingWindow),
			       GTK_WIDGET(main));
      gtkMainShow_window(GTK_WINDOW(main));
    }
}
static gboolean onKillInteractiveDialog(GtkMain *main,
					GdkEvent *event _U_, gpointer data _U_)
{
  onActionsCloseClicked(main, (GtkButton*)0);
  

  return TRUE;
}
static void onMouseActionsClicked(GtkMain *main, GtkButton *button _U_)
{
  gint posx, posy;
  RenderingWindow *window;

  if (!main->interactiveDialog)
    gtkMainBuild_interactiveDialog(main);

  window = RENDERING_WINDOW(main->renderingWindow);

  /* Start new pick & observe session. */
  renderingWindowBlock_defaultIneractiveMode(window);
  gtkInteractiveInit_session(window, renderingWindowGet_interactive(window));

  if (!main->private->oneWindow)
    {
      gtk_window_get_position(GTK_WINDOW(main), &posx, &posy);
      DBG_fprintf(stderr, "Gtk Main: get command panel position"
		  " (%d,%d).\n", posx, posy);
      gtkMainHide_window(GTK_WINDOW(main));
      DBG_fprintf(stderr, "Gtk Main: set observe position (%d,%d).\n", posx, posy);
      gtk_window_move(GTK_WINDOW(main->interactiveDialog), posx, posy);
      renderingWindowSet_panel(window, GTK_WIDGET(main->interactiveDialog));
    }
  gtkMainShow_window(GTK_WINDOW(main->interactiveDialog));
}
static void onSaveButtonClicked(GtkMain *main _U_, GtkButton *button _U_)
{
  gtkSaveInit();
}
static void onQuitButtonClicked(GtkMain *main, GtkButton *button _U_)
{
  gtkMain_quit(main);
}
static gboolean onKillAboutDialog(GtkMain *main, GdkEvent *event _U_,
				  gpointer data _U_)
{
  gtkMainHide_window(GTK_WINDOW(main->aboutDialog));

  return TRUE;
}
static void onAboutButtonClicked(GtkMain *main, GtkButton *button _U_)
{
  if (!main->aboutDialog)
    {
      gtkAboutBuild_window(main);
      g_signal_connect_swapped(G_OBJECT(main->aboutDialog), "delete-event",
			       G_CALLBACK(onKillAboutDialog),
			       (gpointer)main);
      g_signal_connect_swapped(G_OBJECT(main->aboutDialog), "destroy-event",
			       G_CALLBACK(onKillAboutDialog),
			       (gpointer)main);
    }
  gtkMainShow_window(GTK_WINDOW(main->aboutDialog));
}

static void setRenderingButtonSensitive(GtkMain *main, gboolean bool)
{
  g_return_if_fail(IS_GTK_MAIN_TYPE(main));

  gtk_widget_set_sensitive(main->private->pairsButton, bool);
  gtk_widget_set_sensitive(main->private->mouseActions, bool);
}
static void onFileChange(GtkMain *main, VisuData *dataObj, gpointer data _U_)
{ 
  setRenderingButtonSensitive(main, (dataObj != (VisuData*)0));
  toolPanelClassSet_visuData(dataObj);
}
static void onResourcesLoaded(GtkMain *main, VisuData *dataObj _U_, gpointer data _U_)
{
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(main->private->checkPairs),
			       visuPairGet_status());
}
static void onRenderingChanged(GtkMain *main, VisuRendering *method,
			       gpointer data _U_)
{
  DBG_fprintf(stderr, "Gtk Main: caught 'renderingChanged' signal.\n");
  gtk_widget_set_sensitive(main->private->loadButton,
			   (method != (VisuRendering*)0));
  setRenderingButtonSensitive(main, FALSE);
}
static gboolean onKillMainWindowEvent(GtkWidget *widget _U_, GdkEvent *event _U_,
				      gpointer user_data)
{
  gtkMain_quit(GTK_MAIN(user_data));
  return TRUE;
}



/*****************/
/* Miscellaneous */
/*****************/

void gtkMainBuild_interactiveDialog(GtkMain *main)
{
  GtkWidget *wd;

  g_return_if_fail(IS_GTK_MAIN_TYPE(main) && !main->interactiveDialog);

  gtkInteractiveBuild_window(main);
  wd = lookup_widget(main->interactiveDialog, "buttonBackToCommandPanel");
  g_signal_connect_swapped(G_OBJECT(wd), "clicked",
			   G_CALLBACK(onActionsCloseClicked),
			   (gpointer)main);
  g_signal_connect_swapped(G_OBJECT(main->interactiveDialog), "delete-event",
			   G_CALLBACK(onKillInteractiveDialog),
			   (gpointer)main);
  g_signal_connect_swapped(G_OBJECT(main->interactiveDialog), "destroy-event",
			   G_CALLBACK(onKillInteractiveDialog),
			   (gpointer)main);
}

/* This method hides the specified window and save its coordinates if
   it is specified by a parameter. */
static void gtkMainHide_window(GtkWindow *win)
{
  int *val;

  if (!win)
    return;

  if (my_class->rememberWindowPosition)
    {
      val = (int*)g_hash_table_lookup(my_class->windowPosition, win);
      if (!val)
	{
	  val = g_malloc(sizeof(int) * 2);
	  g_hash_table_insert(my_class->windowPosition, (gpointer)win, (gpointer)val);
	}
      gtk_window_get_position(win, &val[0], &val[1]);
      DBG_fprintf(stderr, "Gtk Main : store position (%d,%d) for window %d.\n", 
		  val[0], val[1], GPOINTER_TO_INT(win));
    }
  gtk_widget_hide(GTK_WIDGET(win));
}

/* This method shows the specified window and try to put it at its
   former position if the good parameter is used. */
static void gtkMainShow_window(GtkWindow *win)
{
  int *val;

  if (!win)
    return;

  if (my_class->rememberWindowPosition)
    {
      val = (int*)g_hash_table_lookup(my_class->windowPosition, win);
      if (val)
	{
	  gtk_window_move(win, val[0], val[1]);
	  DBG_fprintf(stderr, "Gtk Main : set position (%d,%d) for window %d.\n", 
		      val[0], val[1], GPOINTER_TO_INT(win));
	}
    }
  gtk_window_present(win);
}

void onHideNextTime(GtkToggleButton *button, gpointer data)
{
  char *posNext;
  gchar *path, *bufferR, *pos;
  GString *bufferW, *bufferW2;
  gboolean resOk;
  int lines;
  GIOChannel *file;
  GError *err;
  gsize taille;
  GIOStatus statOK;

  g_return_if_fail(data);
  path = (gchar*)data;
  
  DBG_fprintf(stderr, "Gtk Main : change the warning dialog parameter in file '%s'.\n", path);
  my_class->warningWhenQuit = !gtk_toggle_button_get_active(button);

  /* If no file exists in the given path, we create it. */
  if (!g_file_test(path, G_FILE_TEST_EXISTS))
    {
      err = (GError*)0;
      resOk = visuConfigFileSave(VISU_CONFIGFILE_PARAMETER, path,
				 &lines, (VisuData*)0, &err);
      if (!resOk)
	{
	  visuGtkRaise_warningLong(_("Saving a file"), err->message,
				   (GtkWindow*)0);
	  g_error_free(err);
	}
      return;
    }

  /* If a parameter file already exist, we then just change the right line. */
  bufferR = (gchar*)0;
  err = (GError*)0;
  if (!g_file_get_contents(path, &bufferR, &taille, &err))
    {
      visuGtkRaise_warningLong(_("Saving a file"), err->message, (GtkWindow*)0);
      g_error_free(err);
      return;
    }

  /* We reopen the channel in write acces. */
  err = (GError*)0;
  file = g_io_channel_new_file(path, "w", &err);
  if (err)
    {
      visuGtkRaise_warningLong(_("Saving a file"), err->message, (GtkWindow*)0);
      g_error_free(err);
      return;
    }

  g_return_if_fail(bufferR);
  bufferW = g_string_new(bufferR);
  g_free(bufferR);
  /* Try to find the flag of the parameter. */
  pos = g_strrstr(bufferW->str, "\n"FLAG_PARAMETER_GTKMAIN_QUIT);
  if (!pos)
    {
      /* We append it at the end of the file. */
      DBG_fprintf(stderr, " | Can't find the option, appending it.\n");
      exportParametersGtkMain(bufferW, (VisuData*)0);

      err = (GError*)0;
      statOK = g_io_channel_write_chars(file, bufferW->str, -1,
                                        &taille, &err);
      if (err)
	{
	  visuGtkRaise_warningLong(_("Saving a file"), err->message, (GtkWindow*)0);
	  g_error_free(err);
	}
    }
  else
    {
      DBG_fprintf(stderr, " | Option found, changing its value.\n");
      /* We erase the line and rewrite it. */
      *(pos + 1) = '\0';
      bufferW2 = g_string_new(bufferW->str);
      g_string_append_printf(bufferW2, "%s[gtk]: %i\n", FLAG_PARAMETER_GTKMAIN_QUIT,
			     (int)my_class->warningWhenQuit);
      posNext = strstr(pos + 2, "\n");
      if (posNext)
	g_string_append(bufferW2, posNext + 1);
      
      err = (GError*)0;
      statOK = g_io_channel_write_chars(file, bufferW2->str, -1,
                                        &taille, &err);
      if (err)
	{
	  visuGtkRaise_warningLong(_("Saving a file"), err->message, (GtkWindow*)0);
	  g_error_free(err);
	}
      g_string_free(bufferW2, TRUE);
    }
  g_io_channel_shutdown(file, TRUE, (GError**)0);
  g_io_channel_unref(file);

  g_string_free(bufferW, TRUE);
}
static void onAddHomedir(GtkButton *button _U_, gpointer quitDialog)
{
  GtkWidget *wd;
  GList *dirs, *tmplst;
  gchar *path;
#if SYSTEM_X11 == 1
#define PERMS (S_IRWXU | S_IRGRP | S_IXGRP)
#endif
#if SYSTEM_WIN32 == 1
#define PERMS 0
#endif

  DBG_fprintf(stderr, "Gtk Main: try to create the local home directory.\n");
#if GLIB_MINOR_VERSION > 7
  if (g_mkdir_with_parents(V_SIM_LOCAL_CONF_DIR, PERMS))
#else
#if SYSTEM_X11 == 1
  if (mkdir(V_SIM_LOCAL_CONF_DIR, PERMS))
#endif
#if SYSTEM_WIN32 == 1
  if (mkdir(V_SIM_LOCAL_CONF_DIR))
#endif
#endif
    /* Failed. */
    visuGtkRaise_warning(_("I/O"),
			 _("Can't create the directory '$XDG_CONFIG_HOME/v_sim'."),
			 (GtkWindow*)0);
  else
    {
      /* Succeed hide the warning. */
      wd = lookup_widget(GTK_WIDGET(quitDialog), "hboxHomedir");
      gtk_widget_hide(wd);
      /* Retest the path. */
      dirs = (GList*)0;
      dirs = g_list_prepend(dirs, (gpointer)V_SIM_DATA_DIR);
      dirs = g_list_prepend(dirs, (gpointer)V_SIM_LOCAL_CONF_DIR);
      tmplst = dirs;

      path = (gchar*)0;
      path = visuConfigFileGet_nextValidPath(VISU_CONFIGFILE_PARAMETER,
					     W_OK, &tmplst, 0);
      if (path)
	{
	  wd = lookup_widget(GTK_WIDGET(quitDialog), "hboxWarning");
	  gtk_widget_hide(wd);
	  wd = lookup_widget(GTK_WIDGET(quitDialog), "checkbuttonHideNextTime");
	  gtk_widget_set_sensitive(wd, TRUE);
	  g_signal_connect(G_OBJECT(wd), "toggled",
			   G_CALLBACK(onHideNextTime), (gpointer)path);
	}
      g_list_free(dirs);
    }
}

static void freeAllAndStop(GtkMain *main)
{
  RenderingWindow *window;

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  renderingWindowSet_visuData(window, (VisuData*)0);

  visuElementFree_all();

  visuBasicFree_all();
  commandLineFree_all();

  if (main->renderingWindow)
    gtk_widget_destroy(main->renderingWindow);

  if (main->pairsDialog)
    gtk_widget_destroy(main->pairsDialog);

  if (main->interactiveDialog)
    gtk_widget_destroy(main->interactiveDialog);

  if (main->aboutDialog)
    gtk_widget_destroy(main->aboutDialog);

  gtk_widget_destroy(GTK_WIDGET(main));

  gtk_main_quit();

#if DEBUG == 1
  g_mem_profile();
#endif
}

void gtkMain_quit(GtkMain *main)
{
  GtkWidget *quitDialog, *wd;
  GList *dirs, *tmplst;
  gchar *path;

  if (!my_class->warningWhenQuit)
    {
      freeAllAndStop(main);
      return;
    }

  quitDialog = create_quitDialog();
  gtk_window_set_transient_for(GTK_WINDOW(quitDialog), GTK_WINDOW(main));
  
  /* Try to find installDir/v_sim.par or $XDG_CONFIG_HOME/v_sim/v_sim.par
     that is writable to store the preference of the hiding
     mode of the dialog. */
  dirs = (GList*)0;
  dirs = g_list_prepend(dirs, (gpointer)V_SIM_DATA_DIR);
  dirs = g_list_prepend(dirs, (gpointer)V_SIM_LOCAL_CONF_DIR);
  tmplst = dirs;

  path = (gchar*)0;
  path = visuConfigFileGet_nextValidPath(VISU_CONFIGFILE_PARAMETER, W_OK, &tmplst, 0);
  if (!path)
    {
      wd = lookup_widget(quitDialog, "hboxWarning");
      gtk_widget_show(wd);
    }
  g_list_free(dirs);

  /* Attach a create the homedir method to the button. */
  wd = lookup_widget(quitDialog, "buttonAddHomedir");
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onAddHomedir), (gpointer)quitDialog);
  /* Show the warning if the homedir is not existing and no path was found. */
  if (!g_file_test(V_SIM_LOCAL_CONF_DIR, G_FILE_TEST_IS_DIR) && !path)
    {
      wd = lookup_widget(quitDialog, "hboxHomedir");
      gtk_widget_show(wd);
    }

  /* Attach a modify the parameter to the checkbox. */
  wd = lookup_widget(quitDialog, "checkbuttonHideNextTime");
  if (!path)
    gtk_widget_set_sensitive(wd, FALSE);
  else
    g_signal_connect(G_OBJECT(wd), "toggled",
		     G_CALLBACK(onHideNextTime), (gpointer)path);
  
  if (gtk_dialog_run(GTK_DIALOG(quitDialog)) == GTK_RESPONSE_OK)
    freeAllAndStop(main);
  else
    gtk_widget_destroy(quitDialog);
}

static gboolean readMainQuitPolicy(gchar **lines, int nbLines, int position,
				   VisuData *dataObj _U_, GError **error)
{
  gboolean bool;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &bool, 1, error))
    return FALSE;
  my_class->warningWhenQuit = bool;
  
  return TRUE;
}
static gboolean readMainPanelStatus(gchar **lines, int nbLines, int position,
				    VisuData *dataObj _U_, GError **error)
{
  gchar **tokens;
  ToolPanel *toolpanel;
  char *pt;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_string(lines[0], position, &tokens, 2, TRUE, error))
    return FALSE;
  toolpanel = toolPanelClassGet_toolPanelById(tokens[0]);
  if (!toolpanel)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: the panel '%s' is unknown.\n"),
			   position, tokens[0]);
      g_strfreev(tokens);
      return FALSE;
    }
  pt = strchr(tokens[1], '\n');
  if (pt)
    *pt = ' ';
  toolPanelSet_containerId(toolpanel, g_strchomp(tokens[1]));

  g_strfreev(tokens);
  return TRUE;
}
static gboolean readMainDock(gchar **lines, int nbLines, int position,
			     VisuData *dataObj _U_, GError **error)
{
  int res;
  gchar **tokens, *values;
  gboolean visible;
  int x, y, width, height;

  g_return_val_if_fail(nbLines == 1, FALSE);

  tokens = g_strsplit(g_strchug(lines[0]), " ", 4);
  if (!tokens[0] || !tokens[1] || !tokens[2] || !tokens[3])
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: awaited "
			     "'id visible pos size'.\n"),
			   position);
      g_strfreev(tokens);
      return FALSE;
    }
  values = g_strjoin(" ", tokens[0], tokens[1], tokens[2], NULL);
  res = sscanf(values, "%d %dx%d %dx%d", (int*)&visible, &x, &y, &width, &height);
  if (res != 5)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: can't read dock"
			     " characteristic values from '%s'.\n"),
			   position, values);
      g_strfreev(tokens);
      g_free(values);
      return FALSE;
    }
  g_free(values);
  toolPanelClassSet_windowSize(g_strchomp(tokens[3]), (guint)width, (guint)height);
  toolPanelClassSet_windowPosition(tokens[3], (guint)x, (guint)y);
  toolPanelClassSet_windowVisibility(tokens[3], visible);

  g_strfreev(tokens);
  return TRUE;
}
static void exportParametersGtkMain(GString *data, VisuData* dataObj _U_)
{
  GList *tmplst, *panelLst, *dockLst;
  gint x, y, width, height;
  gboolean visilibity;
  gchar *id;

  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_GTKMAIN_QUIT);
  g_string_append_printf(data, "%s[gtk]: %i\n\n", FLAG_PARAMETER_GTKMAIN_QUIT,
			 (int)my_class->warningWhenQuit);

  /* Write the panel list. */
  panelLst = toolPanelClassGet_allToolPanels();
  if (panelLst)
    {
      g_string_append_printf(data, "# %s\n", DESC_PARAMETER_GTKMAIN_PANEL);
      for (tmplst = panelLst; tmplst; tmplst = g_list_next(tmplst))
	g_string_append_printf(data, "%s[gtk]: %s %s\n", FLAG_PARAMETER_GTKMAIN_PANEL,
			       toolPanelGet_id(TOOL_PANEL(tmplst->data)),
			       toolPanelGet_containerId(TOOL_PANEL(tmplst->data)));
      g_string_append_printf(data, "\n");
      g_list_free(panelLst);
    }

  /* Write the panel list. */
  dockLst = toolPanelClassGet_allWindows();
  if (dockLst)
    {
      g_string_append_printf(data, "# %s\n", DESC_PARAMETER_GTKMAIN_DOCK);
      for (tmplst = dockLst; tmplst; tmplst = g_list_next(tmplst))
	{ 
	  toolPanelClassGet_windowCharacteristics
	    ((DockWindow*)(tmplst->data), &id,
	     &visilibity, &x, &y, &width, &height);
	  g_string_append_printf(data, "%s[gtk]: %d %dx%d %dx%d %s\n",
				 FLAG_PARAMETER_GTKMAIN_DOCK,
				 (int)visilibity, x, y, width, height, id);
	}
      g_string_append_printf(data, "\n");
      g_list_free(dockLst);
    }
}
gboolean gtkMainInit_panels(gpointer data _U_)
{
  panelBrowserSet_currentDirectory(visuGtkGet_lastOpenDirectory());
  return FALSE;
}
gboolean gtkMainRun_commandLine(gpointer data)
{
  int i, somethingIsLoaded, presetShade, nb;
  int *mapPlaneId;
  gboolean res, new;
  gchar *planeFile, *surfFile, *bgFile, *colorizeFilename, *valueFile;
  float *values;
  int *colUsed;
  float *translations, *extension;
  GList *list;
  Shade *shade;
  VisuData *obj;
  GError *error;
  OptionTable *table;
  gchar **names;
  Plane **planes;
  GtkMain *commandPanel;

  g_return_val_if_fail(IS_GTK_MAIN_TYPE(data), FALSE);

  commandPanel = GTK_MAIN(data);
  obj = renderingWindowGet_visuData(RENDERING_WINDOW(commandPanel->renderingWindow));
  if (!obj)
    return FALSE;

  /* Post loading sequence, corresponding to all other arguments
     given to command line. */
  /* The value File, maybe used later. */
  valueFile = commandLineGet_valueFile();
  /* translate argument */
  translations = commandLineGet_translation();
  if (translations)
    {
      visuDataSet_XYZtranslation(obj, translations);
      visuData_constrainedInTheBox(obj);
    }
  /* A possible shade. */
  presetShade = commandLineGet_colorizePresetColor();
  if (presetShade < 0)
    presetShade = 0;
  /* colorize argument */
  colorizeFilename = commandLineGet_colorizeFileName();
  if (colorizeFilename)
    {
      somethingIsLoaded = panelDataFileLoad_file(obj, colorizeFilename, &new);
      if (somethingIsLoaded)
	{
	  list = toolShadeBuild_presetList();
	  if (list)
	    {
	      shade = (Shade*)g_list_nth_data(list, presetShade);
	      if (shade)
		panelDataFileSet_presetShade(shade);
	    }
	  colUsed = commandLineGet_colorizeColUsed();
	  if (colUsed)
	    for (i = 0; i < 3; i++)
	      dataFileSet_colUsed(obj, colUsed[i] - 1, i);
	  panelDataFileSet_use(TRUE);
	}
    }
  /* plane file argument */
  planeFile = (valueFile)?valueFile:commandLineGet_planesFileName();
  if (planeFile)
    {
      panelPlanesSet_use(obj, TRUE);
      error = (GError*)0;
      res = panelPlanesLoad_file(obj, planeFile, &error);
      if (error)
	{
	  if (error->code != G_MARKUP_ERROR_EMPTY)
	    visuGtkRaise_warning(_("Loading a value file"),
				 error->message, (GtkWindow*)0);
	  g_clear_error(&error);
	}
      else if (res)
	panelPlanesApply_hidingScheme(obj);
    }
  /* iso-surface argument. */
  table = commandLineGet_options();
  surfFile = commandLineGet_isoSurfacesFileName();
  if (surfFile)
    {
      if (panelIsosurfacesLoad_file(surfFile, commandLineGet_fitToBox(),
				    obj, table, (double**)0))
	{
	  panelIsosurfacesShow_all(TRUE);
	  planes = panelPlanesGet_listOfAllPlanes(TRUE);
	  panelIsosurfacesHide(planes);
	  g_free(planes);
	  panelIsosurfacesSet_used(obj, TRUE);
	}
    }
  /* scalar field argument. */
  surfFile = commandLineGet_scalarFieldFileName();
  if (surfFile)
    {
      if (panelIsosurfacesLoad_file(surfFile, commandLineGet_fitToBox(),
				    obj, table, (double**)0))
	{
	  values = commandLineGet_isoValues(&nb);
	  names = commandLineGet_isoNames(&nb);
	  for (i = 0; i < nb; i++)
	    panelIsosurfacesAdd_withValue(surfFile, values[i], names[i]);
	  panelIsosurfacesShow_all(TRUE);
	  /* Mask the surfaces with planes. */
	  planes = panelPlanesGet_listOfAllPlanes(TRUE);
	  panelIsosurfacesHide(planes);
	  g_free(planes);
	  /* Add possible values from a file. */
	  if (valueFile)
	    {
	      error = (GError*)0;
	      if (!panelIsosurfacesParse_XMLFile(valueFile, &error))
		{
		  if (error->code != G_MARKUP_ERROR_EMPTY)
		    visuGtkRaise_warning(_("Loading a value file"),
					 error->message, (GtkWindow*)0);
		  g_clear_error(&error);
		}
	    }

	  panelIsosurfacesSet_used(obj, TRUE);
	}
    }
  /* The extension. */
  extension = commandLineGet_extension();
  if (extension)
    {
      if (!translations)
	visuData_constrainedInTheBox(obj);
      visuData_replicate(obj, extension, (gboolean*)0);
    }
  /* The coloured map argument. */
  mapPlaneId = commandLineGet_coloredMap();
  if (mapPlaneId)
    {
      DBG_fprintf(stderr, "Gtk Main: set a colour map.\n");
      panelMapSet_scale(commandLineGet_logScale());
      panelMapSet_nIsoLines(commandLineGet_nIsoLines());
      panelMapSet_isoLinesColor(commandLineGet_isoLinesColor());
      panelMapSet_precision(commandLineGet_mapPrecision());
      for (i = 1; i <= mapPlaneId[0]; i++)
	{
	  DBG_fprintf(stderr, " | for plane %d (%d).\n", mapPlaneId[i], i);
	  panelMapSet((guint)mapPlaneId[i], 0, (guint)presetShade);
	}
    }
  /* The background image. */
  bgFile = commandLineGet_bgImage();
  if (bgFile)
    panelFogBgColorSet_bgImage(bgFile);
  /* The pick information. */
  if (valueFile)
    {
      gtkMainBuild_interactiveDialog(commandPanel);
      error = (GError*)0;
      if (!gtkPickParse_XMLFile(valueFile, obj, &error))
	{
	  if (error->code != G_MARKUP_ERROR_EMPTY)
	    visuGtkRaise_warning(_("Loading a value file"),
				 error->message, (GtkWindow*)0);
	  g_clear_error(&error);
	}
    }

  visuData_createAllNodes(obj);
  g_idle_add(visuObjectRedraw, (gpointer)0);
  
  return FALSE;
}

/*******************/
/* Class routines. */
/*******************/
GtkMain* gtkMainClassGet_currentPanel()
{
  return currentGtkMain;
}
void gtkMainClassSet_currentPanel(GtkMain *main)
{
  g_return_if_fail(IS_GTK_MAIN_TYPE(main));
  currentGtkMain = main;
}

void gtkMainClassSet_rememberPosition(gboolean val)
{
  g_return_if_fail(my_class);

  DBG_fprintf(stderr, "Gtk Main: set the remember parameter to %d.\n", val);
  my_class->rememberWindowPosition = val;
}
gboolean gtkMainClassGet_rememberPosition()
{
  g_return_val_if_fail(my_class, PARAMETER_GTKMAIN_REMEMBER_DEFAULT);
  return my_class->rememberWindowPosition;
}

void gtkMainClassCreate_main(GtkWindow **panel, GtkWindow **render)
{
  gboolean oneWindow;
  
  oneWindow = (!strcmp(commandLineGet_windowMode(), "oneWindow"));

  DBG_fprintf(stderr, "Gtk Main: create the main command panel.\n");
  *panel = GTK_WINDOW(gtkMainNew(oneWindow));
  DBG_fprintf(stderr, "Gtk Main: command panel -> %p.\n",
              (gpointer)(*panel));
  if (!oneWindow)
    {
      *render = GTK_WINDOW(g_object_get_data
			   (G_OBJECT(GTK_MAIN(*panel)->renderingWindow),
			    RENDERING_WINDOW_ID));
  
      gtk_widget_show(GTK_WIDGET(*render));
    }
  else
    *render = *panel;
  gtk_widget_show(GTK_WIDGET(*panel));

  /* Force the creation of the Shade selector class. */
  g_type_class_ref(shadeComboBox_get_type());

  visuConfigFileAdd_knownTag("gtk");

  return;
}

