/*
 * UnixCW - Unix CW (Morse code) training program
 * Copyright (C) 2001  Simon Baldwin (simonb@caldera.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * Main.cpp - Main module for Qt-based CW tutor program.
 *
 */

#include <cstdio>
#include <cassert>
#include <cstdlib>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#if defined(HAVE_STRING_H)
#	include <string.h>
#endif /* HAVE_STRING_H */
#if defined(HAVE_STRINGS_H)
#	include <strings.h>
#endif /* HAVE_STRINGS_H */

#if defined(HAVE_GETOPT_H)
#	include <getopt.h>			/* Linux */
#endif /* HAVE_GETOPT_H */

#if !defined(MAXPATHLEN)
#	define	MAXPATHLEN	_POSIX_PATH_MAX	/* OpenServer */
#endif /* not MAXPATHLEN */

/* CW library functions header file. */
#include <cwlib.h>

/* Required C++ headers for classes instantiated from here. */
#include <qapplication.h>
#include "XcwcpApplication.h"


/* Strings displayed for the -v option. */
const char	*version	= "Xcwcp version 2.0";
const char	*copyright	= \
"Xcwcp version 2.0  Copyright (C) 2001  Simon Baldwin\n\n"
"This program comes with ABSOLUTELY NO WARRANTY; for details\n"
"please see the file 'COPYING' supplied with the source code.\n"
"This is free software, and you are welcome to redistribute it\n"
"under certain conditions; again, see 'COPYING' for details.\n"	
"This program is released under the GNU General Public License.";

/* Argument handling macros. */
#define	MAXARGS			128		/* Args to exec */
#define	MAXOPTSTR		1024		/* Longest _OPTIONS env var */
#define	ARGS_WHITESPACE		" \t"		/* Argument separators */
#define	ENV_OPTIONS		"XCWCP_OPTIONS"	/* Env string holding options */

/* Base name of the program, from argv[0]. */
static char *argv0	= NULL;


/*
 * print_usage()
 *
 * Print out a brief message directing the user to the help function.
 */
static void
print_usage ()
{
	fprintf (stderr,
#if defined(HAVE_GETOPT_LONG)
		"Try '%s --help' for more information.\n",
#else /* not HAVE_GETOPT_LONG */
		"Try '%s -h' for more information.\n",
#endif /* not HAVE_GETOPT_LONG */
		argv0);
	exit( 1 );
}

/*
 * print_help()
 *
 * Print out a brief page of help information.
 */
static void
print_help ()
{
	int	min_speed, max_speed;		/* Cw library speed limits */
	int	min_frequency, max_frequency;	/* Cw library frequncy limits */
	int	min_gap, max_gap;		/* Cw library gap limits */
	assert (argv0 != NULL);

	/* Read the cw library limits on its operating parameters. */
	cw_get_speed_limits	(&min_speed, &max_speed);
	cw_get_frequency_limits	(&min_frequency, &max_frequency);
	cw_get_gap_limits	(&min_gap, &max_gap);

	printf (
#if defined(HAVE_GETOPT_LONG)
	"Usage: %s [options...]\n\n\
	-d, --device=DEVICE	use DEVICE for sound ioctl [default stdout]\n\
	-w, --wpm=WPM		set initial words per minute [default %d]\n\
				valid WPM values are between %d and %d\n\
	-t, --hz,--tone=HZ	set initial tone to HZ [default %d]\n\
				valid HZ values are between %d and %d\n\
	-g, --gap=GAP		set extra gap between letters [default %d]\n\
				valid GAP values are between %d and %d\n\
	-h, --help		print this message\n\
	-v, --version		output version information and exit\n\n",
#else /* not HAVE_GETOPT_LONG */
	"Usage: %s [options...]\n\n\
	-d		use DEVICE for sound ioctl [default stdout]\n\
	-w WPM		set initial words per minute [default %d]\n\
			valid WPM values are between %d and %d\n\
	-t HZ		set initial tone to HZ [default %d]\n\
			valid HZ values are between %d and %d\n\
	-g GAP		set extra gap between letters [default %d]\n\
			valid GAP values are between %d and %d\n\
	-h		print this message\n\
	-v		output version information and exit\n\n",
#endif /* not HAVE_GETOPT_LONG */
	argv0,
	cw_get_send_speed (),	min_speed,	max_speed,
	cw_get_frequency (),	min_frequency,	max_frequency,
	cw_get_gap (),		min_gap,	max_gap);
}


/*
 * parse_cmdline()
 *
 * Parse the command line options for initial values for the various
 * global and flag definitions. 
 */
static void
parse_cmdline (int argc, char **argv)
{
	int	c;				/* Option character */
	int	sound_fd = 1;			/* Output file descriptor */
	int	intarg;				/* General integer argument */
	int	argind;				/* Loop index */
	char	env_options[ MAXOPTSTR ];	/* Env options string */
	char	*sptr;				/* String pointer */
	char	*local_argv[ MAXARGS ];		/* Local argv array */
	int	local_argc = 0;			/* Local argc */
	char	device[ MAXPATHLEN ];		/* Path to device file */
#if defined(HAVE_GETOPT_LONG)
	int	option_index;			/* Option index */
	static const struct option long_options[] = {	/* Options table */
		{ "device",	1, 0, 'd' }, { "tone",	1, 0, 't' },
		{ "hz",		1, 0, 't' }, { "wpm",	1, 0, 'w' },
		{ "gap",	1, 0, 'g' },
		{ "help",	0, 0, 'h' },
		{ "version",	0, 0, 'v' },
		{ 0, 0, 0, 0 }};
#endif /* HAVE_GETOPT_LONG */
	strcpy (device, "stdout");

	/* Set argv0 to be the basename part of the program name. */
	argv0 = argv[0] + strlen (argv[0]);
	while (*argv0 != '/' && argv0 > argv[0])
		argv0--;
	if (*argv0 == '/')
		argv0++;

	/*
	 * Build a new view of argc and argv by first prepending
	 * the strings from ..._OPTIONS, if defined, then putting the
	 * command line args on (so they take precedence).
	 */
	local_argv[ local_argc++ ] = argv[0];
	if (getenv (ENV_OPTIONS) != NULL)
	    {
		strcpy (env_options, getenv (ENV_OPTIONS));
		sptr = env_options;
		while (local_argc < MAXARGS - 1)
		    {
			while (strchr (ARGS_WHITESPACE, *sptr) != NULL
					&& *sptr != '\0')
				sptr++;
			if ( *sptr == '\0' )
				break;
			else
			    {
				local_argv[ local_argc++ ] = sptr;
				while (strchr (ARGS_WHITESPACE, *sptr)
						== NULL && *sptr != '\0')
					sptr++;
				if (strchr (ARGS_WHITESPACE, *sptr)
						!= NULL && *sptr != '\0')
				    {
					*sptr = '\0';
					sptr++;
				    }
			    }
		    }
	    }
	for (argind = 1; argind < argc; argind++)
	    {
		local_argv[ local_argc++ ] = argv[ argind ];
	    }

	/* Process every option. */
	while (TRUE)
	    {
#if defined(HAVE_GETOPT_LONG)
		c = getopt_long (local_argc, local_argv, "d:t:w:g:hv",
					long_options, &option_index);
#else /* not HAVE_GETOPT_LONG */
		c = getopt (local_argc, local_argv, "d:t:w:g:hv");
#endif /* not HAVE_GETOPT_LONG */
		if (c == -1)
			break;

		switch (c)
		    {

			case 'd':
				if (sscanf (optarg, "%s", device) != 1)
				    {
					fprintf (stderr,
					    "%s: invalid device value\n",
									argv0 );
					exit (1);
				    }
				sound_fd = open (device, O_RDWR);
				if (sound_fd == -1)
				    {
					fprintf (stderr,
					    "%s: error opening output device\n",
									argv0);
					perror (device);
					exit (1);
				    }
				break;

			case 't':
				if (sscanf (optarg, "%d", &intarg) != 1
					|| cw_set_frequency (intarg) != 0)
				    {
					fprintf (stderr,
					    "%s: invalid tone value\n", argv0);
					exit (1);
				    }
				break;

			case 'w':
				if (sscanf (optarg, "%d", &intarg) != 1
					|| cw_set_send_speed (intarg) != 0)
				    {
					fprintf (stderr,
					    "%s: invalid wpm value\n", argv0);
					exit (1);
				    }
				break;

			case 'g':
				if (sscanf (optarg, "%d", &intarg) != 1
					|| cw_set_gap (intarg) != 0)
				    {
					fprintf (stderr,
					    "%s: invalid gap value\n", argv0);
					exit (1);
				    }
				break;

			case 'h':
				print_help ();
				exit (0);

			case 'v':
				printf ("%s, ", version);
				printf ("%s\n", copyright);
				exit (0);

			case '?':
				print_usage ();

			default:
				fprintf (stderr,
					"%s: getopts returned %c\n",
								argv0, c);
				exit (1);
		    }

	    }
	if (optind != local_argc)
		print_usage ();

	/* Check that the output device, default or selected, does sound. */
	if (cw_set_file_descriptor (sound_fd) != 0)
	    {
		fprintf (stderr, "%s: output device won't do sound\n", argv0);
		perror (device);
		exit (1);
	    }
}


/*
 * main
 *
 * Parse the command line, initialize a few things, then instantiate the
 * XcwcpApplication and wait.
 */
int
main (int argc, char **argv)
{
	/* Parse the command line parameters and arguments. */
	parse_cmdline (argc, argv);

	/* Create the application, and display its windows. */
	QApplication application (argc, argv);

	XcwcpApplication *xcwcp = new XcwcpApplication ();
	xcwcp->setCaption ("Xcwcp");
	xcwcp->show ();
	application.connect (&application, SIGNAL (lastWindowClosed ()),
						&application, SLOT (quit ()));

	/* Enter the application event loop. */
	return application.exec ();
}
