/*
 * twlog:  A basic ham loging program using Motif
 * Copyright (C) 1997->2006 Ted Williams WA0EIR 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge. MA 02139,
 * USA. See the COPYING file in this directory.
 *
 * Version: 2.4 -  Oct 2006
 */

/*
 * TWLOG CALLBACKS FUNCTIONS
 */

#include "twlog.h"

/* btn numbers for File */
#define PRINT    0
#define NEW      1
#define CONVERT  2
#define QRT      3

/* btn numbers for Edit */
#define APPEND   0
#define CLEAR    1
#define EDIT     2

/* btn number of About btn */
#define ABOUT    0

extern int shmid;          /* IPC ID for shared memory */
extern void *shareCall;    /* Shared memory ptr - call of current station */

/*
 * dateCB Callback
 * Puts the date into the date text field and forces 
 * focus to the startPB.
 */
void dateCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   Cdata *wids = (Cdata *) cdata;
   time_t current;
   struct tm *timestruct = NULL;
   char date[20];

   time (&current);
   timestruct = localtime (&current);

   strftime (date, sizeof (date), "%d %b %Y", timestruct);
   XtVaSetValues (wids->TF1, XmNvalue, date, NULL);
   XmProcessTraversal (wids->next, XmTRAVERSE_CURRENT);
}


/*
 * startCB Callback
 * Puts the date and time into the date and start time text fields
 * and forces focus to the Call text field.
 */
void startCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   Cdata *wids = (Cdata *) cdata;
   time_t current;
   struct tm *timestruct = NULL;
   char date[20];
   char stime[20];

   time (&current);
   timestruct = localtime (&current);

   strftime (date, sizeof (date), "%d %b %Y", timestruct);
   strftime (stime, sizeof (stime), "%H%M %Z", timestruct);

   XtVaSetValues (wids->TF1, XmNvalue, date, NULL);

   XtVaSetValues (wids->TF2, XmNvalue, stime, NULL);

   XmProcessTraversal (wids->next, XmTRAVERSE_CURRENT);
}


/*
 * traversCB Callback
 * Forces traversal to the next widget.
 */
void traverseCB (Widget w, XtPointer cdata, XtPointer cbstruct)
{
   XmProcessTraversal (w, XmTRAVERSE_NEXT_TAB_GROUP);
}


/*
 * callsignCB Callback
 * modify/verify callback changes all chars to upper case
 * activate callback - Moves focus to next tab group, the Band Menu.
 * losing focus callback - shares call and starts search 
 */
void callsignCB (Widget w, XtPointer cdata, XtPointer cbstruct)
{
   int i;
   XmTextVerifyCallbackStruct *cbs = (XmTextVerifyCallbackStruct *) cbstruct;
   char *hisCall = NULL;

   switch (cbs->reason)
   {
      case XmCR_MODIFYING_TEXT_VALUE:
         for (i = 0; i < cbs->text->length; i++)   /* all to upper case */
         {
            cbs->text->ptr[i] = toupper (cbs->text->ptr[i]);
         }
         break;

      case XmCR_ACTIVATE:      /* force traversal will cause a search */
         XmProcessTraversal (w, XmTRAVERSE_NEXT_TAB_GROUP);
         break;

      case XmCR_LOSING_FOCUS:   /* in case you don't hit NL */

         XtVaGetValues (w,      /* get his call */
            XmNvalue, &hisCall, NULL);

         strncpy (shareCall, hisCall, SHMSIZE); /* and put it in shm */

         /* create popup and search for His call to see if we have */
         /* worked him before */
         /* TJW - causes problem opening logfile if permission are funky */
         if (mwText == NULL)
         {
            build_editor ();
         }
         if (strlen (hisCall) > 1)      /* no search if only one char */
            callSearch (hisCall);

         XmProcessTraversal (w, XmTRAVERSE_NEXT_TAB_GROUP);
         break;
   }
}


/*
 * search for call sign
 */
void callSearch (char *pattern)
{
   int i, k, rowcnt;
   static Widget mw;
   String str;
   String val;

   if (searchSH == NULL)        /* if first time, make the sh */
   {
      mw = build_searchSH ();
   }

   rowcnt = XbaeMatrixNumRows (mw);
   XbaeMatrixDeleteRows (mw, 0, rowcnt);

   XbaeMatrixDisableRedisplay (mw);
   rowcnt = XbaeMatrixNumRows (mwText);

   /* searching from botton up so recent first */
   for (i = rowcnt - 1; i >= 0; i--) 
   {
      str = XbaeMatrixGetCell (mwText, i, 3);
      if (strstr (str, pattern) != NULL)
      {
         XbaeMatrixAddRows (mw, 0, NULL, NULL, NULL, 1);
         for (k = 0; k <= 10; k++)
         {
            val = XbaeMatrixGetCell (mwText, i, k);
            XbaeMatrixSetCell (mw, 0, k, val);
         }
      }
   }

   if (XbaeMatrixNumRows (mw) > 0)
   {
      XtManageChild (searchSH);
   }
   else
   {
      XtUnmanageChild (searchSH);
   }

   XbaeMatrixEnableRedisplay (mw, True);
}


/*
 * build search results SH
 */
Widget build_searchSH (void)
{
   Widget form3, closePB, sep2, mwSearch;

   searchSH = XtVaCreatePopupShell ("searchSH", topLevelShellWidgetClass, logSH,
      XmNmwmDecorations, MWM_DECOR_RESIZEH | MWM_DECOR_TITLE | MWM_DECOR_BORDER,
      XmNtitle, "TWLOG CALL SEARCH",
      NULL);

   form3 = XtVaCreateWidget ("form3", xmFormWidgetClass, searchSH,
      NULL);

   closePB = XtVaCreateManagedWidget ("Close", xmPushButtonWidgetClass,
      form3,
      XmNwidth, PBWIDTH,
      XmNleftAttachment, XmATTACH_POSITION,
      XmNleftPosition, 50,
      XmNleftOffset, -PBWIDTH / 2,
      XmNbottomAttachment, XmATTACH_FORM, XmNbottomOffset, MARGIN, NULL);

   sep2 = XtVaCreateManagedWidget ("sep2", xmSeparatorWidgetClass, form3,
      XmNtopAttachment, XmATTACH_NONE,
      XmNleftAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_WIDGET,
      XmNbottomWidget, closePB,
      XmNbottomOffset, MARGIN, XmNrightAttachment, XmATTACH_FORM, NULL);


   mwSearch =
      XtVaCreateManagedWidget ("mwSearch", xbaeMatrixWidgetClass, form3,
      XmNtopAttachment, XmATTACH_FORM, XmNleftAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_WIDGET, XmNbottomWidget, sep2,
      XmNbottomOffset, MARGIN, XmNrightAttachment, XmATTACH_FORM,
      XmNcalcCursorPosition, True, XmNvisibleRows, 3, NULL);

   XtAddCallback (closePB, XmNactivateCallback, closeCB, (XtPointer) NULL);
   XtAddCallback (mwSearch, XmNenterCellCallback, enterCellCB,
      (XtPointer) NULL);
   XtManageChild (form3);
   return (mwSearch);
}


/*
 * enterCellCB - makes call search matrix non editable
 * setting map in the cbs to false is all it takes
 */
void enterCellCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   XbaeMatrixEnterCellCallbackStruct *pt =
      (XbaeMatrixEnterCellCallbackStruct *) cbs;

   pt->map = False;
}


/*
 * closeCB - closes the call search shell
 */
void closeCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   XtUnmanageChild (searchSH);
}


/*
 * endCB Callback
 * Puts the time into the End Time text field
 * and moves the focus to the Date Pushbutton.
 */
void endCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   Cdata *wids = (Cdata *) cdata;
   time_t current;
   struct tm *timestruct = NULL;
   char stime[20];

   time (&current);
   timestruct = localtime (&current);

   strftime (stime, sizeof (stime), "%H%M %Z", timestruct);
   XtVaSetValues (wids->TF1, XmNvalue, stime, NULL);
   XmProcessTraversal (wids->next, XmTRAVERSE_CURRENT);
}


/*
 * focusCB Callback
 * This callback is called for any loss or gain of focus in any
 * text field.  The I beam cursor will only be visible if the
 * textfield has focus.
 */
void focusCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   XmAnyCallbackStruct *cbstruct = (XmAnyCallbackStruct *) cbs;

   switch (cbstruct->reason)
   {
      case XmCR_FOCUS:
         XtVaSetValues (w, XmNcursorPositionVisible, TRUE, NULL);
         break;

      case XmCR_LOSING_FOCUS:
         XtVaSetValues (w, XmNcursorPositionVisible, FALSE, NULL);
         break;

      default:
         fprintf (stderr, "Invalid focusCB reason\n");
         break;
   }
}


/*
 * fileCB Callback
 * All callbacks on the file pulldown come here.  The button number
 * is checked to see which option was selected.
 */
void fileCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   int btn = (int) cdata;
   char tail[MAXNAME];
   char *newname;
   char *lprstr;
   time_t current;
   struct tm *timestruct = NULL;
   
   switch (btn)
   {
      case PRINT:
         if ((lprstr = malloc (sizeof (PRINT_CMD)
                  + sizeof (logpath) + MAXNAME)) == NULL)
         {
            perror ("fileCB - print malloc failed");
            exit (1);
         }
         strcpy (lprstr, PRINT_CMD);
         strcat (lprstr, logpath);
         system (lprstr);
         free (lprstr);
         break;

      case NEW:
         time (&current);
         timestruct = localtime (&current);
         strftime (tail, MAXNAME, "/log.%Y.%j.%H%M%S", timestruct);

         if ((newname = malloc (strlen (dirpath) + MAXNAME)) == NULL)
         {
            perror ("newname malloc failed:");
            exit (1);
         }
         strcpy (newname, dirpath);
         strcat (newname, tail);

         if (rename (logpath, newname) == 0)
            creat (logpath, 0644);
         else
         {
            perror ("Rename failed");
            exit (1);
         }
         free (newname);
         sleep (1);             /* paranoia - prevents the same name */
         break;
         
      case CONVERT:
            build_convDiag();
            break;

      case QRT:
         shmdt (shareCall);
         shmctl (shmid, IPC_RMID, NULL);
         exit (0);
   }
}


/*
 * editCB Callback
 * All callbacks on the file pulldown come here.  The button number
 * is checked to see which option was selected.
 */
void editCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   int btn = (int) cdata;
   Widget *col_2_wids;

   switch (btn)
   {
      case APPEND:
         XtVaGetValues (XtParent (w), XmNuserData, &col_2_wids, NULL);
         appendlog (logpath, col_2_wids);
         clearform (col_2_wids);
         break;

      case CLEAR:
         XtVaGetValues (XtParent (w), XmNuserData, &col_2_wids, NULL);
         clearform (col_2_wids);
         break;

      case EDIT:
         popupEdit (logpath);
         break;
   }
}


/*
 * appendlog Function
 * Collects the data from the righthand column and writes it
 * to the end of logfile.
 */
#define MAXBOX 20
#define MAXNOTE 80

void appendlog (char logpath[], Widget * col_2_wids)
{
   int i;
   FILE *fp;
   Widget btnwid;
   XmString label_xs;
   char *data;
   char abox[9][MAXBOX + 1];
   char anote[MAXNOTE + 1];

   for (i = 0; i < 9; i++)
   {
      if (i < 3 || i > 5)
      {
         XtVaGetValues (col_2_wids[i],  /* get text from a textfield */
            XmNvalue, &data, NULL);
         strncpy (abox[i], data, MAXBOX);
      }
      else   /* or get XmString from a menu */
      {
         XtVaGetValues (col_2_wids[i],  /* get the menu button wid, then */
            XmNmenuHistory, &btnwid, NULL);
         XtVaGetValues (btnwid,         /* get label for selected button */
            XmNlabelString, &label_xs, NULL);
         data = XmStringUnparse (label_xs, NULL, XmCHARSET_TEXT, XmCHARSET_TEXT,                                 NULL, 0, XmOUTPUT_ALL);
         strncpy (abox[i], data, MAXBOX);
      }
   }
   XtVaGetValues (col_2_wids[i],        /* get the Note field */
      XmNvalue, &data, NULL);
   strncpy (anote, data, MAXNOTE);

   if ((fp = fopen (logpath, "a")) == NULL)     /* open logfile */
   {
      perror ("appendlog - open failed");
      exit (1);
   }

   fprintf (fp, "%-15s%-10s%-15s%-15s%-10s%-10s%-10s%-10s%-10s%-1s",
      abox[0], abox[1], abox[2], abox[3], abox[4],
      abox[5], abox[6], abox[7], abox[8], anote);
   fprintf (fp, "\n");

   fclose (fp);   /* and close logfile */
}


/*
 * clearform Function
 * Clears all text field widgets
 */
void clearform (Widget * col_2_wids)
{
   int i;

   for (i = 0; i <= 9; i++)
   {
      if (i < 3 || i > 5)
         XtVaSetValues (col_2_wids[i], XmNvalue, "", NULL);
   }

   /* clear the callsign shared memory if set */
   if (shareCall != NULL)
   {
      strcpy (shareCall, "");
   }
}


/*
 * helpCB Callback
 * All callbacks on the help pulldown come here.  The button number
 * is checked to see which option was selected.
 */
void helpCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   int btn = (int) cdata;
   switch (btn)
   {
      case ABOUT:
         aboutDiag ();
         break;

      default:
         popupHelp (btn);
         break;
   }
}


/*
 * initCB Callback - focus CB on form1
 * Initialize dirpath and logpath.  If they don't exist, popup a 
 * dialog to see if we really want to create them. 
 */
void initCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   struct stat buf;
   int eflag = 0;
   Widget initDiag;
   XmString xs;
   char *mess =
           "Twlog needs to create .twlogDir/logfile in your home directory.\n"
           "Click OK to create it or Cancel to exit.\n";

   /* remove this callback so it is only called at start up */
   XtRemoveCallback (w, XmNfocusCallback, initCB, NULL);

   if (stat (getenv ("HOME"), &buf) == -1)      /* Do you have a $HOME? */
   {
      fprintf (stderr, "initCB: Can't find your \"$HOME\"\n");
      exit (-1);
   }

   strcpy (dirpath, getenv ("HOME"));   /* $HOME to dir path */
   strcat (dirpath, LOGDIR);            /* then add LOGDIR */
   if ((stat (dirpath, &buf)) == -1)
   {  /* LOGDIR does not exist */
      fprintf (stderr, "initCB: Can't find log directory %s\n", dirpath);
      eflag++;
   }
   else
   {
      /* it exists, but is it a dir? */
      if (!(S_ISDIR(buf.st_mode)))
      {
         fprintf (stderr, "initCB: twlogDir exists, but isn't a directory\n");
         exit (-1);
      }
   }

   strcpy (logpath, dirpath);           /* dirpath to logpath */
   strcat (logpath, LOGFILE);           /* add LOGFILE name   */
   if (stat (logpath, &buf) == -1)      /* and check that it exists */
   {
      fprintf (stderr, "initCB: Can't find %s\n", logpath);
      eflag++;
   }

   if (eflag > 0)
   {
      /* create and popup the dialog */
      //TJW use below - via Jacob KD7YKO xs = XmStringCreateLocalized (mess);
      xs = XmStringCreateLtoR (mess, XmSTRING_DEFAULT_CHARSET);
      initDiag = XmCreateQuestionDialog (logSH, "Create Files", NULL, 0);

      XtVaSetValues (initDiag,
         XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL,
         XmNmessageString, xs,
         NULL);
      XmStringFree (xs);
      XtUnmanageChild (XmMessageBoxGetChild (initDiag, XmDIALOG_HELP_BUTTON));

      /* add its callbacks */
      XtAddCallback (initDiag, XmNokCallback, initDiagCB, (XtPointer)eflag);
      XtAddCallback (initDiag, XmNcancelCallback, initDiagCB, (XtPointer)eflag);
      XtManageChild (initDiag);
   }
}


/*
 * initDiagCB - OK and Cancel callback for initDiag
 * cdate is eflag from initDiag
 * eflag==1: logfile is missing 
 * eflag==2: both logpath and dirpath are missing 
 */
void initDiagCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   int eflag = (int) cdata;
   int modeDir = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
   int modeLog = S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH;

   XmAnyCallbackStruct *ptr = (XmAnyCallbackStruct *) cbs; 

   switch (ptr->reason)
   {
      case XmCR_CANCEL:
         exit (-1);      /* Cancel - quit twlog */
         break;

      case XmCR_OK:
         switch (eflag)
         {
            case 2:
               printf ("case 2 - make dir %s\n", dirpath);
               mkdir (dirpath, modeDir);
               /* No break - we need to fall through */
            case 1:
               printf ("case 1 - make file %s\n", logpath);
               creat (logpath, modeLog);
               break;
            default:
               printf ("initDiagCB: invalid eflag\n");
               exit (-1);
         }
         break;   
      default:
         fprintf (stderr, "initDiagCB - invalid CB reason\n");
         exit (-1);
   }
}


/*
 * popupHandler - Event handler to popup menu
 */
void popupHandler (Widget w, XtPointer cdata, XEvent * event, Boolean * cont)
{
   Widget menuWid = (Widget) cdata;

   if (event->type == ButtonPress && event->xbutton.button == Button3)
   {
      XmMenuPosition (menuWid, &(event->xbutton));
      XtManageChild (menuWid);
   }
}


