/* 
 * twcw - Copyright (C) 2006 by Ted Williams - WA0EIR
 * Derived mostly from qrq.c
 * High speed morse trainer by Fabian Kurz, DJ1YFK, Copyright (C) 2006, and
 * a few bits from unixcw
 * by Simon Baldwin, G0FRD.Copyright (C) 2001-2006
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/ 

#include "libCW.h"

#define TESTMODE 0
#define MESG 0        /* 0 = no message support, 1 =  with message support */
#define DEBUG 0

#if MESG == 1
extern int qid;
#endif

int doReply;                                   /* send reply msg when true */
int key_modes = 0;                             /* no key modes selected */

int rts_on = TIOCM_RTS;                        /* for key down */
int rts_off = 0;                               /* for key up */
int initialized = -1;                          /* set to 1 by cw_init */

char dspdevice[20] = DEFAULT_SC_DEVICE;        /* default dsp */
int  dsp_fd = 0;
char serdevice[20] = DEFAULT_RTS_DEVICE;       /* default serial */
int  ser_fd = 0;
char mixdevice[20] = DEFAULT_MIX_DEVICE;       /* default mixer */
int  mix_fd = 0;
int  rtc_rate = 2048;                          /* RTC interupt rate */

char *codetable[MAX_LETTERS];

int  send_vol = (DEFAULT_VOL<<8)+DEFAULT_VOL;  /* sending vol level */
int  save_vol;                                 /* original vol level */
static int  freq = 500;                        /* freq value for SC */
static long samplerate = SAMPLERATE;

float wpm = DEFAULT_WPM;                       /* current speed in wpm */
float farns_wpm = DEFAULT_FARNS;               /* farnsworth speed in wpm */

float dit_time = 1200.0 / DEFAULT_WPM;
float dah_time = 1200.0 / DEFAULT_WPM * 3;     /* dah time in milliseconds */
float farns_time;                              /* farns time */

float ic_time = 1200 / DEFAULT_WPM;            /* intra letter time */
float il_time = 3 * 1200 / DEFAULT_WPM;        /* inter letter time */
float iw_time = 7 * 1200 / DEFAULT_WPM;        /* inter word time */

static float rise = RISE;                      /* risetime in ms */
static float fall = FALL;                      /* fall time in ms */
static float rt = SAMPLERATE / 1000 * RISE;    /* normalized risetime */
static float ft = SAMPLERATE / 1000 * FALL;    /* normalized falltime */

static pthread_t cwthread;                     /* thread for CW output */
static pthread_attr_t cwattr;                  /* thread attributes */

extern struct cwirc_extension_api *cwirc_api_shm;
struct cwirc_extension_api *cwirc_api_shm;

/*internal use */
void cleanup ();
void *send_cw (void *p);
void keySC (char *code);
void keyRTS (char *code);
void keyIRC (char *code);
int  tonegen (int freq, float length);
int  get_volume (void);
int  set_volume (int vol);
int  open_dsp (char *device);
int  open_mixer (char *device);
int  open_serial (char *device);
int  timer (float usec);
int  close_devs(void);


/*
 * cw_init function
 */
int cw_init (char *dsp, char *mix, char *ser)
{
   int i, fd, rtn;

   /* save the dsp, mixer and serial args if not NULL */
   /* must have both dsp and mix to use the sound card */

   if (dsp != NULL && mix != NULL)
   {
      strcpy (dspdevice, dsp);
      strcpy (mixdevice, mix);
      mix_fd = open_mixer (mixdevice);
      save_vol = get_volume();
      close (mix_fd);
   }
   if (ser != NULL)
   {
      strcpy (serdevice, ser);
      /* open rtc */
      if ((fd = open ("/dev/rtc", O_RDONLY)) < 0)
      {
         perror ("cw_init: open /dev/rtc failed");
         return -1;
      }
      /* find the highest interupt rate */
      /* if 64 is the system default rate, thats kinda slow */
      while (ioctl(fd, RTC_IRQP_SET, rtc_rate) < 0)
      {
         fprintf (stderr, "cw_init RTC_IRQP_SET %d failed\n", rtc_rate);
         rtc_rate = rtc_rate/2;
      }
#if DEBUG == 1
      fprintf (stderr, "RTC rate = %d\n", rtc_rate);
#endif
      close (fd);
   }

  /* Initialize the codetable array */
   for (i = 0; i < MAX_LETTERS; i++)
      codetable[i] = " ";

   /* Pro Signs and Special */   
   codetable[1]         = "......"; /* F1 = Error */
   codetable[2]         = ".-.-.";    /* F2 = <AR>  */
   codetable[3]         = "...-.-";   /* F3 = <SK>  */
   codetable[4]         = "-.--.";    /* F4 = <KN>  */
   codetable[5]         = ".-...";    /* F5 = WAIT  */
   codetable[6]         = "-...-.-";  /* F6 = BREAK */

   /* Numbers */
   codetable[(int) '1'] = ".----";
   codetable[(int) '2'] = "..---";
   codetable[(int) '3'] = "...--";
   codetable[(int) '4'] = "....-";
   codetable[(int) '5'] = ".....";
   codetable[(int) '6'] = "-....";
   codetable[(int) '7'] = "--...";
   codetable[(int) '8'] = "---..";
   codetable[(int) '9'] = "----.";
   codetable[(int) '0'] = "-----";
   codetable[216]       = "-----";    /* 216 or 330h is a zero with a slash */

   /* Alphabet */
   codetable[(int) ' '] = " ";
   codetable[(int) 'A'] = ".-";
   codetable[(int) 'B'] = "-...";
   codetable[(int) 'C'] = "-.-.";
   codetable[(int) 'D'] = "-..";
   codetable[(int) 'E'] = ".";
   codetable[(int) 'F'] = "..-.";
   codetable[(int) 'G'] = "--.";
   codetable[(int) 'H'] = "....";
   codetable[(int) 'I'] = "..";
   codetable[(int) 'J'] = ".---";
   codetable[(int) 'K'] = "-.-";
   codetable[(int) 'L'] = ".-..";
   codetable[(int) 'M'] = "--";
   codetable[(int) 'N'] = "-.";
   codetable[(int) 'O'] = "---";
   codetable[(int) 'P'] = ".--.";
   codetable[(int) 'Q'] = "--.-";
   codetable[(int) 'R'] = ".-.";
   codetable[(int) 'S'] = "...";
   codetable[(int) 'T'] = "-";
   codetable[(int) 'U'] = "..-";
   codetable[(int) 'V'] = "...-";
   codetable[(int) 'W'] = ".--";
   codetable[(int) 'X'] = "-..-";
   codetable[(int) 'Y'] = "-.--";
   codetable[(int) 'Z'] = "--..";

   /* Punctuation */
   codetable[(int) '='] = "-...-";
   codetable[(int) '/'] = "-..-.";
   codetable[(int) '?'] = "..--..";
   codetable[(int) ','] = "--..--";
   codetable[(int) '.'] = ".-.-.-";
   codetable[(int) ':'] = "---...";
   codetable[(int) '\''] = ".----.";
   codetable[(int) '"'] = ".-..-.";
   codetable[(int) '_'] = "..--.-";
   codetable[(int) '('] = "-.--.";
   codetable[(int) ')'] = "-.--.-";
   codetable[(int) '#'] = "-.---";
   codetable[(int) '-'] = "-...-";
   codetable[(int) '|'] = "...-..";
   codetable[(int) '\\'] = "-.....";
   codetable[(int) '*'] = "-----.";
   codetable[(int) ';'] = "-.-.-.";
   codetable[(int) '@'] = ".--.-.";
   codetable[(int) '^'] = "....--.-.";
   codetable[(int) '$'] = "...-..-";
   codetable[(int) '!'] = "....-.";
   codetable[(int) '>'] = "....---.";
   codetable[(int) ']'] = "....-....";
   codetable[(int) '['] = "....-..";
   codetable[(int) '<'] = "....-.-..";
   codetable[(int) '&'] = "....--.";
   codetable[(int) '%'] = "....-.--.";
   codetable[(int) '~'] = "....--";
   codetable[(int) '+'] = ".-.-.";
   codetable[(int) '{'] = "....-.--";
   codetable[(int) '}'] = "....--..-";

   /* Initialize the morse thread */
   pthread_attr_init (&cwattr);
   pthread_attr_setdetachstate (&cwattr, PTHREAD_CREATE_JOINABLE);

   /*
    * trap the following signals for the cleanup function
    */
   signal (SIGINT,  (void *)cleanup);
   signal (SIGTERM, (void *)cleanup);
   signal (SIGQUIT, (void *)cleanup);
   signal (SIGSEGV, (void *)cleanup);

   /* all done so, set initialize flag */
   initialized = 1;

   /* nothing to join right now just so go for it */
   rtn = pthread_create (&cwthread, NULL, &send_cw, (void *) "");
   if (rtn < 0)
   {
      fprintf (stdout, "cw_send_str: pthread_create error\n");
      return -1;
   }
   return 0;
}


/*
 * cleanup
 * Make sure the sidetone is off and the key is up, then exit
 */
void cleanup (void)
{
   signal (SIGINT,  SIG_IGN);
   signal (SIGTERM, SIG_IGN);
   signal (SIGQUIT, SIG_IGN);
   signal (SIGSEGV, SIG_IGN);
   fprintf(stderr, "libCW terminating\n");

   if (ser_fd != 0)
   {
      if (ioctl (ser_fd, TIOCMSET, &rts_off) == -1)
      {
         perror("cleanup: RTS Clear failed");
      }
    }
   fprintf (stderr, "Thread killed\n");
   pthread_kill (cwthread, 9);
   exit (73);
}


/*
 * close_devs - close opened devices
 */
int close_devs (void)
{
   int rtn = 0;

   /* close used devices */
   if (dsp_fd != 0)                      /* if dsp_fd is opened */
   { 
      close (dsp_fd);                    /* close dsp device */
      dsp_fd = 0;
      set_volume (save_vol);             /* set saved vol level */
      rtn = 1;
   }

   if (ser_fd != 0)                      /* if ser_fd is opened */
   {
      /* make sure the key is up */
      if (ioctl (ser_fd, TIOCMSET, &rts_off) < 0)
      {
         perror ("key_other_stuff RTS Clear failed");
         return (-1);
      }
      close (ser_fd);                    /* close ser device */
      ser_fd = 0;
      rtn = 1;
   }
   return (rtn);
}


/*
 * cw_send_str
 * sends a string of characters by starting the send_str thread
 * Called by other cw_send_* functions.
 */
int cw_send_str (char *str)
{
   int rtn;
   static char s[MAX_WORD_LENGTH];

   /* cw_init must be called first */
   if (initialized == -1)
   {
      fprintf (stderr, "cw_send_str: cw_init must be called first.\n");
      return -1;
   }

   /* wait for any current string */
   pthread_join(cwthread, NULL);
   strcpy (s, str);                                 /* save the string */
   rtn = pthread_create (&cwthread, NULL, &send_cw, (void *) s);
   if (rtn < 0)
   {
      fprintf (stdout, "cw_send_str: pthread_create error\n");
      return -1;
   }
   return 0;
}


/*
 * cw_send_word
 * this does the same as send str, but adds a space to the end
 * to make it compatable for twcw.
 */
int cw_send_word (char *str)
{
   static char new_str[MAX_WORD_LENGTH+1];

   strcpy (new_str, str);
   strcat (new_str, " ");
   cw_send_str (new_str);
   return (0);
}


/*
 * cw_send_char - sends one char
 * This method causes a little extra overhead and the timing
 * is off slightly - abt 1 sec longer than it should be over 60 secs. 
 */
int cw_send_char (char ch)
{
   char str[2];

   str[0] = ch;
   str[1] = '\0';
   cw_send_str (str);
   return (0);
}


/*
 * send_cw
 * started as thread by cw_send_str() or cw_send_word()
 */
void *send_cw (void *p)
{
   int i=0;
   char *pt;
   char *code;

#if MESG == 1
   struct tag1a
   {
      long type;
      union
         {
            char word[MAX_WORD_LENGTH];
            int  val;
         } data;
   } reply;
#endif

   /*
    * open needed devices
    */
   if ((key_modes & SC) == SC)
   { 
      dsp_fd = open_dsp (dspdevice);                /* open sound card */
   }
   if ((key_modes & RTS) == RTS)                    /* if RTS mode */
   {
      ser_fd = open_serial (serdevice);             /* open serdevice */
   }

   pt = (char *) p;
   /* now run thru the string, calling keySC or keyTimer or keyIRC */
   while (pt[i] != '\0')
   {
      /* get the ./- representation for the character */
      code = codetable[(int)toupper((int)pt[i])];
      /* key devices */
      if ((key_modes & SC) == SC)
         keySC (code);

      if ((key_modes & RTS) == RTS)
         keyRTS(code);

      if ((key_modes & IRC) == IRC)
         keyIRC(code);

#if MESG == 1
      /* if doReply is true and MESG is 1, we need to send a reply message */
      if (doReply == TRUE)
      {
         reply.type = SENT_MSG;
         reply.data.word[0] = pt[i];
         reply.data.word[1] = (char) NULL;
         if (msgsnd (qid, (struct msgbuf *)&reply,
                     sizeof(reply.data), 0) == -1)
         {
            perror ("keyer.c doWord: msgsnd SENT_MSG failed");
            exit (1);
         }
      }
#endif
      i++;
   }
   close_devs ();
   pthread_exit(0);
}


/*
 * keySC (char *code)
 * uses the morse representation to key the SC
 */
void keySC (char *code)
{ 
   int i = 0;
   float t;

   while (code[i] != '\0')
   {   
      if (code[i] == ' ')                /* do a space or non-valid char */
      {                                  /* this assumes that the that */
         tonegen (0, iw_time - il_time); /* a previous character has done */
         return;                         /* done the il_time already */
      }

      if (code[i] == DIT)
      {
         t = dit_time;
      }
      else
      {
         t = dah_time;
      }

      set_volume (send_vol);             /* set send vol level */
      tonegen (freq, t);                 /* and send the dit/dah */

      i++;
      if (strlen (code) > i)             /* if more ./- to do */
      {                                  /* for this character */ 
         tonegen (0, ic_time);           /* add intra character time */
      }
   }
   tonegen (0, il_time);                 /* do inter letter time */
   return;
}


/*
 * keyRTS (char code)
 * uses the morse representation to key the RTS
 */
void keyRTS (char *code)
{ 
   int i = 0;
   float t;

   while (code[i] != '\0')
   {   
      if (code[i] == ' ')                      /* do a space or non-valid ch */
      {
         timer (iw_time - il_time);
         return;
      }

      if (code[i] == DIT)
         t = dit_time;
      else
         t = dah_time;

      if (ioctl (ser_fd, TIOCMSET, &rts_on) < 0)    /* key down */
      {
         perror("key_other_stuff: RTS Set failed");
         exit (1);
      }

      timer (t);

      if (ioctl (ser_fd, TIOCMSET, &rts_off) < 0)   /* RTS key up */
      {
            perror ("key_other_stuff RTS Clear failed");
      }

      i++;
      if (strlen (code) > i)                   /* if more ./- to do */
      {                                        /* for this character */ 
         timer (ic_time);
      }
   }
      timer (il_time);
}


/*
 * keyIRC
 * uses the morse representation to key the cwIRC
 */
void keyIRC (char *code)
{
   int i = 0;
   float t;

   while (code[i] != '\0')
   {   
      if (code[i] == ' ')                /* do a space or non-valid ch */
      {
         timer (iw_time - il_time);
         return;
      }

      if (code[i] == DIT)
         t = dit_time;
      else
         t = dah_time;

      /* key cwirc */
      cwirc_api_shm->in_key = 1;
      timer (t);
      cwirc_api_shm->in_key = 0;

      i++;
      if (strlen (code) > i)             /* if more ./- to do */
      {                                  /* for this character */ 
         timer (ic_time);
      }
   }
   timer (il_time);
   return;
}


/*
 * tonegen
 * This function was taken form qrq.  Copyright (C) 2006  Fabian Kurz, DJ1YFK
 * Generates a sinus tone of frequency 'freq' and length 'len'
 * based on 'samplerate', 'rise' (rise time), 'fall' (fall time)
 * And no sound card POPS! Thanks, Fabian.
 */
int tonegen (int freq, float len)
{
   int x;
   int out;
   double val;

   /* convert len from milliseconds to samples, determine rise/fall time */
   len = (int) (samplerate * (len / 1000.0));
   for (x=0; x < len-1; x++)
   {
      val = sin(2*PI*freq*x/samplerate);
      if (x < rt)
      {
         val *= sin(PI*x/(2.0*rt));                 /* rising edge */
      }

      if (x > (len-ft))
      {
         val *= sin(2*PI*(x-(len-ft)+ft)/(4*ft));   /* falling edge */
      }
      out = (int) (val * 32500.0);
      out = out + (out<<16);                        /* add second channel */
      write(dsp_fd, &out, sizeof(out));
   }
   return 0;
}


/*
 * timer - use the real time clock for accurate timing
 */
int timer (float msec)
{
   unsigned long rtc_data;
   int fd;
   int i;
   int cnt;

   cnt = msec * rtc_rate / 1000.0;
   /* open rtc */
   if ((fd = open ("/dev/rtc", O_RDONLY)) < 0)
   {
      perror ("timer: open /dev/rtc failed");
      return -1;
   }

   /* configure rtc for periodic interupts */
   if (ioctl (fd, RTC_IRQP_SET, rtc_rate) < 0)
   {
      perror ("timer: RTC_IRQP_SET");
      return -1;
   }

   if (ioctl (fd, RTC_PIE_ON, 0) < 0)
   {
      perror ("timer: PIE_ON");
      return -1;
   }

   i = 0;
   while (cnt - i > 0)
   {
      if (read (fd, &rtc_data, sizeof(rtc_data)) < 0)
      {
         perror ("timer: RTC read");
         exit (errno);
      }
      i = i + (rtc_data >> 8);
   }

   if (ioctl (fd, RTC_PIE_OFF, 0) < 0)
   {
      perror ("timer: PIE_OFF");
      exit (errno);
   }
   close (fd);
   return 0;
}


/*
 * cw_get_send_volume
 */
int cw_get_send_volume (void)
{
   return (send_vol);
}


/*
 * cw_set_send_volume
 * sets the cw volume level (send_vol)
 */
int cw_set_send_volume (int new_volume)
{
   /* don't want to wait for thread here */
   send_vol = (new_volume << 8) + new_volume;
   return 0;
}


/*
 * set doReply - 
 */
int  cw_set_reply_msg (int new_reply)
{
   doReply = new_reply;
   return (doReply);
}


/*
 * get doReply - 
 */
int  cw_get_reply_msg (void)
{
   return (doReply);
}


/*
 * cw_get_key_modes
 */
int cw_get_key_modes (void)
{
   return (key_modes);
}


/*
 * cw_set_key_modes (int modes)
 * turns on key modes specified by modes.  Use IRC, SC and RTS for modes.
 */
int cw_set_key_modes (int modes)
{
   if (key_modes == 4)                /*if IRC mode, don't change it */
      return 0;

   if ((modes & IRC) == IRC)          /* IRC must be exclusive */
   {
      key_modes = IRC;
   }
   else
   {
      if ((modes & SC) == SC)
      {
         key_modes = modes;
      }
      if ((modes & RTS) == RTS)
      {
         key_modes = modes;
      }
   }
   return 0;
}


/*
 * cw_clear_key_modes (int modes)
 * Modes specifies the key modes to turn off.  Use SC and RTS for
 * mode. They may be or'd together.
 */
int cw_clear_key_modes (int modes)
{
   key_modes = key_modes & (~modes & 0x7);
   return 0;
}


/* 
 * cw_get_wpm
 */
int cw_get_wpm (void)
{
   return (wpm);
}


/*
 * cw_set_wpm
 * set wpm.  Calculate the times for all character, including the
 * iw_time and ic_time for farnsworths, if required. Also figure the
 * rise/fall times
 */
int cw_set_wpm (int new_wpm)
{
   float wpm_time;

   /* cw_init must be called first */
   if (initialized == -1)
   {
      fprintf (stderr, "cw_set_wpm: cw_init must be called first.\n");
      return -1;
   }
   /*
    * we don't want to change speed in the middle of a word
    * so wait for the thread if it is running.
    */
    pthread_join(cwthread, NULL);
   /*
    * save the new wpm values if they are >= 5 - that way this function can
    * be used to set only one argument by setting the other argument to 0.
    */
   if (new_wpm >= 5)
   {
      wpm = new_wpm;
   }
   /*
    * speed is in wpm so we have to calculate the dot-length in
    * milliseconds using the well-known formula  dotlength= 1200 / wpm)
    */
   dit_time = 1200 / wpm;
   dah_time = 3 * dit_time;
   ic_time  = dit_time;

   if ((farns_wpm < wpm) && (farns_wpm > 4))
   { /* do farnsworth timing */
      farns_time = 1200 / farns_wpm * 50.0; /* time to send a "word" @ fwpm */ 
      wpm_time = 31 * dit_time;             /* PARIS minus iw_time @ wpm */
      il_time = 3/19.0 * (farns_time - wpm_time);
      iw_time = 7/19.0 * (farns_time - wpm_time);
   }
   else
   { /* use standard timing */
      il_time = 3 * dit_time;
      iw_time = 7 * dit_time;
   }
   /* rise == risetime in milliseconds, we need nr. of samples (rt) */
   rt = (samplerate / 1000.0) * rise;
   ft = (samplerate / 1000.0) * fall;

#if DEBUG == 1
   fprintf (stderr, "BASIC dit=%4.1f dah=%4.1f il=%4.1f iw=%4.1f\n",
           dit_time, dah_time, il_time, iw_time);
#endif
   return 0;
}


/*
 * cw_set_freq
 * sets the frequency of the audio out
 */
int cw_set_freq (int new_freq)
{
   freq = new_freq;
   return 0;
}
   

/*
 * cw_get_freq
 * gets the frequency of the audio out
 */
int cw_get_freq (void)
{
   return (freq);
}


/* cw_get_fwpm 
 * returns current farnsworth wpm
 */
int cw_get_fwpm (void)
{
   return (farns_wpm);
}


/*
 * cw_set_fwpm
 * set farns_wpm.  Calculate the farnsworth times for  il_time and
 * iw_time for farnsworth.
 */
int cw_set_fwpm (int new_fwpm)
{
   float wpm_time;

   farns_wpm = new_fwpm;
   /*
    * we don't want to change speed in the middle of a word
    * so wait for the thread if it is running.
    */
   pthread_join(cwthread, NULL);

   if ((farns_wpm < wpm) && farns_wpm > FARNS_OFF)
   { /* do farnsworth timing */
      farns_time = 1200 / farns_wpm * 50.0; /* time to send a "word" @ fwpm */ 
      wpm_time = 31 * dit_time;             /* PARIS minus iw_time @ wpm */
      
      il_time = 3.0/19.0 * (farns_time - wpm_time);
      iw_time = 7.0/19.0 * (farns_time - wpm_time);
   }
#if DEBUG == 1
   fprintf (stderr, "FARNS dit=%4.1f dah=%4.1f il=%4.1f iw=%4.1f\n",
            dit_time, dit_time*3, il_time, iw_time);
#endif
   return 0;
}


/*
 * open_dsp - derived from qrq - Copyright (C) 2006  Fabian Kurz, DJ1YFK
 */
int open_dsp (char *device)
{
   int tmp;
   int fd;
   int rtn;

   if (dsp_fd != 0)                            /* if already open */
      return (dsp_fd);

   if ((fd = open(device, O_WRONLY, 0)) < 0)
   {
      perror("open_dsp failed");
      exit(1);
   }

   tmp = AFMT_S16_NE; 
   if (ioctl(fd, SNDCTL_DSP_SETFMT, &tmp)==-1)
   {
      perror("SNDCTL_DSP_SETFMT");
      exit(1);
   }

   if (tmp != AFMT_S16_NE)
   {
      fprintf(stderr, "Cannot switch to AFMT_S16_NE\n");
      exit(1);
   }
  
   tmp = 2;	/* 2 channels, stereo */
   if (ioctl(fd, SNDCTL_DSP_CHANNELS, &tmp)==-1)
   {
      perror("SNDCTL_DSP_CHANNELS");
      exit(1);
   }

   if (tmp != 2)
   {
      fprintf(stderr, "No stereo mode possible :(.\n");
      exit(1);
   }

   if ((rtn = ioctl(fd, SNDCTL_DSP_SPEED, &samplerate))==-1)
   {
      perror("SNDCTL_DSP_SPEED");
      exit(1);
   }
   return fd;
}


/*
 * open_mixer
 */
int open_mixer (char *mixdev)
{
   int fd;

   if ((fd = open(mixdev, O_RDWR, 0)) < 0)
   {
      perror("open_mixer failed");
   }
   return fd;   
}


/*
 * open_serial
 */
int open_serial (char serdev[])
{
   int fd = 0;

   if (ser_fd != 0)         /* already opened */
      return (ser_fd);

   if ((fd = open(serdev, O_RDWR, 0)) < 0)
   {
      perror("open_serial;");
   }
   ioctl (fd, TIOCMSET, &rts_off);
   return fd;   
}


/*
 * get_volume -  This function tries to apply the PCM volume read
 * ioctl() to the main soundcard device first.  If this fails, it retries
 * using the PCM channel of the given mixer device, if it has one.
 */
int get_volume (void)
{
   int vol, device_mask;
   int fd;
   /* Try to use the main /dev/dsp device for ioctls first. */
   if ((fd = open_dsp (dspdevice)) < 0)
   {
      perror ("get_volume: open dsp failed");
      return -1;
   }

   if (ioctl (dsp_fd, MIXER_READ (SOUND_MIXER_PCM), &vol) == 0)
   {
      close (fd);
      return vol;
   }
   close (fd);

   /* Volume not found so try the mixer PCM channel volume instead. */
   if ((fd = open_mixer (mixdevice)) < 0)
   {
      perror ("get_volume: open mixer failed");
      return -1;
   }

   if (ioctl (fd, SOUND_MIXER_READ_DEVMASK, &device_mask) == -1)
   {
      perror ("get_volume: read devmask failed");
      close (fd);
      return -1;
   }

   if (device_mask & SOUND_MASK_PCM)
   {
      /* Read the PCM volume from the mixer. */
      if (ioctl (fd, MIXER_READ (SOUND_MIXER_PCM), &vol) == -1)
      {
         perror ("get_volume: read PCM level failed");
         close (fd);
         return -1;
      }
      close (fd);
      return vol;
   }
   else
   {
      if (device_mask & SOUND_MASK_VOLUME)
      {
         /* If in extreme difficulty, try main volume. */
         if (ioctl (fd, MIXER_READ (SOUND_MIXER_VOLUME), &vol) == -1)
         {
            perror ("get_volume  get main volume failed");
            close (fd);
            return -1;
         }
          close (fd);
          return vol;
      }
   }
   close (fd);
   return -1;
}


/*
 * set_volume - used internally to set pcm volume level
 */
int set_volume (int vol)
{
   int fd;

   if ((fd = open_mixer (mixdevice)) < 0)
   {
      perror ("set_volume-open");
   }

   if (ioctl (fd, MIXER_WRITE (SOUND_MIXER_PCM), &vol) < 0)
   {
      perror ("set_volume-write");
   }
   close (fd);
   return 0;
}


#if TESTMODE == 1
int main (int argc, char *argv[])
{
   int i;
   char d;
   int modes = SC;

   /* initialize lib */
   //cw_init ("/dev/dsp", "/dev/mixer", "/dev/ttyS0");         //SoundBlaster
   cw_init ("/dev/dsp1", "/dev/mixer1", "/dev/ttyS0");   //INTEL

   /* do some other funcs */
   cw_set_reply_msg (FALSE);
   cw_set_wpm (20);                                    /* set wpm */
   cw_set_fwpm (4);                                    /* set wpm */
   cw_set_key_modes (modes);
   cw_set_send_volume (32);                            /* set send vol */

   printf ("Enter to start: ");
   scanf (&d);

#if 0
   for (i=0; i<20; i++)
   {
      cw_send_str ("paris ");     /* send "paris" */
   }
#endif

#if 0
   for (i=0; i<20; i++)
   {
      cw_send_word ("paris");
   }
#endif

#if 1
   for (i=0; i<20; i++)
   {
      cw_send_char ('p');
      cw_send_char ('a');
      cw_send_char ('r');
      cw_send_char ('i');
      cw_send_char ('s');
      cw_send_char (' ');
   }
#endif

   cw_send_str ("e");
   pthread_join(cwthread, NULL);       /* wait for it before we terminate */
   return 0;
}
#endif   /* END TESTMODE */

