/*
 * Copyright (c) Tomas Znamenacek
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

package net.sf.turkey;

import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.JButton;
import javax.swing.JTextArea;
import javax.swing.JScrollPane;
import javax.swing.JComboBox;
import javax.swing.JRadioButton;
import javax.swing.JSlider;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.SwingUtilities;
import javax.swing.BoxLayout;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeListener;
import javax.swing.event.ChangeEvent;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;

import java.awt.Dimension;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.Toolkit;
import java.awt.Insets;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.datatransfer.StringSelection;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.EventQueue;

import java.io.InputStreamReader;
import java.io.BufferedReader;

import java.util.prefs.Preferences;

/**
 * Swinging Turkey, ho-ho.
 * The Swing GUI for Turkey.
 */
class GUI extends JFrame implements ActionListener, ChangeListener {

	/** Main window width. */
	protected static final int width = 440;

	/** Main window height. */
	protected static final int height = 500;

	/** Dummy text generator. */
	protected Generator gen = Generator.getInstance();

	/** Runs the GUI. */
	public static void run() {
		
		EventQueue.invokeLater(new Runnable() {

			public void run() {
				
				new GUI().setVisible(true);
			}
		});
	}

	/**
	 * Loads preferences, sets up basic window properties, lays out
	 * the GUI components and generates introductory dummy text.
	 */
	protected GUI() {

		loadPreferences();
		setupWindow();
		setupComponents();
		generateNewText();
	}

	/** Basic window setup. */
	protected void setupWindow() {

		Dimension screenDim = Toolkit.getDefaultToolkit().getScreenSize();
		Dimension ourDim = new Dimension(width, height);
		setSize(ourDim);
		setLocation((int) ((screenDim.width - ourDim.width) / 2), (int) ((screenDim.height - ourDim.height) / 2));

		setTitle("Turkey");
		setResizable(false);
		
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		getContentPane().setLayout(new BorderLayout());

		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent we) { closeWindow(); }
		});

		setIconImage(getIcon("icon.png").getImage());
	}

	/** Generates new dummy text in the output text area. */
	protected void generateNewText() {

		outputArea.setText(gen.getText());

		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				scrollArea.getVerticalScrollBar().setValue(0);
			}
		});
	}

	//
	// Preferences handling
	// 

	/** Save preferences. */
	protected void savePreferences() {

		Preferences root = Preferences.userNodeForPackage(getClass());
		
		root.putInt("sentencesPerPar", gen.getSentencesPerPar());
		root.putInt("paragraphsTotal", gen.getParagraphsTotal());
		root.putBoolean("htmlOutput", gen.isHTMLOutput());
		root.put("usedDictionary", gen.getCurrentDictionary().toString());
	}

	/** Load preferences. */
	protected void loadPreferences() {
		
		Preferences root = Preferences.userNodeForPackage(getClass());
		
		gen.setSentencesPerPar(root.getInt("sentencesPerPar", 5));
		gen.setParagraphsTotal(root.getInt("paragraphsTotal", 10));
		gen.setHTMLOutput(root.getBoolean("htmlOutput", false));
		gen.selectDictionary(root.get("usedDictionary", "default.txt"));
	}
	
	//
	// Boring GUI stuff to the end of file
	//
	
	protected JPanel mainPanel;
	protected JPanel buttonBar;
	protected JPanel outputPanel;
	protected JPanel configPanel;
	protected JPanel mainConfigPanel;
	protected JPanel dictionaryPanel;
	protected JPanel outputTypePanel;
	protected JPanel paramsPanel;
	
	protected JTabbedPane notebook;
	protected JTextArea outputArea;
	protected JScrollPane scrollArea;

	protected JLabel sentenceLabel;
	protected JLabel paragraphLabel;
	
	protected JComboBox dictionaryType;

	protected JSlider sentenceAdjust;
	protected JSlider paraAdjust;
	
	protected JButton closeButton;
	protected JButton copyButton;
	protected JButton goButton;

	protected ButtonGroup bGroup;
	protected JRadioButton otPlain;
	protected JRadioButton otHTML;

	/** Setup the GUI components. */
	protected void setupComponents() {
	
		mainPanel = new JPanel();
		mainPanel.setLayout(new BorderLayout());

		closeButton = new JButton(quitAction);
		goButton = new JButton(goAction);
		copyButton = new JButton(copyAction);
		
		buttonBar = new JPanel();
		buttonBar.setLayout(new FlowLayout(FlowLayout.RIGHT));
		buttonBar.setBorder(new EmptyBorder(new Insets(5, 0, 0, 0)));
		buttonBar.add(goButton);
		buttonBar.add(copyButton);
		buttonBar.add(closeButton);

		outputArea = new JTextArea();
		outputArea.setEditable(false);
		outputArea.setLineWrap(true);
		outputArea.setWrapStyleWord(true);
		outputArea.setMargin(new Insets(5, 5, 5, 5));

		scrollArea = new JScrollPane(outputArea);
		
		outputPanel = new JPanel();
		outputPanel.setBorder(new EmptyBorder(new Insets(5, 5, 5, 5)));
		outputPanel.setLayout(new BorderLayout());
		outputPanel.add(scrollArea, BorderLayout.CENTER);
	
		dictionaryType = new JComboBox(gen.getDictionaryNames());
		dictionaryType.addActionListener(this);
		dictionaryType.setSelectedItem(gen.getCurrentDictionary().toString());
		
		dictionaryPanel = new JPanel();
		dictionaryPanel.setBorder(new TitledBorder("Dictionary"));
		dictionaryPanel.add(new JLabel("Available dictionaries: "));
		dictionaryPanel.add(dictionaryType);

		otPlain = new JRadioButton("Plain text", !gen.isHTMLOutput());
		otPlain.addActionListener(this);
		otHTML = new JRadioButton("HTML", !otPlain.isSelected());
		otHTML.addActionListener(this);
		
		bGroup = new ButtonGroup();
		bGroup.add(otPlain);
		bGroup.add(otHTML);
		
		outputTypePanel = new JPanel();
		outputTypePanel.setBorder(new TitledBorder("Output type"));
		outputTypePanel.add(otPlain);
		outputTypePanel.add(otHTML);

		sentenceAdjust = new JSlider(JSlider.HORIZONTAL, Generator.SENTENCES_MIN, Generator.SENTENCES_MAX, gen.getSentencesPerPar());
		sentenceAdjust.addChangeListener(this);
		sentenceAdjust.setMajorTickSpacing(5);
		sentenceAdjust.setMinorTickSpacing(1);
		sentenceAdjust.setPaintTicks(true);
		sentenceAdjust.setPaintLabels(false);
		sentenceAdjust.setSnapToTicks(true);
		sentenceAdjust.setBorder(new EmptyBorder(0, 5, 0, 5));
		
		paraAdjust = new JSlider(JSlider.HORIZONTAL, Generator.PARAGRAPHS_MIN, Generator.PARAGRAPHS_MAX, gen.getParagraphsTotal());
		paraAdjust.addChangeListener(this);
		paraAdjust.setMajorTickSpacing(20);
		paraAdjust.setMinorTickSpacing(5);
		paraAdjust.setPaintTicks(true);
		paraAdjust.setPaintLabels(false);
		paraAdjust.setSnapToTicks(true);
		paraAdjust.setBorder(new EmptyBorder(0, 5, 5, 5));

		sentenceLabel = new JLabel("Sentences per paragraph: " + gen.getSentencesPerPar(), SwingConstants.CENTER);
		sentenceLabel.setBorder(new EmptyBorder(10, 0, 10, 0));
		paragraphLabel = new JLabel("Paragraphs total: " + gen.getParagraphsTotal(), SwingConstants.CENTER);
		paragraphLabel.setBorder(new EmptyBorder(10, 0, 10, 0));
		
		paramsPanel = new JPanel();
		paramsPanel.setBorder(new TitledBorder("Output settings"));
		paramsPanel.setLayout(new BoxLayout(paramsPanel, BoxLayout.Y_AXIS));
		paramsPanel.add(sentenceLabel);
		paramsPanel.add(sentenceAdjust);
		paramsPanel.add(paragraphLabel);
		paramsPanel.add(paraAdjust);
		
		mainConfigPanel = new JPanel();
		mainConfigPanel.setLayout(new BoxLayout(mainConfigPanel, BoxLayout.Y_AXIS));
		mainConfigPanel.setBorder(new EmptyBorder(new Insets(5, 5, 5, 5)));
		
		mainConfigPanel.add(dictionaryPanel);
		mainConfigPanel.add(outputTypePanel);
		mainConfigPanel.add(paramsPanel);
		
		configPanel = new JPanel();
		configPanel.setLayout(new BorderLayout());
		configPanel.add(mainConfigPanel, BorderLayout.CENTER);
		
		notebook = new JTabbedPane();
		notebook.add("Text", outputPanel);
		notebook.add("Preferences", configPanel);
		notebook.add("About", new JTextPanel("about.txt"));
		notebook.add("License", new JTextPanel("license.txt"));

		mainPanel.add(notebook, BorderLayout.CENTER);
		mainPanel.add(buttonBar, BorderLayout.SOUTH);
		mainPanel.setBorder(new EmptyBorder(new Insets(5, 5, 5, 5)));

		getContentPane().add(mainPanel, BorderLayout.CENTER);
	};

	protected void closeWindow() {

		savePreferences();
		System.exit(0);
	}
	
	protected AbstractAction quitAction = new AbstractAction("Quit", getIcon("quit.gif")) {
		public void actionPerformed(ActionEvent e) {

			closeWindow();
		}
	};

	protected AbstractAction goAction = new AbstractAction("Generate new", getIcon("new.gif")) {
		public void actionPerformed(ActionEvent e) {

			generateNewText();
		}
	};

	protected AbstractAction copyAction = new AbstractAction("Copy", getIcon("copy.gif")) {
		public void actionPerformed(ActionEvent e) {

			setClipboard(
				outputArea.getSelectionStart() == outputArea.getSelectionEnd() ?
				outputArea.getText() : outputArea.getSelectedText());
		}
	};

	public void actionPerformed(ActionEvent e) {

		// The message is from the dictionary combo box
		if (e.getSource().equals(dictionaryType)) {
			gen.selectDictionary((String)dictionaryType.getSelectedItem());
			
		// The message is from the output type radio button	
		} else {

			gen.setHTMLOutput(otHTML.isSelected());
		}
		
		generateNewText();
	}

	public void stateChanged(ChangeEvent e) {
	
		JSlider source = (JSlider)e.getSource();
		
		if (source.getValueIsAdjusting()) {
			sentenceLabel.setText("Sentences per paragraph: " + sentenceAdjust.getValue());
			paragraphLabel.setText("Paragraphs total: " + paraAdjust.getValue());
			return;
		}
		
		if (source == sentenceAdjust) {
			gen.setSentencesPerPar((int)source.getValue());
		} else {
			gen.setParagraphsTotal((int)source.getValue());
		}

		generateNewText();
	}
	
	protected ImageIcon getIcon(String id) {

		return new ImageIcon(getClass().getResource(id));
	}
	
	protected void setClipboard(String s) {
		
		Toolkit.getDefaultToolkit().getSystemClipboard().setContents(
			new StringSelection(s), null);
	}

	/**
	 * Simple panel with text.
	 */
	protected class JTextPanel extends JPanel {

		protected JScrollPane scroller;

		/** Loads and displays some text from external file. */
		public JTextPanel(String resource) {

			super();
			setLayout(new BorderLayout());
			setBorder(new EmptyBorder(new Insets(5, 5, 5, 5)));

			JTextArea text = new JTextArea();
			text.setEditable(false);
			text.setLineWrap(true);
			text.setWrapStyleWord(true);
			text.setMargin(new Insets(5, 5, 5, 5));
			text.setText(getFileContents(resource));
		
			scroller = new JScrollPane(text);
			add(scroller, BorderLayout.CENTER);
			
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					scroller.getVerticalScrollBar().setValue(0);
				}
			});
		}
		
		/** Returns the contents of a file. */
		protected String getFileContents(String fileName) {

			try {
				BufferedReader r = new BufferedReader(new InputStreamReader(
					getClass().getResourceAsStream(fileName)));
			
				StringBuffer s = new StringBuffer();
				String line;
				
				while ((line = r.readLine()) != null)
					s.append(line + "\n");
				
				r.close();
				return s.toString();
				
			} catch (Exception e) {

				return "(could not read contents of " + fileName + ")";
			}
		}
	}
}
