/*************************************************************************
 *	Time Traveling File Manager (TTFM)
 *************************************************************************
 *	Copyright (C) Sandeep Dilip Ranade 2005 - 2006 
 *************************************************************************	
 *	TTFM is free software; you can redistribute it 
 *	and/or modify it under the terms of the GNU General Public 
 *	License	as published by the Free Software Foundation; either 
 *	version 2, or (at your option) any later version.
 *
 *	TTFM is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with GNU CC; see the file COPYING.  If not, write to
 *	the Free Software Foundation, 59 Temple Place - Suite 330,
 *	Boston, MA 02111-1307, USA.
 *************************************************************************/ 
#include <qdir.h>
#include <qfileinfo.h>
#include <qcolor.h>
#include <qtimer.h>
#include <qprocess.h>
#include <qstringlist.h>
#include <qregexp.h>
#include <about_ttfs_form.h>
#include <bookmark_form.h>
#include <qapplication.h>
#include <qfileinfo.h>
#include <qptrstack.h>
#include "ttfs_form.h"

#ifdef TTFS_DEBUG
#define PDEBUG(fmt, arg ...)	printf("ttfs: %s:"fmt, __FUNCTION__, ## arg);
#else
#define PDEBUG(fmt, arg...)
#endif

/******************************************************************************
 * 		global flags
 ******************************************************************************/
bool hide_files = TRUE;		// files stay hidden by default.
#define DEFAULT_DATE_DELTA	1	//month
bool is_in_past = FALSE;	// are we in the present or past ??
uint past_time = 0;		// time to travel to (seconds drom UTC)
bool is_in_init = TRUE;		// are we in the init phase??
char bookmarks_dir_name[] = ".ttfm_bookmarks";	// this is the name of the bookmark directory
QDir bookmarks_dir;

/******************************************************************************
  * 		global objects
  **************************************************************************/
QTimer *refresh_timer = NULL;	// refreshes the browser every so often

/* 
 * This is the date that is today;
 */
QDate today;

/*
 * this is the date that is set at the lower end of the slider.
 * default is a Month
 */
QDate low_date;

/*
 * This starts off by being toda, but can change depending on the Zoom views.
 */
QDate high_date;

/*
 * this is teh fully qualified time shifted path
 */
QDir global_time_shifted_path;

/*
 * this is the path that the browser shows ...
 * defaults at the home directory ...
 */
QDir global_path ("/", "*");

QDir time_path ("/./", "*");
QProcess *p = NULL;

/*
 * this is the structure that holds the entire temporal state
 * to store on stack when going back and forth.
 */
typedef struct {
  QString g_path;
  QString gts_path;
  bool past;
  QDate low;
  QDate high;
  int selected_date;
  QTime selected_time;
} temporal_state_t;

/*
 * stacks to operate the back and forward buttons
 */
QPtrStack < temporal_state_t > go_back_stack;
QPtrStack < temporal_state_t > go_forward_stack;

/*****************************************************************************
 *  functions and slots ...
 *****************************************************************************/

/*
 * refresh_time_and_date:
 *	refreshes the time and date whenever there is a possible change.,
 */
void
ttfs_form::refresh_time_and_date ()
{
  	/* 
	 * get current time if we are in the present 
	 */
  	if (is_in_past == FALSE) {
      		time_edit->setTime (QTime::currentTime ());
    	}

  	/* 
	 * time slider, high end 
	 */
  	high_date_label->setText (high_date.toString 
		(QString ("MMMM d, yyyy")));

  	/* 
	 * time slider, low end 
	 */
  	low_date_label->setText (low_date.toString (QString ("MMMM d, yyyy")));

  	/* 
	 *  delta = high_date - low_date;
	 */
  	QDate delta = high_date;

  	/*
   	 * compute the approximate window size in english :)
   	 */
  	int days, months, years;
  	years = high_date.year () - low_date.year ();
  	months = high_date.month () - low_date.month ();
  	days = high_date.day () - low_date.day ();
  	QString delta_string;

  	if (years) { 
		if (years == 1) {
	  		delta_string = delta_string.sprintf ("about %d Year",
					years);
		} else { 
			delta_string = delta_string.sprintf ("about %d Years",
					years);
		}
    	} else if (months) { 
		if (days < 15) { 
			if (months == 1) { 
				delta_string = delta_string.sprintf (" about"
						" %d Month", months); 
			} else { 
				delta_string = delta_string.sprintf (" about"
						" %d Months", months); 
			} 
		} else { 
			delta_string = delta_string.sprintf (" about %d "
					"Months", months + 1); 
		} 
	} else if (days) { 
		if (days == 1) { 
			delta_string = delta_string.sprintf (" %d Day", days); 
			/* 
			 * here we diable the slider!  
			 * this is because we want the time-edit to 
			 * control time accesses to within a day's worth 
			 */ 
			PDEBUG ("disabling tracking ....\n");

			/*
			 * setTracking(false);
			 */
			date_slider->setDisabled (true);	
		} else { 
			delta_string = delta_string.sprintf (" %d Days", days); 
			
			/* 
			 * here we enable the slider!  
			 * since the "window" is more than a day, we 
			 * can use the slider
			 */ 
			PDEBUG ("enabling tracking ....\n"); 
			/*
			 * Tracking(true);
			 */
			date_slider->setEnabled (true);	
		} 
	} 
	
	/* 
	 * set the string in the label 
	 */ 
	PDEBUG ("date string: %s\n", (const char *) delta_string);
	delta_date_label->setText (delta_string); 
}

/*
 * fileNew:
 *	spawn a new TTFS window ...
 */
void
ttfs_form::fileNew ()
{
  	ttfs_form *w = new ttfs_form;
	w->show ();
	PDEBUG ("New done!\n");
}

/*
 * fileExit:
 * 	exit the main window
 */
void
ttfs_form::fileExit ()
{
	exit (0);
}

/*
 * push_state_to_back_stack()
 *	push the remaining temporal state to stack (s)
 */
void
ttfs_form::push_state_to_back_stack ()
{

	temporal_state_t *state = new temporal_state_t; 
	
	/*
	 * fill in the state structure
	 */ 
	
	state->low = low_date;
	state->high = high_date;
	state->gts_path = global_time_shifted_path.path ().copy ();
	state->g_path = global_path.path ().copy ();
	state->past = is_in_past;
	state->selected_date = date_slider->value ();
	state->selected_time = time_edit->time ();

  	PDEBUG ("---------------------------------------------------"
			"---------------------------------------------\n");
	PDEBUG ("Pushing path %s on back_stack\n", 
			(const char *) state->gts_path);
	PDEBUG ("Pushingtime and date state on stack ...\n");
	PDEBUG ("Pushing low date: %s\n",
			(const char *) state->low.toString 
			(QString ("MMMM d, yyyy")));
	PDEBUG ("Pushing high date: %s\n",
			(const char *) state->high.toString 
			(QString ("MMMM d, yyyy")));
	PDEBUG ("pushed selected date value %d\n", state->selected_date); 
	
	/*
	 * actually push the state on the back stack
	 */
	go_back_stack.push (state);
	PDEBUG ("-----------------------------------------------------"
			"-------------------------------------------\n"); 
	
	/* 
	 * take a snapshot here
	 * HACK: this is so that some one sliding back after no snapshot will
	 * still see the correct current view.
	 */ 
	do_snapshot ();
}

/* 
 * push_state_to_forward_stack()
 *	push the remaining temporal state to forward stack 
 */
void
ttfs_form::push_state_to_forward_stack ()
{ 
	temporal_state_t *state = new temporal_state_t; 
	
	/* 
	 * fill in the state structure
	 */
	state->low = low_date;
	state->high = high_date;
	state->gts_path = global_time_shifted_path.path ().copy (); 
	state->g_path = global_path.path ().copy ();
	state->past = is_in_past;
	state->selected_date = date_slider->value ();
	state->selected_time = time_edit->time ();
	PDEBUG ("----------------------------------------------------------"
			"--------------------------------------\n");
	PDEBUG ("Pushing path %s on forward_stack\n",
			(const char *) state->gts_path);
	PDEBUG ("Pushingtime and date state on stack ...\n");
	PDEBUG ("Pushing low date: %s\n",
			(const char *) state->low.toString 
			(QString ("MMMM d, yyyy")));
	PDEBUG ("Pushing high date: %s\n",
			(const char *) state->high.toString 
			(QString ("MMMM d, yyyy")));
	PDEBUG ("pushed selected date value %d\n", state->selected_date);

	/*
	 * actually push the state on the forward stack
	 */
	go_forward_stack.push (state);
	PDEBUG ("----------------------------------------------------------"
			"--------------------------------------\n");
}

/*
  * restore_state_from_back_stack():
  *	pop the back stack and resore temporal state from it.
  */

void
ttfs_form::restore_state_from_back_stack ()
{
	temporal_state_t *state;

  	/* 
	 * check if the stack is empty 
	 */
	if (go_back_stack.isEmpty () == true) {
		PDEBUG ("go back stack is empty ... doing nothing\n");
		return;
	}

  	PDEBUG ("---------------------------------------------------------"
			"---------------------------------------\n");
	PDEBUG ("restoring time and date state from stack ...\n");
	state = go_back_stack.pop ();

 	/*
	 * restore the low date on the slider display
	 */
	low_date = state->low;
	PDEBUG ("popped low date: %s\n",
			(const char *) low_date.toString 
			(QString ("MMMM d, yyyy")));

	/*
	 * restore the high date on the slider display
	 */
	high_date = state->high;
	PDEBUG ("popped high date: %s\n",
			(const char *) high_date.toString 
			(QString ("MMMM d, yyyy")));

   	/*
	 * reset the date slider's position
	 */
	date_slider->setValue (state->selected_date);
	time_edit->setTime (state->selected_time);

	/*
	 * restore global path.
	 */
	global_path.setPath (state->g_path);
	global_time_shifted_path.setPath (state->gts_path);
	
	PDEBUG ("restored path %s from back_stack\n",
			(const char *) state->gts_path);
  	/*
	 * restore is_in_past flag
	 */
	is_in_past = state->past;
	refresh_time_and_date ();
	ttfs_form::tt_display_path ();
	PDEBUG ("---------------------------------------------------------"
			"---------------------------------------\n");
}

/*
  * restore_state_from_forward_stack():
  *	pop the forward stack and resore temporal state from it.
  */

void
ttfs_form::restore_state_from_forward_stack ()
{
	temporal_state_t *state;

  	/* 
	 * check if the stack is empty 
	 */
	if (go_forward_stack.isEmpty () == true) {
		PDEBUG ("go back stack is empty ... doing nothing\n");
		return;
	}

  	PDEBUG ("-----------------------------------------------------------"
			"-------------------------------------\n");
	PDEBUG ("restoring time and date state from forward stack ...\n");
	state = go_forward_stack.pop ();

   	/*
	 * restore the low date on the slider display
	 */
	low_date = state->low;
	PDEBUG ("popped low date: %s\n",
			(const char *) low_date.toString 
			(QString ("MMMM d, yyyy")));

  	/*
	 * restore the high date on the slider display
	 */
	high_date = state->high;
	PDEBUG ("popped high date: %s\n",
			(const char *) high_date.toString 
			(QString ("MMMM d, yyyy")));
   	/*
	 * reset the date slider's position
	 */
	date_slider->setValue (state->selected_date);
	time_edit->setTime (state->selected_time);

  	/*
	 * restore global path.
	 */
	global_path.setPath (state->g_path);
	global_time_shifted_path.setPath (state->gts_path);

  	PDEBUG ("restored path %s from forward_stack\n",
			(const char *) state->gts_path);

  	/*
	 * restore is_in_past flag
	 */
	is_in_past = state->past;

  	refresh_time_and_date ();
	ttfs_form::tt_display_path ();
	PDEBUG ("---------------------------------------------------------"
			"---------------------------------------\n");
}

/*
  * go_home:
  *	set path to home directory, and update browser.
  */
void
ttfs_form::go_home ()
{
  	/* 
	 * push temporal and spatial state to back stack 
	 */
	push_state_to_back_stack ();

  	/* 
	 * change the path ... 
	 */
	global_path = QDir::home ();
	global_time_shifted_path.setPath (global_path.path ());

  	ttfs_form::tt_display_path ();
}


void
ttfs_form::editPaste ()
{

}

/*
  * helpAbout:
  *	show a nice about dialog 
  */
void
ttfs_form::helpAbout ()
{
	about_ttfs_form about (this);
	about.exec ();
}

/*
  * do_execute:
  *	given a path, try to guess the file type, and then try to execute it
  *	using an application that understands the file format.
  *	Given a file that is a version file, we seperate the time 
  *	stamp from the file name and try to execute the application 
  *	that understands the file.
  *	TODO: add the bindings to a settings file
  */
void
ttfs_form::do_execute (
		QString path)
{
	QString cmd;
	char name[BUFSIZ];
	int time = -1;
	int index = -1;

  	PDEBUG ("Executing %s\n", (const char *) path);

  	QFileInfo finfo = QFileInfo (path);

  	if (finfo.isSymLink ()) {
		path = finfo.readLink ();
		goto try_extension;
	}

  	if (finfo.isFile ()) {
		index = path.find ('@');

		/*
		 * todo: look at double @'s in path 
		 */
		if (index != -1) {
			int sindex = path.findRev ('/');
			if (sindex != -1) {
				if (sindex < index) {
				 	/*
					 * there is an @ in the last 
					 * part of the file ...  
					 * seperate the time stamp from 
					 * the file name..
					 */
					path[index] = ' ';
					sscanf ((const char *) path, 
							"%s %d", name, &time);
					/*
					 * and we restore it 
					 */
					path[index] = '@';	
				} else {
					/*
					 * this is not the last parf of the 
					 * path...
					 */
					sscanf ((const char *) path, "%s", 
							name);
				}
			} else {
			 	/*
				 * there is only a single file ... no path!
				 */
				sscanf ((const char *) path, "%s", name);
				PDEBUG ("Got execute name: %s\n", name);
			}
		} else {
			sscanf ((const char *) path, "%s", name);
			PDEBUG ("Got execute name: %s\n", name);
		}

try_extension:
		QFileInfo extinfo = QFileInfo (name);
		QString ext = extinfo.extension (FALSE);
		PDEBUG ("got extension %s\n", (const char *) ext);
		PDEBUG ("executing final path: %s\n", (const char *) path);
		
		/* 
		 * binary execuables
		 */
		if (finfo.isExecutable ()) {
			cmd = path + " &";
			system ((const char *) cmd);
			return;
		} 
		
		if (ext == "") {
			PDEBUG ("No extension .... guessing\n");
			cmd = "gview " + path + " &";
			system ((const char *) cmd);
			return;
		}
      
		/*
		 * Images
		 */
		if (ext == "jpg" 
				|| ext == "gif" 
				|| ext == "bmp" 
				|| ext == "png") {
			cmd = "kuickshow " + path + " &";
			system ((const char *) cmd);
			return;
		}

		/*
		 * txt files
		 */
		if (ext == "txt") {
			cmd = "gview " + path + " &";
			system ((const char *) cmd);
			return;
		}
	
		/* 
		 * pdf files
		 */
		if (ext == "pdf") {
			cmd = "xpdf " + path + " &";
			system ((const char *) cmd);
			return;
		}
	
		/*
		 * ps files
		 */
		if (ext == "ps") {
			cmd = "ggv " + path + " &";
			system ((const char *) cmd);
			return;
		}
		
		/*
		 * .c files ... cpp. h etc
		 */
		if (ext == "c" ||
				ext == "C" || 
				ext == "cpp" || 
				ext == "cc" || 
				ext == "h") {
			cmd = "gview " + path + " &";
			system ((const char *) cmd);
			return;
		} 
		
		/* 
		 * Doc files
		 */
		if (ext == "doc" 
				|| ext == "sxw") { 
			cmd = "oowriter " + path + " &";
			system ((const char *) cmd);
			return;
		}
	
		/*
		 * html files
		 */
		if (ext == "html" 
				|| ext == "htm") { 
			cmd = "mozilla file://" + path + " &";
			system ((const char *) cmd);
			return;
		}
		return;
	} else {
		PDEBUG ("Not a file!!Cant execute\n");
		time_status_bar->setText ("Cannot Execute this file!\n");
	}
}

/*
 * get_file_pixmap:
 *	deduce the type fo file from its extension, and then return an 
 *	appropriate pixmap
 */
QPixmap
ttfs_form::get_file_pixmap (
		QString path)
{ 
	PDEBUG ("Got path: %s\n", (const char *) path);

	QFileInfo finfo = QFileInfo (path);

  	if (finfo.isFile ()) {
  		QString ext = finfo.extension (FALSE);
  		PDEBUG ("got extension %s\n", (const char *) ext);
      
      		// binary execuables
      		if (finfo.isExecutable ()) {
      			return QPixmap ("images/kservices.png");
      		} 
      	
      		// symbolic links
      		if (finfo.isSymLink ()) {
      			return QPixmap ("images/link.png");
      		} 
      		
      		// Un readble/ locked
      		if (!finfo.isReadable ()) {
      			return QPixmap ("images/encrypted.png");
      		}
      
      		// Images
      		if (ext == "jpg" 
      				|| ext == "gif" 
      				|| ext == "bmp" 
      				|| ext == "png") {
      			return QPixmap ("images/image.png");
      		}
      	
      		// txt files
      		if (ext == "txt") {
      			return QPixmap ("images/txt.png");
      		}
      	
      		// pdf files
      		if (ext == "pdf") {
      			return QPixmap ("images/pdf.png");
      		}
      		
      		// ps files
      		if (ext == "ps") {
      			return QPixmap ("images/postscript.png");
      		}
      	
      		// .c files ...
      		if (ext == "c") {
      			return QPixmap ("images/source_c.png");
      		}
      	
      		//cpp files
      		if (ext == "C" 
      				|| ext == "cpp" 
      				|| ext == "cc") {
      			return QPixmap ("images/source_cpp.png");
      		}
      	
      		//header files
      		if (ext == "h") {
      			return QPixmap ("images/source_h.png");
      		}
      	
      		//object files
      		if (ext == "o") {
      			return QPixmap ("images/source_o.png");
      		}
      	
      		//Doc files
      		if (ext == "doc" || ext == "sxw") {
      			return QPixmap ("images/doc.png");
      		}
      	
      		//html files
      		if (ext == "html" || ext == "htm") {
      			return QPixmap ("images/html.png");
      		}
      	
      		// Jar files
      		if (ext == "jar") {
      			return QPixmap ("images/java_jar.png");
      		}
      	
      		// Wav files
      		if (ext == "wav") {
      			return QPixmap ("images/krec_fileplay.png");
      		}
      	
      		// midi files
      		if (ext == "midi" 
      				|| ext == "mid") {
      			return QPixmap ("images/midi.png");
      		}
      	
      		// real player files
      		if (ext == "ram" 
      				|| ext == "rm") {
      			return QPixmap ("images/real.png");
      		}
      	
      		//Quicktime files
      		if (ext == "mov" 
      				|| ext == "qt") {
      			return QPixmap ("images/quicktime.png");
      		}
      	
      		// other sound files        
      		if (ext == "mp3" 
      				|| ext == "vob" 
      				|| ext == "mp2") {
      			return QPixmap ("images/sound.png");
      		}
      	
      		// movie files
      		if (ext == "mpeg" 
      				|| ext == "mpg" 
      				|| ext == "avi" 
      				|| ext == "wmv") {
      			return QPixmap ("images/multimedia.png");
      		}
      	
      		// RPM packages
      		if (ext == "rpm") {
      			return QPixmap ("images/rpm.png");
      		}
      	
      		// tar 
      		if (ext == "tar") {
      			return QPixmap ("images/tar.png");
      		}
      	
      		// tar.gz or tgz or zip
      		if (ext == "tgz" 
      				|| ext == "zip" 
      				|| ext == "gz") {
      			return QPixmap ("images/tgz.png");
      		}
      	
      		// all remaining ....
      		return QPixmap ("images/binary.png");
      	}

      	/*
      	 * special treatment for directrories and symlinks
      	 */
      	if (finfo.isDir ()) {
      		if (!finfo.isReadable ()) {
      			return QPixmap ("images/folder_locked.png");
      		}
      		return QPixmap ("images/folder_blue.png");
      	}
      	if (finfo.isSymLink ()) {
      		return QPixmap ("images/link.png");
      	}
      	return QPixmap ("images/source_moc.png");
}

/*
 * tt_display_path:
 *	displays the current directory's contents in the file_icon_view
 *	sets the pixmap to the correct image according to file/dir or 
 * 	file mime type one of the most important functions ... 
 *	called almost from all places to refresh the view
 */
void
ttfs_form::tt_display_path ()
{
	PDEBUG ("deleting items ...\n");
	/*
	 * clear the items curently in the view ...
	 */

	file_icon_view->clear ();

	/*
	 *  set the path_edit's text to the current path
	 */
	path_edit->setText ((const char *) global_path.path ());

	QDir time_shifted_path;

	/*
	 * append a different time specific marker if in the past..
	 * /a/b/ .... /x/y@time
	 */
	if (is_in_past) {
		QString tpath;
		tpath.sprintf ("/.@%d", past_time);
		time_path.setPath (tpath);
		PDEBUG ("Setting path to time shifter path: %s/%s\n",
				(const char *) global_path.path (),
				(const char *) time_path.path ());
	} else {
		time_path.setPath ("/./");
		PDEBUG ("Setting path to standard path: %s/%s\n",
				(const char *) global_path.path (),
				(const char *) time_path.path ());
	}

	/*
	 * mark the color of the edit bar differently if the path does not exist
	 */

	if (is_in_past) {
		time_shifted_path.setPath (
				global_path.path () + time_path.path ());
		PDEBUG ("Got time status text : %s\n",
				(const char *) time_shifted_path.path ());
		time_status_bar->setText ("In the Past");
	} else {
		time_shifted_path.setPath (global_path.path ());
		time_status_bar->setText ("Present Time");
	}

	QFileInfo finfo (time_shifted_path.path ());
	if (finfo.exists () == FALSE) {
		// reddish for error
		path_edit->setBackgroundColor (QColor (255, 203, 203));	
		return;
	} else {
		// correct, and in past: blue
		// correct and in present: yellow
		if (is_in_past) {	
			// bluish
			path_edit->setBackgroundColor (QColor (206, 194, 255));
		} else {
			//yellowish
			path_edit->setBackgroundColor (QColor (246, 255, 198));
		}
	}

	/*
	 * for each directory, set the correct font, and add the 
	 * item to the file_icon_view
	 */

	if (hide_files == TRUE) {
		time_shifted_path.setFilter (QDir::Dirs 
				| QDir::Files 
				| QDir::System);
	} else {
		time_shifted_path.
			setFilter (QDir::Dirs 
					| QDir::Files 
					| QDir::Hidden 
					| QDir::System);
	}

	/*
	 * set the global path to correspond with the path...
	 */
	global_time_shifted_path.setPath (time_shifted_path.path ());
	PDEBUG ("setting global_time_shifted_path: %s\n",
			(const char *) global_time_shifted_path.path ());

	/*
	 * iterate over the directory contents, get the pixmap 
	 * for the files/folders and then
	 * add them to the file_con view
	 */
	for (uint i = 0; i < time_shifted_path.count (); i++) {
		QString path = time_shifted_path.path () + "/" + 
			time_shifted_path[i];
		QPixmap pixmap;

		PDEBUG ("Getting pixmaps for path %s\n", (const char *) path);
		pixmap = ttfs_form::get_file_pixmap (path);
		new QIconViewItem (file_icon_view, 
				time_shifted_path[i], pixmap);
	}
	today = QDate::currentDate ();
}


/*
 * init:
 *	Initialize the entire main window, paths and globals.
 */
void
ttfs_form::init ()
{
	PDEBUG ("initializing TTFS form ...\n");
	is_in_init = TRUE;

	/*
	 * Call the path initializer...
	 */

	if (hide_files == TRUE) {
		global_path.setFilter (QDir::Dirs 
				| QDir::Files 
				| QDir::System);
	} else {
		global_path.
			setFilter (QDir::Dirs 
					| QDir::Files 
					| QDir::Hidden 
					| QDir::System);
	}

	global_path = QDir::home ();
	global_time_shifted_path.setPath (global_path.path ());
	is_in_past = FALSE;
	ttfs_form::tt_display_path ();
	PDEBUG ("Current Working Directory: %s\n",
			(const char *) QDir::currentDirPath ());

	/*   add a tooltip dynamically ... */
	QToolTip::add (date_slider, QString ("Date Slider"));

	/*
	 * set the date labels ...
	 */

	today = QDate::currentDate ();
	high_date = today;

	/* low is one month behind today ... */
	low_date = high_date.addMonths (-DEFAULT_DATE_DELTA);

	/* refresh the labels to reflect this ... */
	ttfs_form::refresh_time_and_date ();
	ttfs_form::do_new_date_selected (100);
	time_status_bar->setText ("Present Time");

	connect (time_edit, SIGNAL (valueChanged (const QTime &)),
			this, SLOT (do_time_changed (const QTime &)));

	/*
	 * make sure the .ttfm_bookmarks directory exists
	 */
	QString bm_dir = QDir::home ().path () + "/" + 
		QString (bookmarks_dir_name);

	bookmarks_dir = QDir (bm_dir);

	if (bookmarks_dir.exists ()) {
		PDEBUG ("Bookmarks directory %s exists\n", 
				(const char *) bm_dir);
	} else {
		PDEBUG ("Creating bookmarks directory %s ...\t", 
				(const char *) bm_dir);
		if (bookmarks_dir.mkdir (bm_dir) == true) {
			PDEBUG ("Successful\n");
		} else {
			PDEBUG ("Failed!\n");
		}
	}

	is_in_init = FALSE;

	ttfs_form::do_refresh ();
}

/*
 * tt_change_directory: 	
 *	given an item that was double clicked, figure out the 
 * 	path to change directory to and change directory.
 */
void
ttfs_form::tt_change_directory (
		QIconViewItem * item)
{
	QDir time_shifted_path;
	QString tpath;

	if (item->text () == ".") {
		return;
	}

	if (item->text () == "..") {
		ttfs_form::go_up ();
		return;
	}

	if (is_in_past) {
		/* 
		 * we want to keep the global path and just append 
		 * the item's text.. 
		 */
		time_shifted_path.setPath (global_time_shifted_path.path () + 
				"/" + item->text ());
		PDEBUG ("Got time status text : %s\n",
				(const char *) time_shifted_path.path ());
		time_status_bar->setText ("In the Past");
	} else {
		time_shifted_path.setPath (global_path.path () 
				+ "/" + item->text ());
		time_status_bar->setText ("Present Time");
	}


	QFileInfo finfo = QFileInfo (time_shifted_path.path ());
	if (finfo.isDir ()) {
		/*
		 * push temporal and spatial state to back stack
		 */

		push_state_to_back_stack ();
		/*
		 * if its a directory, CD to it ...
		 */

		PDEBUG ("changing directory to %s\n",
				(const char *) time_shifted_path.path ());
		global_time_shifted_path.setPath (time_shifted_path.path ());

		/*
		 * set the path to the new directory selected ...
		 */

		global_path.setPath (time_shifted_path.path ());

		/*
		 * refresh the brower's file_icon_view...
		 */

		ttfs_form::tt_display_path ();

	} else if (finfo.isFile ()) {
		PDEBUG ("Executing application that understands file %s\n",
				(const char *) time_shifted_path.path ());
		do_execute (time_shifted_path.path ());
	} else {
		PDEBUG ("Ignoring unknown file %s\n",
				(const char *) time_shifted_path.path ());
		do_execute (time_shifted_path.path ());
	}
}

/*
 * go_up:
 *	do a "cd .." essentially...
 */
void
ttfs_form::go_up ()
{
	PDEBUG ("Going up ...\n");
	PDEBUG ("Global path: %s\n", (const char *) global_path.path ());
	PDEBUG ("Global time shifted path: %s\n",
			(const char *) global_time_shifted_path.path ());

	/*
	 * push temporal and spatial state to back stack
	 */
	push_state_to_back_stack ();

	global_time_shifted_path.cdUp ();
	global_path.cdUp ();

	// special protocol for time shifted paths ....
	if (is_in_past == TRUE) {
		/*
		 * we find if there exists an '@" sign after the last "/" sign
		 * if so, then there is a path that is an extra path 
		 * (time path) to take care of    
		 */

		QString time_path = global_path.path ();

		int pos_slash = time_path.findRev ('/');
		int pos_at = time_path.findRev ('@');
		PDEBUG ("pos_slash = %d, pos_at = %d\n", pos_slash, pos_at);

		if (pos_at > pos_slash) {
			PDEBUG ("there is a terminal time shift path\n");
			// go one more level up.
			global_time_shifted_path.cdUp ();
			global_path.cdUp ();
		} else {
			PDEBUG ("no terminal time shift path!\n");
		}
	}
	// refresh the view...
	ttfs_form::tt_display_path ();
}

/*
 * human_readable_file_size:
 *	convert file syse in bytes into a human readable file size.
 */
QString
ttfs_form::human_readable_file_size (
		uint size)
{
	double hsize;
	QString power;

	/*
	 * convert  the size into bytes, KBb, m  or GB representation.
	 */

	if (size < 1024) {
		power = "bytes";
		hsize = (double) size;
	} else if (size < 1024 * 1024) {
		power = "KB";
		hsize = (double) size / 1024;
	} else if (size < 1024 * 1024 * 1024) {
		power = "MB";
		hsize = (double) size / 1024 / 1024;
	} else {
		power = "GB";
		hsize = (double) size / 1024 / 1024 / 1024;
	}

	/*
	 *  return result in form of 5.7 MB, etc
	 */

	QString result;
	return result.sprintf ("%.1f %s", hsize, (const char *) power);
}

/*
 * give_file_tooltip:
 *	show the size and name of the file in the status bar 
 *	at the bottom of the window.
 */
void
ttfs_form::give_file_tooltip (
		QIconViewItem * item)
{
	QString path;

	if (is_in_past) {
		path.sprintf ("%s/.@%d/%s", (const char *) global_path.path (),
				past_time, (const char *) item->text ());
	} else {
		path = global_path.path () + "/" + item->text ();
	}
	QFileInfo finfo = QFileInfo (path);

	PDEBUG ("Showing tooltip for file %s (size %u)\n",
			(const char *) item->text (), finfo.size ());
	QString status;

	/*
	 * show the symlink if it is one
	 */
	if (finfo.isSymLink () == false) {
		status.sprintf ("%s : size(%s)", (const char *) item->text (),
				(const char *) ttfs_form::
				human_readable_file_size (finfo.size ()));
	} else {
		QString link = finfo.readLink ();
		status.sprintf ("%s (symlink) => %s : size(%s)",
				(const char *) item->text (), 
				(const char *) link,
				(const char *) ttfs_form::
				human_readable_file_size (finfo.size ()));
	}

	status_bar->setText (status);
}

/*
 * default_status:
 *	sets the status bar field to the default ... (total files, total size)
 */
void
ttfs_form::do_default_status ()
{
	QString status;
	uint fcount = 0;
	QString path;

	if (is_in_past) {
		path.sprintf ("%s/.@%d", (const char *) global_path.path (), 
				past_time);
	} else {
		path = global_path.path ();
	}

	QDir dir = global_path;	// to preserve set flags, views etc.
	dir.setPath (path);

	if (hide_files == TRUE) {
		dir.setFilter (QDir::Dirs 
				| QDir::Files 
				| QDir::System);
	} else {
		dir.setFilter (QDir::Dirs 
				| QDir::Files 
				| QDir::Hidden 
				| QDir::System);
	}
	for (uint i = 0; i < dir.count (); i++) {
		QString itempath = path + "/" + dir[i];
		QFileInfo finfo = QFileInfo (itempath);

		if (finfo.isFile ()) {
			fcount++;
		}
	}

	status.sprintf ("%u items, %u files", dir.count (), fcount);
	status_bar->setText (status);
}

/*
 * do_change_directory:
 *	do a cd to the specified directory.
 */
void
ttfs_form::do_change_directory (
		QString str_path)
{
	QFileInfo finfo = QFileInfo (str_path);
	if (finfo.isDir ()) { 
		/*
		 * push temporal and spatial state to back stack
		 */
		push_state_to_back_stack ();

		/*
		 * if its a directory, CD to it ...
		 */

		PDEBUG ("changing directory to %s\n", (const char *) str_path);

		/*
		 * set the path to the new directory selected ...
		 */

		global_path.setPath (str_path);

		/*
		 *  refresh the brower's file_icon_view...
		 */

		ttfs_form::tt_display_path ();

	} else if (finfo.isFile ()) {
		PDEBUG ("Executing application that understands file %s\n",
				(const char *) str_path);
		do_execute (str_path);
	} else {
		PDEBUG ("Ignoring unknown file %s\n", (const char *) str_path);
		do_execute (str_path);
	}
}

/*
 * do_snapshot: 
 *	take a snapshot
 *	if not ext3cow, then throw error dialog box.
 */
void
ttfs_form::do_snapshot ()
{
	int ret;
	char cmdline[BUFSIZ];

	PDEBUG ("Taking a Snapshot ....\n");

	PDEBUG ("global path = %s\n", (const char *) global_path.path ());
	snprintf (cmdline, BUFSIZ, "snapshot %s",
			(const char *) global_path.path ());
	ret = system (cmdline);

	if (ret != 0) {
		PDEBUG ("error taking snapshot ...\n");
		PDEBUG ("return value of system: %d\n", ret);
		QString errstr (strerror (ret));
		PDEBUG ("error string: %s\n", (const char *) errstr);
		time_status_bar->
			setText("Snapshot Failed (File system is not ext3cow)");
		return;
	}
	time_status_bar->setText ("Snapshot Successful");
}

/*
 * do_time_travel:
 *	when the button is pressed, collect the time and date, convert it to the
 *	seconds since UTC and use this as an append string to travel back 
 *	in time.
 */
void
ttfs_form::do_time_travel ()
{
	PDEBUG ("Doing TimeTravel ....\n");

	int val = date_slider->value ();
	int rem = 100 - val;
	double fraction = rem / 100.0;

	PDEBUG ("val of slider = %d, remaining = %d, fraction = %f\n", 
			val, rem, fraction);
	int delta_days = low_date.daysTo (high_date);
	PDEBUG ("delta expressed in days is %d\n", delta_days);

	double days_in_past = fraction * (double) delta_days;
	PDEBUG ("days in past =  %f\n", days_in_past);

	/*
	 * the selected date is high date - days_in_past;
	 */

	QDate selected_date = high_date.addDays (-(int) days_in_past);

	PDEBUG ("selected date is %s\n",
			(const char *) selected_date.toString 
			(QString ("d MMMM yyyy")));

	/* get the time selected from the time edit box */
	QTime selected_time = time_edit->time ();

	/* find the number of seconds from UTC to selected date and time. */
	QDateTime dt (selected_date, selected_time);

	past_time = dt.toTime_t ();

	PDEBUG ("Got the seconds from UTC to travel to ...%d\n", past_time100);

	QDateTime current = QDateTime::currentDateTime ();
	uint secs = current.toTime_t ();

	if (past_time < secs) {
		PDEBUG ("Time is in the past!\n");
		is_in_past = TRUE;
	} else {
		PDEBUG ("Time is in the future! BUG()\n");
		past_time = 0;
		is_in_past = FALSE;
	}
	ttfs_form::do_refresh ();
}

/*
 * go_to_current_time:
 *	reset all timer variables and labels, and flags.
 */
void
ttfs_form::go_to_current_time ()
{
	PDEBUG ("Resetting time to current time!\n");

	if (is_in_past == TRUE) {
		ttfs_form::go_home ();
	} else {
		global_time_shifted_path.setPath (global_path.path ());

		/*
		 * push temporal and spatial state to back stack
		 */
		push_state_to_back_stack ();
	}

	/* refresh the labels to reflect this ... */
  	ttfs_form::refresh_time_and_date ();

	is_in_past = FALSE;
	past_time = 0;

	time_path.setPath ("/./");

	high_date = today;

	/* low is one month behind today ... */
	low_date = high_date.addMonths (-DEFAULT_DATE_DELTA);

	// update the slider value and labels
	date_slider->setValue (100);

	// set the current time for the time editor
	time_edit->setTime (QTime::currentTime ());

	ttfs_form::do_refresh ();
}

/*
 * do_time_search:
 *	launch a dialog to help find and grep in time!
 */
void
ttfs_form::do_time_search ()
{
	PDEBUG ("Searching in Time for a file ...\n");
}

/*
 * do_grep_time:
 *	do a grep in time. Launch a dialouge to find a file and grep a string.
 */
void
ttfs_form::do_grep_time ()
{
	PDEBUG ("Grepping in Time for a file ...\n");
}

/* do_refresh:
 *	refresh the browser view
 */
void
ttfs_form::do_refresh ()
{
	if (is_in_past == FALSE) {
		global_path.refresh ();
	}
	ttfs_form::tt_display_path ();

	refresh_time_and_date ();
}

/*
 * do_version_view:
 *	looks at the toggle button's value and shows versions, 
 *	or refreshes the path
 */
void
ttfs_form::do_version_view (bool val)
{
	PDEBUG ("Version View ...\n");
	if (val == TRUE) {
		// button pressed, show versions ...
		if (!is_in_past) {
			ttfs_form::do_versions (
					global_time_shifted_path.path ());
		} else {
			PDEBUG ("Cant do version enumeration in the past"
					" for now!\n");
			time_status_bar->
				setText ("Cant do version enumeration in "
						" the past for now!");
		}
	} else {
		// refresh the view so the standard structure is seen
		ttfs_form::tt_display_path ();
	}
}

/*
 * do_settings:
 *	show a dialog box, and set settings, and write a configuration file!
 */
void
ttfs_form::do_settings ()
{
	PDEBUG ("Settings ....\n");
}

/*
 * do_time_shell:
 *	spawn a shell, with the current directory shifted in time.
 */
void
ttfs_form::do_time_shell ()
{
	int ret = 0;
	QString terminal_binary = "/usr/bin/gnome-terminal";
	QString terminal_option = "--working-directory";
	QString cmd_line;

	/*
	 * notice the & at the end of the command line
	 * some versions of the shell block, some are async!!
	 */
	if (is_in_past == FALSE) {
		PDEBUG ("Spawning shell in current time ...\n");
		cmd_line.sprintf ("%s %s=%s &", 
				(const char *) terminal_binary,
				(const char *) terminal_option,
				(const char *) global_path.path ());
		ret = system ((const char *) cmd_line);
		PDEBUG ("Return value = %d\n", ret);
	} else {
		QString past_path;
		past_path.sprintf ("%s/.@%d", 
				(const char *) global_path.path (),
				past_time);

		PDEBUG ("Spawning shell in past time (for time %d)...\n", 
				past_time);
		cmd_line.sprintf ("%s %s=%s &", 
				(const char *) terminal_binary,
				(const char *) terminal_option,
				(const char *) past_path);
		PDEBUG ("Command line : %s\n", (const char *) cmd_line);

		ret = system ((const char *) cmd_line);
		PDEBUG ("Return value = %d\n", ret);
	}
}

/*
 * do_set_new_path:
 *	cd using the path in the text based browser edit.
 */
void
ttfs_form::do_set_new_path ()
{
	QString new_path = path_edit->text ();
	PDEBUG ("Got new path %s\n", (const char *) new_path);

	QFileInfo finfo (new_path);

	if (finfo.exists () == FALSE) {
		// reddish
		path_edit->setBackgroundColor (QColor (255, 203, 203));	
		return;
	} else {
		// correct, and in past: blue
		// correct and in present: yellow
		if (is_in_past) {
			// bluish
			path_edit->setBackgroundColor (QColor (206, 194, 255));
		} else {
			//yellowish
			path_edit->setBackgroundColor (QColor (246, 255, 198));
		}
	}

	QDir test_dir = QDir (new_path);
	if (test_dir.exists (new_path) == TRUE) {
		/*
		 * change directory ...
		 */
		ttfs_form::do_change_directory (new_path);
	}
}

/*
 * do_zoom_in_date:
 *	zoom in the slider for the date.
 */
void
ttfs_form::do_zoom_in_date ()
{
	PDEBUG ("Zooming IN the date slider ....\n");
	int val = date_slider->value ();
	PDEBUG ("Got value: %d\n", val);

	int rem = 100 - val;
	double fraction = rem / 100.0;

	PDEBUG ("val of slider = %d, remaining = %d, fraction = %f\n", 
			val, rem, fraction);
	int delta_days = low_date.daysTo (high_date);
	PDEBUG ("delta expressed in days is %d\n", delta_days);

	double days_in_past = fraction * (double) delta_days;
	PDEBUG ("days in past =  %f\n", days_in_past);

	int high_part, low_part;

	high_part = rem;
	low_part = val;
	PDEBUG ("Low interval: %d, high interval %d days\n", 
			low_part, high_part);

	/*
	 * the selected date is high date - days_in_past;
	 */

	QDate selected = high_date.addDays (-(int) days_in_past);

	PDEBUG ("selected date is %s\n",
			(const char *) selected.toString 
			(QString ("d MMMM yyyy")));

	/*
	 * shift the days a jalf interval from the slider 
	 * to the left and right.
	 */

	int left_shift = (int) (delta_days - days_in_past) / 2;
	int right_shift = (int) days_in_past / 2;

	if (left_shift == 0) {
		left_shift = 1;
	}
	if (right_shift == 0) {
		right_shift = 1;
	}
	PDEBUG ("val shifts left by %d\n", left_shift);
	PDEBUG ("val shifts right by %d\n", right_shift);

	low_date = selected.addDays (-left_shift);
	high_date = selected.addDays (right_shift);
	if (high_date > today) {
		high_date = today;
	}

	ttfs_form::refresh_time_and_date ();
}

/*
 * do_zoom_out_date:
 *	zoom out the date slider.
 */
void
ttfs_form::do_zoom_out_date ()
{
	PDEBUG ("Zooming OUT of the date slider ....\n");

	int val = date_slider->value ();
	PDEBUG ("Got value: %d\n", val);

	int rem = 100 - val;
	double fraction = rem / 100.0;

	PDEBUG ("val of slider = %d, remaining = %d, fraction = %f\n", val, rem,
			fraction);
	int delta_days = low_date.daysTo (high_date);
	PDEBUG ("delta expressed in days is %d\n", delta_days);

	double days_in_past = fraction * (double) delta_days;
	PDEBUG ("days in past =  %f\n", days_in_past);

	int high_part, low_part;

	high_part = rem;
	low_part = val;
	PDEBUG ("Low interval: %d, high interval %d days\n", low_part, high_part);

	/*
	 * the selected date is high date - days_in_past;
	 */

	QDate selected = high_date.addDays (-(int) days_in_past);

	PDEBUG ("selected date is %s\n",
			(const char *) selected.toString 
			(QString ("d MMMM yyyy")));

	/*
	 * shift it twice the left interval and twice the right interval.
	 */

	int left_shift = 2 * (int) (delta_days - days_in_past);
	int right_shift = 2 * (int) days_in_past;

	if (left_shift == 0) {
		left_shift = 1;
	}
	if (right_shift == 0) {
		right_shift = 1;
	}


	PDEBUG ("val shifts left by %d\n", left_shift);
	PDEBUG ("val shifts right by %d\n", right_shift);

	low_date = low_date.addDays (-left_shift);
	QDate epoch = QDate (1970, 1, 1);	// Jan 1 1970
	if (low_date < epoch) {
		low_date = epoch;
	}

	high_date = high_date.addDays (right_shift);
	if (high_date > today) {
		high_date = today;
	}

	ttfs_form::refresh_time_and_date ();
}

/*
 * do_help:
 * 	spawn a web browser with the correct help page :)
 */
void
ttfs_form::do_help ()
{
	PDEBUG ("Helping ...\n");
}


void
ttfs_form::do_file_copy ()
{
	PDEBUG ("Copying file ...\n");
}


void
ttfs_form::do_file_cut ()
{
	PDEBUG ("Cutting file (moving)\n");
}

void
ttfs_form::do_file_paste ()
{
	PDEBUG ("Pasting file ...\n");
}

/*
 * do_new_date_selected:
 *	when the date slider changes, we look at the value, 
 *	and set the status bar accordingly.
 */
void
ttfs_form::do_new_date_selected (
		int val)
{
	int rem = 100 - val;
	double fraction = rem / 100.0;

	PDEBUG ("val of slider = %d, remaining = %d, fraction = %f\n", 
			val, rem, fraction);
	int delta_days = low_date.daysTo (high_date);
	PDEBUG ("delta expressed in days is %d\n", delta_days);

	double days_in_past = fraction * (double) delta_days;
	PDEBUG ("days in past =  %f\n", days_in_past);

	/*
	 * the selected date is high date - days_in_past;
	 */

	QDate selected = high_date.addDays (-(int) days_in_past);

	PDEBUG ("selected date is %s\n",
			(const char *) selected.toString 
			(QString ("d MMMM yyyy")));
	QString select_str;

	select_str.sprintf ("Selected -- %s",
			(const char *) selected.
			toString (QString ("MMMM d, yyyy")));
	selected_date_label->setText (select_str);


	/*
	 *  the selected date is high date - days_in_past;
	 */

	QDate selected_date = high_date.addDays (-(int) days_in_past);

	PDEBUG ("selected date is %s\n",
			(const char *) selected_date.toString 
			(QString ("d MMMM yyyy")));

	/* get the time selected from the time edit box */
	QTime selected_time = time_edit->time ();

	/* find the number of seconds from UTC to selected date and time. */
	QDateTime dt (selected_date, selected_time);

	past_time = dt.toTime_t ();

	PDEBUG ("Got the seconds from UTC to travel to ...%d\n", past_time);

	QDateTime current = QDateTime::currentDateTime ();
	uint secs = current.toTime_t ();

	if (past_time < secs) {
		if (is_in_init) {
			past_time = 0;
			is_in_past = FALSE;
		} else {
			PDEBUG ("Time is in the past!\n");
			is_in_past = TRUE;
		}
	} else {
		PDEBUG ("Time is in the future!\n");
		past_time = 0;
		is_in_past = FALSE;
	}
	ttfs_form::do_refresh ();
}


/*
 * do_hidden_toggle:
 *	toggle the hidden file flag, and update the view.
 */
void
ttfs_form::do_hidden_toggle (
		bool value)
{
	if (value == TRUE) {
		// button on... so show hidden files
		show_hidden_toggle_button->setText ("Hide");
		hide_files = FALSE;
	} else {
		show_hidden_toggle_button->setText ("Show");
		hide_files = TRUE;
	}
	ttfs_form::tt_display_path ();
}

/*
 * do_context_menu:
 *	show a pop up menu on right click on an intem in the icon view
 */
void
ttfs_form::do_context_menu (
		QIconViewItem * item, 
		const QPoint & pos)
{
	PDEBUG ("giving context menu for file :%s\n", 
			(const char *) item->text ());
	PDEBUG ("context menu at point (%d, %d)\n", pos.x (), pos.y ());
}

/*
 * do_time_changed
 *	update the labels, the delta, the current time etc.
 * as the time changes, also update the file system view....
 */
void
ttfs_form::do_time_changed (
		const QTime & time)
{
	PDEBUG ("Got time %s\n", time.toString ("hh mm ss"));
	PDEBUG ("Changing time value ...\n");
	int val = date_slider->value ();
	PDEBUG ("Got value: %d\n", val);

	int rem = 100 - val;
	double fraction = rem / 100.0;

	PDEBUG ("val of slider = %d, remaining = %d, fraction = %f\n", 
			val, rem, fraction);
	int delta_days = low_date.daysTo (high_date);
	PDEBUG ("delta expressed in days is %d\n", delta_days);

	double days_in_past = fraction * (double) delta_days;
	PDEBUG ("days in past =  %f\n", days_in_past);

	/*
	 * the selected date is high date - days_in_past;
	 */

	QDate selected_date = high_date.addDays (-(int) days_in_past);

	PDEBUG ("selected date is %s\n",
			(const char *) selected_date.toString 
			(QString ("d MMMM yyyy")));

	/* get the time selected from the time edit box */
	QTime selected_time = time_edit->time ();

	/* find the number of seconds from UTC to selected date and time. */
	QDateTime dt (selected_date, selected_time);

	past_time = dt.toTime_t ();

	PDEBUG ("Got the seconds from UTC to travel to ...%d\n", past_time);

	QDateTime current = QDateTime::currentDateTime ();
	uint secs = current.toTime_t ();

	if (past_time < secs) {
		PDEBUG ("Time is in the past!\n");
		is_in_past = TRUE;
	} else {
		PDEBUG ("Time is in the future!Snapping to present time\n");
		past_time = 0;
		is_in_past = FALSE;
	}

	ttfs_form::tt_display_path ();
}

/*
 * get_ls_output:
 *	pipe the ls file2 command's output to this slot handler, and then
 * 	read line by line the output of ls command... 
 *	to find the versions of that file.
 */
void
ttfs_form::get_ls_output ()
{
	int num = 0;
	char str[BUFSIZ];

	file_icon_view->clear ();

	while (1) {
		QString ver = p->readLineStdout ();
		if (ver == QString::null) {
			PDEBUG ("End of output!");
			break;
		}
		PDEBUG ("Got version %s\n", (const char *) ver);
		int index = ver.find ('@');
		if (index == -1) {
			PDEBUG ("No versions for this file\n");
			sscanf ((const char *) ver, "%s", str);
		} else {
			int time = -1;
			ver[index] = ' ';

			sscanf ((const char *) ver, "%s %d", str, &time);
			/*
			 * for each of these, convert it to a string
			 * then add the files to the icon view...
			 */

			num++;

			QPixmap pixmap;
			QString path;
			path.sprintf ("%s/%s",
					(const char *) 
					global_time_shifted_path.path (), str);

			PDEBUG ("Getting pixmaps for path %s\n", 
					(const char *) path);
			pixmap = ttfs_form::get_file_pixmap (path);

			QString name;
			name.sprintf ("%s@%d", str, time);
			new QIconViewItem (file_icon_view, name, pixmap);

		}
	}
	/* show the current object too */
	QPixmap pixmap;
	QString path;
	path.sprintf ("%s/%s", 
			(const char *) global_time_shifted_path.path (),
			str);

	PDEBUG ("Getting pixmaps for path %s\n", (const char *) path);
	pixmap = ttfs_form::get_file_pixmap (path);

	QString name;
	name.sprintf ("%s", str);
	new QIconViewItem (file_icon_view, name, pixmap);
}

/*
 * do_versions:
 *	given a selected file/directory, show all the versions that the 
 *	file system knows about!
 */
void
ttfs_form::do_versions (
		QString path)
{
	PDEBUG ("doing versions ...\n");

	QIconViewItem *selected = NULL;

	selected = file_icon_view->currentItem ();
	if (selected == 0) {
		PDEBUG ("please select a directory\n");
		time_status_bar->setText ("Please select a directory"
				" or file...");
		return;
	}

	p = new QProcess (this);
	p->addArgument ("ls");
	p->addArgument (path + "/" + selected->text () + "@");

	connect (p, SIGNAL (readyReadStdout ()), this, SLOT (get_ls_output ()));
	if (!p->start ()) {
		// error handling
		PDEBUG ("Couldnt execute ls ....\n");
		time_status_bar->setText ("couldn't execute ls <file>@");
		return;
	}
}

/*
 * do_archive:
 *	TODO: capture all the versions in a standard file system tree, 
 *	and then tar.gz it.
 * 	this will preserve the ttfs file manager compatibility since 
 *	the names of the versions will be mangled in the ext3cow manner.
 *
 * 	currently, this function just takes a selected directory or file, 
 *	and archives it, and stores the resulting
 * 	archive in the current working directory.
 */
void
ttfs_form::do_archive ()
{
	/* take the time shifed directory, and archive it */
	QFileInfo finfo (global_time_shifted_path.path ());
	QString cmdline;
	QIconViewItem *selected = NULL;

	return;			// FOR now, unimplemented!

	if (finfo.isWritable ()) {
		selected = file_icon_view->currentItem ();
		if (selected == 0) {
			PDEBUG ("please select a directory\n");
			time_status_bar->setText ("Please select a Directory"
					"/ File");
			return;
		}
		// get extension .... dont tar.gz tgz files!

		/* do it locally! */
		cmdline.sprintf ("tar -czf %s/%s.tgz %s/%s &",
				(const char *) global_time_shifted_path.path (),
				(const char *) selected->text (),
				(const char *) global_time_shifted_path.path (),
				(const char *) selected->text ());
		PDEBUG ("Command line: local: %s\n", (const char *) cmdline);
		system (cmdline);
	} else {
		/* TODO */
		PDEBUG ("show a file open dialog\n");
	}
}


/*
 * do_next_version:
 *	Look at all the versions, look at the current time, 
 *	and snap to the next version time
 */
void
ttfs_form::do_next_version ()
{
	QString path = global_time_shifted_path.path ();

	PDEBUG ("Doing next version from time %d...\n", past_time);
	if (!is_in_past) {
		PDEBUG ("No next version... Present time!\n");
		return;
	}
	QDir dirpath (path);
	QFileInfo finfo (path);
}

/*
 * do_prev_version:
 *	Look at all the versions, look at the current time, 
 *	and snap to the previous version time
 */

void
ttfs_form::do_prev_version ()
{
	PDEBUG ("doing previous version ...\n");
}

/*
 * do_compare:
 * 	given a version view of a file, select a file version, 
 *	and hit the compare view button.
 * 	this function launches kompare on the version file 
 *	and the present time file!
 */
void
ttfs_form::do_compare ()
{
	char name[BUFSIZ];
	int time = -1;


	// get the selected file from the icon view
	QIconViewItem *selected = NULL;

	selected = file_icon_view->currentItem ();
	if (selected == 0) {
		PDEBUG ("please select a file\n");
		time_status_bar->
			setText ("Please select a version file to "
					"compare with current file");
		return;
	}
	QString file (selected->text ());
	QString path = global_path.path () + "/" + file;
	PDEBUG ("got path %s\n", (const char *) path);
	int index = path.find ('@');
	/*
	 * todo: look at double @'s in path 
	 */

	if (index != -1) {
		// seperate the time stamp from the file..
		path[index] = ' ';
		sscanf ((const char *) path, "%s %d", name, &time);
		PDEBUG ("Got execute name: %s, time %d\n", name, time);
		path[index] = '@';	//restore it
	} else {
		PDEBUG ("Cannot compare ... this is the file in present"
				" time!\n");
		time_status_bar->setText ("Cannot compare : this is not "
				"a version!\n");
		return;
    }

	// now the two files are the file and name ...
	QString cmd;
	cmd.sprintf ("kompare %s %s &", (const char *) path, name);
	PDEBUG ("executing: %s\n", (const char *) cmd);

	//    run "kompare" on both the files ...          
	system (cmd);
}

/*
 * go_back: 
 * 	use a stack to push the current address.
 */
void
ttfs_form::go_back ()
{

	/* 
	 * before we make any changes to the current path, 
	 * push the current path to the forward stack too...
	 *  this allows us to go forward to that point in the file system
	 */
	push_state_to_forward_stack ();

	/*
	 * restore temporal state from the back stack
	 */
	restore_state_from_back_stack ();
}


void
ttfs_form::go_forward ()
{
	/*
	 * push any state to the back stack, as we may want to go back to it
	 */

	push_state_to_back_stack ();

	/*
	 * now restore state from the forward stack ...
	 */
	restore_state_from_forward_stack ();
}

/*
 * do_slider_pressed():
 *  	here just as the slider is being moved, (before) we push the 
 *	current path to the back stack.
 */
void
ttfs_form::do_slider_pressed ()
{

	// push the current path to the back stack in case we wanna go back..
	push_state_to_back_stack ();
}

/*
 * do_time_value_changed():
 *	push state when the time changes, so we can go back to that time
 *
 *	ISSUE: 
 *		is this too irritating to go back every second you change??
 * 		there is no other way. if we have a button to "commit" 
 *		time changes, then every time we need to scroll continuosly 
 *		through the time we have to hit the button!
 *		continuous time access is more important than the going 
 *		back granularity of once everytime you change the time.
 */
void
ttfs_form::do_time_value_changed ()
{
	// push the current path to the back stack in case we wanna go back..
	push_state_to_back_stack ();
}

/*
 * do_bookmarks:
 *	view bookmarls directory
 */
void
ttfs_form::do_bookmarks ()
{
	PDEBUG ("Bookmark view mode!\n");
	QString bm_path =
		QDir::home ().path () + "/" + QString (bookmarks_dir_name);

	push_state_to_back_stack ();
	global_path.setPath (bm_path);

	do_refresh ();
}

/*
 * add_to_bookmarks:
 *	add a name label to the bookmarks, the current directory path
 */
void
ttfs_form::add_to_bookmarks ()
{
	PDEBUG ("Adding to bookmarks\n");
	/*
	 * take the current directory/selected object, 
	 * create a link to it in the bookmarks directory
	 */
	QIconViewItem *selected = NULL;
	QString selected_path;

	selected = file_icon_view->currentItem ();
	if (selected == 0) {
		// get the global path. (current dir)
		selected_path = global_time_shifted_path.path ();
	} else {
		selected_path = global_time_shifted_path.path () + 
			"/" + selected->text ();
	}
	PDEBUG ("Adding %s to bookmarks\n", (const char *) selected_path);

	QString bm_dir = QDir::home ().path () + "/" + 
		QString (bookmarks_dir_name) + "/";

	/*
	 * spawn a bookmark dialog box.
	 */
	bookmark_form *bm = new bookmark_form;
	bm->bm_path_name->setText (selected_path);
	bm->bm_dir_name->setText (bm_dir);
	bm->bm_name_line_edit->setText ("");
	bm->show ();
}

void
ttfs_form::do_file_icon_clicked (
		QIconViewItem * item)
{
	/*
	 * this is done so that the selection is cleared 
	 * if some one clicks on nothing
	 */
	if (item == NULL) {
		PDEBUG ("Clearing selection\n");
		do_refresh ();
	}
}
