# -*- python -*-

# Copyright 2005, W. Borgert <debacle@debian.org>
# GPL v2
# Complete BNF: http://people.debian.org/~debacle/ttcn-bnf.html

import pyparsing as pp

def checkLeadingZero(s, loc, toks):
    if len(toks[0]) > 1 and toks[0][0] == '0':
        raise ValueError, "Leading zero not allowed"

def checkOctPair(s, loc, toks):
    flag = False
    for c in toks[0][1]:
        if c in "0123456789abcdefABCDEF":
            flag = not flag
    if flag:
        raise ValueError, "Octetstring not well-formed"

# This is inspired by the original pyparsing.Word class
class ppWord(pp.Word):
    def __init__(self, initChars, bodyChars):
        pp.Word.__init__(self, initChars, bodyChars)
        self.keywords = {}

    def parseImpl(self, instring, loc, doActions=True):
        if not(instring[loc] in self.initChars):
            exc = self.myException
            exc.loc = loc
            exc.pstr = instring
            raise exc
        start = loc
        loc += 1
        bodychars = self.bodyChars
        maxloc = start + self.maxLen
        maxloc = min(maxloc, len(instring))
        while loc < maxloc and instring[loc] in bodychars:
            loc += 1

        if loc - start < self.minLen:
            exc = self.myException
            exc.loc = loc
            exc.pstr = instring
            raise exc

        if instring[start:loc] in self.keywords:
            exc = self.myException
            exc.loc = loc
            exc.pstr = instring
            raise exc
        return loc, [instring[start:loc]]

def BNF():
    _Keywords = {
    # [2]
    "module": "TTCN3ModuleKeyword",
    # [15]
    "type": "TypeDefKeyword",
    # [18]
    "record": "RecordKeyword",
    # [25]
    "optional": "OptionalKeyword",
    # [27]
    "union": "UnionKeyword",
    # [31]
    "set": "SetKeyword",
    # [33]
    "of": "OfKeyword",
    # [37]
    "enumerated": "EnumKeyword",
    # [49]
    "length": "LengthKeyword",
    # [53]
    "port": "PortKeyword",
    # [59]
    "message": "MessageKeyword",
    # [61]
    "all": "AllKeyword",
    # [64]
    "procedure": "ProcedureKeyword",
    # [69]
    "mixed": "MixedKeyword",
    # [74]
    "component": "ComponentKeyword",
    # [85]
    "const": "ConstKeyword",
    # [89]
    "template": "TemplateKeyword",
    # [92]
    "modifies": "ModifiesKeyword",
    # [119]
    "pattern": "PatternKeyword",
    # [121]
    "complement": "ComplementKeyword",
    # [126]
    "superset": "SupersetKeyword",
    # [124]
    "subset": "SubsetKeyword",
    # [132]
    "ifpresent": "IfPresentKeyword",
    # [136]
    "infinity": "InfinityKeyword",
    # [149]
    "function": "FunctionKeyword",
    # [154]
    "return": "ReturnKeyword",
    # [156]
    "runs": "RunsKeyword",
    # [157]
    "on": "OnKeyword",
    # [158]
    "mtc": "MTCKeyword",
    # [171]
    "signature": "SignatureKeyword",
    # [176]
    "exception": "ExceptionKeyword",
    # [178]
    "noblock": "NoBlockKeyword",
    # [181]
    "testcase": "TestcaseKeyword",
    # [187]
    "system": "SystemKeyword",
    # [189]
    "execute": "ExecuteKeyword",
    # [194]
    "altstep": "AltstepKeyword",
    # [202]
    "import": "ImportKeyword",
    # [205]
    "except": "ExceptKeyword",
    # [221]
    "language": "LanguageKeyword",
    # [224]
    "recursive": "RecursiveKeyword",
    # [257]
    "group": "GroupKeyword",
    # [260]
    "external": "ExtKeyword",
    # [265]
    "modulepar": "ModuleParKeyword",
    # [271]
    "control": "ControlKeyword",
    # [280]
    "var": "VarKeyword",
    # [286]
    "timer": "TimerKeyword",
    # [294]
    "self": "SelfOp",
    # [298]
    "done": "DoneKeyword",
    # [300]
    "running": "RunningKeyword",
    # [301]
    "create": "CreateKeyword",
    # [303]
    "connect": "ConnectKeyword",
    # [308]
    "disconnect": "DisconnectKeyword",
    # [310]
    "map": "MapKeyword",
    # [312]
    "unmap": "UnmapKeyword",
    # [314]
    "start": "StartKeyword",
    # [321]
    "send": "SendOpKeyword",
    # [324]
    "to": "ToKeyword",
    # [328]
    "call": "CallOpKeyword",
    # [331]
    "nowait": "NowaitKeyword",
    # [339]
    "reply": "ReplyKeyword",
    # [343]
    "raise": "RaiseKeyword",
    # [347]
    "receive": "ReceiveOpKeyword",
    # [350]
    "from": "FromKeyword",
    # [354]
    "value": "ValueKeyword",
    # [356]
    "sender": "SenderKeyword",
    # [359]
    "trigger": "TriggerOpKeyword",
    # [362]
    "getcall": "GetCallOpKeyword",
    # [366]
    "param": "ParaKeyword",
    # [375]
    "getreply": "GetReplyOpKeyword",
    # [379]
    "check": "CheckOpKeyword",
    # [386]
    "catch": "CatchOpKeyword",
    # [391]
    "clear": "ClearOpKeyword",
    # [396]
    "stop": "StopKeyword",
    # [397]
    "any": "AnyKeyword",
    # [408]
    "timeout": "TimeoutKeyword",
    # [411]
    "bitstring": "BitStringKeyword",
    # [412]
    "boolean": "BooleanKeyword",
    # [413]
    "integer": "IntegerKeyword",
    # [414]
    "octetstring": "OctetStringKeyword",
    # [415]
    "objid": "ObjectIdentifierKeyword",
    # [416]
    "hexstring": "HexStringKeyword",
    # [417]
    "verdicttype": "VerdictTypeKeyword",
    # [418]
    "float": "FloatKeyword",
    # [419]
    "address": "AddressKeyword",
    # [420]
    "default": "DefaultKeyword",
    # [421]
    "anytype": "AnyTypeKeyword",
    # [422]
    "charstring": "CharStringKeyword",
    # [423]
    "universal charstring": "UniversalCharString",
    # [424]
    "universal": "UniversalKeyword",
    # [425]
    "char": "CharKeyword",
    # [426]
    "universal char": "UniversalChar",
    # [479]
    "null": "AddressValue",
    # [481]
    "omit": "OmitKeyword",
    # [482]
    "in": "InParKeyword",
    # [483]
    "out": "OutParKeyword",
    # [484]
    "inout": "InOutParKeyword",
    # [494]
    "with": "WithKeyword",
    # [499]
    "encode": "EncodeKeyword",
    # [500]
    "variant": "VariationKeyword",
    # [501]
    "display": "DisplayKeyword",
    # [502]
    "extension": "ExtensionKeyword",
    # [503]
    "override": "OverrideKeyword",
    # [514]
    "setverdict": "SetVerdictKeyword",
    # [515]
    "getverdict": "GetLocalVerdict",
    # [517]
    "action": "ActionKeyword",
    # [520]
    "alt": "AltKeyword",
    # [527]
    "interleave": "InterleavedKeyword",
    # [533]
    "label": "LabelKeyword",
    # [536]
    "goto": "GotoKeyword",
    # [537]
    "repeat": "RepeatStatement",
    # [539]
    "activate": "ActivateKeyword",
    # [541]
    "deactivate": "DeactivateKeyword",
    # [580]
    "log": "LogKeyword",
    # [583]
    "for": "ForKeyword",
    # [588]
    "while": "WhileKeyword",
    # [590]
    "do": "DoKeyword",
    # [592]
    "if": "IfKeyword",
    # [594]
    "else": "ElseKeyword"
    }
    for kw in _Keywords.keys():
        exec '%s = pp.Literal("%s")' % (_Keywords[kw], kw)
    # [596]
    Dot = pp.Literal(".")
    # [597]
    Dash = pp.Literal("-")
    # [598]
    Minus = Dash
    # [599]
    SemiColon = pp.Literal(";")
    # [600]
    Colon = pp.Literal(":")
    # [601]
    Underscore = pp.Literal("_")
    # [602]
    BeginChar = pp.Literal("{")
    # [603]
    EndChar = pp.Literal("}")
    # [604]
    AssignmentChar = pp.Literal(":=")

    # Identifiers
    # [472]
    Identifier = ppWord(pp.alphas, pp.alphanums + "_")
    Identifier.keywords = _Keywords
    # [4]
    ModuleIdentifier = Identifier
    # [20]
    StructTypeIdentifier = Identifier
    # [24]
    StructFieldIdentifier = Identifier
    # [38]
    EnumTypeIdentifier = Identifier
    # [41]
    EnumerationIdentifier = Identifier
    # [43]
    SubTypeIdentifier = Identifier
    # [54]
    PortTypeIdentifier = Identifier
    # [75]
    ComponentType = Identifier
    # [76]
    ComponentTypeIdentifier = Identifier
    # [81]
    PortIdentifier = Identifier
    # [86]
    ConstIdentifier = Identifier
    # [90]
    TemplateIdentifier = Identifier
    # [150]
    FunctionIdentifier = Identifier
    # [172]
    SignatureIdentifier = Identifier
    # [182]
    TestcaseIdentifier = Identifier
    # [195]
    AltstepIdentifier = Identifier
    # [258]
    GroupIdentifier = Identifier
    # [261]
    ExtFunctionIdentifier = Identifier
    # [263]
    ExtConstIdentifier = Identifier
    # [269]
    ModuleParIdentifier = Identifier
    # [281]
    VarIdentifier = Identifier
    # [287]
    TimerIdentifier = Identifier
    # [444]
    NameForm = Identifier
    # [486]
    ValueParIdentifier = Identifier
    # [488]
    PortParIdentifier = Identifier
    # [490]
    TimerParIdentifier = Identifier
    # [492]
    TemplateParIdentifier = Identifier
    # [534]
    LabelIdentifier = Identifier

    # Grammar
    # [595]
    StatementBlock = pp.Forward()
    ElseClause = pp.Group(ElseKeyword + StatementBlock)
    # [552]
    SingleExpression = pp.Forward()
    BooleanExpression = SingleExpression
    # [593]
    ElseIfClause = pp.Group(ElseKeyword + IfKeyword + pp.Literal("(") \
                            + BooleanExpression + pp.Literal(")") \
                            + StatementBlock)
    # [591]
    ConditionalConstruct = pp.Group(IfKeyword + pp.Literal("(") \
                                    + BooleanExpression + pp.Literal(")") \
                                    + StatementBlock \
                                    + pp.ZeroOrMore(ElseIfClause) \
                                    + pp.Optional(ElseClause))
    # [589]
    DoWhileStatement = pp.Group(DoKeyword + StatementBlock + WhileKeyword \
                                + pp.Literal("(") + BooleanExpression \
                                + pp.Literal(")"))
    # [587]
    WhileStatement = pp.Group(WhileKeyword + pp.Literal("(") \
                              + BooleanExpression + pp.Literal(")") \
                              + StatementBlock)
    # [586]
    Assignment = pp.Forward()
    Step = Assignment
    # [585]
    Final = BooleanExpression
    # [584]
    VarInstance = pp.Forward()
    Initial = (VarInstance | Assignment)
    # [582]
    ForStatement = pp.Group(ForKeyword + pp.Literal("(") + Initial \
                            + SemiColon + Final + SemiColon + Step \
                            + pp.Literal(")") + StatementBlock)
    # [581]
    LoopConstruct = (ForStatement | WhileStatement | DoWhileStatement)
    # [478]
    FreeText = pp.Group(pp.Literal('"') \
                        + pp.ZeroOrMore(pp.CharsNotIn('"\n\r')) \
                        + pp.Literal('"'))
    # [579]
    LogStatement = pp.Group(LogKeyword + pp.Literal("(") \
                            + pp.Optional(FreeText) + pp.Literal(")"))
    # [578]
    ShiftOp = pp.oneOf("<< >> <@ @>")
    # [577]
    #StringOp = pp.Literal("&")
    # [576]
    LogicalOp = pp.oneOf("and xor or")
    # [575]
    BitOp = pp.oneOf("and4b xor4b or4b")
    # [574]
    RelOp = pp.oneOf("== <= >= != > <")
    # [573]
    UnaryOp = pp.oneOf("+ -")
    # [572]
    MultiplyOp = pp.oneOf("* / mod rem")
    # [571]
    AddOp = pp.oneOf("+ - &") # & = StringOp
    # [293]
    SystemOp = SystemKeyword
    # [292]
    CreateOp = pp.Group(ComponentType + Dot + CreateKeyword)
    # [291]
    RunningOp = pp.Forward()
    ConfigurationOps = (SelfOp ^ SystemOp ^ CreateOp ^ RunningOp)
    # [512]
    VerdictOps = GetLocalVerdict
    # [200]
    AltstepRef = AltstepIdentifier
    # [199]
    FunctionActualParList = pp.Forward()
    AltstepInstance = pp.Group(AltstepRef + pp.Literal("(") \
                               + pp.Optional(FunctionActualParList) \
                               + pp.Literal(")"))
    # [538]
    ActivateOp = pp.Group(ActivateKeyword + pp.Literal("(") \
                          + AltstepInstance + pp.Literal(")"))
    # [167]
    PreDefFunctionIdentifier = pp.oneOf(
        "int2char int2unichar int2bit int2hex int2oct int2str int2float " \
        + "float2int char2int unichar2int bit2int bit2hex bit2oct " \
        + "bit2str hex2int hex2bit hex2oct hex2str oct2int oct2bit " \
        + "oct2hex oct2str str2int str2oct lengthof sizeof ispresent " \
        + "ischosen regexp substr rnd")
    # [166]
    FunctionRef = (PreDefFunctionIdentifier | FunctionIdentifier)
    # [165]
    FunctionInstance = pp.Group(FunctionRef + pp.Literal("(") \
                                + pp.Optional(FunctionActualParList) \
                                + pp.Literal(")"))
    # [570]
    TestcaseInstance = pp.Forward()
    OpCall = (ConfigurationOps ^ VerdictOps ^ FunctionInstance ^ ActivateOp \
              ^ TestcaseInstance)
    # [234]
    TypeDefIdentifier = (StructTypeIdentifier ^ EnumTypeIdentifier \
                         ^ PortTypeIdentifier ^ ComponentTypeIdentifier \
                         ^ SubTypeIdentifier)
    # [104]
    FieldOrBitNumber = SingleExpression
    # [103]
    ArrayOrBitRef = pp.Group(pp.Literal("[") + FieldOrBitNumber \
                             + pp.Literal("]"))
    # [569]
    ExtendedFieldReference = pp.OneOrMore(
        pp.Group(Dot + pp.Group(StructFieldIdentifier \
                                ^ ArrayOrBitRef ^ TypeDefIdentifier)) \
        ^ ArrayOrBitRef)
    # [568]
    Value = pp.Forward()
    Primary = (OpCall ^ Value \
               ^ pp.Group(pp.Literal("(") + SingleExpression \
                          + pp.Literal(")")))
    # [567]
    Factor = pp.Group(pp.Optional(UnaryOp) + Primary)
    # [566]
    Term = pp.Group(Factor + pp.ZeroOrMore(MultiplyOp + Factor))
    # [565]
    Product = pp.Group(Term + pp.ZeroOrMore(AddOp + Term))
    # [564]
    SubResult = pp.Group(pp.Optional(pp.Literal("not4b")) + Product)
    # [563]
    Result = pp.Group(SubResult + pp.ZeroOrMore(BitOp + SubResult))
    # [562]
    PartialExpression = pp.Group(Result + pp.Optional(ShiftOp + Result))
    # [561]
    SubExpression = pp.Group(PartialExpression \
                             + pp.Optional(RelOp + PartialExpression))
    # [560]
    SimpleExpression = pp.Group(pp.Optional(pp.Literal("not")) \
                                + SubExpression)
    # [559]
    SingleExpression << pp.Group(SimpleExpression \
                                 + pp.ZeroOrMore(LogicalOp \
                                                 + SimpleExpression))
    # [282]
    VariableRef = pp.Group((VarIdentifier ^ ValueParIdentifier) \
                            + pp.Optional(ExtendedFieldReference))
    # [558]
    Expression = pp.Forward()
    Assignment << pp.Group(VariableRef + AssignmentChar + Expression)
    # [557]
    ConstantExpression = pp.Forward()
    ArrayElementConstExpressionList = pp.Group(
        ConstantExpression + pp.ZeroOrMore(pp.Literal(",") \
                                           + ConstantExpression))
    # [556]
    ArrayConstExpression = pp.Group(
        pp.Literal("{") + pp.Optional(ArrayElementConstExpressionList) \
        + pp.Literal("}"))
    # [555]
    FieldReference = pp.Forward()
    FieldConstExpressionSpec = pp.Group(FieldReference + AssignmentChar \
                                        + ConstantExpression)
    # [554]
    FieldConstExpressionList = pp.Group(
        pp.Literal("{") \
        + FieldConstExpressionSpec \
        + pp.ZeroOrMore(pp.Literal(",") \
                        + FieldConstExpressionSpec) \
        + pp.Literal("}"))
    # [553]
    CompoundConstExpression = (FieldConstExpressionList \
                               ^ ArrayConstExpression)
    # [551]
    SingleConstExpression = SingleExpression
    # [550]
    ConstantExpression << (SingleConstExpression ^ CompoundConstExpression)
    # [109]
    NotUsedSymbol = Dash
    # [549]
    NotUsedOrExpression = (Expression | NotUsedSymbol)
    # [548]
    ArrayElementExpressionList = pp.Group(NotUsedOrExpression \
                                          + pp.ZeroOrMore(
        pp.Literal(",") + NotUsedOrExpression))
    # [547]
    ArrayExpression = pp.Group(pp.Literal("{") \
                               + pp.Optional(ArrayElementExpressionList) \
                               + pp.Literal("}"))
    # [546]
    FieldExpressionSpec = pp.Group(FieldReference + AssignmentChar \
                                   + Expression)
    # [545]
    FieldExpressionList = pp.Group(pp.Literal("{") + FieldExpressionSpec \
                                   + pp.ZeroOrMore(pp.Literal(",") \
                                                   + FieldExpressionSpec) \
                                   + pp.Literal("}"))
    # [544]
    CompoundExpression = (FieldExpressionList ^ ArrayExpression)
    # [543]
    Expression << (SingleExpression ^ CompoundExpression)
    # [542]
    BasicStatements = (LogStatement | LoopConstruct \
                       | ConditionalConstruct | Assignment)
    # [540]
    DeactivateStatement = pp.Group(DeactivateKeyword \
                                   + pp.Optional(pp.Literal("(") \
                                                 + Expression \
                                                 + pp.Literal(")")))
    # [535]
    GotoStatement = pp.Group(GotoKeyword + LabelIdentifier)
    # [532]
    LabelStatement = pp.Group(LabelKeyword + LabelIdentifier)
    # [531]
    InterleavedAction = StatementBlock
    # [530]
    GuardOp = pp.Forward()
    InterleavedGuard = pp.Group(pp.Literal("[") + pp.Literal("]") + GuardOp)
    # [529]
    InterleavedGuardElement = pp.Group(InterleavedGuard + InterleavedAction)
    # [528]
    InterleavedGuardList = pp.OneOrMore(InterleavedGuardElement \
                                        + pp.Optional(SemiColon))
    # [526]
    InterleavedConstruct = pp.Group(InterleavedKeyword + BeginChar \
                                    + InterleavedGuardList + EndChar)
    # [524]
    AltGuardChar = pp.Group(pp.Literal("[") \
                            + pp.Optional(BooleanExpression) \
                            + pp.Literal("]"))
    # [523]
    ElseStatement = pp.Group(pp.Literal("[") + ElseKeyword + pp.Literal("]") \
                             + StatementBlock)
    # [522]
    GuardStatement = pp.Group(AltGuardChar \
                              + pp.Group(AltstepInstance \
                                         ^ pp.Group(GuardOp \
                                                    + StatementBlock)))
    # [521]
    AltGuardList = pp.Group(pp.OneOrMore(GuardStatement \
                                         + pp.Optional(SemiColon)) \
                            + pp.Optional(ElseStatement \
                                          + pp.Optional(SemiColon)))
    # [519]
    AltConstruct = pp.Group(AltKeyword + BeginChar + AltGuardList + EndChar)
    # [518]
    ReturnStatement = pp.Group(ReturnKeyword + pp.Optional(Expression))
    # [516]
    TemplateRefWithParList = pp.Forward()
    SUTStatements = pp.Group(ActionKeyword + pp.Literal("(") \
                             + (FreeText ^ TemplateRefWithParList) \
                             + pp.Literal(")"))
    # [513]
    SetLocalVerdict = pp.Group(SetVerdictKeyword + pp.Literal("(") \
                               + SingleExpression + pp.Literal(")"))
    # [511]
    VerdictStatements = SetLocalVerdict
    # [510]
    _KeywordBehaviourStatements = (ReturnStatement | AltConstruct \
                                   | InterleavedConstruct \
                                   | LabelStatement | GotoStatement \
                                   | RepeatStatement \
                                   | DeactivateStatement)
    BehaviourStatements = (_KeywordBehaviourStatements ^ TestcaseInstance \
                           ^ FunctionInstance ^ AltstepInstance)
    # [509]
    AttribSpec = FreeText
    # [410]
    PredefinedType = (BitStringKeyword ^ BooleanKeyword ^ CharStringKeyword \
                      ^ UniversalCharString ^ CharKeyword ^ UniversalChar \
                      ^ IntegerKeyword ^ OctetStringKeyword \
                      ^ ObjectIdentifierKeyword ^ HexStringKeyword \
                      ^ VerdictTypeKeyword ^ FloatKeyword ^ AddressKeyword \
                      ^ DefaultKeyword ^ AnyTypeKeyword)
    # [506]
    DefOrFieldRef = (Identifier ^ PredefinedType)
    # [505]
    DefOrFieldRefList = pp.Group(DefOrFieldRef \
                                 + pp.ZeroOrMore(pp.Literal(",") \
                                                 + DefOrFieldRef))
    # [504]
    AttribQualifier = pp.Group(pp.Literal("(") + DefOrFieldRefList \
                               + pp.Literal(")"))
    # [498]
    AttribKeyword = (EncodeKeyword | VariationKeyword | DisplayKeyword \
                     | ExtensionKeyword)
    # [497]
    SingleWithAttrib = pp.Group(AttribKeyword \
                                + pp.Optional(OverrideKeyword) \
                                + pp.Optional(AttribQualifier) \
                                + AttribSpec)
    # [496]
    MultiWithAttrib = pp.OneOrMore(SingleWithAttrib + pp.Optional(SemiColon))
    # [495]
    WithAttribList = pp.Group(pp.Literal("{") + MultiWithAttrib \
                              + pp.Literal("}"))
    # [493]
    WithStatement = pp.Group(WithKeyword + WithAttribList)
    # [491]
    Type = pp.Forward()
    FormalTemplatePar = pp.Group(pp.Optional(InParKeyword) + TemplateKeyword \
                                 + Type + TemplateParIdentifier)
    # [489]
    FormalTimerPar = pp.Group(pp.Optional(InOutParKeyword) + TimerKeyword \
                              + TimerParIdentifier)
    # [487]
    FormalPortPar = pp.Group(pp.Optional(InOutParKeyword) \
                             + PortTypeIdentifier \
                             + PortParIdentifier)
    # [485]
    FormalValuePar = pp.Group(pp.Optional(pp.oneOf("in out inout")) \
                              + Type + ValueParIdentifier)
    # [480]
    OmitValue = OmitKeyword
    # [470]
    Cstring = pp.Group(pp.Literal('"') \
                       + pp.ZeroOrMore(pp.Literal('""') \
                                       | pp.CharsNotIn('"\n\r')) \
                       + pp.Literal('"'))
    # [469]
    #Oct = pp.Group(Hex + Hex)
    # [468]
    Ostring = pp.Group(pp.Literal("'") \
                       + pp.Optional(pp.Word("0123456789abcdefABCDEF")) \
                       + pp.Literal("'O"))
    Ostring.setParseAction(checkOctPair)
    # [467]
    #Hex << (Num ^ pp.Literal("A") ^ pp.Literal("B") ^ pp.Literal("C") \
    #        ^ pp.Literal("D") ^ pp.Literal("E") ^ pp.Literal("F") \
    #        ^ pp.Literal("a") ^ pp.Literal("b") ^ pp.Literal("c") \
    #        ^ pp.Literal("d") ^ pp.Literal("e") ^ pp.Literal("f"))
    # [466]
    Hstring = pp.Group(pp.Literal("'") \
                       + pp.Optional(pp.Word("0123456789abcdefABCDEF")) \
                       + pp.Literal("'H"))
    # [465]
    #Bin = (pp.Literal("0") ^ pp.Literal("1"))
    # [464]
    Bstring = pp.Group(pp.Literal("'") \
                       + pp.Optional(pp.Word("01")) \
                       + pp.Literal("'B"))
    # [463]
    #Num << (pp.Literal("0") ^ NonZeroNum)
    # [462]
    DecimalNumber = pp.Word("0123456789", min = 1)
    # [461]
    #NonZeroNum << (pp.Literal("1") ^ pp.Literal("2") ^ pp.Literal("3") \
    #               ^ pp.Literal("4") ^ pp.Literal("5") ^ pp.Literal("6") \
    #               ^ pp.Literal("7") ^ pp.Literal("8") ^ pp.Literal("9"))
    # [460]
    Number = pp.Word("0123456789", min = 1)
    Number.setParseAction(checkLeadingZero)
    # [459]
    ValueReference = (ValueParIdentifier ^ VarIdentifier)
    # [458]
    ReferencedValue = pp.Group(ValueReference \
                               + pp.Optional(ExtendedFieldReference))
    # [457]
    Exponential = pp.Literal("E")
    # [456]
    FloatENotation = pp.Group(Number + pp.Optional(Dot + DecimalNumber) \
                              + Exponential + pp.Optional(Minus) + Number)
    # [455]
    FloatDotNotation = pp.Group(Number + Dot + DecimalNumber)
    # [454]
    FloatValue = (FloatDotNotation ^ FloatENotation)
    # [448]
    CharStringValue = Cstring
    # [446]
    VerdictTypeValue = pp.oneOf("pass fail inconc none error")
    # [445]
    HexStringValue = Hstring
    # [438]
    OctetStringValue = Ostring
    # [437]
    IntegerValue = Number
    # [436]
    BooleanValue = pp.oneOf("true false")
    # [435]
    BitStringValue = Bstring
    # [434]
    PredefinedValue = (BitStringValue ^ BooleanValue ^ CharStringValue \
                       ^ IntegerValue ^ OctetStringValue ^ HexStringValue \
                       ^ VerdictTypeValue ^ FloatValue \
                       ^ AddressValue ^ OmitValue)
    # [433]
    Value << (PredefinedValue ^ ReferencedValue)
    # [432]
    ArrayBounds = SingleConstExpression
    # [431]
    ArrayDef = pp.Group(pp.Literal("[") + ArrayBounds \
                        + pp.Optional(pp.Literal("..") + ArrayBounds) \
                        + pp.Literal("]"))
    # [430]
    TypeActualPar = ConstantExpression
    # [429]
    TypeActualParList = pp.Group(pp.Literal("(") + TypeActualPar \
                                 + pp.ZeroOrMore(pp.Literal(",") \
                                                 + TypeActualPar) \
                                 + pp.Literal(")"))
    # [428]
    TypeReference = (pp.Group(StructTypeIdentifier \
                              + pp.Optional(TypeActualParList) \
                              ^ EnumTypeIdentifier \
                              ^ SubTypeIdentifier \
                              ^ ComponentTypeIdentifier))
    # [427]
    ReferencedType = pp.Group(TypeReference \
                              + pp.Optional(ExtendedFieldReference))
    # [409]
    Type << (PredefinedType ^ ReferencedType)
    # [289]
    TimerRef = TimerIdentifier
    # [407]
    TimerRefOrAny = (TimerRef ^ pp.Group(AnyKeyword + TimerKeyword))
    # [406]
    TimeoutStatement = pp.Group(TimerRefOrAny + Dot + TimeoutKeyword)
    # [402]
    TimerRefOrAll = (TimerRef ^ pp.Group(AllKeyword + TimerKeyword))
    # [401]
    StopTimerStatement = pp.Group(TimerRefOrAll + Dot + StopKeyword)
    # [288]
    TimerValue = Expression
    # [400]
    StartTimerStatement = pp.Group(TimerRef + Dot + StartKeyword \
                                   + pp.Optional(pp.Literal("(") \
                                                 + TimerValue \
                                                 + pp.Literal(")")))
    # [398]
    TimerStatements = (StartTimerStatement ^ StopTimerStatement \
                       ^ TimeoutStatement)
    # [395]
    PortStopOp = StopKeyword
    # [317]
    Port = pp.Group(PortIdentifier + pp.Optional(ArrayOrBitRef))
    # [389]
    PortOrAll = (pp.Group(AllKeyword + PortKeyword) | Port)
    # [394]
    StopStatement = pp.Group(PortOrAll + Dot + PortStopOp)
    # [393]
    PortStartOp = StartKeyword
    # [392]
    StartStatement = pp.Group(PortOrAll + Dot + PortStartOp)
    # [390]
    PortClearOp = ClearOpKeyword
    # [388]
    ClearStatement = pp.Group(PortOrAll + Dot + PortClearOp)
    # [179]
    Signature = SignatureIdentifier
    # [387]
    TemplateInstance = pp.Forward()
    CatchOpParameter = (pp.Group(Signature + pp.Literal(",") \
                                 + TemplateInstance) \
                        ^ TimeoutKeyword)
    # [325]
    AddressRef = (VariableRef ^ FunctionInstance)
    # [349]
    FromClause = pp.Group(FromKeyword + AddressRef)
    # [385]
    PortRedirect = pp.Forward()
    PortCatchOp = pp.Group(CatchOpKeyword + pp.Optional(pp.Literal("(") \
                                                        + CatchOpParameter \
                                                        + pp.Literal(")")) \
                           + pp.Optional(FromClause) \
                           + pp.Optional(PortRedirect))
    # [345]
    PortOrAny = (Port ^ pp.Group(AnyKeyword + PortKeyword))
    # [384]
    CatchStatement = pp.Group(PortOrAny + Dot + PortCatchOp)
    # [352]
    PortRedirectSymbol = pp.Literal("->")
    # [355]
    SenderSpec = pp.Group(SenderKeyword + VariableRef)
    # [382]
    RedirectPresent = pp.Group(PortRedirectSymbol + SenderSpec)
    # [381]
    FromClausePresent = pp.Group(FromClause + pp.Optional(PortRedirectSymbol \
                                                          + SenderSpec))
    # [380]
    CheckPortOpsPresent = pp.Forward()
    CheckParameter = (CheckPortOpsPresent ^ FromClausePresent \
                      ^ RedirectPresent)
    # [378]
    PortCheckOp = pp.Group(CheckOpKeyword + pp.Optional(pp.Literal("(") \
                                                        + CheckParameter \
                                                        + pp.Literal(")")))
    # [377]
    CheckStatement = pp.Group(PortOrAny + Dot + PortCheckOp)
    # [376]
    ValueMatchSpec = pp.Group(ValueKeyword + TemplateInstance)
    # [348]
    ReceiveParameter = TemplateInstance
    # [374]
    PortRedirectWithParam = pp.Forward()
    PortGetReplyOp = pp.Group(GetReplyOpKeyword \
                              + pp.Optional(pp.Literal("(") \
                                            + ReceiveParameter \
                                            + pp.Optional(ValueMatchSpec) \
                                            + pp.Literal(")")) \
                              + pp.Optional(FromClause) \
                              + pp.Optional(PortRedirectWithParam))
    # [373]
    GetReplyStatement = pp.Group(PortOrAny + Dot + PortGetReplyOp)
    # [372]
    VariableEntry = (VariableRef ^ NotUsedSymbol)
    # [371]
    VariableList = pp.Group(VariableEntry + pp.ZeroOrMore(pp.Literal(",") \
                                                          + VariableEntry))
    # [370]
    ParameterIdentifier = (ValueParIdentifier ^ TimerParIdentifier \
                           ^ TemplateParIdentifier ^ PortParIdentifier)
    # [369]
    VariableAssignment = pp.Group(VariableRef + AssignmentChar \
                                  + ParameterIdentifier)
    # [368]
    AssignmentList = pp.Group(VariableAssignment \
                              + pp.ZeroOrMore(pp.Literal(",") \
                                              + VariableAssignment))
    # [367]
    ParaAssignmentList = pp.Group(pp.Literal("(") + pp.Group(AssignmentList \
                                                             ^ VariableList) \
                                  + pp.Literal(")"))
    # [365]
    ParaSpec = pp.Group(ParaKeyword + ParaAssignmentList)
    # [353]
    ValueSpec = pp.Group(ValueKeyword + VariableRef)
    # [364]
    RedirectSpec = (pp.Group(ValueSpec + pp.Optional(ParaSpec) \
                             + pp.Optional(SenderSpec)) \
                    ^ pp.Group(ParaSpec + pp.Optional(SenderSpec)) \
                    ^ SenderSpec)
    # [363]
    PortRedirectWithParam << pp.Group(PortRedirectSymbol + RedirectSpec)
    # [361]
    PortGetCallOp = pp.Group(GetCallOpKeyword \
                             + pp.Optional(pp.Literal("(") \
                                           + ReceiveParameter \
                                           + pp.Literal(")")) \
                             + pp.Optional(FromClause) \
                             + pp.Optional(PortRedirectWithParam))
    # [360]
    GetCallStatement = pp.Group(PortOrAny + Dot + PortGetCallOp)
    # [358]
    PortTriggerOp = pp.Group(TriggerOpKeyword \
                             + pp.Optional(pp.Literal("(") \
                                           + ReceiveParameter \
                                           + pp.Literal(")")) \
                             + pp.Optional(FromClause) \
                             + pp.Optional(PortRedirect))
    # [357]
    TriggerStatement = pp.Group(PortOrAny + Dot + PortTriggerOp)
    # [351]
    PortRedirect << pp.Group(PortRedirectSymbol \
                             + pp.Group(pp.Group(ValueSpec \
                                                 + pp.Optional(SenderSpec)) \
                                        ^ SenderSpec))
    # [346]
    PortReceiveOp = pp.Group(ReceiveOpKeyword \
                             + pp.Optional(pp.Literal("(") \
                                           + ReceiveParameter \
                                           + pp.Literal(")")) \
                             + pp.Optional(FromClause) \
                             + pp.Optional(PortRedirect))
    # [383]
    CheckPortOpsPresent << (PortReceiveOp ^ PortGetCallOp ^ PortGetReplyOp \
                            ^ PortCatchOp)
    # [344]
    ReceiveStatement = pp.Group(PortOrAny + Dot + PortReceiveOp)
    # [323]
    ToClause = pp.Group(ToKeyword + AddressRef)
    # [342]
    PortRaiseOp = pp.Group(RaiseKeyword + pp.Literal("(") + Signature \
                           + pp.Literal(",") + TemplateInstance \
                           + pp.Literal(")") + pp.Optional(ToClause))
    # [341]
    RaiseStatement = pp.Group(Port + Dot + PortRaiseOp)
    # [340]
    ReplyValue = pp.Group(ValueKeyword + Expression)
    # [338]
    PortReplyOp = pp.Group(ReplyKeyword + pp.Literal("(") + TemplateInstance \
                           + pp.Optional(ReplyValue) + pp.Literal(")") \
                           + pp.Optional(ToClause))
    # [337]
    ReplyStatement = pp.Group(Port + Dot + PortReplyOp)
    # [336]
    CallBodyOps = (GetReplyStatement ^ CatchStatement)
    # [335]
    CallBodyGuard = pp.Group(AltGuardChar + CallBodyOps)
    # [334]
    CallBodyStatement = pp.Group(CallBodyGuard + StatementBlock)
    # [333]
    CallBodyStatementList = pp.OneOrMore(CallBodyStatement \
                                         + pp.Optional(SemiColon))
    # [332]
    PortCallBody = pp.Group(BeginChar + CallBodyStatementList + EndChar)
    # [330]
    CallTimerValue = (NowaitKeyword | TimerValue)
    # [329]
    CallParameters = pp.Group(TemplateInstance \
                              + pp.Optional(pp.Literal(",") \
                                            + CallTimerValue))
    # [327]
    PortCallOp = pp.Group(CallOpKeyword + pp.Literal("(") + CallParameters \
                          + pp.Literal(")") + pp.Optional(ToClause))
    # [326]
    CallStatement = pp.Group(Port + Dot + PortCallOp \
                             + pp.Optional(PortCallBody))
    # [322]
    SendParameter = TemplateInstance
    # [320]
    PortSendOp = pp.Group(SendOpKeyword + pp.Literal("(") \
                          + SendParameter + pp.Literal(")") \
                          + pp.Optional(ToClause))
    # [319]
    SendStatement = pp.Group(Port + Dot + PortSendOp)
    # [318]
    CommunicationStatements = (SendStatement ^ CallStatement \
                               ^ ReplyStatement ^ RaiseStatement \
                               ^ ReceiveStatement ^ TriggerStatement \
                               ^ GetCallStatement ^ GetReplyStatement \
                               ^ CatchStatement ^ CheckStatement \
                               ^ ClearStatement ^ StartStatement \
                               ^ StopStatement)
    # [316]
    ComponentIdentifier = VariableRef
    # [315]
    StopTCStatement = StopKeyword
    # [313]
    StartTCStatement = pp.Group(ComponentIdentifier + Dot + StartKeyword \
                                + pp.Literal("(") + FunctionInstance \
                                + pp.Literal(")"))
    # [295]
    MTCOp = MTCKeyword
    # [306]
    ComponentRef = (ComponentIdentifier ^ SystemOp ^ SelfOp ^ MTCOp)
    # [305]
    PortRef = pp.Group(ComponentRef + Colon + Port)
    # [304]
    PortSpec = (pp.Literal("(") + PortRef + pp.Literal(",") + PortRef \
                + pp.Literal(")"))
    # [309]
    MapStatement = pp.Group(MapKeyword + PortSpec)
    # [307]
    DisconnectStatement = pp.Group(DisconnectKeyword + PortSpec)
    # [302]
    ConnectStatement = pp.Group(ConnectKeyword + PortSpec)
    # [297]
    ComponentId = (ComponentIdentifier \
                   ^ pp.Group((AnyKeyword ^ AllKeyword) + ComponentKeyword))
    # [296]
    DoneStatement = pp.Group(ComponentId + Dot + DoneKeyword)
    # [299]
    RunningOp << pp.Group(ComponentId + Dot + RunningKeyword)
    # [525]
    GuardOp << (TimeoutStatement ^ ReceiveStatement ^ CatchStatement \
                ^ CheckStatement ^ GetReplyStatement ^ DoneStatement)
    # [311]
    UnmapStatement = pp.Group(UnmapKeyword + PortSpec)
    # [290]
    _KeywordConfigurationStatements = (ConnectStatement | MapStatement \
                                       | DisconnectStatement \
                                       | UnmapStatement | StopTCStatement)
    ConfigurationStatements = (_KeywordConfigurationStatements \
                               ^ DoneStatement ^ StartTCStatement)
    # [285]
    SingleTimerInstance = pp.Group(TimerIdentifier \
                                   + pp.Optional(AssignmentChar \
                                                 + TimerValue))
    # [284]
    TimerList = pp.Group(SingleTimerInstance \
                         + pp.ZeroOrMore(pp.Literal(",") \
                                         + SingleTimerInstance))
    # [283]
    TimerInstance = pp.Group(TimerKeyword + TimerList)
    # [279]
    VarInitialValue = Expression
    # [278]
    SingleVarInstance = pp.Group(VarIdentifier + pp.Optional(ArrayDef) \
                                 + pp.Optional(AssignmentChar \
                                               + VarInitialValue))
    # [277]
    VarList = pp.Group(SingleVarInstance + pp.ZeroOrMore(pp.Literal(",") \
                                                         + SingleVarInstance))
    # [276]
    VarInstance << pp.Group(VarKeyword + Type + VarList)
    # [275]
    ControlStatement = (TimerStatements ^ BasicStatements \
                        ^ BehaviourStatements)
    # [162]
    FunctionLocalInst = (VarInstance ^ TimerInstance)
    # [274]
    ControlStatementOrDef = (FunctionLocalInst ^ ControlStatement)
    # [273]
    ControlStatementOrDefList = pp.OneOrMore(ControlStatementOrDef \
                                             + pp.Optional(SemiColon))
    # [272]
    ModuleControlBody = pp.Optional(ControlStatementOrDefList)
    # [270]
    ModuleControlPart = pp.Group(ControlKeyword + BeginChar \
                                 + ModuleControlBody + EndChar \
                                 + pp.Optional(WithStatement) \
                                 + pp.Optional(SemiColon))
    # [268]
    ModuleParType = Type
    # [267]
    ModulePar = pp.Group(ModuleParType + ModuleParIdentifier \
                         + pp.Optional(AssignmentChar + ConstantExpression) \
                         + pp.ZeroOrMore(pp.Literal(",") \
                                         + ModuleParIdentifier \
                                         + pp.Optional(AssignmentChar \
                                                       + ConstantExpression)))
    # [266]
    ModuleParList = pp.Group(ModulePar \
                             + pp.ZeroOrMore(SemiColon + ModulePar)) \
                    + pp.Optional(SemiColon)
    # [264]
    ModuleParDef = pp.Group(ModuleParKeyword + BeginChar + ModuleParList \
                            + EndChar)
    # [262]
    ExtConstDef = pp.Group(ExtKeyword + ConstKeyword + Type \
                           + ExtConstIdentifier)
    # [153]
    ReturnType = pp.Group(ReturnKeyword + Type)
    # [259]
    FunctionFormalParList = pp.Forward()
    ExtFunctionDef = pp.Group(ExtKeyword + FunctionKeyword \
                              + ExtFunctionIdentifier + pp.Literal("(") \
                              + pp.Optional(FunctionFormalParList) \
                              + pp.Literal(")") + pp.Optional(ReturnType))
    # [256]
    ModuleDefinitionsPart = pp.Forward()
    GroupDef = pp.Group(GroupKeyword + GroupIdentifier + BeginChar \
                        + pp.ZeroOrMore(ModuleDefinitionsPart) + EndChar)
    # [248]
    FunctionRefList = pp.Group(FunctionIdentifier \
                               + pp.ZeroOrMore(pp.Literal(",") \
                                               + FunctionIdentifier))
    # [249]
    AllFunctionsWithExcept = pp.Group(AllKeyword \
                                      + pp.Optional(ExceptKeyword \
                                                    + FunctionRefList))
    # [247]
    ImportFunctionSpec = pp.Group(FunctionKeyword \
                                  + (FunctionRefList ^ AllFunctionsWithExcept))
    # [245]
    TestcaseRefList = pp.Group(TestcaseIdentifier \
                               + pp.ZeroOrMore(pp.Literal(",") \
                                               + TestcaseIdentifier))
    # [246]
    AllTestcasesWithExcept = pp.Group(AllKeyword \
                                      + pp.Optional(ExceptKeyword \
                                                    + TestcaseRefList))
    # [244]
    ImportTestcaseSpec = pp.Group(TestcaseKeyword \
                                  + (TestcaseRefList ^ AllTestcasesWithExcept))
    # [236]
    TemplateRefList = pp.Group(TemplateIdentifier \
                               + pp.ZeroOrMore(pp.Literal(",") \
                                               + TemplateIdentifier))
    # [237]
    AllTemplsWithExcept = pp.Group(AllKeyword + pp.Optional(ExceptKeyword \
                                                            + TemplateRefList))
    # [235]
    ImportTemplateSpec = pp.Group(TemplateKeyword \
                                  + (TemplateRefList ^ AllTemplsWithExcept))
    # [232]
    TypeRefList = pp.Group(TypeDefIdentifier \
                           + pp.ZeroOrMore(pp.Literal(",") \
                                           + TypeDefIdentifier))
    # [233]
    AllTypesWithExcept = pp.Group(AllKeyword + pp.Optional(ExceptKeyword \
                                                           + TypeRefList))
    # [231]
    ImportTypeDefSpec = pp.Group(TypeDefKeyword + (TypeRefList \
                                                   ^ AllTypesWithExcept))
    # [230]
    ExceptFullGroupIdentifier = pp.Group(GroupIdentifier \
                                         + pp.ZeroOrMore(Dot \
                                                         + GroupIdentifier))
    # [229]
    ExceptGroupRefList = pp.Group(
        ExceptFullGroupIdentifier \
        + pp.ZeroOrMore(pp.Literal(",") \
                        + ExceptFullGroupIdentifier))
    # [228]
    ExceptsDef = pp.Forward()
    FullGroupIdentifier = pp.Group(GroupIdentifier \
                                   + pp.ZeroOrMore(Dot + GroupIdentifier) \
                                   + pp.Optional(ExceptsDef))
    # [227]
    GroupRefList = pp.Forward()
    AllGroupsWithExcept = pp.Group(AllKeyword \
                                   + pp.Optional(ExceptKeyword \
                                                 + GroupRefList))
    # [226]
    GroupRefList << pp.Group(FullGroupIdentifier \
                             + pp.ZeroOrMore(pp.Literal(",") \
                                             + FullGroupIdentifier))
    # [225]
    ImportGroupSpec = pp.Group(GroupKeyword + pp.Group(GroupRefList \
                                                       ^ AllGroupsWithExcept))
    # [222]
    LanguageSpec = pp.Group(LanguageKeyword + FreeText)
    # [220]
    ModuleId = pp.Group(Identifier + pp.Optional(LanguageSpec))
    # [219]
    ImportFromSpec = pp.Group(FromKeyword + ModuleId \
                              + pp.Optional(RecursiveKeyword))
    # [218]
    ImportElement = (ImportGroupSpec ^ ImportTypeDefSpec ^ ImportTemplateSpec \
                     ^ ImportTestcaseSpec ^ ImportFunctionSpec)
    # [217]
    ImportSpec = pp.ZeroOrMore(ImportElement + pp.Optional(SemiColon))
    # [208]
    ExceptGroupSpec = pp.Group(GroupKeyword \
                               + pp.Group(ExceptGroupRefList ^ AllKeyword))
    # [207]
    ExceptElement = ExceptGroupSpec
    # [206]
    ExceptSpec = pp.ZeroOrMore(ExceptElement + pp.Optional(SemiColon))
    # [204]
    ExceptsDef << pp.Group(ExceptKeyword + BeginChar + ExceptSpec + EndChar)
    # [203]
    AllWithExcepts = pp.Group(AllKeyword + pp.Optional(ExceptsDef))
    # [201]
    ImportDef = pp.Group(ImportKeyword + ImportFromSpec \
                         + pp.Group(AllWithExcepts \
                                    ^ pp.Group(BeginChar + ImportSpec \
                                               + EndChar)))
    # [198]
    ConstDef = pp.Forward()
    AltstepLocalDef = (VarInstance ^ TimerInstance ^ ConstDef)
    # [197]
    AltstepLocalDefList = pp.ZeroOrMore(AltstepLocalDef \
                                        + pp.Optional(SemiColon))
    # [196]
    AltstepFormalParList = FunctionFormalParList
    # [155]
    RunsOnSpec = pp.Group(RunsKeyword + OnKeyword + ComponentType)
    # [193]
    AltstepDef = pp.Group(AltstepKeyword + AltstepIdentifier \
                          + pp.Literal("(") \
                          + pp.Optional(AltstepFormalParList) \
                          + pp.Literal(")") + pp.Optional(RunsOnSpec) \
                          + BeginChar + AltstepLocalDefList \
                          + AltGuardList + EndChar)
    # [192]
    TestcaseActualPar = TemplateInstance
    # [191]
    TestcaseActualParList = pp.Group(TestcaseActualPar \
                                     + pp.ZeroOrMore(pp.Literal(",") \
                                                     + TestcaseActualPar))
    # [190]
    TestcaseRef = TestcaseIdentifier
    # [188]
    TestcaseInstance << pp.Group(ExecuteKeyword + pp.Literal("(") \
                                 + TestcaseRef + pp.Literal("(") \
                                 + pp.Optional(TestcaseActualParList) \
                                 + pp.Literal(")") \
                                 + pp.Optional("," + TimerValue) \
                                 + pp.Literal(")"))
    # [186]
    SystemSpec = pp.Group(SystemKeyword + ComponentType)
    # [185]
    ConfigSpec = pp.Group(RunsOnSpec + pp.Optional(SystemSpec))
    # [184]
    TestcaseFormalPar = (FormalValuePar ^ FormalTemplatePar)
    # [183]
    TestcaseFormalParList = pp.Group(TestcaseFormalPar \
                                     + pp.ZeroOrMore(pp.Literal(",") \
                                                     + TestcaseFormalPar))
    # [180]
    TestcaseDef = pp.Group(TestcaseKeyword + TestcaseIdentifier \
                           + pp.Literal("(") \
                           + pp.Optional(TestcaseFormalParList) \
                           + pp.Literal(")") \
                           + ConfigSpec + StatementBlock)
    # [177]
    ExceptionTypeList = pp.Group(Type + pp.ZeroOrMore(pp.Literal(",") + Type))
    # [175]
    ExceptionSpec = pp.Group(ExceptionKeyword + pp.Literal("(") \
                             + ExceptionTypeList + pp.Literal(")"))
    # [174]
    SignatureFormalPar = FormalValuePar
    # [173]
    SignatureFormalParList = pp.Group(SignatureFormalPar \
                                      + pp.ZeroOrMore(pp.Literal(",") \
                                                      + SignatureFormalPar))
    # [170]
    SignatureDef = pp.Group(SignatureKeyword + SignatureIdentifier \
                            + pp.Literal("(") \
                            + pp.Optional(SignatureFormalParList) \
                            + pp.Literal(")") \
                            + pp.Optional(ReturnType ^ NoBlockKeyword) \
                            + pp.Optional(ExceptionSpec))
    # [169]
    FunctionActualPar = (TimerRef ^ TemplateInstance ^ Port ^ ComponentRef)
    # [168]
    FunctionActualParList << pp.Group(FunctionActualPar \
                                      + pp.ZeroOrMore(pp.Literal(",") \
                                                      + FunctionActualPar))
    # [164]
    FunctionStatement = (ConfigurationStatements ^ TimerStatements \
                         ^ CommunicationStatements \
                         ^ BasicStatements ^ BehaviourStatements \
                         ^ VerdictStatements ^ SUTStatements)
    # [163]
    FunctionLocalDef = ConstDef
    # [161]
    FunctionStatementOrDef = (FunctionLocalDef ^ FunctionLocalInst \
                              ^ FunctionStatement)
    # [160]
    FunctionStatementOrDefList = pp.OneOrMore(FunctionStatementOrDef \
                                              + pp.Optional(SemiColon))
    # [159]
    StatementBlock << pp.Group(BeginChar \
                              + pp.Optional(FunctionStatementOrDefList) \
                              + EndChar)
    # [152]
    FunctionFormalPar = (FormalValuePar ^ FormalTimerPar \
                         ^ FormalTemplatePar ^ FormalPortPar)
    # [151]
    FunctionFormalParList << pp.Group(FunctionFormalPar \
                                      + pp.ZeroOrMore(pp.Literal(",") \
                                                      + FunctionFormalPar))
    # [148]
    FunctionDef = pp.Group(FunctionKeyword + FunctionIdentifier \
                           + pp.Literal("(") \
                           + pp.Optional(FunctionFormalParList) \
                           + pp.Literal(")") \
                           + pp.Optional(RunsOnSpec) \
                           + pp.Optional(ReturnType) + StatementBlock)
    # [142]
    TemplateActualPar = TemplateInstance
    # [141]
    TemplateActualParList = pp.Group(pp.Literal("(") + TemplateActualPar \
                                     + pp.ZeroOrMore(pp.Literal(",") \
                                                     + TemplateActualPar) \
                                     + pp.Literal(")"))
    # [139]
    TemplateRef = (TemplateIdentifier ^ TemplateParIdentifier)
    # [91]
    DerivedDef = pp.Group(ModifiesKeyword + TemplateRef)
    # [140]
    TemplateBody = pp.Forward()
    InLineTemplate = pp.Group(pp.Optional(pp.Group((Type ^ Signature)) \
                                          + Colon) \
                              + pp.Optional(DerivedDef + AssignmentChar) \
                              + TemplateBody)
    # [138]
    TemplateRefWithParList << pp.Group(TemplateIdentifier \
                                       + pp.Optional(TemplateActualParList))
    # [137]
    TemplateInstance << InLineTemplate
    # [135]
    UpperBound = (SingleConstExpression ^ InfinityKeyword)
    # [134]
    LowerBound = (SingleConstExpression ^ pp.Group(Minus + InfinityKeyword))
    # [133]
    Range = pp.Group(pp.Literal("(") + LowerBound + pp.Literal("..") \
                     + UpperBound + pp.Literal(")"))
    # [131]
    IfPresentMatch = IfPresentKeyword
    # [48]
    StringLength = pp.Group(LengthKeyword + pp.Literal("(") \
                            + SingleConstExpression \
                            + pp.Optional(pp.Literal("..") + UpperBound) \
                            + pp.Literal(")"))
    # [130]
    LengthMatch = StringLength
    # [129]
    ValueOrAttribList = pp.Group(pp.Literal("(") + TemplateBody \
                                 + pp.OneOrMore(pp.Literal(",") \
                                                + TemplateBody) \
                                 + pp.Literal(")"))
    # [128]
    AnyOrOmit = pp.Literal("*")
    # [127]
    AnyValue = pp.Literal("?")
    # [122]
    ValueList = pp.Group(pp.Literal("(") + ConstantExpression \
                         + pp.ZeroOrMore(pp.Literal(",") \
                                         + ConstantExpression) \
                         + pp.Literal(")"))
    # [125]
    SupersetMatch = pp.Group(SupersetKeyword + ValueList)
    # [123]
    SubsetMatch = pp.Group(SubsetKeyword + ValueList)
    # [120]
    Complement = pp.Group(ComplementKeyword + ValueList)
    # [118]
    CharStringMatch = pp.Group(PatternKeyword + Cstring)
    # [117]
    #OctOrMatch = (Oct | AnyValue | AnyOrOmit)
    # [116]
    OctetStringMatch = pp.Group(
        pp.Literal("'") \
        + pp.Optional(pp.Word("0123456789abcdefABCDEF?*")) \
        + pp.Literal("'O"))
    OctetStringMatch.setParseAction(checkOctPair)
    # [115]
    #HexOrMatch = (Hex | AnyValue | AnyOrOmit)
    # [114]
    HexStringMatch = pp.Group(
        pp.Literal("'") \
        + pp.Optional(pp.Word("0123456789abcdefABCDEF?*")) \
        + pp.Literal("'H"))
    # [113]
    #BinOrMatch = (Bin | AnyValue | AnyOrOmit)
    # [112]
    BitStringMatch = pp.Group(pp.Literal("'") \
                              + pp.Optional(pp.Word("01?*")) \
                              + pp.Literal("'B"))
    # [111]
    ExtraMatchingAttributes = (LengthMatch ^ IfPresentMatch \
                               ^ pp.Group(LengthMatch + IfPresentMatch))
    # [110]
    MatchingSymbol = (Complement ^ AnyValue ^ AnyOrOmit ^ ValueOrAttribList \
                      ^ Range ^ BitStringMatch ^ HexStringMatch \
                      ^ OctetStringMatch ^ CharStringMatch ^ SubsetMatch \
                      ^ SupersetMatch)
    # [108]
    ArrayElementSpec = (TemplateBody | NotUsedSymbol)
    # [107]
    ArrayElementSpecList = pp.Group(ArrayElementSpec \
                                    + pp.ZeroOrMore(pp.Literal(",") \
                                                    + ArrayElementSpec))
    # [106]
    ArrayValueOrAttrib = pp.Group(pp.Literal("{") + ArrayElementSpecList \
                                  + pp.Literal("}"))
    # [105]
    SingleValueOrAttrib = (pp.Group(MatchingSymbol \
                                    + pp.Optional(ExtraMatchingAttributes)) \
                           ^ pp.Group(SingleExpression \
                                      + pp.Optional(
        ExtraMatchingAttributes)) \
                           ^ TemplateRefWithParList)
    # [102]
    SignatureParIdentifier = ValueParIdentifier
    # [101]
    ParRef = SignatureParIdentifier
    # [100]
    StructFieldRef = StructFieldIdentifier
    # [99]
    FieldReference << (StructFieldRef ^ ArrayOrBitRef ^ ParRef)
    # [98]
    FieldSpec = pp.Group(FieldReference + AssignmentChar \
                         + TemplateBody)
    # [97]
    FieldSpecList = pp.Group(pp.Literal("{") \
                             + pp.Optional(FieldSpec \
                                           + pp.ZeroOrMore("," \
                                                           + FieldSpec)) \
                             + pp.Literal("}"))
    # [96]
    SimpleSpec = SingleValueOrAttrib
    # [95]
    TemplateBody << (SimpleSpec ^ FieldSpecList ^ ArrayValueOrAttrib)
    # [94]
    TemplateFormalPar = (FormalValuePar ^ FormalTemplatePar)
    # [93]
    TemplateFormalParList = pp.Group(TemplateFormalPar \
                                     + pp.ZeroOrMore(pp.Literal(",") \
                                                     + TemplateFormalPar))
    # [88]
    BaseTemplate = pp.Group(Type + TemplateIdentifier \
                            + pp.Optional(pp.Literal("(") \
                                          + TemplateFormalParList \
                                          + pp.Literal(")")))
    # [87]
    TemplateDef = pp.Group(TemplateKeyword + BaseTemplate \
                           + pp.Optional(DerivedDef) + AssignmentChar \
                           + TemplateBody)
    # [84]
    SingleConstDef = pp.Group(ConstIdentifier + pp.Optional(ArrayDef) \
                              + AssignmentChar + ConstantExpression)
    # [83]
    ConstList = pp.Group(SingleConstDef \
                         + pp.ZeroOrMore(pp.Literal(",") + SingleConstDef))
    # [82]
    ConstDef << pp.Group(ConstKeyword + Type + ConstList)
    # [80]
    PortElement = pp.Group(PortIdentifier + pp.Optional(ArrayDef))
    # [50]
    PortType = PortTypeIdentifier
    # [79]
    PortInstance = pp.Group(PortKeyword + PortType + PortElement \
                            + pp.ZeroOrMore(pp.Literal(",") + PortElement))
    # [78]
    ComponentElementDef = (PortInstance ^ VarInstance ^ TimerInstance \
                           ^ ConstDef)
    # [77]
    ComponentDefList = pp.ZeroOrMore(ComponentElementDef \
                                     + pp.Optional(SemiColon))
    # [73]
    ComponentDef = pp.Group(ComponentKeyword + ComponentTypeIdentifier \
                            + BeginChar + pp.Optional(ComponentDefList) \
                            + EndChar)
    # [72]
    ProcOrType = (Signature ^ Type)
    # [71]
    ProcOrTypeList = (AllKeyword | pp.Group(ProcOrType \
                                            + pp.ZeroOrMore(pp.Literal(",") \
                                                            + ProcOrType)))
    # [58]
    Direction = pp.oneOf("in out inout")
    # [70]
    MixedList = pp.Group(Direction + ProcOrTypeList)
    # [68]
    MixedAttribs = pp.Group(MixedKeyword + BeginChar \
                            + pp.OneOrMore(MixedList \
                                           + pp.Optional(SemiColon)) \
                            + EndChar)
    # [67]
    SignatureList = pp.Group(Signature + pp.ZeroOrMore(pp.Literal(",") \
                                                       + Signature))
    # [66]
    AllOrSignatureList = (AllKeyword | SignatureList)
    # [65]
    ProcedureList = pp.Group(Direction + AllOrSignatureList)
    # [63]
    ProcedureAttribs = pp.Group(ProcedureKeyword + BeginChar \
                                + pp.OneOrMore(ProcedureList \
                                               + pp.Optional(SemiColon)) \
                                + EndChar)
    # [62]
    TypeList = pp.Group(Type + pp.ZeroOrMore(pp.Literal(",") + Type))
    # [60]
    AllOrTypeList = (AllKeyword | TypeList)
    # [57]
    MessageList = pp.Group(Direction + AllOrTypeList)
    # [56]
    MessageAttribs = pp.Group(MessageKeyword + BeginChar \
                              + pp.OneOrMore(MessageList \
                                             + pp.Optional(SemiColon)) \
                              + EndChar)
    # [55]
    PortDefAttribs = (MessageAttribs ^ ProcedureAttribs ^ MixedAttribs)
    # [52]
    PortDefBody = pp.Group(PortTypeIdentifier + PortDefAttribs)
    # [51]
    PortDef = pp.Group(PortKeyword + PortDefBody)
    # [47]
    RangeDef = pp.Group(LowerBound + pp.Literal("..") + UpperBound)
    # [46]
    ValueOrRange = (RangeDef ^ SingleConstExpression)
    # [45]
    AllowedValues = pp.Group(pp.Literal("(") + ValueOrRange \
                             + pp.ZeroOrMore(pp.Literal(",") \
                                             + ValueOrRange) \
                             + pp.Literal(")"))
    # [44]
    SubTypeSpec = (AllowedValues ^ StringLength)
    # [42]
    SubTypeDef = pp.Group(Type + pp.Group(AddressKeyword \
                                          | SubTypeIdentifier) \
                          + pp.Optional(ArrayDef) + pp.Optional(SubTypeSpec))
    # [40]
    Enumeration = pp.Group(EnumerationIdentifier \
                           + pp.Optional(pp.Literal("(") + Number \
                                         + pp.Literal(")")))
    # [39]
    EnumerationList = pp.Group(Enumeration \
                               + pp.ZeroOrMore(pp.Literal(",") + Enumeration))
    # [36]
    EnumDef = pp.Group(EnumKeyword + pp.Group(AddressKeyword \
                                              | EnumTypeIdentifier) \
                       + BeginChar + EnumerationList + EndChar)
    # [34]
    StructOfDefBody = pp.Group(Type + (AddressKeyword \
                                       | StructTypeIdentifier) \
                               + pp.Optional(SubTypeSpec))
    # [35]
    SetOfDef = pp.Group(SetKeyword + pp.Optional(StringLength) + OfKeyword \
                        + StructOfDefBody)
    # [32]
    RecordOfDef = pp.Group(RecordKeyword + pp.Optional(StringLength) \
                           + OfKeyword + StructOfDefBody)
    # [29]
    UnionFieldDef = pp.Group(Type + StructFieldIdentifier \
                             + pp.Optional(ArrayDef) \
                             + pp.Optional(SubTypeSpec))
    # [23]
    StructFieldDef = pp.Group(Type + StructFieldIdentifier \
                              + pp.Optional(ArrayDef) \
                              + pp.Optional(SubTypeSpec) \
                              + pp.Optional(OptionalKeyword))
    # [22]
    StructDefFormalPar = FormalValuePar
    # [21]
    StructDefFormalParList = pp.Group(pp.Literal("(") + StructDefFormalPar \
                                      + pp.ZeroOrMore(pp.Literal(",") \
                                                      + StructDefFormalPar) \
                                      + pp.Literal(")"))
    # [28]
    UnionDefBody = pp.Group(
        (AddressKeyword \
         | pp.Group(StructTypeIdentifier \
                    + pp.Optional(StructDefFormalParList))) \
        + BeginChar + UnionFieldDef \
        + pp.ZeroOrMore(pp.Literal(",") + UnionFieldDef) \
        + EndChar)
    # [26]
    UnionDef = pp.Group(UnionKeyword + UnionDefBody)
    # [19]
    StructDefBody = pp.Group(
        (AddressKeyword \
         | pp.Group(StructTypeIdentifier + \
                    pp.Optional(StructDefFormalParList))) \
        + BeginChar + pp.Optional(StructFieldDef \
                                  + pp.ZeroOrMore(pp.Literal(",") \
                                                  + StructFieldDef)) \
        + EndChar)
    # [30]
    SetDef = pp.Group(SetKeyword + StructDefBody)
    # [17]
    RecordDef = pp.Group(RecordKeyword + StructDefBody)
    # [16]
    _SetDef = (SetDef ^ SetOfDef)
    _RecordDef = (RecordDef ^ RecordOfDef)
    StructuredTypeDef = (_RecordDef | UnionDef | _SetDef \
                         | EnumDef | PortDef | ComponentDef)
    # [14]
    TypeDefBody = (StructuredTypeDef | SubTypeDef)
    # [13]
    TypeDef = pp.Group(TypeDefKeyword + TypeDefBody)
    # [12]
    _ExtDef = (ExtFunctionDef ^ ExtConstDef)
    ModuleDefinition = (TypeDef | ConstDef | TemplateDef | ModuleParDef \
                        | FunctionDef | SignatureDef | TestcaseDef \
                        | AltstepDef | ImportDef | GroupDef | _ExtDef) \
                        + pp.Optional(WithStatement)
    # [11]
    ModuleDefinitionsList = pp.OneOrMore(ModuleDefinition \
                                         + pp.Optional(SemiColon))
    # [10]
    ModuleDefinitionsPart << ModuleDefinitionsList
    # [8]
    DefinitiveNumberForm = Number
    # [9]
    DefinitiveNameAndNumberForm = pp.Group(Identifier + pp.Literal("(") \
                                           + DefinitiveNumberForm \
                                           + pp.Literal(")"))
    # [7]
    DefinitiveObjIdComponent = (NameForm ^ DefinitiveNumberForm \
                                ^ DefinitiveNameAndNumberForm)
    # [6]
    DefinitiveObjIdComponentList = pp.OneOrMore(DefinitiveObjIdComponent)
    # [5]
    DefinitiveIdentifier = pp.Group(Dot + ObjectIdentifierKeyword \
                                    + pp.Literal("{") \
                                    + DefinitiveObjIdComponentList \
                                    + pp.Literal("}"))
    # [3]
    TTCN3ModuleId = pp.Group(ModuleIdentifier \
                             + pp.Optional(DefinitiveIdentifier))
    # [1]
    TTCN3Module = pp.Group(TTCN3ModuleKeyword + TTCN3ModuleId + BeginChar \
                           + pp.Optional(ModuleDefinitionsPart) \
                           + pp.Optional(ModuleControlPart) + EndChar \
                           + pp.Optional(WithStatement))
    bnf = pp.Group(pp.OneOrMore(TTCN3Module) + pp.StringEnd())
    singleLineComment = "//" + pp.restOfLine
    bnf.ignore(singleLineComment)
    bnf.ignore(pp.cStyleComment)
    return bnf
