/*
 *          TTCN-3 Parser
 *
 * Copyright (C) 2001 Institute for Telematics
 *
 *    Michael Schmitt <schmitt@itm.mu-luebeck.de>
 *    Roman Koch <rkoch@itm.mu-luebeck.de>
 *
 *    Medical University of Luebeck,
 *    Ratzeburger Allee 160,
 *    23538 Luebeck,
 *    Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: TTCNMain.cpp,v 1.20 2001/08/05 16:44:23 schmitt Exp $
 *
 */

#include <vector>
#include <fstream>
#include "antlr/AST.hpp"
#include "antlr/ANTLRException.hpp"
#include "ScopeAST.hpp"
#include "TTCNLexer.hpp"
#include "TTCNParser.hpp"
#include "TTCNTreeParser.hpp"

using std::istream;
using std::ifstream;
using std::cout;
using std::cin;
using std::cerr;
using std::endl;
using std::vector;
using std::string;
using antlr::RefAST;
using antlr::InfoAST;
using antlr::ScopeAST;
using antlr::EnvAST;


void printAST( RefAST node, const vector<string> &tokenNames,
               bool flatten, bool symTabs, int level = 0 )
{
  RefAST          node2;
  const InfoAST   *infoAST;
  const ScopeAST  *scopeAST;
  const EnvAST    *envAST;

  for ( node2 = node; node2 != 0; node2 = node2 -> getNextSibling() )
  {
    if ( node2 -> getFirstChild() != antlr::nullAST )
    {
      flatten = false;
      break;
    }
  }

  for ( node2 = node; node2 != 0; node2 = node2 -> getNextSibling() )
  {
    if ( !flatten || node2 == node )
    {
      for ( int i = 0; i < level; i++ )
        cout << "  ";
    }

    if ( node2 -> getText() != "" )
      cout << "'" << node2 -> getText() << "' ("
           << tokenNames[ node2 -> getType() ] << ")";
    else
      cout << tokenNames[ node2 -> getType() ];

    if ( ( infoAST = dynamic_cast< const InfoAST * >( node2.get() ) ) != 0 )
    {
      if ( infoAST -> getLine() != 0 )
        cout << " (" << infoAST -> getLine() << "," << infoAST -> getColumn() << ")";

      if ( ( scopeAST = dynamic_cast< const ScopeAST * >( node2.get() ) ) != 0 )
        cout << " is ScopeAST ";
      else
        if ( ( envAST = dynamic_cast< const EnvAST * >( node2.get() ) ) != 0 )
          cout << " is EnvAST ";
    }
    else
      scopeAST = 0;

    if ( flatten )
      cout << " ";
    else
      cout << endl;

    if ( scopeAST && symTabs )
    {
      if ( !flatten || node2 == node )
      {
        for ( int i = 0; i <= level; i++ )
          cout << "  ";
      }
      cout << "BEGIN SYMBOL TABLE";
      if ( flatten )
        cout << " ";
      else
        cout << endl;

      printAST( RefAST( scopeAST -> getEnv() ), tokenNames, flatten, symTabs, level + 2 );

      if ( !flatten )
      {
        for ( int i = 0; i <= level; i++ )
          cout << "  ";
        cout << "END SYMBOL TABLE" << endl;
      }
    }

    if ( node2 -> getFirstChild() != antlr::nullAST )
      printAST( node2 -> getFirstChild(), tokenNames, flatten, symTabs, level + 1 );
  }

  if ( flatten )
    cout << endl;
}


struct
{
  bool            isHelp;
  bool            isVersion;
  bool            isReadStdin;
  bool            isPrintTree;
  bool            isPrintSymbolTables;
  bool            isNoFlattening;
  bool            isPrettyPrint;
  vector< int >   fileNameArgs;
} options = { false, false, false, false, false, false, false, vector< int >() };


int main( int argc, char *argv[] )
{
  for ( int i = 1; i < argc; i++ )
  {
    if ( strcmp( "--help", argv[ i ] ) == 0 || strcmp( "-h", argv[ i ] ) == 0 )
    {
      options.isHelp = true;
      continue;
    }

    if ( strcmp( "--version", argv[ i ] ) == 0 || strcmp( "-v", argv[ i ] ) == 0 )
    {
      options.isVersion = true;
      continue;
    }

    if ( strcmp( "--read-stdin", argv[ i ] ) == 0 || strcmp( "-s", argv[ i ] ) == 0 )
    {
      options.isReadStdin = true;
      continue;
    }

    if ( strcmp( "--print-tree", argv[ i ] ) == 0 || strcmp( "-t", argv[ i ] ) == 0 )
    {
      options.isPrintTree = true;
      continue;
    }

    if ( strcmp( "--print-symbol-tables", argv[ i ] ) == 0 || strcmp( "-T", argv[ i ] ) == 0 )
    {
      options.isPrintTree = true;
      options.isPrintSymbolTables = true;
      continue;
    }

    if ( strcmp( "--no-flattening", argv[ i ] ) == 0 || strcmp( "-f", argv[ i ] ) == 0 )
    {
      options.isNoFlattening = true;
      continue;
    }

    if ( strcmp( "--pretty-print", argv[ i ] ) == 0 || strcmp( "-p", argv[ i ] ) == 0 )
    {
      options.isPrettyPrint = true;
      continue;
    }

    if ( strncmp( "--", argv[ i ], 2 ) == 0 )
    {
      cerr << endl
           << "Error: Invalid command line option '" << argv[ i ] << "'." << endl
           << endl;
      exit( 1 );
    }

    options.fileNameArgs.push_back( i );
  }

  cout << "TTCN-3 Parser, compiled on " << __DATE__ << endl
       << endl
       << "Copyright (C) 2001 Institute for Telematics, University of Luebeck" << endl
       << "                   M. Schmitt, J. Grabowski, A. Hansen, R. Koch" << endl
       << "                   {schmitt,grabowsk,rkoch,hansen}@itm.mu-luebeck.de" << endl
       << endl;

  if ( options.isVersion || options.isHelp || ( options.fileNameArgs.empty() && !options.isReadStdin ) )
  {
    cout << "The TTCN-3 Parser comes with ABSOLUTELY NO WARRANTY; for details see the" << endl
         << "GNU General Public License (GPL). This is free software, and you are welcome" << endl
         << "to redistribute it under the conditions specified in the GNU GPL." << endl
         << endl;
  }

  if ( options.isHelp )
  {
    cout << "Usage:" << endl
         << endl
         << "  TTCN3Parser [ options ] [ file ... ]" << endl
         << endl
         << "Options:" << endl
         << endl
         << "  -h, --help                display this overview and exit" << endl
         << "  -v, --version             display program version and exit" << endl
         << "  -s, --read-stdin          read from standard input" << endl
         << "  -t, --print-tree          print abstract syntax tree" << endl
         << "  -T, --print-symbol-tables print abstract syntax tree with symbol tables" << endl
         << "  -f, --no-flattening       disable flattened output of AST" << endl
         << "  -p, --pretty-print        format the TTCN-3 document" << endl
         << endl;
    exit( 0 );
  }

  if ( options.isVersion )
  {
    exit( 0 );
  }

  if ( options.fileNameArgs.empty() && !options.isReadStdin )
  {
    cout << "Usage:" << endl
         << endl
         << "  TTCN3Parser [ --help ] [ --version ] [ --read-stdin ] [ --print-tree ]" << endl
         << "              [ --print-symbol-tables ] [ --no-flattening ] [ --pretty-print ]" << endl
         << "              [ file ... ]" << endl
         << endl;
    exit( 0 );
  }

  if ( options.isReadStdin && options.fileNameArgs.size() > 0 )
  {
    cerr << "Error: Cannot read from 'stdin' and from files at the same time." << endl
         << endl;
    exit( 1 );
  }

  for ( unsigned int i = 0; i < options.fileNameArgs.size() || ( options.isReadStdin && i == 0 ); i++ )
  {
    ifstream   *file = 0;

    if ( !options.isReadStdin )
    {
      file = new ifstream( argv[ options.fileNameArgs[ i ] ] );

      if ( !*file )
      {
        cerr << "Error: Cannot open file '" << argv[ options.fileNameArgs[ i ] ] << "'." << endl
             << endl;
        exit( 1 );
      }
    }

    try
    {
      unsigned int                  noOfErrors;
      TTCNGrammar::TTCNLexer        lexer( ( file != 0 ) ? *dynamic_cast< istream * >( file ) : cin );
      TTCNGrammar::TTCNParser       parser( lexer );
      TTCNGrammar::TTCNTreeParser   treeparser;

      lexer.setFilename( ( file != 0 ) ? argv[ options.fileNameArgs[ i ] ] : "<stdin>" );
      parser.setFilename( ( file != 0 ) ? argv[ options.fileNameArgs[ i ] ] : "<stdin>" );

      lexer.resetErrors();
      parser.resetErrors();

      parser.setASTNodeFactory( &antlr::InfoAST::factory );

      if ( options.fileNameArgs.size() > 1 )
        cout << "*** Input file: " << argv[ options.fileNameArgs[ i ] ] << " ***" << endl
             << endl;

      cout << "Running the TTCN-3 parser..." << endl
           << endl;

      parser.ttcn3Document();

      noOfErrors = lexer.numberOfErrors() + parser.numberOfErrors();

      RefAST tree = parser.getAST();

      if ( noOfErrors == 0 && tree != antlr::nullAST )
      {
        // treeparser.ttcn3Document( tree );
      }

      // noOfErrors = noOfErrors + treeparser.numberOfErrors()

      cout << noOfErrors << " error(s) found." << endl << endl;

      if ( noOfErrors > 0 )
        continue;

      if( options.isPrintTree )
      {
        if ( tree != antlr::nullAST )
        {
          cout << "Printing the abstract syntax tree..." << endl
               << endl;

          printAST( tree, parser.getTokenNames(),
                    !options.isNoFlattening, options.isPrintSymbolTables );

          cout << endl;
        }
        else
        {
          cout << "No tree generated." << endl
               << endl;
        }
      }

      if ( options.isPrettyPrint )
      {
        cout << "Sorry, option '--pretty-print' is not supported so far." << endl
             << "Please check the web page of the TTCN-3 parser for the most recent updates." << endl
             << "http://www.itm.mu-luebeck.de/english/research/specification/ttcn3parser" << endl
             << endl;
      }
    }
    catch( antlr::ANTLRException& e )
    {
      cerr << e.toString() << endl;
    }

    if ( file != 0 )
      delete file;
  }

  cout << "Any problem? Send an email to 'schmitt@itm.mu-luebeck.de'!" << endl
       << endl;
}
