/*
 * @(#)Track.cpp 3.00 8 July 2000
 *
 * Copyright (c) 2000 Pete Goodliffe (pete.goodliffe@pace.co.uk)
 *
 * This file is part of TSE3 - the Trax Sequencer Engine version 3.00.
 *
 * This library is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "tse3/cmd/Track.h"

#include "tse3/Track.h"
#include "tse3/Part.h"

#include <algorithm>

using namespace TSE3;
using namespace TSE3::Cmd;

/******************************************************************************
 * Track_SetInfo class
 *****************************************************************************/

Track_SetInfo::Track_SetInfo(TSE3::Track               *track,
                             const std::string         &title,
                             const TSE3::MidiFilter    &smef,
                             const TSE3::MidiParams    &mp,
                             const TSE3::DisplayParams &dp)
: Command("track info"), track(track),
  newTitle(title), smef(smef), mp(mp), dp(dp)
{
}


void Track_SetInfo::executeImpl()
{
    oldTitle = track->title();

    track->setTitle(newTitle);
    std::swap(smef, *(track->filter()));
    std::swap(mp,   *(track->params()));
    std::swap(dp,   *(track->displayParams()));
}


void Track_SetInfo::undoImpl()
{
    track->setTitle(oldTitle);
    std::swap(smef, *(track->filter()));
    std::swap(mp,   *(track->params()));
    std::swap(dp,   *(track->displayParams()));
}


/******************************************************************************
 * Track_Snip class
 *****************************************************************************/

Track_Snip::Track_Snip(TSE3::Track *track,
                       TSE3::Clock  snipTime)
: Command("snip part"), track(track), snipTime(snipTime), valid(false),
  shouldDelete(false)
{
    size_t pos = track->index(snipTime);
    if (pos < track->size() && (*track)[pos]->start() < snipTime)
    {
        oldPart      = (*track)[pos];
        oldEndTime   = oldPart->end();
        newPart      = new Part(*oldPart);
        valid        = true;
        shouldDelete = true;

        // Work out new Part's parameters
        newPart->setStart(snipTime);
        TSE3::Clock phraseStart = oldPart->start();
        if (oldPart->repeat())
        {
            while (phraseStart+oldPart->repeat() <= snipTime)
            {
                phraseStart += oldPart->repeat();
            }
        }
        newPart->filter()->setOffset(snipTime - phraseStart
                                     + oldPart->filter()->offset());
        if (oldPart->repeat())
        {
            while (newPart->filter()->offset() >= newPart->repeat())
            {
                newPart->filter()->setOffset(newPart->filter()->offset()
                                             - newPart->repeat());
            }
        }
    }
}


Track_Snip::~Track_Snip()
{
    if (shouldDelete)
    {
        delete newPart;
    }
}


void Track_Snip::executeImpl()
{
    if (valid)
    {
        oldPart->setEnd(snipTime);
        track->insert(newPart);
        shouldDelete = false;
    }
}


void Track_Snip::undoImpl()
{
    if (valid)
    {
        track->remove(newPart);
        oldPart->setEnd(oldEndTime);
        shouldDelete = true;
    }
}


/******************************************************************************
 * Track_Glue class
 *****************************************************************************/

bool Track_Glue::valid(TSE3::Track *track,
                       TSE3::Clock  glueTime)
{
    size_t pos = track->index(glueTime);
    if (pos != 0 && pos != track->size() && (*track)[pos]->start() <= glueTime)
    {
        if ((*track)[pos-1]->end() == (*track)[pos]->start())
        {
            return true;
        }
    }
    return false;
}


Track_Glue::Track_Glue(TSE3::Track *track,
                       TSE3::Clock  glueTime)
: Command("glue parts"), track(track), glueTime(glueTime),
  _valid(valid(track, glueTime)), oldPart(0)
{
    if (_valid)
    {
        pos = track->index(glueTime);
        oldEndTime = (*track)[pos]->start();
    }
}


Track_Glue::~Track_Glue()
{
    delete oldPart;
}


void Track_Glue::executeImpl()
{
    if (_valid)
    {
        oldPart = (*track)[pos];
        track->remove(pos);
        (*track)[pos-1]->setEnd(oldPart->end());
    }
}


void Track_Glue::undoImpl()
{
    if (_valid)
    {
        (*track)[pos-1]->setEnd(oldEndTime);
        track->insert(oldPart);
        oldPart = 0;
    }
}


/******************************************************************************
 * Track_RemovePart class
 *****************************************************************************/

Track_RemovePart::Track_RemovePart(Part *p)
: Command("remove part"), track(p->parent()), part(p), partno(0)
{
    if (!track) part=0;
}


Track_RemovePart::Track_RemovePart(Track *t, size_t p)
: Command("remove part"), track(t), part(NULL), partno(p)
{
}


Track_RemovePart::Track_RemovePart(Track *t, Part *p)
: Command("remove part"), track(t), part(p), partno(0)
{
}


Track_RemovePart::~Track_RemovePart()
{
    if (done())
    {
        delete part;
    }
}


void Track_RemovePart::executeImpl()
{
    if (!part && partno < (int)track->size())
    {
        part = (*track)[partno];
    }
    else
    {
        partno = track->index(part);
    }
    track->remove(part);
}


void Track_RemovePart::undoImpl()
{
    if (part)
    {
        track->insert(part);
    }
}

