/* vim: set tabstop=4: */
/* This file is part of TraceProto. */

/** @mainpage
 * @author Eric Hope and others
 *
 * Traceproto is a traceroute replacement written in C that allows
 * the user to specify the protocol and port to trace to. It currently
 * supports TCP, UDP, and ICMP traces with the possibility of others
 * in the future.
 *
 * @see http://traceproto.sf.net/
 *
 * Traceproto is Copyright 2004-2005 Eric Hope and others; see the AUTHORS file
 * for details.
 *
 * TraceProto is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * TraceProto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TraceProto; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <libnet.h>
#include <pcap.h>
#include <errno.h>
#include <sys/time.h>
#include <stdio.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <netdb.h>
#include <ctype.h>
#include <unistd.h>
#include <assert.h>

#include "traceproto.h"
#include "config.h"
#include "tp_as.h"
#include "tp_miscfunc.h"
#include "tp_output.h"
#include "tp_packet.h"

#ifdef HAVE_LIBDMALLOC
#include <dmalloc.h>
#endif /* HAVE_LIBDMALLOC */

int main( int argc, char * argv[] )
{


	int cmd_arg, i, l;
	u_int on = 1;
	char * opt_string;
	char * env_string;

#ifdef HAVE_LIBCAP
	dropexcesscapabilities();
#endif /* HAVE_LIBCAP */

/*
 * regularize the program name
 */
	if ( reg_name( argv[ 0 ] ) == 0 )
	{
		printf ( "setup error, unable to parse program name\n" );
		tixe ( tixe_cleanup, 1 );
	}

/*
 * set the defaults
 */
	behavior.continuous = NO;
	behavior.packets_per_hop = 3;
	behavior.src_port_incr = 1;	/* src port */
	behavior.dst_port_incr = 0;	/* dst port */
	behavior.min_src_port = 10240;
	behavior.rndm_src_port = YES;
	behavior.min_dst_port = 80;
	behavior.max_ttl = 30;			/* hops */
	behavior.min_ttl = 1;			/* hop(s) */
	behavior.wait_timeout = 5;		/* seconds */
	behavior.wait_between_packets = 100;	/* milliseconds */
	behavior.protocol = "tcp";
	behavior.tcp_resets = YES;
	behavior.account_level = TP_ACCOUNT_FULL;
	behavior.report = report_std;
	behavior.do_audit = NO;
	behavior.do_audit_exit = NO;
	behavior.do_skip = NO;
	behavior.as_discovery = NO;
	behavior.output_style = TP_STD_OUTPUT;
	behavior.payload_size = 12;
	behavior.libnet_resolve_choice = LIBNET_RESOLVE;
	behavior.hop_incr_unit = 1;
	behavior.timestamp = NO;
	behavior.timestamp_style = TP_TIMESTAMP_STD;
	behavior.filter_text = "( host %s and dst port %d and src port %d ) or ( icmp[12:2] == %d )";
	behavior.default_if = YES;
	memset ( behavior.interface, '\0', TP_IF_ARRAY );
	strncpy ( behavior.interface, "any", TP_IF_ARRAY - 1 ); 

	packet.protocol_number = IPPROTO_TCP;
	packet.payload = 0;
	packet.frag_bit = 0;
	packet.tcp_flags = TH_SYN;
	packet.ip_packet_len = LIBNET_IPV4_H;

	state.tcp_h  = LIBNET_PTAG_INITIALIZER;
	state.udp_h  = LIBNET_PTAG_INITIALIZER;
	state.icmp_h = LIBNET_PTAG_INITIALIZER;
	state.ip_h   = LIBNET_PTAG_INITIALIZER;
	state.incr_error = NO;
	state.target_response = NO;
	state.continuous_count = ( int ) NULL;

	/*
	 * If called as "traceroute" try to be a drop-in replacement.
	 * Currently we aren't an exact copy, but most of the differences
	 * are a matter of different command line flags for the original
	 * features.  We can more or less produce the same output behavior. 
	 */
	if ( strncmp ( state.prog, "traceroute", 11 ) == 0 )
	{
		behavior.output_style = TP_CLASSIC_OUTPUT;
		behavior.protocol = "udp";
		behavior.min_dst_port = 32768 + 666;
		behavior.account_level = TP_ACCOUNT_NONE;
		behavior.src_port_incr = 0;	/* src port */
		behavior.dst_port_incr = 1;	/* dst port */
	}

/*
 * grab the relavent env variables
 *
 * NOTE: the values gathered from the env
 * overwrite the defaults set above and are 
 * overwritten in turn by any cmd line flags 
 * that happen to be set below
 *
 * These should always have an else or a default so that the
 * prog still does rational things if the user puts in a weird
 * value.
 */

	env_string = NULL;
	if ( ( env_string = getenv ( "TP_TIMESTAMP_STYLE" ) ) != NULL )
	{
		if ( strncmp ( env_string, "std", 3 ) == 0 )
			behavior.timestamp_style = TP_TIMESTAMP_STD;
		else if ( strncmp ( env_string, "us", 2 ) == 0 )
			behavior.timestamp_style = TP_TIMESTAMP_US;
		else if ( strncmp ( env_string, "desc", 4 ) == 0 )
			behavior.timestamp_style = TP_TIMESTAMP_DESCEND;
		else if ( strncmp ( env_string, "epoch", 5 ) == 0 )
			behavior.timestamp_style = TP_TIMESTAMP_EPOCH;
		else
			behavior.timestamp_style = TP_TIMESTAMP_STD;
	}
	
	env_string = NULL;
	if ( ( env_string = getenv ( "TP_DEFAULT_IF" ) ) != NULL )
	{
		strncpy ( behavior.interface, env_string, TP_IF_ARRAY - 1 );
		behavior.default_if = NO;
	}

	env_string = NULL;
	if ( ( env_string = getenv ( "TP_OUTPUT_STYLE" ) ) != NULL )
	{
		if ( 
			strncmp ( env_string, "stand", 5 ) == 0 ||
			strncmp ( env_string, "std",   3 ) == 0 ||
			strncmp ( env_string, "s",     1 ) == 0    )
		{
				behavior.output_style = TP_STD_OUTPUT;
				behavior.report = report_std;
		} else if (	
			strncmp ( env_string, "graph", 5 ) == 0 ||
			strncmp ( env_string, "g",     1 ) == 0    )
		{
				behavior.output_style = TP_GRAPHIC_OUTPUT;
				behavior.report = report_graphic;
		} else if (
			strncmp ( env_string, "class", 5 ) == 0 ||
			strncmp ( env_string, "c",     1 ) == 0    )
		{
				behavior.output_style = TP_CLASSIC_OUTPUT;
				behavior.report = report_classic;
		} else if (
			strncmp ( env_string, "no",    2 ) == 0 ||
			strncmp ( env_string, "n",     1 ) == 0    )
		{
				behavior.output_style = TP_NO_OUTPUT;
				behavior.report = report_none;
		} else if (
			strncmp ( env_string, "min",   3 ) == 0 ||
			strncmp ( env_string, "m",     1 ) == 0    )
		{
				behavior.output_style = TP_MIN_OUTPUT;
				behavior.report = report_minimum;
		} else if ( 
			strncmp ( env_string, "script", 6 ) == 0 ||
			strncmp ( env_string, "p",      1 ) == 0    )
		{
				behavior.output_style = TP_SCRIPT_OUTPUT;
				behavior.report = report_scriptable;
		} else if ( 
			strncmp ( env_string, "curs",  4 ) == 0 ||
			strncmp ( env_string, "ncurs", 5 ) == 0 ||
			strncmp ( env_string, "C",     1 ) == 0    )
		{
				behavior.output_style = TP_CURSES_OUTPUT;
				behavior.report = report_curses;
				tixe_cleanup.curses_end = YES;
		} else {
				behavior.output_style = TP_STD_OUTPUT;
				behavior.report = report_std;
		}
	} /* end if getenv ( "TP_OUTPUT_STYLE" ) */
	


/*
 * get command line args
 */
	while ( ( cmd_arg = getopt ( argc, argv,
			"cCvnhRfATp:i:I:D:r:t:k:o:a:s:S:H:d:M:m:w:W:F:p:P:z:"
		) ) != EOF )
	{
		switch ( cmd_arg )
		{
		case 'c':
			behavior.continuous = YES;
			behavior.continuous_accounting = NO;
			break;
		case 'C':
			behavior.continuous = YES;
			behavior.continuous_accounting = YES;
			break;
		case 'I':
			state.continuous_count = atoi ( optarg );
			behavior.continuous = YES;
			break;
		case 'H':
			behavior.packets_per_hop = atoi ( optarg );
			break;
		case 'i':
			/*
			 * controls the port number on the two ends
			 * possiblities are src/dst and incr/decr/static
			 */
			opt_string = optarg;
			for ( i = strlen ( opt_string ); i >= 0; i-- )
			{
				switch ( opt_string [ i ] )
				{
				case 's':
					behavior.src_port_incr = -1;
					break;
				case 'S':
					behavior.src_port_incr = 1;	/* default */
					break;
				case 'd':
					behavior.dst_port_incr = -1;
					break;
				case 'D':
					behavior.dst_port_incr = 1;
					break;
				case 'n':
					behavior.src_port_incr = 0;
					break;
				case 'N':
					behavior.dst_port_incr = 0;	/* default */
					break;
				case '\0':
					break;
				default:
					state.incr_error = YES;
					break;
				}
			}
			break;
		case 'd':
			behavior.min_dst_port = atoi( optarg );
			break;
		case 'D':
			behavior.max_dst_port = atoi( optarg );
			break;
		case 'm':
			behavior.min_ttl = atoi ( optarg );
			break;
		case 'M':
			behavior.max_ttl = atoi( optarg );
			break;
		case 'n':
			behavior.libnet_resolve_choice = LIBNET_DONT_RESOLVE;
			break;
		case 'w':
			/* how long to wait to see if a response turns up */
			behavior.wait_timeout = atoi ( optarg );
			break;
		case 'W':
			/* how long to wait after a response ( or timeout )
			 * before sending the next probe */
			behavior.wait_between_packets = atoi ( optarg );
			break;
		case 's':
			behavior.min_src_port = atoi ( optarg );
			behavior.rndm_src_port = NO;
			break;
		case 'S':
			behavior.max_src_port = atoi ( optarg );
			behavior.rndm_src_port = NO;
			break;
		case 'p':
			behavior.protocol = optarg;
			break;
		case 'a':
			behavior.account_level = atoi ( optarg );
			break;
		case 'k':
			/* a list of hops ( ie ttl numbers ) to skip */
			behavior.skip_str = optarg;
			behavior.do_skip = YES;
			break;
		case 'h':
			usage ( );
			tixe ( tixe_cleanup, 0 );
			break;
		case 'v':
			version ( );
			tixe ( tixe_cleanup, 1 );
			break;
		case 'o':
			/* what should the printout look like */
			opt_string = optarg;
			if ( opt_string [ 0 ] == 's' )
			{
				behavior.output_style = TP_STD_OUTPUT;
				behavior.report = report_std;
				break;
			} else if ( opt_string [ 0 ] == 'g' ) {
				behavior.output_style = TP_GRAPHIC_OUTPUT;
				behavior.report = report_graphic;
				break;
			} else if ( opt_string [ 0 ] == 'c' ) {
				behavior.output_style = TP_CLASSIC_OUTPUT;
				behavior.report = report_classic;
				break;
			} else if ( opt_string [ 0 ] == 'C' ) {
				behavior.output_style = TP_CURSES_OUTPUT;
				behavior.report = report_curses;
				break;
			} else if ( opt_string [ 0 ] == 'n' ) {
				behavior.output_style = TP_NO_OUTPUT;
				behavior.report = report_none;
				break;
			} else if ( opt_string [ 0 ] == 'm' ) {
				behavior.output_style = TP_MIN_OUTPUT;
				behavior.report = report_minimum;
				break;
			} else if ( opt_string [ 0 ] == 'p' ) {
				behavior.output_style = TP_SCRIPT_OUTPUT;
				behavior.report = report_scriptable;
				break;
			} else {
				printf ( "invalid output style: %c\n", * opt_string );
				usage ( );
				tixe ( tixe_cleanup, 1 );
			}
			break;
		case 't':
			/* the tcp SYN/ACK/FIN/etc flags */
			opt_string = optarg;
			packet.tcp_flags = parse_flags ( opt_string );
			break;
		case 'P':
			behavior.payload_size = atoi ( optarg );
			break;
		case 'f':
			packet.frag_bit = TP_DONT_FRAG;
			break;
		case 'F':
			strncpy ( behavior.interface, optarg, TP_IF_ARRAY - 1 );
			behavior.default_if = NO;
			break;
		case 'A':
			behavior.as_discovery = YES;
			break;
		case 'T':
			behavior.timestamp = YES;
			make_timestamp ( behavior.timestamp_str );
			break;
		case 'R':
			/* start at the distant end and trace consecutively
			 * closer until we get here */
			behavior.hop_incr_unit = -1;
			break;
		case 'z':
			/* undocumented debug flag */
			parse_debug ( optarg );
			break;
		default:
			usage( );
			tixe ( tixe_cleanup, 1 );
			break;
		}
	}
	behavior.target = argv[optind];

/*
 * set the max ports and starting ports
 */
	if ( ! behavior.max_src_port )
		behavior.max_src_port = behavior.min_src_port
					+ ((( behavior.max_ttl - behavior.min_ttl )
					+ behavior.packets_per_hop )
					* behavior.packets_per_hop );

	if ( ! behavior.max_dst_port )
		behavior.max_dst_port = behavior.min_dst_port
					+ ((( behavior.max_ttl - behavior.min_ttl )
					+ behavior.packets_per_hop )
					* behavior.packets_per_hop );

	if ( behavior.rndm_src_port == YES )
	{
		/* this needs to be random, just not very random */
		srand ( ( unsigned int ) getpid() );
		behavior.rndm_src_port = abs ( rand() % 256 );
		behavior.min_src_port += behavior.rndm_src_port;
		behavior.max_src_port += behavior.rndm_src_port;
	}

	if ( behavior.src_port_incr == -1 )
		packet.src_port = behavior.max_src_port - behavior.src_port_incr;
	else
		packet.src_port = behavior.min_src_port - behavior.src_port_incr;

	if ( behavior.dst_port_incr == -1 )
		packet.dst_port = behavior.max_dst_port - behavior.dst_port_incr;
	else
		packet.dst_port = behavior.min_dst_port - behavior.dst_port_incr;

/*
 * sanity checks
 */

	/* must be root or the euivalent to use the raw sockets */
	if ( geteuid() != 0 )
	{
		printf ( "error: root privileges required\n" );
		tixe ( tixe_cleanup, 1 );
	}

	/* we really do need to know where to trace to... */
	if ( behavior.target == NULL )
	{
		printf ( "error: invalid target\n" );
		usage( );
		tixe ( tixe_cleanup, 1 );
	}

	if ( state.incr_error != NO )
	{
		printf ( "invalid incriment option specified with -i\n" );
		usage ( );
		tixe ( tixe_cleanup, 1 );
	}

	/* limit the number of probes per hop */
	if ( behavior.packets_per_hop <= 0 || behavior.packets_per_hop > 10 )
	{
		printf("packets per hop must be between 1 and 10\n");
		behavior.packets_per_hop = 3;
	}

/*
	if ( behavior.continuous == YES && behavior.wait_between_packets < 200 )
	{
		behavior.wait_between_packets = 200;
	}
*/

	/* limits on the continuous count */
	if ( state.continuous_count > 2048 )
	{
		printf ( "warning: unusually large iteration number set with -I\n" );
	}

	/* set the protocol */
	if ( strncmp( behavior.protocol, "udp", 3 ) == 0 )
	{
		behavior.protocol = "udp";
		packet.protocol_number = IPPROTO_UDP;
		packet.ip_packet_len += LIBNET_UDP_H;
	}
	else if ( strncmp( behavior.protocol, "tcp", 3 ) == 0 )
	{
		behavior.protocol = "tcp";
		packet.protocol_number = IPPROTO_TCP;
		packet.ip_packet_len += LIBNET_TCP_H;
	}
	else if ( strncmp( behavior.protocol, "icm", 3 ) == 0 )
	{
		behavior.protocol = "icmp";
		packet.protocol_number = IPPROTO_ICMP;
		behavior.filter_text = "( icmp[12:2] == %d ) or ( host %s and icmp[0] == 0 )";
		packet.ip_packet_len += LIBNET_ICMPV4_ECHO_H;
	}
	else
	{
		printf("error: invalid protocol specified with -p\n");
		usage( );
		tixe ( tixe_cleanup, 1 );
	}

	if ( packet.dst_port <= 0 || packet.dst_port > 65535 )
	{
		printf("error: illegal destination port specified with -d\n");
		usage( );
		tixe ( tixe_cleanup, 1 );
	}

	if ( packet.src_port <= 0 || packet.dst_port > 65535 )
	{
		printf("error: illegal source port specified with -s\n");
		usage( );
		tixe ( tixe_cleanup, 1 );
	}

	if ( behavior.max_ttl <= 0 )
	{
		printf("error: illegal maximum time to live value specified with -m\n");
		usage( );
		tixe ( tixe_cleanup, 1 );
	} else if ( behavior.max_ttl > 256 ) {
		printf("warning: using an unusually high max ttl value\n");
	}

	if ( behavior.min_ttl <= 0 )
	{
		printf("error: illegal minimum time to live value specified with -n\n");
		behavior.min_ttl = 1;
	} else if ( behavior.min_ttl > behavior.max_ttl ) {
		printf("error: minimum ttl (-n) must be less than maximum ttl (-m)\n");
		usage( );
		tixe ( tixe_cleanup, 1 );
	}

	if ( behavior.max_src_port < behavior.min_src_port )
	{
		printf ( "error: maximum src port ( specified with -S )\n" );
		printf ( "\tmust be greater than the minimum src port\n" );
		printf ( "\t( specified with -s or default 10240 )\n" );
		usage( );
		tixe ( tixe_cleanup, 1 );
	}

	if ( behavior.wait_timeout <= 0 )
	{
		printf("error: illegal timeout specified with -w\n");
		usage( );
		tixe ( tixe_cleanup, 1 );
	} else if ( behavior.wait_timeout > 300 ) {
		printf("warning: using and unusually high wait timeout specified with -w\n");
	}

	if ( behavior.wait_between_packets < 0 )
	{
		printf("error: illegal timeout specified with -z\n");
		usage( );
		tixe ( tixe_cleanup, 1 );
	} else if ( behavior.wait_between_packets > 10000 ) {
		printf("warning: using an unusually high wait time between packets\n");
	}

	if ( packet.src_port == 0 )
	{
		printf("warning: using source port 0\n");
	}

	if ( behavior.account_level > TP_ACCOUNT_FULL )
	{
		behavior.account_level = TP_ACCOUNT_FULL;
	}
	else if ( behavior.account_level < TP_ACCOUNT_NONE )
	{
		behavior.account_level = TP_ACCOUNT_NONE;
	}

	/* there will be no overflows here, thank you. */
	if ( strlen ( behavior.filter_text )
			+ strlen ( behavior.target )
			+ 12 > FILTERSIZE )
	{
		printf ( "error: bpf filter size limit exceeded.  "
			"Specify a shorter target name (perhaps the ip address).\n" );
		tixe ( tixe_cleanup, 1 );
	}

	/*
	 * Explicitly turn off continuous_accounting with the curses
	 * It won't work well and is redundant.
	 */
	if ( behavior.output_style == TP_CURSES_OUTPUT )
		behavior.continuous_accounting = NO;

	/*
	 * note that the default payload is 12 bytes due to a udp
	 * peculiarity.  less than 12 bytes and no udp response
	 * is returned.  Don't know if its the stack, libnet, pcap
	 * or something else completely.
	 */
	if ( behavior.payload_size != 0 )
	{
		if ( behavior.payload_size > 8192 )
		{
			printf ( "illegal payload size\n" );
			usage ( );
			tixe ( tixe_cleanup, 1 );
		}
		packet.payload = ( u_char * ) malloc ( behavior.payload_size );
		assert(packet.payload != NULL);
		if ( packet.payload == NULL )
		{
			printf ( "error allocating payload memory for payload\n" );
			tixe ( tixe_cleanup, 1 );
		}
		tixe_cleanup.payload_free = YES;
		memset ( packet.payload, '\0', behavior.payload_size );

		packet.ip_packet_len += behavior.payload_size;
	}

/*
 * Setup the free list for the packet alignment func
 */
	tp_align_freelist.next = ( struct tp_align_ref * ) malloc ( sizeof ( struct tp_align_ref ) );
	if ( tp_align_freelist.next == NULL )
	{
		printf ( "error allocating memory for the free list\n" );
		tixe (  tixe_cleanup, 1 );
	}
	tixe_cleanup.free_list_free = YES;


/*
 * DNS stuff
 */
#ifdef HAVE_GETADDRINFO
	/* Use address family independent resolver function, so we can start 
	   to handle IPv6 as well */
	behavior.hint.ai_flags = 0;
	behavior.hint.ai_family = PF_UNSPEC;
	behavior.hint.ai_socktype = SOCK_STREAM;
	behavior.hint.ai_protocol = IPPROTO_IP;
	behavior.hint.ai_addrlen = 0;
	behavior.hint.ai_addr = NULL;
	behavior.hint.ai_canonname = NULL;
	behavior.hint.ai_next = NULL;
	behavior.target_addrinfo_list_start = NULL;

	if ( (i = getaddrinfo( behavior.target, NULL, &behavior.hint, &behavior.target_addrinfo_list_start )) != 0 ) {
		printf ( "error resolving target address %s: %s\n", behavior.target, gai_strerror(i));
		tixe ( tixe_cleanup, 1 );
	}
	/* moving this after the getaddrinfo call to fix a segfault on freebsd when the target is invalid */
	tixe_cleanup.addrinfo_cleanup = YES;

	/* We should really think about what the proper way for traceproto of
	   dealing with multiple DNS entries is.
	   For now, we just pick the first IPv4 address, if any, in the list.

	   FIXME: This is too simplistic: we lose the pointer to the start of
	   the list, so we lose the chance to clean up after ourselves with
	   freeaddrinfo(3).
	 */
	behavior.target_addrinfo = behavior.target_addrinfo_list_start;
	while ( (behavior.target_addrinfo->ai_addr->sa_family != AF_INET) &&
		(behavior.target_addrinfo->ai_next != NULL) ) {
		behavior.target_addrinfo = behavior.target_addrinfo->ai_next;
	}

# ifdef DEBUG
	fprintf ( stderr, "resolving %s: success\n", behavior.target );
# endif

	if ( behavior.target_addrinfo->ai_addr->sa_family == AF_INET ) {
# ifdef DEBUG
		fprintf ( stderr, "Address family: AF_INET\n" );
# endif
		l = sizeof(struct sockaddr_in);
	} else if ( behavior.target_addrinfo->ai_addr->sa_family == AF_INET6 ) {
# ifdef DEBUG
		fprintf ( stderr, "Address family: AF_INET6\n" );
# endif
		l = sizeof(struct sockaddr_in6);
	} else {
		fprintf ( stderr, "Address family %d not supported\n", behavior.target_addrinfo->ai_addr->sa_family );
		l = 0; /* gets rid of a stupid compiler warning --skippy */
		tixe ( tixe_cleanup, 1 );
	}
	behavior.target_reverse = malloc( NI_MAXHOST );
	assert(behavior.target_reverse != NULL);
	if ( (i = getnameinfo( behavior.target_addrinfo->ai_addr, l, behavior.target_reverse, NI_MAXHOST-1, NULL, 0, NI_NUMERICHOST )) != 0 ) {
		fprintf ( stderr, "getnameinfo(...) failed: %s\n", gai_strerror(i) );
	}

	/* Sorry, this is it so far for IPv6 */
	if ( behavior.target_addrinfo->ai_addr->sa_family != AF_INET ) {
		fprintf ( stderr, "Sorry, target address %s did not resolve to an IPv4 address (address family is %d)\n", behavior.target, behavior.target_addrinfo->ai_addr->sa_family );
		tixe ( tixe_cleanup, 1 );
	}
#else
	/* IPv4 specific implementation */
	behavior.packed_target_reverse = gethostbyname ( behavior.target );
	if ( behavior.packed_target_reverse == NULL )
	{
		printf ( "error resolving target address %s\n", behavior.target );
		tixe ( tixe_cleanup, 1 );
	}

	behavior.target_reverse = inet_ntoa (
			*(struct in_addr *)( behavior.packed_target_reverse->h_addr_list[0]));
#endif /* HAVE_GETADDRINFO */


/*
 * initialize the skips list
 */
	if ( behavior.do_skip == YES && parse_skips ( behavior.skip_str ) != 0 )
	{
		fprintf ( stderr, "error with option k arguments\n" );
		usage ( );
		tixe ( tixe_cleanup, 1 );
	}

/*
 * set up accounting
 */
	state.account_hops = behavior.max_ttl + 1;
	state.hop_record = ( struct hop_record * ) calloc (
						state.account_hops,
						sizeof ( struct hop_record ) );
	if ( state.hop_record == NULL )
	{
		perror ( "memory allocation error" );
		tixe ( tixe_cleanup, 1 );
	}
	tixe_cleanup.hop_record_free = YES;

/*
 * find interface if needed and start packet
 */
	/* find_interface ( ); */

#ifdef OS_FREEBSD
	/*
	 * temporary kludge: freebsd (and probably others) dump core
	 * when calling pcap_open_live with a first arguement (the interface)
	 * as NULL.  This causes it to revert to the interface "any" on
	 * FreeBSD.  The "any" interface doesn't exist, but at least it
	 * errors out rather than dumping core.
	 */
	behavior.default_if = NO;
#endif /* OS_FREEBSD */

	if ( debug.interface == YES )
	{
		printf ( "debug: using default interface = %s\n",
			behavior.default_if ? "YES"  : "NO" );
		printf ( "debug: interface flag is %s\n",
			behavior.default_if == NO ? behavior.interface : "NULL" );
		
	}

	state.packet = libnet_init (
		LIBNET_RAW4,	    /* inject type */
	/*	NULL,	*/	   /* device */
		behavior.default_if == NO ? behavior.interface : NULL,
		( char * ) state.error_buff );
	if (!state.packet)
	{
		printf("Error: %s\n", * state.error_buff);
		tixe ( tixe_cleanup, 1 );
	}
	tixe_cleanup.libnet_cleanup = YES;

	packet.packed_target = libnet_name2addr4(
		state.packet, behavior.target, LIBNET_RESOLVE );
	if ( packet.packed_target == -1 )
	{
		printf("error resolving destination (%s): %s\n",
			behavior.target, libnet_geterror( state.packet ) );
		usage( );
		tixe ( tixe_cleanup, 1 );
	}

	packet.packed_src = libnet_get_ipaddr4( state.packet );
	if ( packet.packed_src == -1 )
	{
		printf("error resolving source: %s\n",
			libnet_geterror( state.packet ) );
		tixe ( tixe_cleanup, 1 );
	}

	libnet_seed_prand( state.packet );

/*
 * do the pcap stuff and the initial filter
 */
	/* state.psocket = pcap_open_live( NULL, */
	/* state.psocket = pcap_open_live( behavior.interface, */
	state.psocket = pcap_open_live (
				behavior.default_if == NO ? behavior.interface : NULL,
				SNAPLEN,
				NO_PROMISC,
				( behavior.wait_timeout * 1000 ),
				state.pc_error );
	if ( state.psocket == NULL )
	{
		printf("pcap error: %s\n", state.pc_error);
		tixe ( tixe_cleanup, 1 );
	}
	tixe_cleanup.pcap_cleanup = YES;

	/*
	 * this is a specific weirdness.  See the README for more details.
	 *
	 * The pcap_fileno is a bit borrowed from Michael Toren
	 * and his tcptraceroute.  Appearently the pcap documentation
	 * is blatently wrong here and the pcap_fileno actually returns
	 * the socket fd, not the fd of the logfile.
	 *
	 * So now you have a choice.  You can use my crude kludge to
	 * break the encapsulation, or you can use the "legit" but
	 * badly mis-documented library function....see the configure
	 * options.
	 */
#ifdef USE_PCAP_FILENO
	state.fake_psocket = ( struct fake_pcap * )
		malloc ( sizeof ( struct fake_pcap ) );
	if ( state.fake_psocket == NULL )
	{
		printf ( "error allocating pcap_fileno memory\n" );
		tixe ( tixe_cleanup, 1 );
	}
	tixe_cleanup.fake_psocket_free = YES;

	state.fake_psocket->fd  = pcap_fileno ( state.psocket );
#else
	state.fake_psocket = ( struct fake_pcap * ) state.psocket;
#endif /* USE_PCAP_FILENO */

#ifdef OS_FREEBSD
	/*
	 * Appearently, BSD bpf fds won't return immediately without this
	 * set.  Without it, select waits until its timeout and THEN
	 * grabs and parses the packet.  Poof, instant round trip times
	 * of whatever -w is set to.
	 */
	if ( ioctl ( state.fake_psocket->fd, BIOCIMMEDIATE, & on ) > 0 )
	{
		printf ( "error setting BSD BPF fd to immediate\n" );
		tixe ( tixe_cleanup, 1 );
	}
#endif /* OS_FREEBSD */

/*
 * set up the initial filter
 */
	if ( do_filter() != 0 )
	{
		tixe ( tixe_cleanup, 1 );
	}

/*
 * drop the root privileges if possible
 */
	if ( setuid ( getuid ( ) ) != 0 )
	{
		printf ( "error dropping privileges\n" );
		tixe ( tixe_cleanup, 1 );
	}
	if ( setgid ( getgid ( ) ) != 0 )
	{
		printf ( "error dropping privileges\n" );
		tixe ( tixe_cleanup, 1 );
	}

/*
 * if we're doing as_discovery, set it up here
 */
	if ( behavior.as_discovery == YES && setup_as ( ) != 0 )
	{
		printf ( "error doing AS discovery: %s\n", as_string );
		behavior.as_discovery = NO;
	}

/*
 * set up the signals
 */
	signal ( SIGINT, ctrl_c );
	signal ( SIGTSTP, ctrl_z );

/*
 * ttl setup depending on if we're tracing forward or backwards
 */
	if ( behavior.hop_incr_unit == 1 )
		state.current_hop = behavior.min_ttl;
	else
		state.current_hop = behavior.max_ttl;
/*
 * MAIN LOOP STARTS HERE
 *
 * loop through, building the ip
 * and sending it
 */
	behavior.report( TP_OUT_HEADER, NULL, 0 );


	/*
	 * conditional: ( A && B ) || C
	 */
	for ( state.packets_this_hop = 0;

		( state.current_hop >= behavior.min_ttl
		&&
		state.current_hop <= behavior.max_ttl )
		||
		state.packets_this_hop < behavior.packets_per_hop;

		++( state.packets_this_hop ) )
	{
		/*
		 * reset if we've sent enough probes for this ttl
		 */
		if ( state.packets_this_hop >= behavior.packets_per_hop )
		{
			state.current_hop += behavior.hop_incr_unit;
			state.packets_this_hop = 0;
		}

		/*
		 * skip the hops marked on the cmd line
		 */
		if ( behavior.do_skip == YES && behavior.skips[ state.current_hop ] == YES )
		{
			state.current_hop += behavior.hop_incr_unit;
			state.packets_this_hop = -1;
			continue;
		}

		state.packet_match = TP_PACKET_NO;
		state.low_ttl = NO;

/*
 * increment the src and/or dst ports
 */
		packet.src_port += behavior.src_port_incr;
		packet.dst_port += behavior.dst_port_incr;

		build_packet();

/*
 * build the ip
 * ( this should probably be rolled in build_packet() )
 */
		packet.ip_id = libnet_get_prand(LIBNET_PRu16);
		if ( debug.packet_length == YES )
			printf ( "debug: packet.ip_id=%d", packet.ip_id );

		state.ip_h = libnet_build_ipv4(
			packet.ip_packet_len,	/* IP packet len */
			0,						/* tos */
			packet.ip_id,			/* frag id */
			packet.frag_bit,		/* frag offset */
			state.current_hop,		/* ttl */
			packet.protocol_number,
			0,						/* checksum */
			packet.packed_src,		/* packed source */
			packet.packed_target,		/* packed target */
			NULL,					/* payload */
			0,						/* payload size */
			state.packet,
			state.ip_h );

		if ( debug.packet_length == YES )
			printf ( "\nHEADER_LENGTH IP:%d payload:%d, total:%d\n",
				LIBNET_IPV4_H, behavior.payload_size, packet.ip_packet_len );

		if ( debug.send_buf == YES )
		{
			printf ( "debug: send buffer:\n" );
			debug_packet ( libnet_getpbuf ( state.packet, state.ip_h ),
				  libnet_getpbuf_size ( state.packet, state.ip_h ) );
			debug_packet ( libnet_getpbuf ( state.packet, state.tcp_h ),
				  libnet_getpbuf_size ( state.packet, state.tcp_h ) );
			debug_packet ( packet.payload, behavior.payload_size );
		}
/*
 * and change the filter
 */
		if ( do_filter() != 0 )
		{
			tixe ( tixe_cleanup, 1 );
		}

/*
 * spit out the hop number if this is the first probe this hop
 * then start timer and send the packet
 */
		if ( state.packets_this_hop == 0 )
			behavior.report( TP_OUT_HOP_NUMBER, ( struct in_addr * ) NULL, 0 );

		gettimeofday ( & state.start_time, NULL );

		if ( debug.loop == YES ) { printf ( "debug: time checked, writing packet\n" ); }

		if ( libnet_write( state.packet ) == -1 )
		{
			printf("error at send: %s\n",
				libnet_geterror( state.packet ) );
			tixe ( tixe_cleanup, 1 );
		}

		if ( debug.loop == YES ) { printf ( "debug: packet written, waiting\n" ); }

/*
 * wait for a packet to get
 * through the filter, timeout
 * after the wait_timeout
 */
		while ( state.packet_match == TP_PACKET_NO )
		{
			state.packet_wait.tv_sec  = behavior.wait_timeout;
			state.packet_wait.tv_usec = 0;
			FD_ZERO( & ( state.wheel ) );
			FD_SET( state.fake_psocket->fd, & ( state.wheel ) );

			if ( debug.loop == YES ) { printf ( "debug: select\n" ); }

			if ( select( ( state.fake_psocket->fd ) + 1,
					& ( state.wheel ),
					NULL,
					NULL,
					& state.packet_wait ) > 0 )
			{
				/*
				 * if we get a bite, note the time,
				 * test the response, and report it if OK
				 */
				gettimeofday ( & ( state.end_time ), NULL );

				if ( debug.loop == YES )
					printf ( "debug: select returned, capturing\n" );

				strcpy ( state.pc_error, "unknown" );

				/*
				 * If we've gotten here, it means that the select returned > 0
				 * meaning that there's a packet waiting for us.
				 */
				state.capture_buf = pcap_next(
						state.psocket,
						& ( state.psock_hdr ) );
				/*
				 * Under certain conditions, pcap_next returns NULL ( meaning no packet )
				 * even though the select returned > 0.
				 * This if is a somewhat heavy handed workaround while sorting this out.
				 * Effectively if pcap_next returns NULL, try again and only error if we
				 * get NULL a second time.  The second pacp_next seems to see the packet 
				 * consistantly.
				 */
				if ( state.capture_buf == NULL )
				{
					state.capture_buf = pcap_next(
						state.psocket,
						& ( state.psock_hdr ) );

					if ( state.capture_buf == NULL )
					{
						printf ( "error receiving packet: %s\n",
							state.pc_error );
						tixe ( tixe_cleanup, 1 );
					}
				}

				if ( debug.loop == YES )
					printf ( "debug: captured, parsing\n" );

				state.trip_time = diff_time (
					& ( state.start_time ),
					& ( state.end_time ) );

				if ( parse_packet( state.capture_buf ) > 0 )
				{
					freelist_cleaner ( );
				}

				if ( debug.loop == YES )
					printf ( "debug: parsed, resets\n" );

				if ( packet.protocol_number == IPPROTO_TCP
					&& behavior.tcp_resets == YES
					&& state.packet_match == TP_PACKET_DONE )
				{
					if ( send_tcp_reset ( ) != 0 )
						printf("error at send: %s\n",
							libnet_geterror( state.packet ) );
				}

				/* blank the buffer to prevent confusion */
				memset ( (char *) state.capture_buf, '\0', sizeof ( state.capture_buf ) );
				state.trip_time = ( double ) 0.0;
			} else {
				/*
				 * if we get here,  no one wanted to talk to us for this probe
				 */
				account_packet ( ( double ) 0 );
				behavior.report( TP_OUT_HOP_INFO, ( struct in_addr * ) NULL, TP_TYPE_NR );
				state.packet_match = TP_PACKET_TIMEOUT;

				if ( debug.loop == YES )
					printf ( "debug: no answer (%d)\n", FD_ISSET ( state.fake_psocket->fd, & ( state.wheel ) ) );
			}

			/*
			 * if there was an interupt during the wait
			 */
			if ( behavior.do_audit == YES || behavior.do_audit_exit == YES )
			{
				if ( behavior.do_audit_exit == YES )
					behavior.report( TP_OUT_FOOTER, NULL, 0 );
				else
					hop_audit();
			}
		} /* end of while ( packet_match == TP_PACKET_NO ) loop */

/*
 * an acceptable response in hand,
 * quit and clean up, or start over
 */
		/*
		 * in technical terms, this is "icky"
		 * it does however, seem to operate correctly
		 *
		 * conditional: ( ( A || B ) && C && D ) || ( E && F && G )
		 * I told you it was icky...
		 */
		if ( ( ( state.target_response == YES
					||
					state.current_hop == behavior.max_ttl )
				&&
				state.packets_this_hop >= behavior.packets_per_hop - 1
				&&
				behavior.hop_incr_unit == 1 )
			||
			( behavior.hop_incr_unit == -1
				&&
				state.current_hop == behavior.min_ttl
				&&
				state.packets_this_hop >= behavior.packets_per_hop - 1 ) )
		{
			/*
			 * If we get here, we're at the end of a run
			 * either with one or more responses from the target
			 * or because we've reached the ttl limit.
			 */
			state.current_hop = behavior.max_ttl + 1;

			/*
			 * if we're supposed to be continuous,
			 * reset the loop and start again
			 */
			if ( state.continuous_count != (int) NULL )
			{
				state.continuous_count--;
				if ( state.continuous_count == 0 )
					behavior.continuous = NO;
			}
			if ( behavior.continuous == YES )
			{
				state.packet_match = TP_PACKET_NO;
				state.target_response = NO;

				if ( behavior.hop_incr_unit == 1 )
					state.current_hop = behavior.min_ttl;
				else
					state.current_hop = behavior.max_ttl;

				/*
				 * setting packets_this_hop to -1 allows the
				 * reset if at the beginning of the loop to function
				 */
				state.packets_this_hop = -1;

				if ( behavior.continuous_accounting == YES )
					hop_audit();
			}
		}

		/*
		 * after a run pause for -W usecs
		 * (at least I think it usecs)
		 */
		if ( behavior.wait_between_packets != 0 )
			usleep ( behavior.wait_between_packets * 1000 );

/*
 * reset the ports as needed
 */
		if ( packet.src_port >= behavior.max_src_port && behavior.src_port_incr == 1 )
			packet.src_port = behavior.min_src_port - 1;
		else if ( packet.src_port <= behavior.min_src_port && behavior.src_port_incr == -1 )
			packet.src_port = behavior.max_src_port + 1;

		if ( packet.dst_port >= behavior.max_dst_port && behavior.dst_port_incr == 1)
			packet.dst_port = behavior.min_dst_port - 1;
		else if ( packet.dst_port <= behavior.min_dst_port && behavior.dst_port_incr == -1)
			packet.dst_port = behavior.max_dst_port + 1;

		if ( debug.loop == YES ) { printf ( "debug, looping back to the beginning\n" ); }

	} /* end of the main send/recv loop */

/*
 * yahoo, we're done
 * 
 * spit out the stats, sweep up after ourselves, and close up shop
 * last packet out please shut off the lights
 */
	behavior.report ( TP_OUT_FOOTER, NULL, 0 );
	/* hop_audit(); */

	tixe ( tixe_cleanup, 0 );	/* cleanup and exit, status 0 */

	exit ( 0 ); /* we should never get here */
}

