// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef MEASURER_INC
#define MEASURER_INC

#include "Object.h"

namespace Torch {

class Machine;
class DataSet;

/** Used to measure what you want during training/testing.
    Usually, trainers call measurers.
    The #DataSet# associated to the measurer allow us
    to know when the measurer should be called.
    (if the #DataSet# is the train dataset, the
    measurer will be called during the train phase...)

    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
class Measurer : public Object
{
  public:

    /** The measurer put here what it measures.
        As it could be everything, you should
        know what's inside before using it.
    */
    void *measure;

    /// The measurer save measures in this file.
    FILE *file;

    // -- internal --
    bool file_is_opened;

    /// The associated #DataSet#.
    DataSet *data;

    //-----

    /** Measurer with the associated #DataSet# #data_#,
        and put results in the file #file_#.
    */
    Measurer(DataSet *data_, FILE *file_);

    /** Open the file #filename# for writing measures.        
        #data_# is the associated #DataSet#.
    */
    Measurer(DataSet *data_, const char *filename);

    /** Measure something for the current example.
        (This example has been selected in #data#
         by the trainer)
    */
    virtual void measureEx();

    /** Measure something after the current iteration.
        (After the call of #measureEx()# for each
        example of #data#)
    */
    virtual void measureIter();

    /// Measure something at the end of the training/testing phase.
    virtual void measureEnd();

    //-----

    virtual void reset();
    virtual ~Measurer();
};


}

#endif
