// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef KERNEL_INC
#define KERNEL_INC

#include "general.h"
#include "DataSet.h"
#include "EuclideanDataSet.h"

namespace Torch {

/** Kernel class.
    Note that all kernels are based on the #dotProduct#
    of their associated #DataSet#.

    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
class Kernel : public Object
{
  public:
    DataSet *data;

    //-----

    ///
    Kernel(DataSet *data_);

    /** Kernel between the example #i# and #j# of the dataset.
        This function takes in account the #selected_examples#
        of the dataset.
    */
    virtual real eval(int i, int j) = 0;

    /** Kernel between the example #i# of the dataset and
        one example #y#. The structure of #y# \emph{must}
        be the same as those generated in #inputs# by
        the dataset #data#.
        Note that this function don't care about the #selected_examples#
        of the dataset: #i# is a \emph{real} index.
    */
    virtual real realEval(int i, List *y) = 0;

    /// Reset the kernel (Default, do nothing)
    virtual void reset();

    //-----

    virtual ~Kernel();
};

/** DotProduct

    @author Ronan Collobert (collober@iro.umontreal.ca)
 */
class DotKernel : public Kernel
{
  public:
    EuclideanDataSet *edata;

    ///
    DotKernel(EuclideanDataSet *edata_);

    virtual real eval(int i, int j);
    virtual real realEval(int i, List *y);
    virtual ~DotKernel();
};


/** Polynomial $k(x,y) = (s*x.y+r)^d$.
    
    Options:
    \begin{tabular}{lcll}
      "degree"  &  int  &   degree of the polynome & [2]\\
      "mul cst" &  real &   s                      & [1]\\
      "add cst" &  real &   r                      & [1]
    \end{tabular}

    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
class PolynomialKernel : public Kernel
{
  public:
    EuclideanDataSet *edata;
    int d;
    real s, r;

    ///
    PolynomialKernel(EuclideanDataSet *edata_);
    virtual real eval(int i, int j);
    virtual real realEval(int i, List *y);
    virtual ~PolynomialKernel();
};

/** Gaussian $k(x,y) = exp(-g * ||x-y||^2)$
    
    Options:
    \begin{tabular}{lcll}
      "gamma"  &  real  &   g & [0.01]
    \end{tabular}

    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
class GaussianKernel : public Kernel
{
  public:
    EuclideanDataSet *edata;
    real g;
    real *precalc;
    real *precalc_real;
    bool precalc_alloc;

    ///
    GaussianKernel(EuclideanDataSet *edata_);
    virtual void reset();
    virtual real eval(int i, int j);
    virtual real realEval(int i, List *y);
    virtual ~GaussianKernel();
};

/** Sigmoid $k(x,y) = tanh(s*x.y+r)$
    
    Options:
    \begin{tabular}{lcll}
      "mul cst" &  real &    s        &          [1]\\
      "add cst" &  real &    r        &          [1]
    \end{tabular}

    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
class SigmoidKernel : public Kernel
{
  public:
    EuclideanDataSet *edata;
    real s, r;

    ///
    SigmoidKernel(EuclideanDataSet *edata_);
    virtual real eval(int i, int j);
    virtual real realEval(int i, List *y);
    virtual ~SigmoidKernel();
};


}

#endif
