// Copyright (C) 2002 Johnny Mariethoz  (Johnny.Mariethoz@idiap.ch)
//                and Bison Ravi  (francois.belisle@idiap.ch)
//                and Samy Bengio (bengio@idiap.ch)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef SEQ_DATASET_INC
#define SEQ_DATASET_INC

#include "DataSet.h"
#include "normalize.h"

namespace Torch {
class Distribution;

/// This structure keeps a sequence example
struct SeqExample
{
    /// the number of frames of the sequence
    int n_frames;

    /// if there are targets, the number of targets (the length of the matrix)
    int n_seqtargets;

    /// if there are inputs, the matrix containing the inputs
    real** inputs;

    /// if there are observations, the matrix containing the observations
    real** observations;

    /// if there are targets, the matrix containing the targets
    real** seqtargets;

    /** alignment between observations and seqtargets, when available.
        alignment[i] gives the index of the inputs/observations
        corresponding to phoneme alignment_phoneme[i].
    */
    int *alignment;

    /// alignment phoneme index
    int *alignment_phoneme;

    /// the size of the alignment vector (needs not to coincide with seqtargets)
    int n_alignments;

    /** some examples have names. this is used to make a relation between
        alignment datafiles and observation datafiles
    */
    char* name;

    /// if only some frames of the example is selected
    int *selected_frames;

    /// how many real frames are there in the example
    int n_real_frames;

    /// which frame are we handling now
    int current_frame;

};

void setFrameExample(SeqExample* ex, int frame);

/** This class defines the standard framework of 
   Sequence Data processing
  
   DataSet with a vector of reals for inputs, vector of 
   reals for observations and vector of reals for targets
   
   Options: 
   
   "normalize inputs"          bool    normalize the inputs by mean/stdv       [ false ]
   "normalize observations"    bool    normalize the observations by mean/stdv [ false ]
   "normalize targets"         bool    normalize the targets by mean/stdv      [ false ]
  
	 p
  
   @author Bison Ravi  (francois.belisle@idiap.ch)
   @author Johnny Mariethoz  (Johnny.Mariethoz@idiap.ch)
   @author Samy Bengio (bengio@idiap.ch)
 */
class SeqDataSet : public DataSet
{
  public:
    
    /// the vector of examples of the dataset
    SeqExample* examples;

    /// names of data files
    char** file_names;

    /// the number of data files to read
    int n_file_names;

    /// do we normalize the inputs?
    bool norm_inputs;

    /// do we normalize the observations?
    bool norm_observations;

    /// if we normalize the inputs, this vector contains the mean of the inputs
    real* mean_in;

    /// if we normalize the inputs, this vector contains the mean of the observations
    real* mean_ob;

    /// if we normalize the inputs, this vector contains the stdev of the inputs
    real* stdv_in;

    /// if we normalize the inputs, this vector contains the stdev of the observations
    real* stdv_ob;

    /// the current frame we are working on
    int current_frame;

    /// the number of frames of the current example
    int n_frames;

    /// the number of observations of the examples
    int n_observations;
    
    /// the total number of frames in the dataset
    int tot_n_frames;

    /** some dataset keeps an information of the kind of the data.
        this is used in Htk data files for instance
    */
    char* kind;

    /** if there are targets, the length of the seqtargets matrix.
        Note that its width is equal to n_targets.
    */
    int n_seqtargets;

    /// the targets of the current example
    real** seqtargets;

    ////
    SeqDataSet();

    virtual ~SeqDataSet();
    
    /** each Subclass of #SeqDataSet# is forced 
       to allocate/free its own memory. This is 
       forced upon them for historical reasons
       eg IOTorch allocates/frees its own memory
       which is used in #StaticMatSeqDataSet#
     */
    virtual void freeMemory() = 0; 

    // declaration of inherited virtual functions
    virtual void setRealExample( int );
    virtual void init();

    /** Remove the frames considering as unlinkely and return the number of
		    removed frames*/
    int removeUnlikelyFrames(Distribution* likely_distr, Distribution* unlikely_distr,int obs_offset = 0,int range=5);
    int xwavesRemoveUnlikelyFrames(Distribution* likely_distr, Distribution* unlikely_distr,bool* frame_to_keep,int obs_offset = 0, int range=5);
		//int removeUnlikelyFrames(Distribution* likely_distr, real thrd);
    
    /** Remove the column of data unused */
    void removeUnusedData(bool* mask);

    // The following method should be called *after* <init()>

    /** Normalize the #DataSet# using the normalization values
        of another #DataSet#. If the other #DataSet# isn't 
       normalized, nothing happens...
     */
    void normalizeUsingDataSet( SeqDataSet* );
    /// normalize the database
    void normalize();

    /// set the current frame of the current example
    void setFrame( int );
    /** compute the total number of frames in the database.
        keep this value in tot_n_frames
    */
    void totNFrames(); 
    void selectBootstrap();
    void unsetAllSelectedFrames();
    void linearSegmentation(int ith_segment,int n_segment);

    /** transform the database such that there is one frame per example.
        by default, each file corresponds to one sequence.
    */
    void toOneFramePerExample();

    /// read seqtargets from a file
    virtual void readTargets(char* file);

    /// read alignments from a file
    virtual void readAlignments(char* file, bool needs_all_examples = false);

    /// for the current example, consider only the #selected_frames#
    void setSelectedFrames( int *selected_frames, int n_selected_frames );

    /// for the current example, consider all the frames
    void unsetSelectedFrames();
    
    /// print the database, for debugging purposes only
    void display();

    /// Load the normalization parameters
    virtual void loadFILE(FILE *file);

    /// Save the normalization parameters
    virtual void saveFILE(FILE *file);

};


}

#endif
