/*
 *  ISEM - Instructional Sparc EMulator and tkisem
 *  Copyright (C) 1993, 1994, 1995, 1996
 *	Department of Computer Science,
 *      The University of New Mexico
 *
 *  Please send questions, comments, and bug reports to: isem@cs.unm.edu
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
//
// MMU.h -- definition of the  MMU  class
//
//   The MMU supports 8 address spaces:
//
//	 0 - User Text Page Table
//	 1 - Supervisor Text Page Table
//
//	 2 - User Data Page Table
//	 3 - Supervisor Data Page Table
//
//
//	 8 - User Text
//	 9 - Supervisor Text
//
//	10 - User Data
//	11 - Supervisor Data
//
//
//   The current implementation does not keep track of whether a page
//   is data, text, read-only, read-write, etc.  This probably should
//   be added at sometime in the future
//

#if __GNUC__
#define UNUSED __attribute__ ((unused)) 
#else
#define UNUSED
#endif

static char rcsid[] UNUSED = "$Id: MMU.cpp 1.1 Fri, 25 Oct 1996 18:04:04 -0600 maccabe $";

#include "Assert.h"
#include "sizedefs.h"
#include "Config.h"

#include "sys_bus.h"
#include "MMU.h"

//
// public Constructors / Destructors
//
MMU::MMU(SystemBus& sb)
    : sbus(sb)
{
    // calculate number of last table entry
    //
    lastPage = (1 << (Config::ADDR_BITS - Config::ISEM_PAGE_BITS)) - 1;

    // allocate both a user and supervisor page table
    //
    userTextPageTable = new UInt32[ lastPage + 1 ];
    Assert(userTextPageTable, "new failed");

    unsigned i;
    for ( i = 0; i < lastPage; i++)
	userTextPageTable[i] = ~0;

    superTextPageTable = new UInt32[ lastPage + 1 ];
    Assert(superTextPageTable, "new failed");

    for (i = 0; i < lastPage; i++)
	superTextPageTable[i] = ~0;

    userDataPageTable = new UInt32[ lastPage + 1 ];
    Assert(userDataPageTable, "new failed");

    for (i = 0; i < lastPage; i++)
	userDataPageTable[i] = ~0;

    superDataPageTable = new UInt32[ lastPage + 1] ;
    Assert(superDataPageTable, "new failed");

    for (i = 0; i < lastPage; i++)
	superDataPageTable[i] = ~0;
}



//
// public Member functions
//

UInt32 MMU::read(int as, UInt32 address) {
    UInt32	page, offset, a;
    UInt32*	pt;

    //  if 0, 1, 2, 3
    //	    return page table entry
    //  else
    //	    determine correct page table
    //
    switch (as) {
    case 0:
	return userTextPageTable[ address >> 2 ];

    case 1:
	return superTextPageTable[ address >> 2 ];

    case 2:
	return userDataPageTable[ address >> 2 ];

    case 3:
	return superDataPageTable[ address >> 2 ];

    case 8:
	pt = userTextPageTable;
	break;

    case 9:
	pt = superTextPageTable;
	break;

    case 10:
	pt = userDataPageTable;
	break;

    case 11:
	pt = superDataPageTable;
	break;


    default:
	// This should actually be data_access_MMU_miss, but as of
	//   now, we don't have any mechanism to do that

	sbus.bp_memory_exception(1);

	return 0;
    }


    // make sure page table entry is valid
    //
    page = address >> Config::ISEM_PAGE_BITS;

    if ( page > lastPage ) {
	// This should actually be data_access_MMU_miss, but as of
	//   now, we don't have any mechanism to do that

	sbus.bp_memory_exception(1);
	return 0;
    }

    if (pt[page] == ~0U) {
	// This should actually be data_access_MMU_miss, but as of
	//   now, we don't have any mechanism to do that

	sbus.bp_memory_exception(1);
	return 0;
    }

    offset = address & (Config::ISEM_PAGE_SIZE - 1);

    // map address via correct page table and return value
    //
    a = (pt[page] << Config::ISEM_PAGE_BITS) | offset;
    return sbus.read(a);
}

void MMU::write(int as, UInt32 address, int bitMask, UInt32 value) {
    UInt32	page, offset, a;
    UInt32*	pt;

    if ((as < 4) && (value > lastPage)) {

	// This should actually be data_access_MMU_miss, but as of
	//   now, we don't have any mechanism to do that

	sbus.bp_memory_exception(1);
	return;
    }

    switch (as) {
    case 0:
	// write page table entry - ignore bitMask
	//
	userTextPageTable[ address >> 2 ] = value;
	return;

    case 1:
	// write page table entry - ignore bitMask
	//
	superTextPageTable[ address >> 2 ] = value;
	return;

    case 2:
	// write page table entry - ignore bitMask
	//
	userDataPageTable[ address >> 2 ] = value;
	return;

    case 3:
	// write page table entry - ignore bitMask
	//
	superDataPageTable[ address >> 2 ] = value;
	return;

    case 8:
	pt = userTextPageTable;
	break;

    case 9:
	pt = superTextPageTable;
	break;

    case 10:
	pt = userDataPageTable;
	break;

    case 11:
	pt = superDataPageTable;
	break;

    default:
	// This should actually be data_access_MMU_miss, but as of
	//   now, we don't have any mechanism to do that

	sbus.bp_memory_exception(1);
	return;
    }


    // make sure page table entry is valid
    //
    page = address >> Config::ISEM_PAGE_BITS;

    if ( page > lastPage ) {
	// This should actually be data_access_MMU_miss, but as of
	//   now, we don't have any mechanism to do that

	sbus.bp_memory_exception(1);
	return;
    }

    if (pt[page] == ~0U) {
	// This should actually be data_access_MMU_miss, but as of
	//   now, we don't have any mechanism to do that

	sbus.bp_memory_exception(1);
	return;
    }

    offset = address & (Config::ISEM_PAGE_SIZE - 1);

    // map address via correct page table and return value
    //
    a = (pt[page] << Config::ISEM_PAGE_BITS) | offset;

    sbus.write(a, bitMask, value);
    return;
}
