/*
 *  Copyright (C) 2000 by Marco G"otze.
 *
 *  This code is part of the ThoughtTracker source package, which is
 *  distributed under the terms of the GNU GPL2.
 */

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <iostream>

#include <gtk--/box.h>
#include <gtk--/buttonbox.h>
#include <gtk--/label.h>
#include <gtk--/main.h>
#include <gtk--/pixmap.h>
#include <gtk--/table.h>
#include <gtk--/window.h>

#include "thoughttracker.h"
#include "querydlg.h"
#include "xpm.h"

#ifdef ENABLE_DEBUG
#undef DMSG 
#define DMSG cerr << "TTQueryDialog::" << __FUNCTION__ << "(): "
#endif  /* ENABLE_DEBUG */

using SigC::bind;
using SigC::slot;

using namespace Gtk;

#define TTQD_BTN_MAX 1<<4-1 // maximal value for a TTQDButtons variable

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
TTQueryDialog::TTQueryDialog(Window *transient, char **xpm,
  const string &title, const string &text, int buttons,
  TTQDButtons default_button)
  : Window(GTK_WINDOW_DIALOG), pressed(TTQD_BTN_NONE)
{
  // configure self
  set_title(title);
  set_wmclass(WM_CLASS_NAME, "dialog");
  set_transient_for(*transient);
  realize();
  GdkWindowHints hints = GDK_HINT_BASE_SIZE;
  GdkGeometry geometry;
  geometry.min_width = 150;
  geometry.min_height = 100;
  set_geometry_hints(*this, &geometry, hints);
  set_position(GTK_WIN_POS_MOUSE);
  set_policy(false, false, true);
  set_border_width(7);

  // configure table for setting up all the other widgets
  Table *table = manage(new Table(2, 2));
  table->set_col_spacing(0, 5);
  table->set_row_spacing(0, 10);
  add(*table);

  // configure pixmap and text (label)
  Pixmap *icon = manage(new Pixmap(xpm ? xpm : dlg_info_xpm));
  table->attach(*icon, 0, 1, 0, 1, 0, GTK_EXPAND | GTK_FILL, 5, 5);
  Label *label = manage(new Label(text));
  label->set_line_wrap(true);
  table->attach(*label, 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, 0, 3, 0);

  // configure button box and buttons
  ButtonBox *bbox = manage(new HButtonBox);
  table->attach(*bbox, 0, 2, 1, 2, GTK_EXPAND | GTK_FILL, 0, 0, 0);
  // check button parameters for sanity
  if (buttons == TTQD_BTN_NONE) buttons = TTQD_BTN_OK;  // that's the minimum
  if (default_button == TTQD_BTN_NONE)  // make sure there's EXACTLY ONE
    for (int i = 1; i <= TTQD_BTN_MAX; i <<= 1)
      if (i & buttons) {
        default_button = TTQDButtons(i);
        break;
      }
  // create buttons
  for (int i = 1; i <= TTQD_BTN_MAX; i <<= 1) {
    if (i & buttons) {
      string s;
      switch (i) {
        case TTQD_BTN_OK:     s = _("OK"); break;
        case TTQD_BTN_YES:    s = _("Yes"); break;
        case TTQD_BTN_NO:     s = _("No"); break;
        case TTQD_BTN_CANCEL: s = _("Cancel"); break;
      }
      Button *btn = manage(new Button(s));
      btn->clicked.connect(bind(slot(this, &TTQueryDialog::button_press),
        (TTQDButtons) i));
	  bbox->pack_start(*btn, false, false, false);
      btn->set_flags(GTK_CAN_DEFAULT);
      if (i == default_button) def_btn = btn;
    }
  }
  def_btn->grab_default();
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
TTQDButtons
TTQueryDialog::run()
{
  show_all();
  set_modal(true);

  while (!pressed) Main::iteration();

#ifdef ENABLE_DEBUG
  char *s;
  switch (pressed) {
    case TTQD_BTN_OK:     s = "TTQD_BTN_OK"; break;
    case TTQD_BTN_YES:    s = "TTQD_BTN_YES"; break;
    case TTQD_BTN_NO:     s = "TTQD_BTN_NO"; break;
    case TTQD_BTN_CANCEL: s = "TTQD_BTN_CANCEL"; break;
    default:              s = "unknown code";
  }
  DMSG << "returning " << pressed << " (" << s << ')' << endl;
#endif  // ENABLE_DEBUG
  return pressed;
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
void
message_popup(Gtk::Window *transient, const string &title, const string &text)
{
  TTQueryDialog dlg(transient, dlg_info_xpm, title, text);
  dlg.run();
}

//.............................................................................
void
error_popup(Gtk::Window *transient, const string &text)
{
  TTQueryDialog dlg(transient, dlg_error_xpm, _("Error"), text);
  dlg.run();
}

//.............................................................................
void
internal_error_popup(Gtk::Window *transient, TTQDInternalError code)
{
  string text;
  switch (code) {
    case TTQD_ERROR_UNIMPLEMENTED:
      text = _("This feature has not been implemented yet.");
      break;
    case TTQD_ERROR_DB_READ:
      text = _("There has been an error reading from the database.");
      break;
    case TTQD_ERROR_DB_WRITE:
      text = _("There has been an error writing to the database.");
      break;
    case TTQD_ERROR_SEARCH:
      text = _("Search failed internally.");
      break;
    case TTQD_ERROR_IMPOSSIBLE_SITUATION:
      text = _("The program has reached a state that's actually impossible, "
        "which makes it seem likely that there's some bug in the user "
        "interface or some conditional loophole.  Please report this as a "
        "bug, and try to describe what you've done that caused this.");
      break;
  }
  TTQueryDialog dlg(transient, dlg_internal_error_xpm, _("Internal error..."),
    text);
  dlg.run();
}

//.............................................................................
TTQDButtons
query_popup(Gtk::Window *transient, const string &title,
  const string &text, int buttons, TTQDButtons default_button)
{
  TTQueryDialog dlg(transient, dlg_info_xpm, title, text, buttons,
    default_button);
  return dlg.run();
}

